/* tie sndlib into guile (Scheme) */

#if defined(HAVE_CONFIG_H)
  #include "config.h"
#endif

#if HAVE_GUILE

#include <stddef.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "sndlib.h"
#include "sndlib-strings.h"
#include "vct.h"
#include <guile/gh.h>

#define RTNINT(a) return(gh_int2scm(a))
#define RTNSTR(a) return(gh_str02scm(a))

#define ERRS1(a,b) SCM_ASSERT((gh_string_p(a)),a,SCM_ARG1,b)
#define ERRN1(a,b) SCM_ASSERT((gh_number_p(a)),a,SCM_ARG1,b)

#ifdef SUN
  static int bool_or_arg_p(SCM a) {return((gh_number_p(a)) || (a == SCM_BOOL_F) || (a == SCM_BOOL_T) || (a == SCM_UNDEFINED) || (a == SCM_UNSPECIFIED));}
  #define ERRB1(a,b) SCM_ASSERT((bool_or_arg_p(a)),a,SCM_ARG1,b)
  #define ERRB2(a,b) SCM_ASSERT((bool_or_arg_p(a)),a,SCM_ARG2,b)
  #define ERRB3(a,b) SCM_ASSERT((bool_or_arg_p(a)),a,SCM_ARG3,b)
#endif

static char *full_filename(SCM file)
{
  char *urn,*filename;
  urn = gh_scm2newstr(file,NULL);
  filename = mus_complete_filename(urn); /* static -- don't free this */
  FREE(urn);
  return(filename);
}


static SCM g_sound_samples(SCM filename) {ERRS1(filename,S_sound_samples); RTNINT(sound_samples(full_filename(filename)));}
static SCM g_sound_frames(SCM filename) {ERRS1(filename,S_sound_frames); RTNINT(sound_frames(full_filename(filename)));}
static SCM g_sound_datum_size(SCM filename) {ERRS1(filename,S_sound_datum_size); RTNINT(sound_datum_size(full_filename(filename)));}
static SCM g_sound_data_location(SCM filename) {ERRS1(filename,S_sound_data_location); RTNINT(sound_data_location(full_filename(filename)));}
static SCM g_sound_chans(SCM filename) {ERRS1(filename,S_sound_chans); RTNINT(sound_chans(full_filename(filename)));}
static SCM g_sound_srate(SCM filename) {ERRS1(filename,S_sound_srate); RTNINT(sound_srate(full_filename(filename)));}
static SCM g_sound_header_type(SCM filename) {ERRS1(filename,S_sound_header_type); RTNINT(sound_header_type(full_filename(filename)));}
static SCM g_sound_data_format(SCM filename) {ERRS1(filename,S_sound_data_format); RTNINT(sound_data_format(full_filename(filename)));}
static SCM g_sound_length(SCM filename) {ERRS1(filename,S_sound_length); RTNINT(sound_length(full_filename(filename)));}
static SCM g_sound_type_specifier(SCM filename) {ERRS1(filename,S_sound_type_specifier); RTNINT(sound_type_specifier(full_filename(filename)));}
static SCM g_sound_comment(SCM filename) {ERRS1(filename,S_sound_comment); RTNSTR(sound_comment(full_filename(filename)));}
static SCM g_sound_type_name(SCM type) {ERRN1(type,S_sound_type_name); RTNSTR(sound_type_name(gh_scm2int(type)));}
static SCM g_sound_format_name(SCM format) {ERRN1(format,S_sound_format_name); RTNSTR(sound_format_name(gh_scm2int(format)));}
static SCM g_sound_bytes_per_sample(SCM format) {ERRN1(format,S_sound_bytes_per_sample); RTNINT(sound_bytes_per_sample(gh_scm2int(format)));}
static SCM g_audio_error(void) {RTNINT(audio_error());}
static SCM g_audio_error_name(SCM err) {ERRN1(err,S_audio_error_name); RTNSTR(audio_error_name(gh_scm2int(err)));}
static SCM g_report_audio_state(void) {RTNSTR(report_audio_state());}
static SCM g_sound_duration(SCM filename) {ERRS1(filename,S_sound_duration); return(gh_double2scm(sound_duration(full_filename(filename))));}

static SCM g_audio_outputs(SCM speakers, SCM headphones, SCM line_out)
{
#ifdef SUN
  SCM_ASSERT(gh_number_p(speakers),speakers,SCM_ARG1,S_audio_outputs);
  SCM_ASSERT(gh_number_p(headphones),headphones,SCM_ARG2,S_audio_outputs);
  SCM_ASSERT(gh_number_p(line_out),line_out,SCM_ARG3,S_audio_outputs);
  sun_outputs(gh_scm2int(speakers),gh_scm2int(headphones),gh_scm2int(line_out));
#endif
  return(SCM_BOOL_F);
}

static SCM g_sound_max_amp(SCM file)
{
  int chans,rtn,i;
  int *vals;
  char *filename;
  SCM vect = SCM_BOOL_F;
  ERRS1(file,S_sound_max_amp);
  filename = full_filename(file);
  chans = sound_chans(filename);
  if (chans > 0)
    {
      vals = (int *)CALLOC(chans*2,sizeof(int));
      rtn = sound_max_amp(filename,vals);
      if (rtn > 0)
	{
	  vect = gh_make_vector(chans*2,gh_int2scm(0));
	  for (i=0;i<chans*2;i+=2)
	    {
	      gh_vector_set_x(vect,gh_int2scm(i),gh_int2scm(vals[i]));
	      gh_vector_set_x(vect,gh_int2scm(i+1),gh_double2scm((double)(vals[i+1] * SNDLIB_SNDFLT)));
	    }
	}
    }
  return(vect);
}

/* to support the actual sound file/audio port stuff, we need an "smob" for the int** arrays */

typedef struct {
  int length,chans;
  int **data;
} sound_data;

static int sound_data_tag = 0;

static SCM mark_sound_data(SCM obj)
{
  SCM_SETGC8MARK(obj);
  return(SCM_BOOL_F);
}

static int sound_data_p(SCM obj)
{
  return((SCM_NIMP(obj)) && (SCM_TYP16(obj) == (SCM)sound_data_tag));
}

static SCM g_sound_data_p(SCM obj) 
{
  return((sound_data_p(obj)) ? SCM_BOOL_T : SCM_BOOL_F);
}

static int **get_sound_data(SCM arg)
{
  sound_data *sd;
  if (sound_data_p(arg))
    {
      sd = (sound_data *)(gh_cdr(arg));
      return(sd->data);
    }
  return(NULL);
}

static scm_sizet free_sound_data(SCM obj)
{
  int i;
  sound_data *v = (sound_data *)gh_cdr(obj);
  if (v->data) 
    {
      for (i=0;i<v->chans;i++) if (v->data[i]) FREE(v->data[i]);
      FREE(v->data);
    }
  v->data = NULL;
  FREE(v);
  return(0);
}

static int print_sound_data(SCM obj, SCM port, scm_print_state *pstate)
{
  char *buf;
  sound_data *v = (sound_data *)gh_cdr(obj);
  buf = (char *)CALLOC(64,sizeof(char));
  sprintf(buf,"#<sound_data: %d chans, %d frames>",v->chans,v->length);
  scm_puts(buf,port);
  FREE(buf);
  return(1);
}

static SCM equalp_sound_data(SCM obj1, SCM obj2)
{
  sound_data *v1,*v2;
  v1 = (sound_data *)gh_cdr(obj1);
  v2 = (sound_data *)gh_cdr(obj2);
  if (v1 == v2) return(SCM_BOOL_T);
  return(SCM_BOOL_F);
}

static SCM sound_data_length(SCM obj)
{
  sound_data *v = (sound_data *)gh_cdr(obj);
  return(gh_int2scm(v->length));
}

static SCM sound_data_chans(SCM obj)
{
  sound_data *v = (sound_data *)gh_cdr(obj);
  return(gh_int2scm(v->chans));
}

static SCM make_sound_data(int chans, int frames)
{
  SCM ans;
  int i;
  sound_data *new_sound_data;
  new_sound_data = (sound_data *)CALLOC(1,sizeof(sound_data));
  new_sound_data->length = frames;
  new_sound_data->chans = chans;
  new_sound_data->data = (int **)CALLOC(chans,sizeof(int *));
  for (i=0;i<chans;i++) new_sound_data->data[i] = (int *)CALLOC(frames,sizeof(int));
  SCM_NEWCELL(ans);
  SCM_SETCAR(ans,sound_data_tag);
  SCM_SETCDR(ans,(SCM)new_sound_data);
  return(ans);
}

#if (!HAVE_MAKE_SMOB_TYPE)
static scm_smobfuns sound_data_smobfuns = {
  &mark_sound_data,
  &free_sound_data,
  &print_sound_data,
  &equalp_sound_data};
#endif

static SCM g_make_sound_data(SCM chans, SCM frames)
{
  ERRN1(chans,S_make_sound_data);
  SCM_ASSERT(gh_number_p(frames),frames,SCM_ARG1,S_make_sound_data);
  return(make_sound_data(gh_scm2int(chans),gh_scm2int(frames)));
}

static SCM sound_data_ref(SCM obj, SCM chan, SCM frame)
{
  sound_data *v = (sound_data *)gh_cdr(obj);
  int loc,chn;
  SCM_ASSERT(sound_data_p(obj),obj,SCM_ARG1,S_sound_data_ref);
  SCM_ASSERT(gh_number_p(chan),chan,SCM_ARG2,S_sound_data_ref);
  SCM_ASSERT(gh_number_p(frame),frame,SCM_ARG3,S_sound_data_ref);
  if (v)
    {
      chn = gh_scm2int(chan);
      if ((chn >= 0) && (chn < v->chans))
	{
	  loc = gh_scm2int(frame);
	  if ((loc >= 0) && (loc < v->length))
	    return(gh_double2scm((double)(SNDLIB_SNDFLT * v->data[chn][loc])));
	  else scm_misc_error(S_sound_data_ref,"invalid frame",SCM_LIST2(obj,frame));
	}
      else scm_misc_error(S_sound_data_ref,"invalid channel",SCM_LIST2(obj,chan));
    }
  else scm_misc_error(S_sound_data_ref,"nil sound-data?",SCM_EOL);
  return(gh_double2scm(0.0));
}

static SCM sound_data_set(SCM obj, SCM chan, SCM frame, SCM val)
{
  sound_data *v = (sound_data *)gh_cdr(obj);
  int loc,chn;
  if (v)
    {
      chn = gh_scm2int(chan);
      if ((chn >= 0) && (chn < v->chans))
	{
	  loc = gh_scm2int(frame);
	  if ((loc >= 0) && (loc < v->length))
	    v->data[chn][loc] = (int)(SNDLIB_SNDFIX * gh_scm2double(val));
	  else scm_misc_error(S_sound_data_setB,"invalid frame",SCM_LIST3(obj,chan,frame));
	}
      else scm_misc_error(S_sound_data_setB,"invalid channel",SCM_LIST3(obj,chan,frame));
    }
  else scm_misc_error(S_sound_data_setB,"nil sound-data?",SCM_EOL);
  return(val);
}

static SCM sound_data2vct(SCM sdobj, SCM chan, SCM vobj)
{
  vct *v;
  sound_data *sd;
  int len,i,chn;
  SCM_ASSERT(sound_data_p(sdobj),sdobj,SCM_ARG1,S_sound_data2vct);
  SCM_ASSERT(gh_number_p(chan),chan,SCM_ARG2,S_sound_data2vct);
  SCM_ASSERT(vct_p(vobj),vobj,SCM_ARG3,S_sound_data2vct);
  v = (vct *)get_vct(vobj);
  sd = (sound_data *)gh_cdr(sdobj);
  chn = gh_scm2int(chan);
  if (chn < sd->chans)
    {
      if (sd->length < v->length) len = sd->length; else len = v->length;
      for (i=0;i<len;i++) v->data[i] = (float)(sd->data[chan][i] * SNDLIB_SNDFLT);
    }
  else scm_misc_error(S_sound_data2vct,"invalid channel",SCM_LIST3(sdobj,chan,vobj));
  return(vobj);
}

static SCM vct2sound_data(SCM vobj, SCM sdobj, SCM chan)
{
  vct *v;
  sound_data *sd;
  int len,i,chn;
  SCM_ASSERT(vct_p(vobj),vobj,SCM_ARG1,S_vct2sound_data);
  SCM_ASSERT(sound_data_p(sdobj),sdobj,SCM_ARG2,S_vct2sound_data);
  SCM_ASSERT(gh_number_p(chan),chan,SCM_ARG3,S_vct2sound_data);
  v = (vct *)get_vct(vobj);
  sd = (sound_data *)gh_cdr(sdobj);
  chn = gh_scm2int(chan);
  if (chn < sd->chans)
    {
      if (sd->length < v->length) len = sd->length; else len = v->length;
      for (i=0;i<len;i++) sd->data[chan][i] = (int)(v->data[i] * SNDLIB_SNDFIX);
    }
  else scm_misc_error(S_vct2sound_data,"invalid channel",SCM_LIST3(vobj,chan,sdobj));
  return(vobj);
}


static SCM g_open_sound_input(SCM file)
{
  int fd;
  ERRS1(file,S_open_sound_input);
  fd = open_sound_input(full_filename(file));
  return(gh_int2scm(fd));
}

static SCM g_open_sound_output(SCM file, SCM srate, SCM chans, SCM data_format, SCM header_type, SCM comment)
{
  int fd;
  char *com = NULL;
  SCM_ASSERT(gh_string_p(file),file,SCM_ARG1,S_open_sound_output);
  SCM_ASSERT(gh_number_p(srate),srate,SCM_ARG2,S_open_sound_output);
  SCM_ASSERT(gh_number_p(chans),chans,SCM_ARG3,S_open_sound_output);
  SCM_ASSERT(gh_number_p(data_format),data_format,SCM_ARG4,S_open_sound_output);
  SCM_ASSERT(gh_number_p(header_type),header_type,SCM_ARG5,S_open_sound_output);
  SCM_ASSERT((gh_string_p(comment) || (comment == SCM_UNSPECIFIED)),comment,SCM_ARG6,S_open_sound_output);
  if (gh_string_p(comment)) com = gh_scm2newstr(comment,NULL);
  fd = open_sound_output(full_filename(file),gh_scm2int(srate),gh_scm2int(chans),gh_scm2int(data_format),gh_scm2int(header_type),com);
  if (com) FREE(com);
  return(gh_int2scm(fd));
}

static SCM g_reopen_sound_output(SCM file, SCM chans, SCM data_format, SCM header_type, SCM data_loc)
{
  int fd;
  SCM_ASSERT(gh_string_p(file),file,SCM_ARG1,S_reopen_sound_output);
  SCM_ASSERT(gh_number_p(chans),chans,SCM_ARG2,S_reopen_sound_output);
  SCM_ASSERT(gh_number_p(data_format),data_format,SCM_ARG3,S_reopen_sound_output);
  SCM_ASSERT(gh_number_p(header_type),header_type,SCM_ARG4,S_reopen_sound_output);
  SCM_ASSERT(gh_number_p(data_loc),data_loc,SCM_ARG5,S_reopen_sound_output);
  fd = reopen_sound_output(full_filename(file),gh_scm2int(chans),gh_scm2int(data_format),gh_scm2int(header_type),gh_scm2int(data_loc));
  return(gh_int2scm(fd));
}

static SCM g_close_sound_input(SCM fd)
{
  SCM_ASSERT(gh_number_p(fd),fd,SCM_ARG1,S_close_sound_input);
  return(gh_int2scm(close_sound_input(gh_scm2int(fd))));
}

static SCM g_close_sound_output(SCM fd, SCM bytes)
{
  SCM_ASSERT(gh_number_p(fd),fd,SCM_ARG1,S_close_sound_output);
  SCM_ASSERT(gh_number_p(bytes),bytes,SCM_ARG2,S_close_sound_output);
  return(gh_int2scm(close_sound_output(gh_scm2int(fd),gh_scm2int(bytes))));
}

static SCM g_read_sound(SCM fd, SCM beg, SCM end, SCM chans, SCM sv)
{
  SCM_ASSERT(gh_number_p(fd),fd,SCM_ARG1,S_read_sound);
  SCM_ASSERT(gh_number_p(beg),beg,SCM_ARG2,S_read_sound);
  SCM_ASSERT(gh_number_p(end),end,SCM_ARG3,S_read_sound);
  SCM_ASSERT(gh_number_p(chans),chans,SCM_ARG4,S_read_sound);
  SCM_ASSERT(sound_data_p(sv),sv,SCM_ARG5,S_read_sound);
  return(gh_int2scm(read_sound(gh_scm2int(fd),gh_scm2int(beg),gh_scm2int(end),gh_scm2int(chans),get_sound_data(sv))));
}

static SCM g_write_sound(SCM fd, SCM beg, SCM end, SCM chans, SCM sv)
{
  SCM_ASSERT(gh_number_p(fd),fd,SCM_ARG1,S_write_sound);
  SCM_ASSERT(gh_number_p(beg),beg,SCM_ARG2,S_write_sound);
  SCM_ASSERT(gh_number_p(end),end,SCM_ARG3,S_write_sound);
  SCM_ASSERT(gh_number_p(chans),chans,SCM_ARG4,S_write_sound);
  SCM_ASSERT(sound_data_p(sv),sv,SCM_ARG5,S_write_sound);
  return(gh_int2scm(write_sound(gh_scm2int(fd),gh_scm2int(beg),gh_scm2int(end),gh_scm2int(chans),get_sound_data(sv))));
}

static SCM g_seek_sound(SCM fd, SCM offset, SCM origin)
{
  SCM_ASSERT(gh_number_p(fd),fd,SCM_ARG1,S_seek_sound);
  SCM_ASSERT(gh_number_p(offset),offset,SCM_ARG2,S_seek_sound);
  SCM_ASSERT(gh_number_p(origin),origin,SCM_ARG3,S_seek_sound);
  return(gh_int2scm(seek_sound(gh_scm2int(fd),gh_scm2int(offset),gh_scm2int(origin))));
}

static SCM g_seek_sound_frame(SCM fd, SCM offset)
{
  SCM_ASSERT(gh_number_p(fd),fd,SCM_ARG1,S_seek_sound_frame);
  SCM_ASSERT(gh_number_p(offset),offset,SCM_ARG2,S_seek_sound_frame);
  return(gh_int2scm(seek_sound_frame(gh_scm2int(fd),gh_scm2int(offset))));
}

static SCM g_open_audio_output(SCM dev, SCM srate, SCM chans, SCM format, SCM size)
{
  SCM_ASSERT(gh_number_p(dev),dev,SCM_ARG1,S_open_audio_output);
  SCM_ASSERT(gh_number_p(srate),srate,SCM_ARG2,S_open_audio_output);
  SCM_ASSERT(gh_number_p(chans),chans,SCM_ARG3,S_open_audio_output);
  SCM_ASSERT(gh_number_p(format),format,SCM_ARG4,S_open_audio_output);
  SCM_ASSERT(gh_number_p(size),size,SCM_ARG5,S_open_audio_output);
  return(gh_int2scm(open_audio_output(gh_scm2int(dev),gh_scm2int(srate),gh_scm2int(chans),gh_scm2int(format),gh_scm2int(size))));
}

static SCM g_open_audio_input(SCM dev, SCM srate, SCM chans, SCM format, SCM size)
{
  SCM_ASSERT(gh_number_p(dev),dev,SCM_ARG1,S_open_audio_input);
  SCM_ASSERT(gh_number_p(srate),srate,SCM_ARG2,S_open_audio_input);
  SCM_ASSERT(gh_number_p(chans),chans,SCM_ARG3,S_open_audio_input);
  SCM_ASSERT(gh_number_p(format),format,SCM_ARG4,S_open_audio_input);
  SCM_ASSERT(gh_number_p(size),size,SCM_ARG5,S_open_audio_input);
  return(gh_int2scm(open_audio_input(gh_scm2int(dev),gh_scm2int(srate),gh_scm2int(chans),gh_scm2int(format),gh_scm2int(size))));
}

static SCM g_close_audio(SCM line)
{
  SCM_ASSERT(gh_number_p(line),line,SCM_ARG1,S_close_audio);
  return(gh_int2scm(close_audio(gh_scm2int(line))));
}

static SCM g_save_audio_state (void) {save_audio_state(); return(SCM_BOOL_F);}
static SCM g_restore_audio_state (void) {restore_audio_state(); return(SCM_BOOL_F);}
static SCM g_audio_systems (void) {return(gh_int2scm(audio_systems()));}


/* these take a sndlib int** buffer (sound_data) and handle the conversion to the interleaved char* internally */
/* so, they take "frames", not "bytes", and a sound_data object, not char* etc */

static SCM g_write_audio(SCM line, SCM sdata, SCM frames)
{
  /* assuming 16-bit output here for now */
  short *obuf;
  int **bufs;
  sound_data *sd;
  int i,j,k,outbytes,val,frms;
  SCM_ASSERT(gh_number_p(line),line,SCM_ARG1,S_write_audio);
  SCM_ASSERT(sound_data_p(sdata),sdata,SCM_ARG2,S_write_audio);
  SCM_ASSERT(gh_number_p(frames),frames,SCM_ARG3,S_write_audio);
  sd = (sound_data *)gh_cdr(sdata);
  bufs = sd->data;
  frms = gh_scm2int(frames);
  outbytes = frms * sd->chans * 2;
  obuf = (short *)CALLOC(frms * sd->chans,sizeof(short));
  if (sd->chans == 1)
    {
      for (k=0;k<frms;k++) obuf[k] = bufs[0][k];
    }
  else
    {
      for (k=0,j=0;k<frms;k++,j+=sd->chans)
	{
	  for (i=0;i<sd->chans;i++) obuf[j+i] = bufs[i][k];
	}
    }
  val = write_audio(gh_scm2int(line),(char *)obuf,outbytes);
  FREE(obuf);
  return(gh_int2scm(val));
}

static SCM g_read_audio(SCM line, SCM sdata, SCM frames)
{
  short *inbuf;
  sound_data *sd;
  int val,inbytes,i,j,k,frms;
  int **bufs;
  SCM_ASSERT(gh_number_p(line),line,SCM_ARG1,S_read_audio);
  SCM_ASSERT(sound_data_p(sdata),sdata,SCM_ARG2,S_read_audio);
  SCM_ASSERT(gh_number_p(frames),frames,SCM_ARG3,S_read_audio);
  sd = (sound_data *)gh_cdr(sdata);
  bufs = sd->data;
  frms = gh_scm2int(frames);
  inbytes = frms * sd->chans * 2;
  inbuf = (short *)CALLOC(frms * sd->chans,sizeof(short));
  val = read_audio(gh_scm2int(line),(char *)inbuf,inbytes);
  if (sd->chans == 1)
    {
      for (k=0;k<frms;k++) bufs[0][k] = inbuf[k];
    }
  else
    {
      for (k=0,j=0;k<frms;k++,j+=sd->chans)
	{
	  for (i=0;i<sd->chans;i++) bufs[i][k] = inbuf[j+i];
	}
    }
  FREE(inbuf);
  return(gh_int2scm(val));
}

static SCM g_read_audio_state(SCM dev, SCM field, SCM chan, SCM vals)
{
  int val,i,len;
  float *fvals;
  SCM_ASSERT(gh_number_p(dev),dev,SCM_ARG1,S_read_audio_state);
  SCM_ASSERT(gh_number_p(field),field,SCM_ARG2,S_read_audio_state);
  SCM_ASSERT(gh_number_p(chan),chan,SCM_ARG3,S_read_audio_state);
  SCM_ASSERT(gh_vector_p(vals),vals,SCM_ARG4,S_read_audio_state);
  len = gh_vector_length(vals);
  if (len == 0)
    fvals = (float *)CALLOC(1,sizeof(float));
  else fvals = (float *)CALLOC(len,sizeof(float));
  val = read_audio_state(gh_scm2int(dev),gh_scm2int(field),gh_scm2int(chan),fvals);
  for (i=0;i<len;i++) gh_vector_set_x(vals,gh_int2scm(i),gh_double2scm(fvals[i]));
  FREE(fvals);
  return(gh_int2scm(val));
}

static SCM g_write_audio_state(SCM dev, SCM field, SCM chan, SCM vals)
{
  int i,len,res;
  float *fvals;
  SCM_ASSERT(gh_number_p(dev),dev,SCM_ARG1,S_write_audio_state);
  SCM_ASSERT(gh_number_p(field),field,SCM_ARG2,S_write_audio_state);
  SCM_ASSERT(gh_number_p(chan),chan,SCM_ARG3,S_write_audio_state);
  SCM_ASSERT(gh_vector_p(vals),vals,SCM_ARG4,S_write_audio_state);
  len = gh_vector_length(vals);
  if (len == 0)
    fvals = (float *)CALLOC(1,sizeof(float));
  else
    {
      fvals = (float *)CALLOC(len,sizeof(float));
      for (i=0;i<len;i++) fvals[i] = gh_scm2double(gh_vector_ref(vals,gh_int2scm(i)));
    }
  res = write_audio_state(gh_scm2int(dev),gh_scm2int(field),gh_scm2int(chan),fvals);
  FREE(fvals);
  return(gh_int2scm(res));
}

#ifdef __cplusplus
  static SCM gh_new_procedure5_1 (char *proc_name,SCM (*fn)(...)) {return(gh_new_procedure(proc_name,fn,5,1,0));}
#else
  static SCM gh_new_procedure5_1 (char *proc_name,SCM (*fn)()) {return(gh_new_procedure(proc_name,fn,5,1,0));}
#endif

void init_sndlib2scm(void)
{
  initialize_sndlib();

#if HAVE_MAKE_SMOB_TYPE
  sound_data_tag = scm_make_smob_type_mfpe("sound-data",sizeof(sound_data),mark_sound_data,free_sound_data,print_sound_data,equalp_sound_data);
#else
  sound_data_tag = scm_newsmob(&sound_data_smobfuns);
#endif

  gh_new_procedure1_0(S_sound_data_length,sound_data_length);
  gh_new_procedure1_0(S_sound_data_chans,sound_data_chans);
  gh_new_procedure3_0(S_sound_data_ref,sound_data_ref);
  gh_new_procedure4_0(S_sound_data_setB,sound_data_set);
  gh_new_procedure2_0(S_make_sound_data,g_make_sound_data);
  gh_new_procedure1_0(S_sound_data_p,g_sound_data_p);
  gh_new_procedure3_0(S_sound_data2vct,sound_data2vct);
  gh_new_procedure3_0(S_vct2sound_data,vct2sound_data);

  gh_define(S_next_sound_file,gh_int2scm(NeXT_sound_file));
  gh_define(S_aiff_sound_file,gh_int2scm(AIFF_sound_file));
  gh_define(S_riff_sound_file,gh_int2scm(RIFF_sound_file));
  gh_define(S_nist_sound_file,gh_int2scm(NIST_sound_file));
  gh_define(S_raw_sound_file,gh_int2scm(raw_sound_file));
  gh_define(S_ircam_sound_file,gh_int2scm(IRCAM_sound_file));

  gh_define(S_snd_16_linear,gh_int2scm(SNDLIB_16_LINEAR));
  gh_define(S_snd_16_linear_little_endian,gh_int2scm(SNDLIB_16_LINEAR_LITTLE_ENDIAN));
  gh_define(S_snd_8_mulaw,gh_int2scm(SNDLIB_8_MULAW));
  gh_define(S_snd_8_alaw,gh_int2scm(SNDLIB_8_ALAW));
  gh_define(S_snd_8_linear,gh_int2scm(SNDLIB_8_LINEAR));
  gh_define(S_snd_8_unsigned,gh_int2scm(SNDLIB_8_UNSIGNED));
  gh_define(S_snd_32_float,gh_int2scm(SNDLIB_32_FLOAT));
  gh_define(S_snd_32_float_little_endian,gh_int2scm(SNDLIB_32_FLOAT_LITTLE_ENDIAN));
  gh_define(S_snd_32_linear,gh_int2scm(SNDLIB_32_LINEAR));
  gh_define(S_snd_32_linear_little_endian,gh_int2scm(SNDLIB_32_LINEAR_LITTLE_ENDIAN));
  gh_define(S_snd_24_linear,gh_int2scm(SNDLIB_24_LINEAR));
  gh_define(S_snd_24_linear_little_endian,gh_int2scm(SNDLIB_24_LINEAR_LITTLE_ENDIAN));
  gh_define(S_snd_64_double,gh_int2scm(SNDLIB_64_DOUBLE));
  gh_define(S_snd_64_double_little_endian,gh_int2scm(SNDLIB_64_DOUBLE_LITTLE_ENDIAN));
  gh_define(S_snd_16_unsigned,gh_int2scm(SNDLIB_16_UNSIGNED));
  gh_define(S_snd_16_unsigned_little_endian,gh_int2scm(SNDLIB_16_UNSIGNED_LITTLE_ENDIAN));

  gh_define(S_sndlib_default_device,gh_int2scm(SNDLIB_DEFAULT_DEVICE));
  gh_define(S_sndlib_read_write_device,gh_int2scm(SNDLIB_READ_WRITE_DEVICE));
  gh_define(S_sndlib_line_out_device,gh_int2scm(SNDLIB_LINE_OUT_DEVICE));
  gh_define(S_sndlib_line_in_device,gh_int2scm(SNDLIB_LINE_IN_DEVICE));
  gh_define(S_sndlib_microphone_device,gh_int2scm(SNDLIB_MICROPHONE_DEVICE));
  gh_define(S_sndlib_speakers_device,gh_int2scm(SNDLIB_SPEAKERS_DEVICE));
  gh_define(S_sndlib_dac_out_device,gh_int2scm(SNDLIB_DAC_OUT_DEVICE));
  gh_define(S_sndlib_adat_in_device,gh_int2scm(SNDLIB_ADAT_IN_DEVICE));
  gh_define(S_sndlib_aes_in_device,gh_int2scm(SNDLIB_AES_IN_DEVICE));
  gh_define(S_sndlib_digital_in_device,gh_int2scm(SNDLIB_DIGITAL_IN_DEVICE));
  gh_define(S_sndlib_digital_out_device,gh_int2scm(SNDLIB_DIGITAL_OUT_DEVICE));
  gh_define(S_sndlib_adat_out_device,gh_int2scm(SNDLIB_ADAT_OUT_DEVICE));
  gh_define(S_sndlib_aes_out_device,gh_int2scm(SNDLIB_AES_OUT_DEVICE));
  gh_define(S_sndlib_dac_filter_device,gh_int2scm(SNDLIB_DAC_FILTER_DEVICE));
  gh_define(S_sndlib_mixer_device,gh_int2scm(SNDLIB_MIXER_DEVICE));
  gh_define(S_sndlib_line1_device,gh_int2scm(SNDLIB_LINE1_DEVICE));
  gh_define(S_sndlib_line2_device,gh_int2scm(SNDLIB_LINE2_DEVICE));
  gh_define(S_sndlib_line3_device,gh_int2scm(SNDLIB_LINE3_DEVICE));
  gh_define(S_sndlib_aux_input_device,gh_int2scm(SNDLIB_AUX_INPUT_DEVICE));
  gh_define(S_sndlib_cd_in_device,gh_int2scm(SNDLIB_CD_IN_DEVICE));
  gh_define(S_sndlib_aux_output_device,gh_int2scm(SNDLIB_AUX_OUTPUT_DEVICE));
  gh_define(S_sndlib_spdif_in_device,gh_int2scm(SNDLIB_SPDIF_IN_DEVICE));
  gh_define(S_sndlib_spdif_out_device,gh_int2scm(SNDLIB_SPDIF_OUT_DEVICE));

  gh_define(S_sndlib_amp_field,gh_int2scm(SNDLIB_AMP_FIELD));
  gh_define(S_sndlib_srate_field,gh_int2scm(SNDLIB_SRATE_FIELD));
  gh_define(S_sndlib_channel_field,gh_int2scm(SNDLIB_CHANNEL_FIELD));
  gh_define(S_sndlib_format_field,gh_int2scm(SNDLIB_FORMAT_FIELD));
  gh_define(S_sndlib_device_field,gh_int2scm(SNDLIB_DEVICE_FIELD));
  gh_define(S_sndlib_imix_field,gh_int2scm(SNDLIB_IMIX_FIELD));
  gh_define(S_sndlib_igain_field,gh_int2scm(SNDLIB_IGAIN_FIELD));
  gh_define(S_sndlib_reclev_field,gh_int2scm(SNDLIB_RECLEV_FIELD));
  gh_define(S_sndlib_pcm_field,gh_int2scm(SNDLIB_PCM_FIELD));
  gh_define(S_sndlib_pcm2_field,gh_int2scm(SNDLIB_PCM2_FIELD));
  gh_define(S_sndlib_ogain_field,gh_int2scm(SNDLIB_OGAIN_FIELD));
  gh_define(S_sndlib_line_field,gh_int2scm(SNDLIB_LINE_FIELD));
  gh_define(S_sndlib_mic_field,gh_int2scm(SNDLIB_MIC_FIELD));
  gh_define(S_sndlib_line1_field,gh_int2scm(SNDLIB_LINE1_FIELD));
  gh_define(S_sndlib_line2_field,gh_int2scm(SNDLIB_LINE2_FIELD));
  gh_define(S_sndlib_line3_field,gh_int2scm(SNDLIB_LINE3_FIELD));
  gh_define(S_sndlib_synth_field,gh_int2scm(SNDLIB_SYNTH_FIELD));
  gh_define(S_sndlib_bass_field,gh_int2scm(SNDLIB_BASS_FIELD));
  gh_define(S_sndlib_treble_field,gh_int2scm(SNDLIB_TREBLE_FIELD));
  gh_define(S_sndlib_cd_field,gh_int2scm(SNDLIB_CD_FIELD));

  gh_new_procedure1_0(S_sound_samples,g_sound_samples);
  gh_new_procedure1_0(S_sound_frames,g_sound_frames);
  gh_new_procedure1_0(S_sound_duration,g_sound_duration);
  gh_new_procedure1_0(S_sound_datum_size,g_sound_datum_size);
  gh_new_procedure1_0(S_sound_data_location,g_sound_data_location);
  gh_new_procedure1_0(S_sound_chans,g_sound_chans);
  gh_new_procedure1_0(S_sound_srate,g_sound_srate);
  gh_new_procedure1_0(S_sound_header_type,g_sound_header_type);
  gh_new_procedure1_0(S_sound_data_format,g_sound_data_format);
  gh_new_procedure1_0(S_sound_length,g_sound_length);
  gh_new_procedure1_0(S_sound_type_specifier,g_sound_type_specifier);
  gh_new_procedure1_0(S_sound_type_name,g_sound_type_name);
  gh_new_procedure1_0(S_sound_format_name,g_sound_format_name);
  gh_new_procedure1_0(S_sound_comment,g_sound_comment);
  gh_new_procedure1_0(S_sound_bytes_per_sample,g_sound_bytes_per_sample);
  gh_new_procedure0_0(S_audio_error,g_audio_error);
  gh_new_procedure1_0(S_audio_error_name,g_audio_error_name);
  gh_new_procedure0_0(S_report_audio_state,g_report_audio_state);
  gh_new_procedure3_0(S_audio_outputs,g_audio_outputs);
  gh_new_procedure1_0(S_sound_max_amp,g_sound_max_amp);

  gh_new_procedure1_0(S_open_sound_input,g_open_sound_input);
  gh_new_procedure5_1(S_open_sound_output,g_open_sound_output);
  gh_new_procedure5_0(S_reopen_sound_output,g_reopen_sound_output);
  gh_new_procedure1_0(S_close_sound_input,g_close_sound_input);
  gh_new_procedure2_0(S_close_sound_output,g_close_sound_output);
  gh_new_procedure5_0(S_read_sound,g_read_sound);
  gh_new_procedure5_0(S_write_sound,g_write_sound);
  gh_new_procedure3_0(S_seek_sound,g_seek_sound);
  gh_new_procedure2_0(S_seek_sound_frame,g_seek_sound_frame);

  gh_new_procedure5_0(S_open_audio_output,g_open_audio_output);
  gh_new_procedure5_0(S_open_audio_input,g_open_audio_input);
  gh_new_procedure1_0(S_close_audio,g_close_audio);
  gh_new_procedure0_0(S_save_audio_state,g_save_audio_state);
  gh_new_procedure0_0(S_restore_audio_state,g_restore_audio_state);
  gh_new_procedure0_0(S_audio_systems,g_audio_systems);
  gh_new_procedure3_0(S_write_audio,g_write_audio);
  gh_new_procedure3_0(S_read_audio,g_read_audio);
  gh_new_procedure4_0(S_read_audio_state,g_read_audio_state);
  gh_new_procedure4_0(S_write_audio_state,g_write_audio_state);
}

#endif
