/*
 *  apcaccess.c  -- Text based IPC management tool for apcupsd package.
 *
 *  apcupsd.c    -- Simple Daemon to catch power failure signals from a
 *                  BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *               -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick
 *                        <hedrick@astro.dyer.vanderbilt.edu>
 *  All rights reserved.
 *
 */

/*
 *                     GNU GENERAL PUBLIC LICENSE
 *                        Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *                           675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR 
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

/*
 *  Changed the code to output the informative messages to stdout.
 *   -RF
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <fcntl.h>
#include <limits.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/ipc.h>
#include <ctype.h>
#include <pwd.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <apc_config.h>
#include <apc_i18n.h>
#include <apc_version.h>
#include <apc_defines.h>
#include <apc_struct.h>
#include <apc_extern.h>

#define BUFFER_SIZE PIPE_BUF

struct passwd *pwentry;

#ifndef NEW_THREADS
int server_fifo_fd;
int client_fifo_fd;
#endif /* NEW_THREADS */

char msg[100];
time_t nowtime;
char m_char;

UPSINFO myUPS;

DATAINFO myDATA = {
	"\0",	/* data->apcmagic             */
	0,   	/* data->update_master_config */
	0,   	/* data->get_master_status    */
	-1,  	/* data->slave_status         */
	0,   	/* data->call_master_shutdown */
	"\0"	/* data->accessmagic          */
};

CONFIGINFO myCONFIG = {
	0,	/* config->new_annoy          */
	0,	/* config->new_maxtime        */
	0,	/* config->new_delay          */
	0,	/* config->new_proctime       */
	0,	/* config->new_logtime        */
	0,	/* config->new_nettime        */
	0,	/* config->new_percent        */

	"\0",	/* config->new_batt_cmd       */
	"\0",	/* config->new_time_cmd       */
	"\0",	/* config->new_load_cmd       */
	"\0",	/* config->new_pwr_cmd        */
	"\0",	/* config->new_ret_cmd        */
	"\0" 	/* config->new_remote_cmd     */
};

/*********************************************************************/
void output_status (UPSINFO *ups);
void reconfig_data (CONFIGINFO *config);
void do_main_menu (void);
void do_config_menu (void);

/*********************************************************************/
void output_status (UPSINFO *ups)
{
	time(&nowtime);
	strftime(msg, 100, "%b %d %X", localtime(&nowtime));

	printf("\nAPC      : %s", msg);
	printf("\nCABLE    : %s", ups->cable.long_name);
	printf("\nUPSMODEL : %s", ups->mode.long_name);
	printf("\nUPSMODE  : %s", ups->class.long_name);

	if (ups->sharenet.type != DISABLE)
		printf("\nSHARE    : %s", ups->sharenet.long_name);

	switch (ups->mode.type) {
		case BK:
		case SHAREBASIC:
		case NETUPS:
			if (ups->LineUp == 0) {
				printf("\nLINEFAIL : OK");
				printf("\nBATTSTAT : OK\n");
			} else {
				printf("\nLINEFAIL : DOWN");
				if (ups->BattUp == 0)
					printf("\nBATTSTAT : RUNNING\n");
				else
					printf("\nBATTSTAT : FAILING\n");
			}
			break;
		case BKPRO:
		case VS:
			if (ups->LineUp == 0) {
				printf("\nLINEFAIL : OK");
				printf("\nBATTSTAT : OK");
				if (ups->LineLevel == -1)
					printf("\nLINEVOLT : LOW");
				else if (ups->LineLevel == 1)
					printf("\nLINEVOLT : HIGH");
				else
					printf("\nLINEVOLT : OK");
			} else {
				printf("\nLINEFAIL : DOWN");
				if (ups->BattUp == 0)
					printf("\nBATTSTAT : RUNNING");
				else
					printf("\nBATTSTAT : FAILING");
			}
	case NBKPRO:
	case SMART:
	case SHARESMART:
	case MATRIX:
			if (ups->UPSlink_flags & 0x01000000)
				printf("\nUPSNAME  : %s", ups->name);

			if (ups->UPSlink_flags & 0x00000010)
				printf("\nULINE    : %05.1f Volts", ups->LineVoltage);

			if (ups->UPSlink_flags & 0x00000020)
				printf("\nMLINE    : %05.1f Volts", ups->LineMax);

			if (ups->UPSlink_flags & 0x00000040)
				printf("\nNLINE    : %05.1f Volts", ups->LineMin);

			if (ups->UPSlink_flags & 0x00000800)
				printf("\nFLINE    : %03.1f Hz", ups->LineFreq);

			if (ups->UPSlink_flags & 0x00000080)
				printf("\nVOUTP    : %05.1f Volts", ups->OutputVoltage);

			if (ups->UPSlink_flags & 0x00000400)
				printf("\nLOUTP    : %05.1f Load Capacity", ups->UPSLoad);

			if (ups->UPSlink_flags & 0x00000200)
				printf("\nBOUTP    : %04.1f Volts", ups->BattVoltage);

			if (ups->UPSlink_flags & 0x00000100)
				printf("\nBCHAR    : %05.1f Batt. Charge", ups->BatLoad);

			if (ups->UPSlink_flags & 0x00001000)
				printf("\nTIME     : %04.1f Minutes", ups->TimeLeft);

			if (ups->UPSlink_flags & 0x00010000) {
				switch ((*ups).sensitivity[0]) {
					case 'L':
						printf("\nSENSE    : LOW");
						break;
					case 'M':
						printf("\nSENSE    : MEDIUM");
						break;
					case 'H':
					default :
						printf("\nSENSE    : HIGH");
						break;
				}
			}

			if (ups->UPSlink_flags & 0x00020000)
				printf("\nWAKEUP   : %03d Cycles", ups->wakeup);

			if (ups->UPSlink_flags & 0x00040000)
				printf("\nSLEEP    : %03d Cycles", ups->sleep);

			if (ups->UPSlink_flags & 0x00080000)
				printf("\nLOTRANS  : %03d.0 Volts", ups->lotrans);

			if (ups->UPSlink_flags & 0x00100000)
				printf("\nHITRANS  : %03d.0 Volts", ups->hitrans);

			if (ups->UPSlink_flags & 0x00200000)
				printf("\nCHARGE   : %03d.0 Percent", ups->minon);

			if (ups->UPSlink_flags & 0x00000001)
				printf("\nBFAIL    : 0x%02X Status Flag", ups->Status);

			if (ups->UPSlink_flags & 0x00002000)
				printf("\nUTEMP    : %04.1f C Internal", ups->UPSTemp);

			if (ups->UPSlink_flags & 0x00400000) {
				switch ((*ups).beepstate[0]) {
					case 'T':
						printf("\nALARM    : Low Batt + 30");
						break;
					case 'L':
						printf("\nALARM    : Low Batt");
						break;
					case 'N':
						printf("\nALARM    : Never");
						break;
					case '0':
					default :
						printf("\nALARM    : Always");
						break;
				}
			}

			if (ups->UPSlink_flags & 0x00000004) {
				switch (ups->G[0]) {
					case 'O':
						printf("\nLASTEVNT : POWER UP");
						break;
					case 'S':
						printf("\nLASTEVNT : SELF TEST");
						break;
					case 'L':
						printf("\nLASTEVNT : LINE VOLTAGE DECREASE");
						break;
					case 'H':
						printf("\nLASTEVNT : LINE VOLTAGE INCREASE");
						break;
					case 'T':
						printf("\nLASTEVNT : POWER FAILURE");
						break;
					case 'R':
						printf("\nLASTEVNT : R-EVENT");
						break;
					default :
						printf("\nLASTEVNT : UNKNOWN EVENT %c %c", ups->G[0], ups->G[1]);
						break;
				}
			}

			if (ups->UPSlink_flags & 0x00800000)
				printf("\nLOWBATT  : %02d Minutes", ups->lowbatt);

			if (ups->UPSlink_flags & 0x00004000) {
				if (ups->seven & 0x08)
					printf("\nDIPSW    : 0x%d", TRUE);
				else
					printf("\nDIPSW    : 0x%d", FALSE);
				if (ups->seven & 0x04)
					printf("%d", TRUE);
				else
					printf("%d", FALSE);
				if (ups->seven & 0x02)
					printf("%d", TRUE);
				else
					printf("%d", FALSE);
				if (ups->seven & 0x01)
					printf("%d", TRUE);
				else
					printf("%d", FALSE);
			}

			printf("\n");
			break;
		default:
			break;
	}
}

/*********************************************************************/
void reconfig_data (CONFIGINFO *config)
{
	/* config->new_annoy          */
	/* config->new_maxtime        */
	/* config->new_delay          */
	/* config->new_proctime       */
	/* config->new_logtime        */
	/* config->new_nettime        */
	/* config->new_percent        */
	/* config->new_batt_cmd       */
	/* config->new_time_cmd       */
	/* config->new_load_cmd       */
	/* config->new_pwr_cmd        */
	/* config->new_ret_cmd        */
	/* config->new_remote_cmd     */
}

/*********************************************************************/
void do_main_menu (void)
{
	char ch, temp_comment[256];
	int comment_count = 0;

	while ((ch = getchar()) != '\n')
		temp_comment[comment_count++] = ch;
	temp_comment[comment_count] = '\0';
	m_char = temp_comment[0];
	printf("\n");

	switch (m_char) {
		case '~':
			m_char = 'S';
			break;
		case 'R':
		case 'r':
			do_config_menu();
			break;
		case 'S':
		case 's':
			break;
		case 'K':
		case 'k':
			break;
		case 'E':
		case 'e':
			m_char = '~';
			break;
		default:
			break;
	}

	system("clear");
}
	
/*********************************************************************/
void do_config_menu (void)
{
	system("clear");
}

#ifndef NEW_THREADS

/*********************************************************************/
int main (int argc, char **argv)
{
	int bytes_read = 0;
	int bytes_sent = 0;
	int test = 0;
	int mode = 0;
	int step = 0;

	zero_ups_struct(&myUPS);

	/*
	* The old logging out business was childish:
	*   code replaced by fair report and exit.
	* JHNC
	*
	* We must use the getuid and not rely on pipe permissions because one day we
	* will be able to reconfigure networked UPSes with wich hardly we will have
	* an open pipe.
	* -RF
	*/
	if ((getuid() != 0) && (geteuid() != 0)) {
		fprintf(stderr, "%s: (fifo) needs superuser privileges.\n", argv[0]);
		exit(1);
	}

	if (argc < 2) {
		fprintf(stderr, "Usage: %s (fifo) < reconfig | status | slave | shutdown >\n", argv[0]);
		exit(1);
	}

	if (strcmp(argv[1], "reconfig") == 0) {
		mode = 1;
		fprintf(stderr, "%s (fifo) <%s> : is not functional yet.\n", argv[0], argv[1]);
		myDATA.update_master_config = TRUE;
		myDATA.get_master_status    = FALSE;
		myDATA.slave_status         = -1;
		myDATA.call_master_shutdown = FALSE;
		exit(1);
	} else if (strcmp(argv[1], "status") == 0) {
		mode = 2;
		myDATA.update_master_config = FALSE;
		myDATA.get_master_status    = TRUE;
		myDATA.slave_status         = -1;
		myDATA.call_master_shutdown = FALSE;
		fprintf(stderr, "%s (fifo) : polling apcupsd for %s.\n", argv[0], argv[1]);
	} else if (strcmp(argv[1], "slave") == 0) {
		mode = 3;
		myDATA.update_master_config = FALSE;
		myDATA.get_master_status    = FALSE;
		myDATA.slave_status         = TRUE;
		myDATA.call_master_shutdown = FALSE;
		fprintf(stderr, "%s (fifo) : calling apcupsd for %s\n", argv[0], argv[1]);
	} else if (strcmp(argv[1], "shutdown") == 0) {
		mode = 4;
		myDATA.update_master_config = FALSE;
		myDATA.get_master_status    = FALSE;
		myDATA.slave_status         = -1;
		myDATA.call_master_shutdown = TRUE;
		fprintf (stderr, "%s (fifo) : calling apcupsd for %s.\n", argv[0], argv[1]);
	} else {
		mode = 0;
		exit(1);
	}

/*
	while (m_char != '~')
		do_main_menu();
*/

	mkfifo(ACCESS_FIFO, 0777);
	if ((server_fifo_fd = open(ACCESS_FIFO, O_WRONLY)) == -1) {
#ifdef DEBUG
		fprintf(stderr, "%s: (fifo) Server fifo failure\n", argv[0]);
#endif
		unlink(ACCESS_FIFO);
		exit(1);
	}

	strcpy(myDATA.apcmagic, APC_MAGIC);
	strcpy(myDATA.accessmagic, ACCESS_MAGIC);

	for (step=0; step<12; step++) {
		if ((client_fifo_fd = open(PIPE_FIFO, O_RDONLY)) != -1) {
#ifdef DEBUG
			fprintf(stderr, "%s: (fifo) Got Client FIFO Success\n", argv[0]);
#endif
			goto fifo_success;
		}
#ifdef DEBUG
		fprintf(stderr, "%s: (fifo) Waiting on Client FIFO, count = %d.\n", argv[0], step);
#endif
		sleep(5);
	}


	close(server_fifo_fd);
#ifdef DEBUG
	fprintf(stderr, "%s: (fifo) client fifo failure\n", argv[0]);
#endif
	unlink(ACCESS_FIFO);
	exit(1);

fifo_success:

	switch (mode) {
		case 1:
			reconfig_data(&myCONFIG);
			if ((test = write(server_fifo_fd, &myDATA, sizeof(myDATA))) != sizeof(myDATA)) {
				fprintf(stderr, "\n%s: (fifo) %d size of myDATA", argv[0], test);
				close(server_fifo_fd);
				close(client_fifo_fd);
				unlink(ACCESS_FIFO);
				exit(1);
			}
			while (bytes_sent < sizeof(myCONFIG)) {
				test = write(server_fifo_fd, &myCONFIG, BUFFER_SIZE);
				bytes_sent += test;
			}
			close(server_fifo_fd);
			if ((test = read(client_fifo_fd, &myDATA, sizeof(myDATA))) != sizeof(myDATA)) {
				fprintf(stderr, "\n%s: (fifo) %d size of myDATA", argv[0], test);
				close(client_fifo_fd);
				unlink(ACCESS_FIFO);
				exit(1);
			}
			close(client_fifo_fd);
			break;
		case 2:
			if ((test = write(server_fifo_fd, &myDATA, sizeof(myDATA))) != sizeof(myDATA)) {
				fprintf(stderr, "\n%s: (fifo) %d size of myDATA", argv[0], test);
				close(server_fifo_fd);
				close(client_fifo_fd);
				unlink(ACCESS_FIFO);
				exit(1);
			}
			do {
				test = read(client_fifo_fd, &myUPS, BUFFER_SIZE);
				bytes_read += test;
			} while (test > 0) ;
			close(client_fifo_fd);
			output_status(&myUPS);
			break;
		case 3:
		case 4:
			if ((test = write(server_fifo_fd, &myDATA, sizeof(myDATA))) != sizeof(myDATA)) {
				fprintf(stderr, "\n%s: (fifo) %d size of myDATA", argv[0], test);
				close(server_fifo_fd);
				close(client_fifo_fd);
				unlink(ACCESS_FIFO);
				exit(1);
			}
			close(client_fifo_fd);
			break;
	}

	close(server_fifo_fd);
	unlink(ACCESS_FIFO);

	return(0);
}

#else /* NEW_THREADS */

/*********************************************************************/
int main (int argc, char **argv)
{
	int mode = 0;

	zero_ups_struct(&myUPS);

	if ((getuid() != 0) && (geteuid() != 0)) {
		fprintf(stderr, "%s: (ipc) needs superuser privileges.\n", argv[0]);
		exit(1);
	}

	if (argc < 2) {
		fprintf(stderr,	"Usage: %s (ipc) < reconfig | status | slave | shutdown >\n", argv[0]);
		exit(1);
	}

	if (strcmp(argv[1], "reconfig") == 0) {
		mode = 1;
		fprintf(stderr, "%s (ipc) <%s> : is not functional yet.\n", argv[0], argv[1]);
		myDATA.update_master_config = TRUE;
		myDATA.get_master_status    = FALSE;
		myDATA.slave_status         = -1;
		myDATA.call_master_shutdown = FALSE;
		exit(1);
	} else if (strcmp(argv[1], "status") == 0) {
		mode = 2;
		myDATA.update_master_config = FALSE;
		myDATA.get_master_status    = TRUE;
		myDATA.slave_status         = -1;
		myDATA.call_master_shutdown = FALSE;
		fprintf(stderr, "%s (ipc) : polling apcupsd for %s.\n", argv[0], argv[1]);
	} else if (strcmp(argv[1], "slave") == 0) {
		mode = 3;
		myDATA.update_master_config = FALSE;
		myDATA.get_master_status    = FALSE;
		myDATA.slave_status         = TRUE;
		myDATA.call_master_shutdown = FALSE;
		fprintf(stderr, "%s (ipc) <%s> : is not functional yet.\n", argv[0], argv[1]);
	} else if (strcmp(argv[1], "shutdown") == 0) {
		mode = 4;
		myDATA.update_master_config = FALSE;
		myDATA.get_master_status    = FALSE;
		myDATA.slave_status         = -1;
		myDATA.call_master_shutdown = TRUE;
		fprintf(stderr, "%s (ipc) <%s> : is not functional yet.\n", argv[0], argv[1]);
	} else {
		mode = 0;
		exit(1);
	}

/*
	while (m_char != '~')
		do_main_menu();
*/

	strcpy(myDATA.apcmagic, APC_MAGIC);
	strcpy(myDATA.accessmagic, ACCESS_MAGIC);

	switch (mode) {
		case 1:
			break;
		case 2:
			if (attach_ipc() != SUCCESS) {
				fprintf(stderr, "Can not attach SYSV IPC.\n");
				break;
			}

			if (read_shmarea(&myUPS) != SUCCESS) {
				fprintf(stderr,	"Can not read shm data area.\n");
				break;
			}

			output_status(&myUPS);
			detach_ipc();
			break;
		case 3:
			break;
		case 4:
			if (attach_ipc() != SUCCESS) {
				fprintf(stderr, "Can not attach SYSV IPC.\n");
				break;
			}
			if (write_shmarea(&myUPS) != SUCCESS) {
				fprintf(stderr, "Can not read shm data area.\n");
				detach_ipc();
				break;
			}

			detach_ipc();
			break;
		default:
			break;
	}
	return(0);
}
#endif /* NEW_THREADS */
