/*
 *  apcaction.c -- Actions taken when something is happen to UPS.
 *
 *  apcupsd.c   -- Simple Daemon to catch power failure signals from a
 *                 BackUPS, BackUPS Pro, or SmartUPS (from APCC).
 *              -- Now SmartMode support for SmartUPS and BackUPS Pro.
 *
 *  Copyright (C) 1996-99 Andre M. Hedrick
 *                        <hedrick@astro.dyer.vanderbilt.edu>
 *  All rights reserved.
 *
 */

/*
 *                     GNU GENERAL PUBLIC LICENSE
 *                        Version 2, June 1991
 *
 *  Copyright (C) 1989, 1991 Free Software Foundation, Inc.
 *                           675 Mass Ave, Cambridge, MA 02139, USA
 *  Everyone is permitted to copy and distribute verbatim copies
 *  of this license document, but changing it is not allowed.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 *  IN NO EVENT SHALL ANY AND ALL PERSONS INVOLVED IN THE DEVELOPMENT OF THIS
 *  PACKAGE, NOW REFERRED TO AS "APCUPSD-Team" BE LIABLE TO ANY PARTY FOR
 *  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
 *  OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF ANY OR ALL
 *  OF THE "APCUPSD-Team" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  THE "APCUPSD-Team" SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 *  ON AN "AS IS" BASIS, AND THE "APCUPSD-Team" HAS NO OBLIGATION TO PROVIDE
 *  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 *  THE "APCUPSD-Team" HAS ABSOLUTELY NO CONNECTION WITH THE COMPANY
 *  AMERICAN POWER CONVERSION, "APCC".  THE "APCUPSD-Team" DID NOT AND
 *  HAS NOT SIGNED ANY NON-DISCLOSURE AGREEMENTS WITH "APCC".  ANY AND ALL
 *  OF THE LOOK-A-LIKE ( UPSlink(tm) Language ) WAS DERIVED FROM THE
 *  SOURCES LISTED BELOW.
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <stdlib.h>
#include <stdarg.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <apc_config.h>
#include <apc_i18n.h>
#include <apc_version.h>
#include <apc_defines.h>
#include <apc_struct.h>
#include <apc_extern.h>

#ifdef NEW_THREADS
#undef RE_SAMPLE_SERIAL

/*********************************************************************/
void timer_action (int sig) {
	UPSINFO *ups = &myUPS;

	read_andlock_shmarea(ups);

	if (ups->reports) {
		if (ups->logtime)
			alarm(ups->logtime);

		if (ups->proctime)
			alarm(ups->proctime);
	}

	/*
	 *  Must SHUTDOWN Remote System Calls
	 */
	if (ups->remotedown) {
		ups->BatteryUp = 0;
		ups->ShutDown = 1;
		if (ups->remote_cmd[0])
			system(ups->remote_cmd);
		delete_lockfile(ups);
		if (ups->nologin_file != TRUE)
			make_file(NOLOGIN);
		ups->nologin_file = TRUE;
		logonfail(0);
		system(ups->remote_shutdown);
		make_file(PWRFAIL);
		powerfail(3);
	}
	/*
	 *  Did status_bit go high? Then the power is failing.
	 */
	if (ups->OldLineUp == 0 && ups->LineUp != 0) {
		system(ups->powerout);
		if (ups->pwr_cmd[0])
			system(ups->pwr_cmd);
		/*
		 *  Now enable the DTR. (power_bit) for _CUSTOM
		 */
		if (ups->cable.type == CUSTOM_SIMPLE)
			(void) ioctl(ups->fd, TIOCMBIS, &dtr_bit);
		ups->OldLineUp = ups->LineUp;
		ups->OldBattUp = ups->BattUp;
		ups->OldLineDown = ups->LineDown;

#ifdef RE_SAMPLE_SERIAL
		/*
		 *  Now re-sample the status_bit line.
		 *
		 *  This is done to prevent over reaction of the daemon,
		 *  if the local power is fuzzy.
		 *  Usually stormy nights/
		 */
		write_andunlock_shmarea(ups);
		alarm(TIMER_SERIAL);
		(void) check_serial(ups);
		read_andlock_shmarea(ups);
#endif /* RE_SAMPLE_SERIAL */

		/*
		 *  Did the second test verify the power is failing?
		 */
		if ((ups->LineUp) && (ups->OldLineUp)) {
			ups->BatteryUp = 1;
#if 0
			if (ups->nologin.type == ALWAYS) {
				make_file(NOLOGIN);
				ups->nologin_file = TRUE;
				logonfail(0);
			}
#endif
			system(ups->onbatt);
		}
	}
	/*
	 *  Did lowbatt_bit go high? Then the battery power is failing.
	 *  Normal Power down during Power Failure
	 */
	if ((ups->OldBattUp == 0 && ups->BattUp > 0) &&
	    (ups->BatteryUp == 1 && ups->ShutDown == 0)) {
		ups->BatteryUp = 0;
		ups->ShutDown = 1;
		delete_lockfile(ups);
		system(ups->failing);
		if (ups->batt_cmd[0]) system(ups->batt_cmd);
			make_file(PWRFAIL);
		powerfail(2);
		if (ups->nologin_file != TRUE)
			make_file(NOLOGIN);
		logonfail(0);
		system(ups->doshutdown);
	}
	/*
	 *  Did MaxTimeOnBattery Expire?
	 *  Normal Power down during Power Failure
	 */
	if ((ups->maxtime != 0) && ups->LineUp &&
	    ((time(NULL) - last_time_on_line) > ups->maxtime)) {
		ups->ShutDown = 1;
		ups->timedout = TRUE;
		if (ups->time_cmd[0])
			system(ups->time_cmd);
		delete_lockfile(ups);
		system(ups->timeout);
		make_file(PWRFAIL);
		powerfail(2);
		if (ups->nologin_file != TRUE)
			make_file(NOLOGIN);
		logonfail(0);
		system(ups->doshutdown);
	}
	/*
	 *  Did Battery Charge go below percent cutoff?
	 *  Normal Power down during Power Failure
	 */
	if ((ups->mode.type >= NBKPRO) && ups->LineUp &&
	    (ups->BatLoad <= ups->percent)) {
		ups->ShutDown = 1;
		ups->load = TRUE;
		if (ups->load_cmd[0])
			system(ups->load_cmd);
		delete_lockfile(ups);
		system(ups->loadlimit);
		make_file(PWRFAIL);
		powerfail(2);
		if (ups->nologin_file != TRUE)
			make_file(NOLOGIN);
		logonfail(0);
		system(ups->doshutdown);
	}
	/*
	 *  Did Battery Runtime go below percent cutoff?
	 *  Normal Power down during Power Failure
	 */
	if ((ups->mode.type >= NBKPRO) && ups->LineUp &&
	    (ups->TimeLeft <= ups->runtime)) {
		ups->ShutDown = 1;
		ups->timelout = TRUE;
		if (ups->limit_cmd[0])
			system(ups->limit_cmd);
		delete_lockfile(ups);
		system(ups->runlimit);
		make_file(PWRFAIL);
		powerfail(2);
		if (ups->nologin_file != TRUE)
			make_file(NOLOGIN);
		logonfail(0);
		system(ups->doshutdown);
	}
	/*
	 *  Did status_bit go down again? Then the power is back.
	 *  No shutdown to cancel.
	 */
	if ((ups->OldLineUp > 0 && ups->LineUp == 0) && ups->ShutDown == 0) {
		ups->BatteryUp = 0;
		system(ups->mainsback);
		if (ups->ret_cmd[0])
			system(ups->ret_cmd);
		logonfail(1);
		remove_file(NOLOGIN);
		ups->nologin_file = FALSE;
		if (ups->cable.type == CUSTOM_SIMPLE) {
			(void) ioctl(ups->fd, TIOCMBIC, &dtr_bit);
			/*
			 *  Clearing DTR and TxD (ST).
			 */
			(void) ioctl(ups->fd, TIOCMBIC, &st_bit);
		} 
	}
	/*
	 *  Did status_bit go down again? Then the power is back.
	 *  Shutdown to cancel.
	 */
	if ((ups->OldLineUp > 0 && ups->LineUp == 0) && ups->ShutDown == 1) {
		ups->BatteryUp = 0;
		ups->ShutDown = 0;
		system(ups->waitasec);
		if (ups->ret_cmd[0])
			system(ups->ret_cmd);
		powerfail(1);
		remove_file(PWRFAIL);
		logonfail(1);
		remove_file(NOLOGIN);
		ups->nologin_file = FALSE;
		if (ups->cable.type == CUSTOM_SIMPLE) {
			(void) ioctl(ups->fd, TIOCMBIC, &dtr_bit);
			/*
			 *  Clearing DTR and TxD (ST).
			 */
			(void) ioctl(ups->fd, TIOCMBIC, &st_bit);
		}
	}
	/*
	 *  Announce to LogOff, with initial delay ....
	 */
	if ((ups->BatteryUp == 1 && ups->ShutDown == 0 && ups->LineUp > 0) &&
	    ((time(NULL) - last_time_delay) > ups->delay) &&
	    ((time(NULL) - last_time_annoy) > ups->annoy) &&
	    (ups->nologin_file == TRUE)) {
		system(ups->annoyme);
		if (ups->pwr_cmd[0])
			system(ups->pwr_cmd);
		time(&last_time_annoy);
	}
	/*
	 *  Delay NoLogons....
	 */
	if ((ups->BatteryUp == 1) && (ups->ShutDown == 0) &&
	    (ups->LineUp > 0) && (ups->nologin_file != TRUE)) {
		switch(ups->nologin.type) {
			case NEVER:
				break;
			case TIMEOUT:
				if ((time(NULL) - last_time_nologon) > ups->nologin_time) {
					make_file(NOLOGIN);
					ups->nologin_file = TRUE;
					logonfail(0);
				}
				break;
			case PERCENT:
				if (ups->nologin_time >= ups->BatLoad) {
					make_file(NOLOGIN);
					ups->nologin_file = TRUE;
					logonfail(0);
				}
				break;
			case MINUTES:
				if (ups->nologin_time >= ups->TimeLeft) {
					make_file(NOLOGIN);
					ups->nologin_file = TRUE;
					logonfail(0);
				}
				break;
			case ALWAYS:
			default:
				make_file(NOLOGIN);
				ups->nologin_file = TRUE;
				logonfail(0);
				break;
		}
	}
	/*
	 *  Did lowbatt_bit never go low then high?
	 *  Then the battery power has failed!!!
	 *  Must SHUTDOWN NOW
	 */
	if ((ups->LineUp != 0 && ups->BattUp == 1 && ups->LineDown == 0) &&
	    (ups->BatteryUp == 1 && ups->ShutDown == 0)) {
		ups->BatteryUp = 0;
		ups->ShutDown = 1;
		ups->emergencydown = TRUE;
		delete_lockfile(ups);
		system(ups->emergency);
		if (ups->batt_cmd[0])
			system(ups->batt_cmd);
		if (ups->nologin_file != TRUE)
			make_file(NOLOGIN);
		ups->nologin_file = TRUE;
		logonfail(0);
		make_file(PWRFAIL);
		powerfail(2);
	}

	/*
	 *  Remember status
	 */

	ups->OldLineUp = ups->LineUp;
	ups->OldBattUp = ups->BattUp;
	ups->OldLineDown = ups->LineDown;

	write_andunlock_shmarea(ups);
	alarm(TIMER_ACTION);
}

/*********************************************************************/
void do_action (void) {
	init_thread_signals();
	init_timer(TIMER_ACTION, timer_action);
	sleep_forever();
}
#endif /* NEW_THREADS */
