/*
 * dproc.c - Pyramid DC/OSx and Reliant UNIX process access functions for lsof
 */


/*
 * Copyright 1996 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1996 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dproc.c,v 1.13 99/06/22 08:11:02 abe Exp $";
#endif

#include "lsof.h"
#include <sys/systeminfo.h>


/*
 * Local function prototypes.
 */

#if	defined(RELIANTV)
_PROTOTYPE(static void get_cdevs,(void));
#endif	/* defined(RELIANTV) */

_PROTOTYPE(static void get_kernel_access,(void));
_PROTOTYPE(static void readfsinfo,(void));
_PROTOTYPE(static void process_text,(KA_T pa));


/*
 * Local static values
 */

static KA_T Sgvnops;			/* kernel's segvn_ops address */


/*
 * ckkv - check kernel version
 */

static void
ckkv(d, er, ev, ea)
	char *d;			/* dialect */
	char *er;			/* expected release */
	char *ev;			/* expected version */
	char *ea;			/* expected architecture */
{

#if	defined(HASKERNIDCK)
	char r[64];

	if (Fwarn)
	    return;
/*
 * Read Pyramid kernel information.
 */
	if (sysinfo(SI_RELEASE, r, (long)sizeof(r)) < 0) {
	    (void) fprintf(stderr, "%s: can't get sysinfo: %s\n",
		Pn, strerror(errno));
	    Exit(1);
	}
/*
 * Warn if the actual and expected releases don't match.
 */
	r[sizeof(r)-1] = '\0';
	if (!er || strcmp(er, r))
	    (void) fprintf(stderr,
		"%s: WARNING: compiled for %s release %s; this is %s.\n",
		Pn, d, er ? er : "UNKNOWN", r);
#endif	/* defined(HASKERNIDCK) */

}


/*
 * gather_proc_info() -- gather process information
 */

void
gather_proc_info()
{
	struct dirent *dp;
	KA_T f;
	int i, j, pgrp, pid;
	MALLOC_S len;
	static struct proc *p = (struct proc *)NULL;
	int pfd = -1;
	static DIR *pfsfd = (DIR *)NULL;
	struct pid pids;
	static char *pn = (char *)NULL;
	static MALLOC_S pna = 0;
	static char *pnp;
	static MALLOC_S pnpl;
	static MALLOC_S pnsp;
	short pss, sf;
	static struct user *u = (struct user *)NULL;
	uid_t uid;
/*
 * Allocate proc structure buffer.
 */
	if (!p) {
	    if (!(p = (struct proc *)malloc(sizeof(struct proc)))) {
		(void) fprintf(stderr,
		    "%s: can't allocate %d bytes for proc structure\n",
		    Pn, sizeof(struct proc));
		Exit(1);
	    }
	}
/*
 * Allocate a properly aligned user structure buffer.
 */
	if (!u) {
	    i = sysconf(_SC_PAGESIZE);
	    if (!(u = (struct user *)memalign(i, (sizeof(struct user) + i)))) {
		(void) fprintf(stderr,
		    "%s: can't memalign user structure buffer\n", Pn);
		Exit(1);
	    }
	}
/*
 * Prepare the path name buffer for reading individual /proc/<number> files.
 */
	if (!pn) {
	    pnpl = strlen(PROCFS) + 1;
	    pna = (MALLOC_S)(pnpl + 1 + 10);
	    if (!(pn = (char *)malloc(pna))) {
		(void) fprintf(stderr,
		    "%s: can't allocate %d bytes for %s name buffer\n",
		    Pn, pna, PROCFS);
		Exit(1);
	    }
	    (void) sprintf(pn, "%s/", PROCFS);
	    pnp = &pn[pnpl];
	    pnsp = (MALLOC_S)(pna - pnpl - 1);
	}

#if     defined(HASNCACHE)
/*
 * Read the kernel name cache.
 */
	ncache_load();
#endif  /* defined(HASNCACHE) */

/*
 * Open access to the /proc directory, or position it to its beginning.
 */
	if (!pfsfd) {
	    if (!(pfsfd = opendir(PROCFS))) {
		(void) fprintf(stderr, "%s: can't open %s\n", Pn, PROCFS);
		Exit(1);
	    }
	} else
	    (void) rewinddir(pfsfd);
/*
 * Examine proc structures and their associated information.
 */
	while ((dp = readdir(pfsfd))) {

	/*
	 * Process next procfs directory entry.
	 */
	    if (pfd >= 0) {
		(void) close(pfd);
		pfd = -1;
	    }
	    for (i = len = 0; dp->d_name[i]; i++) {
		if (dp->d_name[i] < '0' || dp->d_name[i] > '9')
		    break;
		len++;
	    }
	    if (dp->d_name[i] || !len)
		continue;
	    if (len > pnsp) {

	    /*
	     * Expand the directory entry buffer.
	     */
		pna = (MALLOC_S)(pnpl + len + 10 + 1);
		if (!(pn = (char *)realloc(pn, pna))) {
		    (void) fprintf(stderr,
			"%s: can't realloc %d bytes for %s path name buffer\n",
			Pn, pna, PROCFS);
		    Exit(1);
		}
		pnp = &pn[pnpl];
		pnsp = (MALLOC_S)(pna - pnpl - 1);
	    }
	    (void) strcpy(pnp, dp->d_name);
	    if ((pfd = open(pn, O_RDONLY, 0)) < 0)
		continue;
	    if (ioctl(pfd, PIOCGETPR, p) < 0)
		continue;
	    if (p->p_stat == 0 || p->p_stat == SZOMB

#if	defined(RELIANTV) && (defined(RELIANTRMP) || RELIANTV>=54400)
	    ||  p->p_flag & SSYS
#endif	/* defined(RELIANTV) && (defined(RELIANTRMP) || RELIANTV>=54400) */

	    )
		continue;
	/*
	 * Get Process ID, Process group ID, and User ID.
	 */
	    if (!p->p_pidp
	    ||  kread((KA_T)p->p_pidp, (char *)&pids, sizeof(pids)))
		continue;
	    pid = (int)pids.pid_id;
	    if (!p->p_pgidp
	    ||  kread((KA_T)p->p_pgidp, (char *)&pids, sizeof(pids)))
		continue;
	    pgrp = (int)pids.pid_id;
	    uid = (uid_t)p->p_uid;
	    if (is_proc_excl(pid, pgrp, (UID_ARG)uid, &pss, &sf))
		continue;
	/*
	 * Get the user area associated with the process.
	 */
	    if (ioctl(pfd, PIOCGETU, u) < 0)
		continue;
	/*
	 * Allocate a local process structure.
	 */
	    if (is_cmd_excl(p->p_comm, &pss, &sf))
		continue;
	    alloc_lproc(pid, pgrp, (int)p->p_ppid, (UID_ARG)uid, p->p_comm,
		(int)pss, (int)sf);
	    Plf = (struct lfile *)NULL;
	/*
	 * Save current working directory information.
	 */
	    if (u->u_cdir) {
		alloc_lfile(CWD, -1);
		process_node((KA_T)u->u_cdir);
		if (Lf->sf)
		    link_lfile();
	    }
	/*
	 * Save root directory information.
	 */
	    if (u->u_rdir) {
		alloc_lfile(RTD, -1);
		process_node((KA_T)u->u_rdir);
		if (Lf->sf)
		    link_lfile();
	    }
	/*
	 * Print information on the text file.
	 */
	    if (Sgvnops && p->p_as)
		process_text((KA_T)p->p_as);
	/*
	 * Save information on file descriptors.
	 */
	    for (i = 0, j = 0; i < u->u_nofiles; i++) {
		if (++j > NFPCHUNK) {
		    if (!u->u_flist.uf_next)
			break;
		    if (kread((KA_T)u->u_flist.uf_next, (char *)&u->u_flist,
		    	      sizeof(struct ufchunk)))
			break;
		    j = 1;
		}
		f = (KA_T)u->u_flist.uf_ofile[j-1];
		if (f) {
		    alloc_lfile(NULL, i);
		    process_file(f);
		    if (Lf->sf) {

#if	defined(HASFSTRUCT)
		    if (Fsv & FSV_FG)
			Lf->pof = (long)u->u_flist.uf_pofile[j-1];
#endif	/* defined(HASFSTRUCT) */

			link_lfile();
		    }
		}
	    }
	/*
	 * Examine results.
	 */
	    if (examine_lproc())
		break;
	}
/*
 * Close directory accesses before returning.
 */
	if (pfd >= 0) {
	    (void) close(pfd);
	    pfd = -1;
	}
	if (!RptTm && pfsfd) {
	    (void) closedir(pfsfd);
	    pfsfd = (DIR *)NULL;
	}
}


#if	defined(RELIANTV)
/*
 * get_cdevs() - get major device numbers from cdevsw[]
 */

static void
get_cdevs()
{
	char buf[128];
	struct cdevsw *c, *cd;
	int i, sz;
	MALLOC_S len;
	KA_T v;
/*
 * Read the cdevsw[] size and allocate temporary space for it.
 */
	if (get_Nl_value("ncdev", Drive_Nl, &v) < 0 || !v
	||  kread(v, (char *)&sz, sizeof(sz)) || !sz)
	    return;
	len = (MALLOC_S)(sz * sizeof(struct cdevsw));
	if (!(cd = (struct cdevsw *)malloc(len))) {
	    (void) fprintf(stderr, "%s: can't allocate %d bytes for cdevsw\n",
		Pn);
	    Exit(1);
	}
/*
 * Read the cdevsw[] from kernel memory.
 */
	if (get_Nl_value("cdev", Drive_Nl, &v) < 0 || !v
	||  kread(v, (char *)cd, (int)len)) {
	    (void) free((MALLOC_P *)cd);
	    return;
	}
/*
 * Scan the cdevsw[], reading it's names, looking for:
 *	"clone",
 *	"mipc".
 *
 * Record their cdevsw[] indexes (i.e., major device numbers).
 */
	len = sizeof(buf) - 1;
	buf[len] = '\0';
	for (c = cd, i = 0; i < sz; c++, i++) {
	    if (!c->d_name || kread((KA_T)c->d_name, buf, len))
		continue;
	    if (strcmp(buf, "clone") == 0) {
		CloneMaj = i;
		HaveCloneMaj = 1;
		continue;
	    }

#if	defined(RELIANTV) && defined(HASMIPCH)
	    if (strcmp(buf, "mipc") == 0) {
		MipcMaj = i;
		HaveMipcMaj = 1;
	    }
#endif	/* defined(RELIANTV) && defined(HASMIPCH) */

	}
	(void) free((MALLOC_P *)cd);
}
#endif	/* defined(RELIANTV) */


/*
 * get_kernel_access() - get access to kernel memory
 */

static void
get_kernel_access()
{
	KA_T v;
/*
 * Check kernel version.
 */
	(void) ckkv(

#if	defined(RELIANTV)
		    "Reliant",
#else	/* !defined(RELIANTV) */
		    "DC/OSx",
#endif	/* defined(RELIANTV) */

		    LSOF_VSTR, (char *)NULL, (char *)NULL);

#if	defined(WILLDROPGID)
/*
 * If kernel memory isn't coming from KMEM, drop setgid permission
 * before attempting to open the (Memory) file.
 */
	if (Memory)
	    (void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the non-KMEM memory file is readable.
 */
	if (Memory && !is_readable(Memory, 1))
	    Exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Open kernel memory access.
 */
	if ((Kd = open(Memory ? Memory : KMEM, O_RDONLY, 0)) < 0) {
	    (void) fprintf(stderr, "%s: can't open %s: %s\n", Pn,
		Memory ? Memory : KMEM, strerror(errno));
	    Exit(1);
	}

#if	defined(WILLDROPGID)
/*
 * Drop setgid permission, if necessary.
 */
	if (!Memory)
	    (void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the name list file is readable.
 */
	if (Nmlst && !is_readable(Nmlst, 1))
	    Exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Access kernel symbols and values.
 */
	(void) build_Nl(Drive_Nl);
        if (nlist(Nmlst ? Nmlst : N_UNIX, Nl) < 0) {
	    (void) fprintf(stderr, "%s: can't read kernel name list from %s\n",
		Pn, Nmlst ? Nmlst : N_UNIX);
	    Exit(1);
	}

#if	defined(RELIANTV)
	(void) get_cdevs();
#else	/* !defined(RELIANTV) */
	if (get_Nl_value("clmaj", Drive_Nl, &v) >= 0 && v
	&&  kread((KA_T)v, (char *)&CloneMaj, sizeof(CloneMaj)) == 0)
	    HaveCloneMaj = 1;
#endif	/* defined(RELIANTV) */

	if (get_Nl_value("sgvnops", Drive_Nl, &Sgvnops) < 0 && !Sgvnops)
	    Sgvnops = (KA_T)0;
}


/*
 * initialize() - perform all initialization
 */

void
initialize()
{
	get_kernel_access();
	readfsinfo();
}


/*
 * kread() - read from kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel memory address */
	char *buf;			/* buffer to receive data */
	READLEN_T len;			/* length to read */
{
	KA_T a = addr;
	char *b = buf;
	int br, b2r, tb;

	for (tb = 0; tb < len; a += (KA_T)br, b += br, tb += br) {

	/*
	 * Break reads of /dev/kmem into small enough lengths that none
	 * causes the crossing of an 0xfff to 0x1000 boundary.
	 */
	    b2r = len - tb;
	    if (((a & 0xfff) + b2r) > 0x1000)
		b2r = 0x1000 - (a & 0xfff);
	    if (lseek(Kd, (off_t)a, L_SET) == (off_t)-1)
		return(-1);
	    if ((br = read(Kd, b, b2r)) != b2r)
		return(1);
	}
	return(0);
}


/*
 * process_text() - process text access information
 */

static void
process_text(pa)
	KA_T pa;			/* kernel address space description
					 * pointer */
{
	struct as as;
	int i, j, k, l;
	struct seg s;
	struct segvn_data vn;
	KA_T v[MAXSEGS];
/*
 * Get address space description.
 */
	if (kread(pa, (char *)&as, sizeof(as)))
	    return;
/*
 * Loop through the segments.  The loop should stop when the segment
 * pointer returns to its starting point, but just in case, it's stopped
 * when MAXSEGS have been recorded or 2*MAXSEGS have been examined.
 */
	s.s_next = as.a_segs;
	for (i = j = k = 0; i < MAXSEGS && j < 2*MAXSEGS; j++) {
	    if (!s.s_next
	    ||  kread((KA_T)s.s_next, (char *)&s, sizeof(s)))
		break;
	    if (Sgvnops == (KA_T)s.s_ops && s.s_data) {
		if (kread((KA_T)s.s_data, (char *)&vn, sizeof(vn)))
		    break;
		if (vn.vp) {

		/*
		 * This is a virtual node segment.
		 *
		 * If its vnode pointer has not been seen already,
		 * print its information.
		 */
		    for (l = 0; l < k; l++) {
			if (v[l] == (KA_T)vn.vp)
			    break;
		    }
		    if (l >= k) {
			alloc_lfile(" txt", -1);
			process_node((KA_T)vn.vp);
			if (Lf->sf) {
			    link_lfile();
			    i++;
			}
		    }
		    v[k++] = (KA_T)vn.vp;
		}
	    }
	}
}


/*
 * readfsinfo() - read file system information
 */

static void
readfsinfo()
{
	char buf[FSTYPSZ+1];
	int i;

	if ((Fsinfomax = sysfs(GETNFSTYP)) == -1) {
	    (void) fprintf(stderr, "%s: sysfs(GETNFSTYP) error: %s\n",
		Pn, strerror(errno));
	    Exit(1);
	} 
	if (Fsinfomax == 0)
	    return;
	if (!(Fsinfo = (char **)malloc((MALLOC_S)(Fsinfomax * sizeof(char *)))))
	{
	    (void) fprintf(stderr, "%s: no space for sysfs info\n", Pn);
	    Exit(1);
	}
	for (i = 1; i <= Fsinfomax; i++) {
	    if (sysfs(GETFSTYP, i, buf) == -1) {
		(void) fprintf(stderr, "%s: sysfs(GETFSTYP) error: %s\n",
		    Pn, strerror(errno));
		Exit(1);
	    }
	    buf[FSTYPSZ] = '\0';
	    if (!(Fsinfo[i-1] = (char *)malloc((MALLOC_S)(strlen(buf) + 1)))) {
		(void) fprintf(stderr,
		    "%s: no space for file system entry %s\n", Pn, buf);
		Exit(1);
	    }
	    (void) strcpy(Fsinfo[i-1], buf);
	}
}
