/*
 * "$Id: html.cpp,v 1.17 1999/01/04 17:45:23 mike Exp $"
 *
 *   HTML exporting functions for HTMLDOC, a HTML document processing program.
 *
 *   Copyright 1997-1999 by Michael Sweet.
 *
 *   HTMLDOC is distributed under the terms of the GNU General Public License
 *   which is described in the file "COPYING-2.0".
 *
 * Contents:
 *
 *   html_export()   - Export to HTML...
 *   count_headers() - Count the number of first and second level headings
 *                     in a document tree.
 *   write_header()  - Output the standard "header" for a HTML file.
 *   write_footer()  - Output the standard "footer" for a HTML file.
 *   write_all()     - Write all markup text for the given tree.
 *   get_title()     - Get the title string for the given document...
 *   add_link()      - Add a named link...
 *   find_link()     - Find a named link...
 *   compare_links() - Compare two named links.
 */

/*
 * Include necessary headers.
 */

#include "htmldoc.h"


/*
 * Named link structure...
 */

typedef struct
{
  int		chapter;	/* Chapter for link */
  uchar		name[124];	/* Reference name */
} link_t;


/*
 * Local globals...
 */


static int	chapter,
		heading,
		num_headings;
static uchar	headings[MAX_HEADINGS][32];
static int	num_links;
static link_t	links[MAX_LINKS];


/*
 * Local functions...
 */

static void	count_headers(tree_t *t);
static FILE	*write_header(uchar *title, uchar *author, uchar *copyright,
		              uchar *docnumber, tree_t *t);
static void	write_footer(FILE **out, tree_t *t);
static int	write_all(tree_t *t, uchar *title, uchar *author,
		          uchar *copyright, uchar *docnumber, FILE **out,
			  int col);
static uchar	*get_title(tree_t *doc);

static void	add_link(uchar *name);
static link_t	*find_link(uchar *name);
static int	compare_links(link_t *n1, link_t *n2);


/*
 * 'html_export()' - Export to HTML...
 */

int				/* O - 0 = success, -1 = failure */
html_export(tree_t *document,	/* I - Document to export */
            tree_t *toc)	/* I - Table of contents for document */
{
  uchar	*title,			/* Title text */
	*author,		/* Author name */
	*copyright,		/* Copyright text */
	*docnumber;		/* Document number */
  FILE	*out;			/* Output file */


  if (OutputFiles && LogoImage[0] != '\0')
    image_copy(LogoImage, OutputPath);

  if (OutputFiles && TitleImage[0] != '\0' && TitlePage)
    image_copy(TitleImage, OutputPath);

  heading      = 0;
  num_headings = 0;
  num_links    = 0;
  title        = get_title(document);
  author       = htmlGetMeta(document, (uchar *)"author");
  copyright    = htmlGetMeta(document, (uchar *)"copyright");
  docnumber    = htmlGetMeta(document, (uchar *)"docnumber");

  chapter = 0;
  count_headers(document);

  out = NULL;
  if (TitlePage)
  {
    chapter = -1;
    out = write_header(title, author, copyright, docnumber, NULL);
    if (OutputFiles)
      write_footer(&out, NULL);
    else
      fclose(out);
  };

  out     = NULL;
  chapter = 0;
  heading = -1;
  write_all(toc, title, author, copyright, docnumber, &out, 0);
  if (out != NULL)
    write_footer(&out, NULL);

  out = NULL;
  write_all(document, title, author, copyright, docnumber, &out, 0);
  if (out != NULL)
    write_footer(&out, NULL);

  if (title != NULL)
    free(title);

  return (0);
}


/*
 * 'count_headers()' - Count the number of first and second level headings
 *                     in a document tree.
 */

static void
count_headers(tree_t *t)	/* I - Document tree */
{
  uchar	*link;			/* Link name */


  while (t != NULL)
  {
    if (t->markup == MARKUP_H1 || t->markup == MARKUP_H2)
    {
      if (t->child != NULL &&
          (link = htmlGetVariable(t->child, (uchar *)"NAME")) != NULL)
        strcpy((char *)headings[num_headings], (char *)link);

      num_headings ++;
    };

    if (t->markup == MARKUP_H1)
      chapter ++;

    if (t->markup == MARKUP_A)
      link = htmlGetVariable(t, (uchar *)"NAME");
    else
      link = htmlGetVariable(t, (uchar *)"ID");

    if (link != NULL)
      add_link(link);

    count_headers(t->child);

    t = t->next;
  };
}


/*
 * 'write_header()' - Output the standard "header" for a HTML file.
 */

static FILE *			/* O - Output file pointer */
write_header(uchar  *title,	/* I - Title for document */
             uchar  *author,	/* I - Author for document */
             uchar  *copyright,	/* I - Copyright for document */
             uchar  *docnumber,	/* I - ID number for document */
             tree_t *t)		/* I - Tree entry */
{
  char		filename[255];	/* Output filename */
  FILE		*out;		/* Output file */


  if (OutputFiles)
  {
    if (chapter <= 0)
    {
      if (TitlePage && chapter == 0)
        sprintf(filename, "%s/toc.html", OutputPath);
      else
        sprintf(filename, "%s/index.html", OutputPath);
    }
    else
      sprintf(filename, "%s/doc%d.html", OutputPath, chapter);

    if (t == NULL || t->markup == MARKUP_H1)
      out = fopen(filename, "w");
    else
      out = fopen(filename, "a");
  }
  else if (OutputPath[0] != '\0')
    out = fopen(OutputPath,
                (chapter == -1 || (chapter == 0 && !TitlePage)) ? "w" : "a");
  else
    out = stdout;

  if (out == NULL)
    return (NULL); /**** NEED TO ADD ERROR MESSAGE HOOK ****/

  if ((OutputFiles && (t == NULL || t->markup == MARKUP_H1)) || chapter <= 0)
  {
    if (!TitlePage || chapter == -1)
    {
      fputs("<HTML>\n", out);
      fputs("<HEAD>\n", out);
      fprintf(out, "<TITLE>%s</TITLE>\n", title);
      if (author != NULL)
        fprintf(out, "<META NAME=\"AUTHOR\" CONTENT=\"%s\">\n", author);
      if (copyright != NULL)
        fprintf(out, "<META NAME=\"COPYRIGHT\" CONTENT=\"%s\">\n", copyright);
      if (docnumber != NULL)
        fprintf(out, "<META NAME=\"DOCNUMBER\" CONTENT=\"%s\">\n", docnumber);
      fputs("</HEAD>\n", out);

      if (BodyImage[0] != '\0')
        fprintf(out, "<BODY BACKGROUND=\"%s\">\n", BodyImage);
      else if (BodyColor[0] != '\0')
        fprintf(out, "<BODY BGCOLOR=\"%s\">\n", BodyColor);
      else
        fputs("<BODY>\n", out);
    };

    if (chapter == -1)
    {
      if (OutputFiles)
	fputs("<CENTER><A HREF=toc.html>", out);
      else
	fputs("<CENTER><A HREF=#contents>", out);

      if (TitleImage[0] != '\0')
      {
	if (OutputFiles && strrchr((char *)TitleImage, '/') != NULL)
	  fprintf(out, "<IMG SRC=\"%s\" BORDER=0><BR>\n", strrchr((char *)TitleImage, '/') + 1);
	else
	  fprintf(out, "<IMG SRC=\"%s\" BORDER=0><BR>\n", TitleImage);
      };

      if (title != NULL)
        fprintf(out, "<H1>%s</H1></A><BR>\n", title);
      else
        fputs("</A>\n", out);

      if (docnumber != NULL)
        fprintf(out, "%s<BR>\n", docnumber);

      if (author != NULL)
        fprintf(out, "%s<BR>\n", author);

      if (copyright != NULL)
        fprintf(out, "%s<BR>\n", copyright);

      fputs("</CENTER>\n", out);
    };

    if (chapter >= 0)
      fputs("<TABLE WIDTH=100% BORDER=0 CELLSPACING=0 CELLPADDING=4>\n", out);
  };

  if (chapter >= 0)
  {
    fputs("<TR>\n", out);
    if (LogoImage[0] == '\0' || (t != NULL && t->markup != MARKUP_H1))
    {
      fprintf(out, "<TD VALIGN=TOP ALIGN=CENTER BGCOLOR=%s>\n", BarColor);
      fputs("&nbsp;<BR>\n", out);
    }
    else
    {
      fprintf(out, "<TD VALIGN=TOP ALIGN=CENTER BGCOLOR=%s>\n", BarColor);
      if (OutputFiles && strrchr((char *)LogoImage, '/') != NULL)
	fprintf(out, "<IMG SRC=%s><BR>\n", strrchr((char *)LogoImage, '/') + 1);
      else
	fprintf(out, "<IMG SRC=%s><BR>\n", LogoImage);
    };

    if (OutputFiles)
    {
      if (heading > 0)
	fprintf(out, "<A HREF=%s.html>Contents</A><BR>\n",
        	TitlePage ? "toc" : "index");
      if (heading > 1)
	fprintf(out, "<A HREF=doc%d.html#%s>Previous</A><BR>\n",
        	atoi((char *)headings[heading - 1]), headings[heading - 1]);
      if (heading < num_headings && heading > 0)
	fprintf(out, "<A HREF=doc%d.html#%s>Next</A><BR>\n",
        	atoi((char *)headings[heading + 1]), headings[heading + 1]);
    }
    else
    {
      if (heading > 0)
	fputs("<A HREF=#contents>Contents</A><BR>\n", out);
      if (heading > 1)
	fprintf(out, "<A HREF=#%s>Previous</A><BR>\n", headings[heading - 1]);
      if (heading < num_headings && heading > 0)
	fprintf(out, "<A HREF=#%s>Next</A><BR>\n", headings[heading + 1]);
    };

    fputs("</TD><TD WIDTH=90% VALIGN=TOP>\n", out);
  };

  return (out);
}


/*
 * 'write_footer()' - Output the standard "footer" for a HTML file.
 */

static void
write_footer(FILE   **out,	/* IO - Output file pointer */
             tree_t *t)		/* I - Tree entry */
{
  heading ++;
  if (t == NULL || t->markup == MARKUP_H1)
    chapter ++;

  if (!OutputFiles && t != NULL && t->markup == MARKUP_H1)
    fputs("<HR>\n", *out);

  fputs("</TD>\n", *out);
  fputs("</TR>\n", *out);

  if ((OutputFiles && (t == NULL || t->markup == MARKUP_H1)) ||
      heading == num_headings)
  {
    fputs("</TABLE>\n", *out);
    fputs("</BODY>\n", *out);
    fputs("</HTML>\n", *out);
  };

  if (*out != stdout)
    fclose(*out);

  *out = NULL;
}


/*
 * 'write_all()' - Write all markup text for the given tree.
 */

static int			/* O - Current column */
write_all(tree_t *t,		/* I - Document tree */
          uchar  *title,	/* I - Document title */
          uchar  *author,	/* I - Author for document */
          uchar  *copyright,	/* I - Copyright for document */
          uchar  *docnumber,	/* I - ID number for document */
          FILE   **out,		/* IO - Output file */
          int    col)		/* I - Current column */
{
  int		i;		/* Looping var */
  uchar		*ptr,		/* Pointer to output string */
		*src,		/* Source image */
		newsrc[1024];	/* New source image filename */
  link_t	*link;		/* Link name */


  while (t != NULL)
  {
    if (t->markup == MARKUP_H1 || t->markup == MARKUP_H2)
    {
      if (*out != NULL && chapter > 0)
      {
        if (col > 0)
          putc('\n', *out);

        write_footer(out, t);
      };

      *out = write_header(title, author, copyright, docnumber, t);
      col  = 0;
    };

    if (*out == NULL)
    {
      if (t->child != NULL)
        col = write_all(t->child, title, author, copyright, docnumber, out, col);
      t = t->next;
      continue;
    };

    if (t->markup == MARKUP_NONE)
    {
      if (t->preformatted)
      {
        for (ptr = t->data; *ptr != '\0'; ptr ++)
          if (*ptr > 126)
            fprintf(*out, "&#%d;", *ptr);
          else
            putc(*ptr, *out);

	if (t->data[strlen((char *)t->data) - 1] == '\n')
          col = 0;
	else
          col += strlen((char *)t->data);
      }
      else
      {
	if ((col + strlen((char *)t->data)) > 72 && col > 0)
	{
          putc('\n', *out);
          col = 0;
	};

        for (ptr = t->data; *ptr != '\0'; ptr ++)
          if (*ptr > 126)
            fprintf(*out, "&#%d;", *ptr);
          else
            putc(*ptr, *out);

	col += strlen((char *)t->data);

	if (col > 72)
	{
          putc('\n', *out);
          col = 0;
	};
      };
    }
    else if (t->markup == MARKUP_COMMENT)
      fprintf(*out, "\n<!--%s-->\n", t->data);
    else if (t->markup != MARKUP_AREA &&
             t->markup != MARKUP_HEAD &&
             t->markup != MARKUP_HTML &&
             t->markup != MARKUP_MAP &&
             t->markup != MARKUP_META &&
             t->markup != MARKUP_TITLE &&
             t->markup != MARKUP_BODY)
    {
      switch (t->markup)
      {
        case MARKUP_BR :
        case MARKUP_CENTER :
        case MARKUP_COMMENT :
        case MARKUP_DD :
        case MARKUP_DL :
        case MARKUP_DT :
        case MARKUP_H1 :
        case MARKUP_H2 :
        case MARKUP_H3 :
        case MARKUP_H4 :
        case MARKUP_H5 :
        case MARKUP_H6 :
        case MARKUP_H7 :
        case MARKUP_HR :
        case MARKUP_LI :
        case MARKUP_OL :
        case MARKUP_P :
        case MARKUP_PRE :
        case MARKUP_TABLE :
        case MARKUP_TR :
        case MARKUP_UL :
            if (col > 0)
            {
              putc('\n', *out);
              col = 0;
            };
            break;
      };

      if (t->markup == MARKUP_IMG && OutputFiles &&
          (src = htmlGetVariable(t, (uchar *)"SRC")) != NULL)
      {
       /*
        * Update local images...
        */

        if (strncmp((char *)src, "http:", 5) != 0 &&
            strncmp((char *)src, "file:", 5) != 0 &&
            strncmp((char *)src, "ftp:", 4) != 0 &&
            src[0] != '/')
        {
          image_copy((char *)src, OutputPath);
          if (strrchr((char *)src, '/') != NULL)
          {
            strcpy((char *)newsrc, strrchr((char *)src, '/') + 1);
            htmlSetVariable(t, (uchar *)"SRC", newsrc);
          };
        };
      };

      if (t->markup == MARKUP_A &&
          (src = htmlGetVariable(t, (uchar *)"HREF")) != NULL)
      {
       /*
        * Update local links...
        */

        if (strncmp((char *)src, "http:", 5) != 0 &&
            strncmp((char *)src, "file:", 5) != 0 &&
            strncmp((char *)src, "ftp:", 4) != 0 &&
            strncmp((char *)src, "doc", 3) != 0 &&
            strrchr((char *)src, '#') != NULL)
        {
          strcpy((char *)newsrc, strrchr((char *)src, '#'));
	  link = find_link(newsrc + 1);

          if (OutputFiles && link != NULL)
	    sprintf((char *)newsrc, "doc%d.html#%s", link->chapter,
	            link->name);

          htmlSetVariable(t, (uchar *)"HREF", newsrc);
        };
      };

      col += fprintf(*out, "<%s", _htmlMarkups[t->markup]);
      for (i = 0; i < t->nvars; i ++)
      {
	if (col > 72)
	{
          putc('\n', *out);
          col = 0;
	};

        if (col > 0)
        {
          putc(' ', *out);
          col ++;
        };

	if (t->vars[i].value == NULL)
          col += fprintf(*out, "%s", t->vars[i].name);
	else if (strchr((char *)t->vars[i].value, ' ') != NULL ||
        	 strchr((char *)t->vars[i].value, '\t') != NULL ||
        	 strchr((char *)t->vars[i].value, '\n') != NULL ||
        	 strchr((char *)t->vars[i].value, '\r') != NULL)
          col += fprintf(*out, "%s=\"%s\"", t->vars[i].name, t->vars[i].value);
	else
          col += fprintf(*out, "%s=%s", t->vars[i].name, t->vars[i].value);
      };

      putc('>', *out);
      col ++;

      if (col > 72)
      {
	putc('\n', *out);
	col = 0;
      };

      if (t->child != NULL)
      {
	col = write_all(t->child, title, author, copyright, docnumber, out, col);

	if (col > 72)
	{
	  putc('\n', *out);
	  col = 0;
	};
	
        col += fprintf(*out, "</%s>", _htmlMarkups[t->markup]);
        switch (t->markup)
        {
          case MARKUP_BR :
          case MARKUP_CENTER :
          case MARKUP_COMMENT :
          case MARKUP_DD :
          case MARKUP_DL :
          case MARKUP_DT :
          case MARKUP_H1 :
          case MARKUP_H2 :
          case MARKUP_H3 :
          case MARKUP_H4 :
          case MARKUP_H5 :
          case MARKUP_H6 :
          case MARKUP_H7 :
          case MARKUP_HR :
          case MARKUP_LI :
          case MARKUP_OL :
          case MARKUP_P :
          case MARKUP_PRE :
          case MARKUP_TABLE :
          case MARKUP_TR :
          case MARKUP_UL :
              putc('\n', *out);
              col = 0;
              break;
        };
      };
    }
    else if (t->child != NULL &&
             (t->markup == MARKUP_HTML || t->markup == MARKUP_BODY))
      col = write_all(t->child, title, author, copyright, docnumber, out, col);

    t = t->next;
  };

  return (0);
}


/*
 * 'get_title()' - Get the title string for the given document...
 */

static uchar *		/* O - Title string */
get_title(tree_t *doc)	/* I - Document tree */
{
  uchar	*temp;		/* Temporary pointer to title */


  while (doc != NULL)
  {
    if (doc->markup == MARKUP_TITLE)
      return (htmlGetText(doc->child));
    else if (doc->child != NULL)
      if ((temp = get_title(doc->child)) != NULL)
        return (temp);

    doc = doc->next;
  };

  return (NULL);
}


/*
 * 'add_link()' - Add a named link...
 */

static void
add_link(uchar *name)	/* I - Name of link */
{
  link_t	*temp;	/* New name */


  if ((temp = find_link(name)) != NULL)
    temp->chapter = chapter;
  else if (num_links < MAX_LINKS)
  {
    temp = links + num_links;
    num_links ++;

    strncpy((char *)temp->name, (char *)name, sizeof(temp->name) - 1);
    temp->name[sizeof(temp->name) - 1] = '\0';
    temp->chapter = chapter;

    if (num_links > 1)
      qsort(links, num_links, sizeof(link_t),
            (int (*)(const void *, const void *))compare_links);
  };
}


/*
 * 'find_link()' - Find a named link...
 */

static link_t *
find_link(uchar *name)	/* I - Name to find */
{
  link_t	key,	/* Search key */
		*match;	/* Matching name entry */


  if (name == NULL || num_links == 0)
    return (NULL);

  strncpy((char *)key.name, (char *)name, sizeof(key.name) - 1);
  key.name[sizeof(key.name) - 1] = '\0';
  match = (link_t *)bsearch(&key, links, num_links, sizeof(link_t),
                            (int (*)(const void *, const void *))compare_links);

  return (match);
}


/*
 * 'compare_links()' - Compare two named links.
 */

static int			/* O - 0 = equal, -1 or 1 = not equal */
compare_links(link_t *n1,	/* I - First name */
              link_t *n2)	/* I - Second name */
{
  return (strcasecmp((char *)n1->name, (char *)n2->name));
}


/*
 * End of "$Id: html.cpp,v 1.17 1999/01/04 17:45:23 mike Exp $".
 */
