
/*
 * DREADERD/DNS.C - dns resolution and authentication task
 *
 *	DNS authenticator for new connections.
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 */

#include "defs.h"

#define MAXKPAIR 32

Prototype void DnsTask(int fd, const char *id);
Prototype void DnsTest(DnsReq *dreq, DnsRes *dres, const char *fqdn, char **ary, const char *ipname);

int TestDnsNames(const char *wildHost, const char *name, struct sockaddr_in *rsin, int testWild, int testFwd);
int TestForwardLookup(const char *hostName, const struct sockaddr_in *rsin);
void GetAuthUser(const struct sockaddr_in *lsin, const struct sockaddr_in *rsin, char *ubuf, int ulen);
int readExactly(int fd, void *buf, int bytes);
void sigSegVDNS(int sigNo);
int readtoeof(int fd, void *buf, int bytes, int secs);
int cidrMatch (const char *c_line, const struct in_addr *c_host);

int DFd = -1;
MemPool	*DnsMemPool;

void
DnsTask(int fd, const char *id)
{
    /*
     * read remote address to resolve.  Note: fd is not set to non-blocking
     */
    DnsReq dreq;
    int n;

    DFd = fd;

    if (CoreDebugOpt)
        signal(SIGSEGV, sigSegVDNS);

    /*
     * Make dummy call to lock buffer so we do not inefficiently free and
     * reallocate it for every DNS request.
     */
    (void)zalloc(&DnsMemPool, 8);

    /*
     * Loop on DNS requests, resolve them, and respond.
     */

    while ((n = readExactly(fd, &dreq, sizeof(dreq))) == sizeof(dreq)) {
	struct hostent *he;
	char *ipname;
	DnsRes dres;

	bzero(&dres, sizeof(dres));

	stprintf("reverse auth %s", inet_ntoa(dreq.dr_RSin.sin_addr));

	if (dreq.performident)
	    GetAuthUser(&dreq.dr_LSin, &dreq.dr_RSin, dres.dr_User, sizeof(dres.dr_User));

	stprintf("dns lookup %s", inet_ntoa(dreq.dr_RSin.sin_addr));

	dres.dr_Addr = dreq.dr_RSin.sin_addr;
	dres.dr_Port = dreq.dr_RSin.sin_port;
	dres.dr_Flags &= ~DF_HOSTISIP;

	he = gethostbyaddr(
	    (char *)&dreq.dr_RSin.sin_addr,
	    sizeof(dreq.dr_RSin.sin_addr),
	    AF_INET
	);

	/*
	 * NOTE: we have to copy the fields returned by gethostbyaddr()
	 * because we will be doing other gethost*() calls in our tests and
	 * the original fields may get replaced out from under us.
	 */

	ipname = zallocStr(&DnsMemPool, inet_ntoa(dreq.dr_RSin.sin_addr));

	if (he != NULL) {
	    char *hname;
	    char **haliases;
	    int nary;
	    int i;

	    for (nary = 0; he->h_aliases && he->h_aliases[nary]; ++nary)
		;

	    /*
	     * allocate
	     */
	    hname = zallocStr(&DnsMemPool, he->h_name);
	    haliases = zalloc(&DnsMemPool, sizeof(char *) * (nary + 1));
	    for (i = 0; i < nary; ++i) 
		haliases[i] = zallocStr(&DnsMemPool, he->h_aliases[i]);

	    /*
	     * call test
	     */
	    DnsTest(&dreq, &dres, hname, haliases, ipname);

	    /*
	     * free
	     */
	    if (DebugOpt)
		printf("X1 %s\n", hname);
	    zfreeStr(&DnsMemPool, &hname);
	    if (DebugOpt)
		printf("X2\n");
	    for (i = 0; i < nary; ++i)
		zfreeStr(&DnsMemPool, &haliases[i]);
	    if (DebugOpt)
		printf("X3\n");
	    zfree(&DnsMemPool, haliases, sizeof(char *) * (nary + 1));
	    if (DebugOpt)
		printf("X4\n");
	} else {
	    DnsTest(&dreq, &dres, NULL, NULL, ipname);
	    dres.dr_Flags |= DF_HOSTISIP;
	}
	write(fd, &dres, sizeof(dres));
	zfreeStr(&DnsMemPool, &ipname);
	stprintf("dns idle");
    }
    logit(LOG_NOTICE, "DNS process exiting n=%d/%d", n, sizeof(dreq));
}

/*
 * DnsTest() - test access file entry against host.
 *
 *	fqdn	reverse lookup of name, if known, or NULL
 *	ary	aliases, if known, or NULL
 *	ipname	name of host as dotted quad
 */

void 
DnsTest(DnsReq *dreq, DnsRes *dres, const char *fqdn, char **ary, const char *ipname)
{
    FILE *fi;
    int ok = 0;

    /*
     * open dreader.hosts
     */

    if ((fi = fopen(PatLibExpand(DReaderHostsPat), "r")) != NULL) {
	char buf[256];
	char *flags = "";

	while (fgets(buf, sizeof(buf), fi) != NULL) {
	    int needauth = 0;
	    int testIp = -1;	/* -1=no, 0=yes,nowild, 1=yes */
	    int testFqdn = -1;	/* -1=no, 0=yes,nowild, 1=yes */
	    char *wildHost;

	    wildHost = strtok(buf, " \t\n");
	    if (wildHost == NULL || wildHost[0] == '#' || wildHost[0] == '\n')
		continue;
	    flags = strtok(NULL, " \t\n");
	    if (flags == NULL || flags[0] == '#') {
		flags = "";
	    }

	    /*
	     * Determine which IP & FQDN tests will be done.
	     */

	    if (strncmp(wildHost, "IP:", 3) == 0) {
		if (ipname)
		    testIp = 1;
		wildHost += 3;
	    } else if (strncmp(wildHost, "FQDN:", 5) == 0) {
		if (fqdn)
		    testFqdn = 1;
		wildHost += 5;
	    } else {
		if (ipname)
		    testIp = 1;
		if (fqdn && wildHost) {
		    if (isdigit(wildHost[0])) {
			testFqdn = 0;
		    } else {
			testFqdn = 1;
		    }
		}
	    }

	    /*
	     * Extract auth information now because
	     * we may use strtok() later and destroy our
	     * current strtok context.
	     */

	    if (strchr(flags, 'a') != NULL) {
		char *kt;
		char *user = NULL;
		char *pass = NULL;

		while ((kt = strtok(NULL, " \t\n")) != NULL) {
		    if (strncmp(kt, "user=", 5) == 0) {
			user = kt + 5;
		    } else if (strncmp(kt, "pass=", 5) == 0) {
			if (kt[5])
			    pass = kt + 5;
		    }
		}
		if (DebugOpt)
		    printf("USER %s PASS %s DR_USER %s\n", user, pass, dres->dr_User);
		if (user == NULL) {
		    needauth = 1;
		} else if (strcasecmp(dres->dr_User, user) == 0) {
		    /*
		     * reverse-auth auto authorization if password is blank
		     */
		    if (pass && strcmp(dreq->dr_AuthPass, pass) != 0)
			needauth = 1;
		} else if (strcasecmp(dreq->dr_AuthUser, user) == 0) {
		    /*
		     * password required
		     */
		    if (pass == NULL || strcmp(dreq->dr_AuthPass, pass) != 0) {
			needauth = 1;
		    }
		} else {
		    needauth = 1;
		}
	    }

	    if (DebugOpt)
		printf("needauth is %d\n", needauth);

	    /*
	     * note: strtok used by subroutines, invalid after this point.
	     */

	    /*
	     * Test entry against IP address
	     */
	    if (testIp >= 0 && TestDnsNames(wildHost, ipname, &dreq->dr_RSin, testIp, 0) == 0) {
		snprintf(dres->dr_Host, sizeof(dres->dr_Host), "%s", ipname);
		ok = 1;
	    }

	    /*
	     * Test entry against FQDN (from reverse lookup)
	     */
	    if (testFqdn >= 0 && ok == 0 && TestDnsNames(wildHost, fqdn, &dreq->dr_RSin, testFqdn, 1) == 0) {
		snprintf(dres->dr_Host, sizeof(dres->dr_Host), "%s", fqdn);
		ok = 1;
	    }

	    /*
	     * Save the current wildHost for later connection limiting
	     */
	    snprintf(dres->dr_Match, sizeof(dres->dr_Match), "%s", wildHost);

	    /*
	     * If we still do not have a match, test entry against wildHost
	     * aliases (but only if testFqdn is valid)
	     */

	    if (testFqdn >= 0 && ok == 0 && ary) {
		int i;

		for (i = 0; ary[i]; ++i)  {
		    if (TestDnsNames(wildHost, ary[i], &dreq->dr_RSin, testFqdn, 1) == 0) {
			snprintf(dres->dr_Host, sizeof(dres->dr_Host), "%s", ary[i]);
			ok = 1;
			break;
		    }
		}
	    }

	    if (ok) {
		if (needauth)
		    dres->dr_Flags |= DF_AUTHREQUIRED;
		break;
	    }
	}
	fclose(fi);

	/*
	 * munge dr_Host if it's illegal
	 */
	SanitizeString(dres->dr_Host);

	/*
	 * deal with flags
	 */

	if (ok) {
	    char c;

	    while ((c = *flags) != 0) {
		switch(c) {
		case 'a':
		    /* already dealt with */
		    break;
		case 'q':
		    dres->dr_Flags |= DF_QUIET;
		    break;
		case 'f':
		    dres->dr_Flags |= DF_FEED;
		    break;
		case 'r':
		    dres->dr_Flags |= DF_READ;
		    break;
		case 'p':
		    dres->dr_Flags |= DF_POST;
		    break;
		case 'w':
		    dres->dr_Flags |= DF_POST;
		    break;
		case 'l':
		    dres->dr_PerArtRateLim = strtol(flags + 1, NULL, 0);
		    break;
		case 'L':
		    dres->dr_GlobalRateLim = strtol(flags + 1, NULL, 0);
		    break;
		case 'h':
		    dres->dr_MaxFromHost = strtol(flags + 1, NULL, 0);
		    break;
		case 'H':
		    dres->dr_MaxFromMatch = strtol(flags + 1, NULL, 0);
		    break;
		case 'u':
		    dres->dr_MaxFromUserHost = strtol(flags + 1, NULL, 0);
		    break;
		case 's':
		    dres->dr_Flags |= DF_FETCHSERVER;
		    dres->dr_FetchPri = strtol(flags + 1, NULL, 0);
		    break;
		case 'o':
		    dres->dr_Flags |= DF_PUSHSERVER;
		    dres->dr_PushPri = strtol(flags + 1, NULL, 0);
		    break;
		case 'g':
		    dres->dr_Flags |= DF_GROUPID;
		    dres->dr_GroupId = strtol(flags + 1, NULL, 0);
		    break;
		default:
		    /*
		     * number or unknown flag ignored (unless number retrieved
		     * above, in which case we skip it here).
		     */
		    break;
		}
		++flags;
	    }
	}
    } else {
	logit(LOG_NOTICE, "connect: file %s not found", PatLibExpand(DReaderHostsPat));
    }
    dres->dr_Code = ok;
}

int
TestDnsNames(const char *wildHost, const char *name, struct sockaddr_in *rsin, int testWild, int testFwd)
{
    int r = -1;
    if (strchr(wildHost, '/') != NULL && cidrMatch(wildHost, &rsin->sin_addr)) { 
	if (testFwd == 0 || TestForwardLookup(name, rsin) == 0);
	r = 0;
    } else if (strchr(wildHost, '?') == NULL && strchr(wildHost, '*') == NULL) {
	if (testWild >= 0 && strcasecmp(wildHost, name) == 0) {
	    if (testFwd == 0 || TestForwardLookup(name, rsin) == 0) {
		r = 0;
	    }
	}
    } else if (testWild > 0 && WildCaseCmp(wildHost, name) == 0) {
	if (testFwd == 0 || TestForwardLookup(name, rsin) == 0) {
	    r = 0;
	}
    }
    return(r);
}

/*
 * TestForwardLookup(). 
 */

int
TestForwardLookup(const char *hostName, const struct sockaddr_in *rsin)
{
    struct hostent *he;
    int r = -1;

    stprintf("forward auth %s", hostName);
    if ((he = gethostbyname(hostName)) != NULL) {
	int i;

	for (i = 0; he->h_addr_list && he->h_addr_list[i]; ++i) {
	    const struct in_addr *haddr = (const void *)he->h_addr_list[i];
	    if (rsin->sin_addr.s_addr == haddr->s_addr) {
		r = 0;
		break;
	    }
	}
	if (he->h_addr_list[i] == NULL)
	    logit(LOG_NOTICE, "DNS Fwd/Rvs mismatch: %s/%s", hostName, inet_ntoa(rsin->sin_addr));
    } else {
	logit(LOG_NOTICE, "DNS Fwd/Rvs mismatch: lookup of %s failed", hostName);
    }
    return(r);
}

/*
 * GetAuthUser() - authenticate the remote username by connecting to port
 *		   113 and requesting the user id of the remote port.  This
 *		   is used by X-Trace:
 */

void
GetAuthUser(const struct sockaddr_in *plsin, const struct sockaddr_in *prsin, char *ubuf, int ulen)
{
    int cfd;
    int lport;
    int rport;
    int n;
    char buf[256];
    struct sockaddr_in lsin = *plsin;
    struct sockaddr_in rsin = *prsin;

    rport = ntohs(rsin.sin_port);
    lport = ntohs(lsin.sin_port);

    lsin.sin_port = 0;
    lsin.sin_family = AF_INET;

    if ((cfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        return;
    }
    if (bind(cfd, (struct sockaddr *)&lsin, sizeof(lsin)) < 0) {
	perror("bind");
        close(cfd);
        return;
    }
    rsin.sin_port = htons(113);
    rsin.sin_family = AF_INET;

    /*
     * Do asynchronous connection with 10 second timeout.  The timeout is
     * necessary to deal with broken clients or firewalls.
     */

    fcntl(cfd, F_SETFL, O_NONBLOCK);
    errno = 0;
    if (connect(cfd, (struct sockaddr *)&rsin, sizeof(rsin)) < 0) {
	fd_set wfds;
	struct timeval tv = { 10, 0 };

	if (errno != EINPROGRESS) {
	    close(cfd);
	    return;
	}
	FD_ZERO(&wfds);
	FD_SET(cfd, &wfds);
	if (select(cfd + 1, NULL, &wfds, NULL, &tv) == 0) {
	    close(cfd);
	    cfd = -1;
	}
    }

    /*
     * Send query, interpret response
     *
     * write() may fail if connect() failed to establish a connection.
     */

    if (cfd >= 0) {
	fcntl(cfd, F_SETFL, 0);
	sprintf(buf, "%d, %d\r\n", rport, lport);
	if (DebugOpt)
	    printf("IDENTD COMMAND %s\n", buf);
	if (write(cfd, buf, strlen(buf)) != strlen(buf)) {
	    n = -1;
	} else {
	    fcntl(cfd, F_SETFL, O_NONBLOCK);
	    n = readtoeof(cfd, buf, sizeof(buf) - 1, 10);
	}
    } else {
	n = -1;
    }
    if (n > 0) {
	char *uid;

        buf[n] = 0;

	if (DebugOpt)
	    printf("IDENTD RESPONSE %s\n", buf);

        if ((uid = strstr(buf, "USERID")) != NULL) {
            uid = strchr(uid + 1, ':');
            if (uid) {
                uid = strchr(uid + 1, ':');
                if (uid) {
                    uid = strtok(uid + 1, "\r\n: \t");
                    if (uid && strlen(uid) < ulen) {
                        strcpy(ubuf, uid);
			SanitizeString(ubuf);
                    }
                }
            }
        }
    }
    if (cfd >= 0)
	close(cfd);
}

int
readExactly(int fd, void *buf, int bytes)
{
    int r = 0;

    while (bytes > 0) {
	int n = read(fd, buf, bytes);

	if (n <= 0) {
	    if (n < 0 && r == 0)
		r = -1;
	    break;
	}
	buf = (char *)buf + n;
	bytes -= n;
	r += n;
    }
    return(r);
}

int
readtoeof(int fd, void *buf, int bytes, int secs)
{
    fd_set rfds;
    int r = 0;

    FD_ZERO(&rfds);

    for (;;) {
	struct timeval tv = { 0, 0 };
	int n;

	tv.tv_sec = secs;
	FD_SET(fd, &rfds);

	if (select(fd + 1, &rfds, NULL, NULL, &tv) == 0)
	    break;
	errno = 0;
	n = read(fd, buf, bytes);
	if (n == 0)
	    break;
	if (n < 0) {
	    if (errno == EWOULDBLOCK ||
		errno == EINTR ||
		errno == EAGAIN
	    ) {
		continue;
	    }
	    break;
	}
	r += n;
	buf = (char *)buf + n;
	bytes -= n;
    }
    return(r);
}

void
sigSegVDNS(int sigNo)
{
    if (DFd >= 0)
        close(DFd);
    nice(20);
    for (;;)
        ;
}   
    
int
cidrMatch(const char *c_line, const struct in_addr *c_host)
{
    unsigned long line, host;
    char *p, *q;
    int mask;

    q = strdup(c_line);
    if (q == NULL)
	return(0);
    mask = -1;
    if ((p = strchr(q, '/')) != NULL) {
	mask = atoi(p + 1);
	*p = '\0';
    } else {
	fprintf(stderr, "invalid cidr %s", c_line);
	free(q);
	return(0);
    }
    if ((mask < 0) || (mask > 32)) {
	fprintf(stderr, "invalid mask \"%s\"\n", p + 1);
	free(q);
	return(0);
    }

    if ((line = inet_addr (q)) == 0) {
	fprintf (stderr, "invalid address \"%s\"\n", q);
	free(q);
	return(0);
    }
    free(q);
    line = ntohl(line);

    host = c_host->s_addr;

    line >>= (32 - mask);
    host >>= (32 - mask);

    if (line == host)
	return(1);

    return (0);
}
