/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/
/*
** filestack.c
**
** keep track of files for pcpp
** this isn't really a stack!  we have to keep track of
** pop'd things also
**
*/

# include "lclintMacros.nf"
# include "basic.h"
# include "fileStack.h"

static bool fileStack_oldcontains (fileStack f, fileId fid);

/*@only@*/ fileStack
fileStack_create ()
{
  fileStack f = (fileStack) dmalloc (sizeof (*f));
  
  f->nentries = 0;
  f->nfree = FILESTACKBASESIZE;
  f->oentries = 0;
  f->ofree = FILESTACKBASESIZE;
  
  f->entries = (fentry *) dmalloc (sizeof (*f->entries) * FILESTACKBASESIZE);
  f->oldentries = (fileId *) dmalloc (sizeof (*f->oldentries) * FILESTACKBASESIZE);
  
  return (f);
}

static void
fileStack_grow_entries (fileStack f)
{
  o_fentry *ofe = f->entries;
  int i;

  f->nfree += FILESTACKBASESIZE;
  f->entries = (fentry *) dmalloc (sizeof (*f->entries) * (f->nentries + f->nfree + 1));
  
  for (i = 0; i < f->nentries; i++)
    {
      f->entries[i] = ofe[i];
    }

  sfree (ofe);
}

static void
fileStack_grow_oentries (fileStack f)
{
  fileId *ofe = f->oldentries;
  int i;

  f->ofree += FILESTACKBASESIZE;
  f->oldentries = (fileId *) dmalloc (sizeof (*f->oldentries) * (f->oentries + f->ofree + 1));

  for (i = 0; i < f->oentries; i++)
    {
      f->oldentries[i] = ofe[i];
    }
  sfree (ofe);
}

static /*@only@*/ fentry
make_fentry (/*@null@*/ /*@dependent@*/ FILE *in,
	     /*@null@*/ /*@dependent@*/ FILE *out, 
	     fileId curf, int curline,
	     /*@null@*/ /*@dependent@*/ FILE *def)
{
  fentry fe = (fentry) dmalloc (sizeof (*fe));
  
  fe->fin = in;
  fe->fout = out;
  fe->fdef = def;
  fe->curline = curline;
  fe->curfile = curf;

  return (fe);
}

void
fileStack_push (fileStack f, /*@dependent@*/ FILE *in, 
		/*@dependent@*/ FILE *out, 
		fileId curf, int curline,
		/*@dependent@*/ FILE *def)
{
  if (f->nfree == 0)
    {
      fileStack_grow_entries (f);
    }

  llassert (f->nfree > 0);

  f->nfree--;
  f->entries[f->nentries] = make_fentry (in, out, curf, curline, def);
  f->nentries++;
}

fileId
fileStack_pop (fileStack f, dFILE *in, dFILE *out, dFILE * def)
{
  fentry fe;
  fileId ret;

  f->nentries--;
  f->nfree++;

  llassert (f->nentries >= 0);
  
  /*@-deparrays@*/ fe = f->entries[f->nentries]; /*@=deparrays@*/

  *in = fe->fin;
  *out = fe->fout;
  *def = fe->fdef;

  ret = fe->curfile;

  if (!fileStack_oldcontains (f, ret))
    {
      if (f->ofree <= 0)
	fileStack_grow_oentries (f);

      f->ofree--;
      f->oldentries[f->oentries] = ret;
      f->oentries++;
    }

  sfree (fe);
  return (ret);
}

void
fileStack_pushOld (fileStack f, fileId curf)
{
  if (!fileStack_oldcontains (f, curf))
    {
      if (f->ofree <= 0)
	fileStack_grow_oentries (f);

      f->ofree--;
      f->oldentries[f->oentries] = curf;
      f->oentries++;
    }
}

bool
fileStack_contains (fileStack f, fileId fn)
{
  int i;

  for (i = 0; i < f->nentries; i++)
    {
      if (fileId_equal (f->entries[i]->curfile, fn))
	return TRUE;
    }

  for (i = 0; i < f->oentries; i++)
    {
      if (fileId_equal (f->oldentries[i], fn))
	return TRUE;
    }

  return FALSE;
}

static bool
fileStack_oldcontains (fileStack f, fileId fn)
{
  int i;

  for (i = 0; i < f->oentries; i++)
    {
      if (fileId_equal (f->oldentries[i], fn))
	return TRUE;
    }

  return FALSE;
}

void
fileStack_annihilate (/*@only@*/ fileStack f)
{
  int i;

  for (i = 0; i < f->nentries; i++)
    {
      llcontbug (message ("Warning: still on stack: %d", 
			  /*@access fileId@*/ 
			  (int) f->entries[i]->curfile) /*@noaccess fileId@*/);
      
      sfree (f->entries[i]);
    }
  
  sfree (f->entries);
  sfree (f->oldentries);
  sfree (f);
}

void
fileStack_print (fileStack f)
{
  int i;

  printf ("\nnentries: %d  free: %d\n", f->nentries, f->nfree);

  for (i = 0; i < f->nentries; i++)
    {
      printf ("%d %s\n", i, cstring_toCharsSafe (fileName (f->entries[i]->curfile)));
    }

  printf ("======\n");
  for (i = 0; i < f->oentries; i++)
    {
      printf ("%d %s\n", i, cstring_toCharsSafe (fileName (f->oldentries[i])));
    }
}

