/*
** Copyright (c) 1996 Thorsten Kukuk 
**
** This file is part of the NYS YP Server.
**
** The NYS YP Server is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
** The NYS YP Server is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
** 
** You should have received a copy of the GNU General Public
** License along with the NYS YP Server; see the file COPYING.  If
** not, write to the Free Software Foundation, Inc., 675 Mass Ave,
** Cambridge, MA 02139, USA.
**
** Author: Thorsten Kukuk <kukuk@uni-paderborn.de>
*/

static char rcsid[] = "$Id: ypxfr.c,v 1.0.4.1 1996/07/14 15:10:45 kukuk Exp $";

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "system.h"

#if HAVE_BROKEN_LINUX_LIBC
#define xdr_ypreq_nokey _xdr_ypreq_nokey
#endif

#include <rpc/rpc.h>
#ifdef HAVE_RPC_CLNT_SOC_H
#include <rpc/clnt_soc.h>
#endif
#include "yp.h"
#include <rpcsvc/ypclnt.h>
#include <netdb.h>
#include <fcntl.h>
#include <ctype.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <memory.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <gdbm.h>
#include <unistd.h>
#if defined(HAVE_GETOPT_H) && defined(HAVE_GETOPT_LONG)
#include <getopt.h>
#else
#include <compat/getopt.h>
#endif

#include "version.h"
#include "yp_msg.h"
#include "ypxfr.h"

#ifndef HAVE_GETOPT_LONG
#include <compat/getopt.c>
#include <compat/getopt1.c>
#endif

#ifndef YPMAPDIR
#define YPMAPDIR "/var/yp"
#endif

static char SourceHost[MAXHOSTNAMELEN], *TargetDomain=NULL, *SourceDomain=NULL;
static struct in_addr IpAddress;
static int Force=0, NoClear=0, TaskId=0, ProgramNumber=0,
	PortNumber=0;
static GDBM_FILE dbm;

int debug_flag=0;

/* NetBSD has a different prototype in struct ypall_callback */
#if defined(__NetBSD__)
static int ypxfr_foreach(u_long status, char *key, int keylen, 
			 char *val, int vallen, void *data)
#else
static int ypxfr_foreach(int status, char *key, int keylen, 
			 char *val, int vallen, int *data)
#endif
{
  datum outKey, outData;

  if(debug_flag>1)
    yp_msg("ypxfr_foreach: key=%.*s, val=%.*s\n",keylen, key, vallen, val);
  
  if (status==YP_NOMORE)
    return 0;

  if (status!=YP_TRUE) 
    {
      int s=ypprot_err(status);
      yp_msg("%s\n", yperr_string(s));
      return 1;
    }
	
  outKey.dptr=keylen?key:""; outKey.dsize=keylen;
  outData.dptr=vallen?val:""; outData.dsize=vallen;
  if(gdbm_store(dbm,outKey, outData, GDBM_REPLACE) != GDBM_NO_ERROR)
    return 1;
  
  return 0;
}

static enum ypxfrstat ypxfr(char *mapName) 
{
  int localOrderNum=0;
  int masterOrderNum=0;
  char *masterName = NULL;
  struct sockaddr_in local_sockaddr;
  struct sockaddr_in source_sockaddr;
  struct ypall_callback callback;
  char dbName[MAXPATHLEN+1];
  char dbName2[MAXPATHLEN+1];
  int y, sourceSock;
  CLIENT *sourceClient;
  struct ypreq_nokey req_nokey;
  struct ypreq_key req_key;
  struct ypresp_order *resp_order;
  struct ypresp_master *resp_master;
  struct ypresp_val *resp_match;
  datum outKey, outData;
  char orderNum[12];

  memset(&local_sockaddr, '\0', sizeof(local_sockaddr));
  local_sockaddr.sin_family=AF_INET;
  local_sockaddr.sin_addr.s_addr=htonl(INADDR_LOOPBACK);

  /* 
  ** Get the Master hostname for the map, if no explicit 
  ** sourcename is given.
  */
  if (SourceHost[0] == '\0') 
    {
      if ((y=__yp_master(SourceDomain, mapName, &masterName)))
	return YPXFR_MADDR;
      strcpy(SourceHost,masterName);
    }

  /*
  ** Build a connection to the host with the master map, not the
  ** the nearest ypserv, because this map could be out of date.
  */
  memset(&source_sockaddr, '\0', sizeof(source_sockaddr));
  source_sockaddr.sin_family=AF_INET;
  {
    struct hostent *h=gethostbyname(SourceHost);
    if (!h) 
      {
	return YPXFR_MADDR;
      }
    memcpy(&source_sockaddr.sin_addr, h->h_addr, sizeof source_sockaddr.sin_addr);
  }

  if ((y=ypbind_host(&source_sockaddr, SourceDomain))) 
    return YPXFR_RPC;

  sourceSock=RPC_ANYSOCK;
  sourceClient=clnttcp_create(&source_sockaddr, YPPROG, YPVERS, &sourceSock, 0, 0);
  if (sourceClient==NULL) 
    {
      clnt_pcreateerror("YPXFR");
      return YPXFR_RPC;
    }

  req_nokey.domain=SourceDomain;
  req_nokey.map=mapName;
  if ((resp_order=ypproc_order_2(&req_nokey, sourceClient))==NULL)
    {
      clnt_perror(sourceClient, "masterOrderNum");
      masterOrderNum=0x7fffffff;
    } 
  else 
    {
      masterOrderNum=resp_order->ordernum;
      xdr_free((xdrproc_t) xdr_ypresp_order, (char *)resp_order);
    }

  if (masterName == NULL)
    {
      req_nokey.domain=SourceDomain;
      req_nokey.map=mapName;
      if ((resp_master=ypproc_master_2(&req_nokey, sourceClient))==NULL) 
	{
	  clnt_perror(sourceClient, "ypmaster");
	}
      else
	{
	  masterName = strdup(resp_master->peer);
	  xdr_free((xdrproc_t) xdr_ypresp_master, (char *)resp_master);
	}
    }
  /*
  ** If we doesn't force the map, look, if the new map is really newer
  */
  if (!Force) 
    {
      datum inKey, inVal;

      if(strlen(YPMAPDIR)+strlen(TargetDomain)+strlen(mapName)+3 < MAXPATHLEN)
	sprintf(dbName, "%s/%s/%s", YPMAPDIR, TargetDomain, mapName);
      else
	{
	  yp_msg("ERROR: Path to long: %s/%s/%s\n",YPMAPDIR, TargetDomain,mapName);
	  return YPXFR_RSRC;
	}
      if ((dbm=gdbm_open(dbName,0,GDBM_READER, 0600, NULL))==NULL)
	{
	  yp_msg("Cannot open old %s - ignored.\n", dbName);
	  localOrderNum=0;
	} 
      else 
	{
	  inKey.dptr="YP_LAST_MODIFIED"; inKey.dsize=strlen(inKey.dptr);
	  inVal=gdbm_fetch(dbm,inKey);
	  if (inVal.dptr) 
	    {
	      int i;
	      char *d=inVal.dptr;
	      for (i=0; i<inVal.dsize; i++, d++) 
		{
		  if (!isdigit(*d)) 
		    {
		      gdbm_close(dbm);
		      return YPXFR_SKEW;
		    }
		}
	      localOrderNum=atoi(inVal.dptr);
	    }
	  gdbm_close(dbm);
	}
      if(debug_flag>1)
	yp_msg("masterOrderNum=%d, localOrderNum=%d\n",
	       masterOrderNum, localOrderNum);
      if (localOrderNum>=masterOrderNum) return YPXFR_AGE;
    }
  
  if(strlen(YPMAPDIR)+strlen(TargetDomain)+strlen(mapName)+4 < MAXPATHLEN)
    sprintf(dbName, "%s/%s/%s~", YPMAPDIR, TargetDomain, mapName);
  else
    {
      yp_msg("ERROR: Path to long: %s/%s/%s~\n",YPMAPDIR, TargetDomain,mapName);
      return YPXFR_RSRC;
    }
  if ((dbm=gdbm_open(dbName,0,GDBM_NEWDB,0600,NULL))==NULL) 
    {
      yp_msg("Cannot open %s\n", dbName);
      return YPXFR_DBM;
    }
  
  outKey.dptr="YP_MASTER_NAME"; outKey.dsize=strlen(outKey.dptr);
  outData.dptr=masterName; outData.dsize=strlen(outData.dptr);
  if(gdbm_store(dbm,outKey, outData, GDBM_REPLACE) != GDBM_NO_ERROR)
    {
      gdbm_close(dbm);
      unlink(dbName);
      return YPXFR_DBM;
    }
  sprintf(orderNum, "%d", masterOrderNum);
  outKey.dptr="YP_LAST_MODIFIED"; outKey.dsize=strlen(outKey.dptr);
  outData.dptr=orderNum; outData.dsize=strlen(outData.dptr);
  if(gdbm_store(dbm, outKey, outData, GDBM_REPLACE) != GDBM_NO_ERROR)
    {
      gdbm_close(dbm);
      unlink(dbName);
      return YPXFR_DBM;
    }

  /* 
  ** Get the YP_INTERDOMAIN field. This is needed from the SunOS ypserv.
  ** We ignore this, since we have the "dns" option. But a Sun could be
  ** a NIS slave server and request the map from us.
  */
  req_key.domain=SourceDomain;
  req_key.map=mapName;
  req_key.key.keydat_val="YP_INTERDOMAIN";
  req_key.key.keydat_len=strlen("YP_INTERDOMAIN");
  if ((resp_match=ypproc_match_2(&req_key, sourceClient))==NULL) 
    {
      clnt_perror(sourceClient, "yproc_match");
    }
  else
    {
      if (resp_match->stat == YP_TRUE)
	{
	  outKey.dptr="YP_INTERDOMAIN"; outKey.dsize=strlen(outKey.dptr);
	  outData.dptr=resp_match->val.valdat_val; 
	  outData.dsize=resp_match->val.valdat_len;
	  if (gdbm_store(dbm, outKey, outData, GDBM_REPLACE) != GDBM_NO_ERROR)
	    {
	      gdbm_close(dbm);
	      unlink(dbName);
	      return YPXFR_DBM;
	    }
	}
      xdr_free((xdrproc_t) xdr_ypresp_val, (char *)resp_match);
    }

  /* 
  ** Get the YP_SECURE field. This is needed by the SunOS ypserv.
  ** We ignore this in the moment, since we have the /etc/ypserv.conf for this.
  */
  req_key.domain=SourceDomain;
  req_key.map=mapName;
  req_key.key.keydat_val="YP_SECURE";
  req_key.key.keydat_len=strlen("YP_SECURE");
  if ((resp_match=ypproc_match_2(&req_key, sourceClient))==NULL) 
    {
      clnt_perror(sourceClient, "yproc_match");
    }
  else
    {
      if (resp_match->stat == YP_TRUE)
	{
	  outKey.dptr="YP_SECURE"; outKey.dsize=strlen(outKey.dptr);
	  outData.dptr=resp_match->val.valdat_val; 
	  outData.dsize=resp_match->val.valdat_len;
	  if (gdbm_store(dbm, outKey, outData, GDBM_REPLACE) != GDBM_NO_ERROR)
	    {
	      gdbm_close(dbm);
	      unlink(dbName);
	      return YPXFR_DBM;
	    }
	}
      xdr_free((xdrproc_t) xdr_ypresp_val, (char *)resp_match);
    }
  
  callback.foreach=ypxfr_foreach;
  callback.data=NULL;
  y=__yp_all(SourceDomain, mapName, &callback);
  
  gdbm_close(dbm);
  if (strlen(YPMAPDIR)+strlen(TargetDomain)+strlen(mapName)+4 < MAXPATHLEN)
    sprintf(dbName, "%s/%s/%s~", YPMAPDIR, TargetDomain, mapName);
  else
    {
      yp_msg("ERROR: Path to long: %s/%s/%s~\n",YPMAPDIR, TargetDomain,mapName);
      return YPXFR_RSRC;
    }
  if(strlen(YPMAPDIR)+strlen(TargetDomain)+strlen(mapName)+3 < MAXPATHLEN)
    sprintf(dbName2, "%s/%s/%s", YPMAPDIR, TargetDomain, mapName);
  else
    {
      yp_msg("ERROR: Path to long: %s/%s/%s\n",YPMAPDIR, TargetDomain,mapName);
      return YPXFR_RSRC;
    }
  unlink(dbName2);
  rename(dbName, dbName2);
  
  if (!NoClear) 
    {
      memset(&local_sockaddr, '\0', sizeof local_sockaddr);
      local_sockaddr.sin_family=AF_INET;
      local_sockaddr.sin_addr.s_addr=htonl(INADDR_LOOPBACK);
      if (ypbind_host(&local_sockaddr, TargetDomain) ||
	  _yp_clear(TargetDomain)) 
	return YPXFR_CLEAR;
    }
  return y==0?YPXFR_SUCC:YPXFR_YPERR;
}

void Usage(int exit_code)
{
  fprintf(stderr,"usage: ypxfr [-f] [-c] [-d target domain] [-h source host] [-s source domain]\n");
  fprintf(stderr,"             [-C taskid program-number ipaddr port] mapname ...\n");
  fprintf(stderr,"       ypxfr --version\n");
  fprintf(stderr,"\n");
  fprintf(stderr,"where\n");
  fprintf(stderr,"\t-f forces transfer even if the master's copy is not newer.\n");
  fprintf(stderr,"\thost may be either a name or an internet\n");
  fprintf(stderr,"\t     address of form ww.xx.yy.zz\n");
  fprintf(stderr,"\t-c inhibits sending a \"Clear map\" message to the local ypserv.\n");
  fprintf(stderr,"\t-C is used by ypserv to pass callback information.\n");
  exit(exit_code);
}

int main (int argc, char **argv)
{
  static char res;

  if (argc < 2)
    Usage(1);

  if(!isatty(fileno(stderr)))
    openlog("ypxfr", LOG_PID, LOG_DAEMON);
  else
    debug_flag=1;

  memset(SourceHost, '\0', sizeof(SourceHost));

  while(1) 
    {
      int c;
      int option_index = 0;
      static struct option long_options[] =
      {
        {"version", no_argument, NULL, '\255'},
	{"debug", no_argument, NULL, '\254'},
        {"help", no_argument, NULL, 'u'},
        {"usage", no_argument, NULL, 'u'},
        {NULL, 0, NULL, '\0'}
      };

      c=getopt_long(argc, argv, "ufcd:h:s:C:S",long_options, &option_index);
      if (c==EOF) break;
      switch (c) 
	{
	case 'f':
	  Force++;
	  break;
	case 'c':
	  NoClear++;
	  break;
	case 'd':
	  TargetDomain=optarg;
	  break;
	case 'h':
	  strcpy(SourceHost,optarg);
	  break;
	case 's':
	  SourceDomain=optarg;
	  break;
	case 'C':
	  if(optind+3 > argc) 
	    {
	      Usage(1);
	      break;
	    }
	  TaskId=atoi(optarg);
	  ProgramNumber=atoi(argv[optind++]);
	  IpAddress.s_addr=inet_addr(argv[optind++]);
	  PortNumber=atoi(argv[optind++]);
	  break;
	case 'u':
	  Usage(0);
	  break;
	case '\254':
	  debug_flag=2;
	  break;
	case '\255':
	  fprintf(stderr,"ypxfr - NYS YP Server version %s\n",version);
          exit(0);
	default:
	  Usage(1);
	  break;
	}
    }
  argc-=optind;
  argv+=optind;

  if (!TargetDomain) 
    yp_get_default_domain(&TargetDomain);
    
  if (!SourceDomain) 
    SourceDomain=TargetDomain;
    

  for (; *argv; argv++) 
    {
      enum ypxfrstat y;

      /*
      ** Start the map Transfer
      */
      if ((y=ypxfr(*argv))!=YPXFR_SUCC) 
	{
	  yp_msg("ypxfr: %s\n", ypxfr_err_string(y));
	}

      /*
      ** Now send the status to the yppush program, so it can display a 
      ** message for the sysop and do not timeout.
      */
      if (TaskId) 
	{
	  struct sockaddr_in addr;
	  struct timeval wait;
	  CLIENT *clnt;
	  int s;
	  ypresp_xfr resp;
	  static struct timeval tv={10,0};
	  
	  memset(&addr, '\0', sizeof addr);
	  addr.sin_addr=IpAddress;
	  addr.sin_port=htons(PortNumber);
	  addr.sin_family=AF_INET;
	  wait.tv_sec=25; wait.tv_usec=0;
	  s=RPC_ANYSOCK;

	  clnt=clntudp_create(&addr, ProgramNumber, 1, wait, &s);
	  if (!clnt) 
	    {
	      clnt_pcreateerror("ypxfr_callback");
	      continue;
	    }

	  resp.transid=TaskId;
	  resp.xfrstat=y;

	  if(clnt_call(clnt, 1, (xdrproc_t) xdr_ypresp_xfr, (void *) &resp,
		       (xdrproc_t) xdr_void, &res, tv) != RPC_SUCCESS) 
	    clnt_perror(clnt, "ypxfr_callback");

	  clnt_destroy(clnt);
	}
    }
  return 0;
}
