/* IluJava_JGC.h */
/* Chris Jacobi, August 15, 1997 5:13 pm PDT */

/*
 * Copyright (c) 1997 Xerox Corporation.  All Rights Reserved.  
 * Unlimited use, reproduction, and distribution of this software is
 * permitted.  Any copy of this software must include both the above
 * copyright notice of Xerox Corporation and this paragraph.  Any
 * distribution of this software must comply with all applicable United
 * States export control laws.  This software is made available AS IS,
 * and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
 * INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
 * PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
 * THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
 * CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
 * XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 */
 
/* $Id: IluJava_JGC.h,v 1.8 1997/08/16 01:31:53 jacobi Exp $ */

/*
 * Macros to make it possible to switch the implementation
 * of java garbage collection primitives.
 */

#ifndef _ILUJAVA_JGC_
#define _ILUJAVA_JGC_

#include "IluJava_Includes.h"


/* Stuffing Java refs into the C heap is not extremely
 * portable. These macros or routines should make this
 * undertaking a little bit more portable.
 *
 * In Sun's JDK these are no-ops; nevertheless the 
 * REVEAL operations carries the risk of accessing
 * an object which has already been collected.
 *
 * With Microsoft Java or with the Boehm collector, these 
 * operations will be real.
 *
 * With Microsoft, even stuffing Java refs on the C STACK
 * will cause problems.
 */
 
/* 
 * About ILUOInt's
 * The LSR keeps the invariant that a LSO is not garbage collected
 * while the kernel hangs on to a KO-LSO pair.  This pair is
 * modified only with the server lock held.
 *
 * Note that the invariant is about garbage collection, not about
 * finalization.  All thinkable implementation with finalization
 * will provide the further invariant that an object is not garbage 
 * collected before it is finalized.
 */



/* 
 * JGC_ENABLE, JGC_DISABLE and JGC_FRAME are modelled
 * to be a thin veneer over RNI's mechanisms and noops otherwise.
 */

#if (defined(RNI))

    #define JGC_ENABLE \
        GCEnable();
    #define JGC_DISABLE \
        GCDisable();

    #define JGC_FRAME_DECL(frameName) \
        GCFrame frameName;
    #define JGC_FRAME_PUSH(frameName, addr, sz) \
        GCFramePush(& frameName, addr, sz);
    #define JGC_FRAME_POP(frameName)  \
        GCFramePop(& frameName);

#else

    #define JGC_ENABLE 
    #define JGC_DISABLE 

    #define JGC_FRAME_DECL(frameName)  
    #define JGC_FRAME_PUSH(frameName, addr, sz)
    #define JGC_FRAME_POP(frameName)

#endif



/*
 * GLOBALOBJ is used for permanent static objects. 
 * The macros keep track of code moving but do not
 * necessarily prevent GC; this must be done on the 
 * java side.
 * Consider GLOBALOBJ to be assign-once. 
 * ASSIGNTO and GETFROM may need an implied JENV parameter
 */

#if (defined(RNI))

    #define JGC_GLOBALOBJ_DECL(name) \
        HObject** name = 0
    #define JGC_GLOBALOBJ_ASSIGNTO(name, jh_obj) \
        name = GCGetPtr((HObject *) jh_obj);
    #define JGC_GLOBALOBJ_GETFROM(jh_obj, name) \
         ((HObject*)jh_obj) = *name;

#elif (defined(ONI))

    #define JGC_GLOBALOBJ_DECL(name) \
        HObject* name = 0
    #define JGC_GLOBALOBJ_ASSIGNTO(name, jh_obj) \
        name = (HObject*) jh_obj;
    #define JGC_GLOBALOBJ_GETFROM(jh_obj, name) \
        ( * (HObject**) & jh_obj ) = name;

#elif (defined(JNI))

    #define JGC_GLOBALOBJ_DECL(name) \
        jglobal name = 0
    #define JGC_GLOBALOBJ_ASSIGNTO(name, jh_obj) \
        name = NewGlobalRef(JENV_ACTUAL jh_obj);
    #define JGC_GLOBALOBJ_GETFROM(jh_obj, name) \
        jh_obj = GlobalToLocalRef(JENV_ACTUAL name);

#else

    #error "None of RNI, ONI or JNI defined"

#endif




/*
 * Weak pointers
 * 
 *
 *
 * JGC_WP_TYPE(javatype)
 *     Expands to the type used for weak pointers.
 *
 * JGC_WP_REVEAL(jwp_obj)
 *     Expands to an expression revealing the contents of a weak 
 *     pointer.
 *     NIL is legal but I prefer explicite NIL tests
 *     Returned value needs to be cast in some runtime impls.
 *    
 * JGC_WP_MAKE(jh_obj)
 *     Expands to an expression creating a weak pointer to 
 *     the argument.
 *     NIL is legal but I prefer explicite NIL tests
 *    
 * JGC_WP_RELEASE(jwp_obj) 
 *     Expands to an expression telling language vm that this weak 
 *     pointer doesn't need to be updated anymore.
 *     NIL is legal but I prefer explicite NIL tests.
 *
 * JGC_GET_WP_FROM_JAVA_OBJECT(jh_obj)
 *     Special convention:  For java objects extending IluWPBase only.
 *     Gets the weak pointer allocated on object creation
 *     NIL is legal but I prefer explicite NIL tests.
 */
 
#if (defined(RNI))

    /* rni is actually compacting */

    #define JGC_WP_TYPE(javatype) \
        HObject**
    #define JGC_WP_MAKE(jh_obj) \
		(jh_obj ? GCGetPtr((HObject*)jh_obj) : (HObject**) 0)
    #define JGC_WP_REVEAL(jwp_obj) \
		( (HObject*) (jwp_obj ? *jwp_obj : 0) )
    #define JGC_WP_RELEASE(jwp_obj) \
		(jwp_obj ? GCFreePtr(((HObject**)jwp_obj)) : 0)
    #define JGC_GET_WP_FROM_JAVA_OBJECT(jh_obj) \
		(jh_obj ? GET_IluWPBase_ywpx(jh_obj) : 0)

#elif (defined (ONI))
    
    /* oni is NOT compacting */
    
    #define JGC_WP_TYPE(javatype) \
        javatype
    #define JGC_WP_MAKE(jh_obj) \
        jh_obj
    #define JGC_WP_REVEAL(jwp_obj) \
        (jwp_obj)
    #define JGC_WP_RELEASE(jwp_obj) \
        /* noop */
    #define JGC_GET_WP_FROM_JAVA_OBJECT(jh_obj) \
        jh_obj

#else

    #error "Weak pointers not defined for other than ONI or RNI"

#endif


#endif /* _ILUJAVA_JGC_ */
