#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <iostream.h>
#include <time.h>

#include "icq.h"


unsigned long ipToIcq(unsigned long l)
/* return IP address in a format suitable to be transmitted to ICQ via
   packet class. ICQ uses big endian format, packet class return little
   endian longs, so we need a conversion from little to big endian.
   IP is represented in native format and converted to big-endian by
   htonl();
   "aa.bb.cc.dd" must be passed to packet class as 0xddccbbaa
   on entry, l is in native format:
      0xaabbccdd on Sparcs (big endian)
      0xddccbbaa on Intels (little endian)
   after htonl(), l is in big-endian (net format)
      0xaabbccdd
   to pass to ICQ, l is converted to little endian) */
{
	l=htonl(l);
	return (l << 24) | ((l & 0xff00) << 8) | ((l & 0xff0000) >> 8) | (l >> 24);
}


unsigned long icqToIp(unsigned long l)
/* return IP address in a format suitable to be used by socket from a long
   returned by packet class. ICQ uses big endian format, packet class return
   little endian longs, so we need a conversion from little to big endian.
   IP is returned in native format by ntohl();
   "aa.bb.cc.dd" is received from packet class as 0xddccbbaa.
   0xddccbbaa is converted to net format (big endian) and than back to host
   via ntohl(). */
{
	return ntohl((l << 24) | ((l & 0xff00) << 8) | ((l & 0xff0000) >> 8) | (l >> 24));
}


//-----ICQ::constructor----------------------------------------------------------------------------
ICQ::ICQ(void) : tcpPorts(10)
{
   char filename[64];
   char userKey[8];
   unsigned long uin;
   unsigned short usersList;
   char errorFileName[MAX_FILENAME_LEN];
   char TranslationTableFileName[MAX_FILENAME_LEN];

   autoReconnect = true;
   allowUpdateUsers = true;

   sprintf(filename, "%s%s%s", BASE_DIR, CONF_DIR, "licq.conf");
   File licqConf(filename);
   if (licqConf.error()) { cerr << "Fatal error!  Unable to open " << filename << "." << endl; exit(1); }
   
   // Network configuration
   if (DEBUG_LEVEL >= 1) cout << "-> Network configuration." << endl;

   unsigned short numRemoteServers, remoteServerPort;
   char remoteServerID[32], remotePortID[32], remoteServerName[64];
   bool foundDefaultPort;

   licqConf.section("network").key("NumOfServers").read(numRemoteServers, true);
   if (DEBUG_LEVEL >= 1) cout << "   " << numRemoteServers << " servers stated." << endl;
   
   // Check for a default server port - if it's not there, we set it to the internal default
   if (!licqConf.section("network").key("DefaultServerPort").read(defaultRemotePort))
   { 
      cerr << "Using internal default server port (" << DEFAULT_SERVER_PORT << ")." << endl;
      defaultRemotePort = DEFAULT_SERVER_PORT;
      foundDefaultPort = false;
   } 
   else
   {
      if (DEBUG_LEVEL >= 1) cout << "   " << "Default server port set to " << defaultRemotePort << "." << endl;
      foundDefaultPort = true;
   }

   // read in all the servers
   for(int i = 0; i < numRemoteServers; i++)
   {
      if (DEBUG_LEVEL >= 1) cout << "   " << "Reading in server " << i + 1 << ": ";
      sprintf(remoteServerID, "Server%d", i + 1);
      sprintf(remotePortID, "ServerPort%d", i + 1);
      if (!licqConf.section("network").key(remoteServerID).read(remoteServerName))
      {
         cerr << "Skipping server." << endl;
         continue;
      } 
      if (DEBUG_LEVEL >= 1) cout << remoteServerName << ":";
      if (!licqConf.section("network").key(remotePortID).read(remoteServerPort, false, false))
      {
         // if no default port set and no port for this server, print out an error
         if (!foundDefaultPort) cerr << "Using default port " << defaultRemotePort << " for " << remoteServerName << "." << endl;
         remoteServerPort = defaultRemotePort;
      }
      if (DEBUG_LEVEL >= 1) cout << remoteServerPort << "." << endl;
      icqServers.addServer(remoteServerName, remoteServerPort);
   }

   licqConf.section("network").key("TCPServerPort").read(tcpServerPort);
   if (DEBUG_LEVEL >= 1) 
   {
      cout << "   TCP server port ";
      if (tcpServerPort == 0) cout << "will be assigned by the system." << endl;
      else cout << "set to " << tcpServerPort << "." << endl;
   }

  licqConf.section("network").key("Errors").read(errorFileName, true);
   
   // Users configuration
   if (DEBUG_LEVEL >= 1) cout << "-> User configuration." << endl; 
   
   sprintf(filename, "%s%s%s", BASE_DIR, CONF_DIR, "users.conf");
   File usersConf(filename);
   if (usersConf.error()) { cerr << "Fatal error!  Unable to open " << filename << "." << endl; exit(1); }
   
   usersConf.section("users").key("NumOfUsers").read(usersList, true);
   if (DEBUG_LEVEL >= 1) cout << "   " << usersList << " users stated..." << flush;
   
   numUsersVal = 0;  // will eventually equal usersList if all users are there
   for(unsigned short i = 0; i < usersList; i++)
   {
      sprintf(userKey, "User%d", i + 1);
      if (!usersConf.section("users").key(userKey).read(uin))
      {   
         cerr << "Skipping user." << endl;
         continue;
      }
      sprintf(filename, "%s%s%li%s", BASE_DIR, CONF_DIR, uin, ".uin"); 
      addUser(uin, filename, false);
   } 
   if (DEBUG_LEVEL >= 1) cout << numUsersVal << " users found." << endl;
   
   // sound configuration
   char soundMessageFile[MAX_FILENAME_LEN], soundPlayerFile[MAX_FILENAME_LEN],
        soundUrlFile[MAX_FILENAME_LEN], soundChatFile[MAX_FILENAME_LEN],
        soundFileFile[MAX_FILENAME_LEN], soundNotifyFile[MAX_FILENAME_LEN];
   
   if (DEBUG_LEVEL >= 1) cout << "-> Sound configuration." << endl;
   licqConf.section("sound").key("Enable").read(soundEnabledVal);
   if (DEBUG_LEVEL >= 1) cout << "   Sound is " <<  (soundEnabledVal == 0 ? "disabled." : "enabled.") << endl;
   soundPlayer = soundMsg = soundUrl = soundChat = soundFile = soundNotify = NULL;
   licqConf.section("sound").key("Player").read(soundPlayerFile, true);
   licqConf.section("sound").key("Message").read(soundMessageFile);
   licqConf.section("sound").key("Url").read(soundUrlFile);
   licqConf.section("sound").key("Chat").read(soundChatFile);
   licqConf.section("sound").key("File").read(soundFileFile);
   licqConf.section("sound").key("OnlineNotify").read(soundNotifyFile);
   setSounds(soundPlayerFile, soundMessageFile, soundUrlFile, soundChatFile, soundFileFile, soundNotifyFile);
   
   if (DEBUG_LEVEL >= 1) 
   {
      cout << "   User defined sound player is \"" << soundPlayer << " <sound file>\"" << endl;
      if (soundEnabledVal == 1) cout << "   Sound will be played using user defined player." << endl;
      else if (soundEnabledVal == 2) cout << "   Sound will be played using pc speaker." << endl;
   
      cout << "   Sounds: " <<  soundMsg << " (message), " << endl
           << "           " <<  soundUrl << " (url), "  << endl
           << "           " << soundChat << " (chat), " << endl
           << "           " << soundFile << " (file)." << endl
           << "           " << soundNotify << " (online notify)." << endl;
   }
   
   // open the error file is there is one
   if (strncmp(errorFileName, "none", 4) == 0)
   {
      if (DEBUG_LEVEL >= 1) cout << "   No error log will be kept." << endl;
      errorFile = NULL;
   }
   else
   {
      char errorFileNameFull[256];
      sprintf(errorFileNameFull, "%s%s", BASE_DIR, errorFileName);
      if (DEBUG_LEVEL >= 1) cout << "   Opening " << errorFileNameFull << " as error log." << endl;
      errorFile = fopen(errorFileNameFull, "a");
   }

   icqServers.setServer(1);    // set the initial UDP remote server (opened in logon)
   if (DEBUG_LEVEL >= 1) cout << "   Starting TCP server..." << flush;
   if (!icqOwner.tcpSocket.startServer(tcpServerPort))    // start up the TCP server
   { 
      if (DEBUG_LEVEL >= 1) cout << "unable to allocate port!" << endl; 
      else cerr << "Unable to allocate TCP port for local server!" << endl;
      exit(1); 
   }
   if (DEBUG_LEVEL >= 1) cout << "done." << endl;
   
   for (unsigned short i = 0; i < 10; i++) tcpPorts[i] = false;

   connect (icqOwner.tcpSocket.sn, SIGNAL(activated(int)), this, SLOT(recvNewTCP(int)));
   connect (&pingTimer, SIGNAL(timeout()), this, SLOT(ping()));

   // Loading translation table from file
   licqTrans = new LicqTranslator;

   licqConf.section("network").key("Translation").read(TranslationTableFileName, true);
   if (strncmp(TranslationTableFileName, "none", 4) == 0)
   {
      if (DEBUG_LEVEL >= 1) cout << "   Will not translate." << endl;   
   }
   else
   {
      char TranslationTableFileNameFull[1024];
      sprintf(TranslationTableFileNameFull, "%s%s%s", BASE_DIR, TRANSLATION_DIR, TranslationTableFileName);
      if (DEBUG_LEVEL >= 1) cout << "   Opening " << TranslationTableFileNameFull << " as translation table." << endl;
      licqTrans->setTranslationMap (TranslationTableFileNameFull);
   }
   
}


//-----ICQ::destructor------------------------------------------------------------------------------
ICQ::~ICQ(void)
{
   // udpServer might not exist at this point. Think about it.
   if (icqOwner.status() != ICQ_STATUS_OFFLINE) logoff(false);
   if (errorFile != NULL) fclose(errorFile);
}


//-----ICQ::simple stuff---------------------------------------------------------------------------
void ICQ::getStatusInfo(struct UserStatusline &us)  { icqOwner.getStatusInfo(us); }
void ICQ::setOutputWin(OutputWin *w)  { outputWindow = w; }
int ICQ::numUsers(void)  { return(numUsersVal); }
void ICQ::setAwayMessage(char *m)  { icqOwner.setAwayMessage(m); }
char *ICQ::awayMessage(void)  { return(icqOwner.awayMessage()); }
void ICQ::setSoundEnabled(unsigned short s)  { soundEnabledVal = s; }
unsigned short ICQ::soundEnabled(void)  { return(soundEnabledVal); }


//-----ICQ::saveUsers------------------------------------------------------------------------------
void ICQ::saveUsers(void)
/* Write the user list to the conf/users.conf file, thus adding any new users and updating
   the order they appear in */
{
   char filename[128];
   sprintf(filename, "%s%s%s", BASE_DIR, CONF_DIR, "users.conf");
   FILE *usersConf = fopen(filename, "w");
   fprintf(usersConf, "[users]\nNumOfUsers = %d\n", numUsersVal);
   unsigned short j = 1;  // counter to keep track of which total user we are at
   for (unsigned short i = 0; i < users.size(); i++) 
      fprintf(usersConf, "User%d = %ld\n", j++, users[i]->uin());
   fclose(usersConf);
}


//-----ICQ::sendICQ--------------------------------------------------------------------------------
ICQEvent *ICQ::sendICQ(INetSocket &sock, Packet &packet, unsigned short cmd, unsigned long theSequence, 
                       unsigned long uin = 0, unsigned short subCmd = 0)
{
   if (!sock.connected()) return(NULL);  // don't send if not connected
   
   icqEvents.push_back(new ICQEvent(cmd, subCmd, theSequence, uin, &sock, &packet, outputWindow));
   connect (icqEvents[icqEvents.size() - 1], SIGNAL(timedOut(bool, int, int)), this, SLOT(doneEvent(bool, int, int)));
   if (!icqEvents[icqEvents.size() - 1]->start()) 
   {
      cancelEvent(icqEvents[icqEvents.size() - 1]);
      return(NULL);
   }
   return (icqEvents[icqEvents.size() - 1]);
}


//-----ICQ::addUser--------------------------------------------------------------------------------
void ICQ::addUser(unsigned long id, TCPSocket &sock, bool newUser = false)  
{ 
   numUsersVal++; 
   users.push_back(new ICQUser(id, sock, newUser));
   connect (users[users.size() - 1]->tcpSocket.sn, SIGNAL(activated(int)), this, SLOT(recvTCP(int)));   
   if (newUser) addNewUser(users[users.size() - 1]);
}


//-----ICQ::addUser--------------------------------------------------------------------------------
void ICQ::addUser(unsigned long id, char *filename, bool newUser = false)
{ 
   numUsersVal++; 
   users.push_back(new ICQUser(id, filename, newUser));
   if (newUser) addNewUser(users[users.size() - 1]);
}


//-----ICQ::addNewUser-----------------------------------------------------------------------------
void ICQ::addNewUser(ICQUser *u)
{
   // update the users info from the server
   if (icqOwner.status() != ICQ_STATUS_OFFLINE) 
   {
      Packet packet;  // alert server to new user
      /* 02 00 3C 05 06 00 50 A5 82 00 8F 76 20 00 */
      packet << ICQ_VERSION
             << ICQ_CMDxSND_USERxADD
             << (unsigned short)icqOwner.sequence()
             << icqOwner.uin()
             << u->uin()
      ;
      outputWindow->wprintf("%sAlerting server to new user (%d)...", UDP, icqOwner.sequenceVal - 1);
      sendICQ(udpServer, packet, ICQ_CMDxSND_USERxADD, icqOwner.sequenceVal - 1);
      
      getUserInfo(u);
   }

   // save the new info
   saveUsers();
   u->saveInfo();
   moveToTop(u);
}



//-----ICQ::removeUser-----------------------------------------------------------------------------
void ICQ::removeUser(ICQUser *u)
{
   char filename[MAX_FILENAME_LEN];
   sprintf(filename, "%s%s%ld.uin", BASE_DIR, CONF_DIR, u->uin());
   remove(filename);
   if (u->history != NULL) 
   {
      u->history->close();
      remove(u->history->name());
   }
   
   unsigned short uNum = 0;
   while (uNum < users.size() && u != users[uNum]) uNum++;
   for (unsigned short i = uNum; i < users.size() - 1; i++) users[i] = users[i + 1];
   delete u;
   users.pop_back();
   
   numUsersVal--;
   saveUsers();
   emit updatedUsers();
}


//-----ICQ::doneEvent------------------------------------------------------------------------------
void ICQ::doneEvent(bool gotAck, int sockfd, int theSeq)
{    
   unsigned short i;
   for (i = 0; i < icqEvents.size(); i++) if (icqEvents[i]->isEvent(sockfd, theSeq)) break;
   if (i == icqEvents.size()) return;  // didn't find a relevant event, ignore ack

   unsigned short cmd = icqEvents[i]->cmd;                  // extract the useful information from the event
   icqEvents[i]->ackTimer.stop();
   
   switch (cmd) 
   {
   case ICQ_CMDxTCP_START: emit doneUserFcn(gotAck, icqEvents[i]); break;
   case ICQ_CMDxSND_THRUxSERVER: emit doneUserFcn(gotAck, icqEvents[i]); break;
   case ICQ_CMDxSND_USERxGETINFO: if (!gotAck) emit doneUserInfo(false, icqEvents[i]->uin); break;
   case ICQ_CMDxSND_SETxSTATUS: 
      icqOwner.statusVal = desiredStatus;
      emit doneOwnerFcn(gotAck, cmd);
      break;
   case ICQ_CMDxSND_PING: if (!gotAck) emit doneOwnerFcn(false, cmd); break;
   case ICQ_CMDxSND_USERxADD: if (!gotAck) emit doneOwnerFcn(false, cmd); break;
   case ICQ_CMDxSND_AUTHORIZE: emit doneOwnerFcn(gotAck, cmd); break;
   case ICQ_CMDxSND_LOGON: 
      if (!gotAck) 
      {
         logoff(false);
         emit doneOwnerFcn(false, cmd); 
      }
      break;
   case ICQ_CMDxSND_USERxLIST: if (!gotAck) emit doneOwnerFcn(false, cmd); break;
   case ICQ_CMDxSND_SYSxMSGxREQ: if (!gotAck) emit doneOwnerFcn(false, cmd); break;
   case ICQ_CMDxSND_SYSxMSGxDONExACK: if (!gotAck) emit doneOwnerFcn(false, cmd); break;
   default: break;
   }
   
   if (icqEvents.size() > 0)   // possibly all events have been deleted in logoff
   {
      delete icqEvents[i];                          // delete the event
      icqEvents[i] = icqEvents[icqEvents.size() - 1];     // move the last event to the position of the old one
      icqEvents.pop_back();                         // clear the last event
   }
   

   // logoff if server timed out in anything except first logon packet
   if (!gotAck && cmd != ICQ_CMDxTCP_START && cmd != ICQ_CMDxSND_LOGON) logoff(autoReconnect);
}


//-----ICQ::cancelEvent------------------------------------------------------------------------------
void ICQ::cancelEvent(ICQEvent *&e)
{    
   unsigned short i;
   for (i = 0; i < icqEvents.size(); i++) if (icqEvents[i] == e) break;
   if (i == icqEvents.size()) return;  // didn't find a relevant event, ignore cancel

   icqEvents[i]->ackTimer.stop();
   if (icqEvents[i]->subCmd == ICQ_CMDxTCP_CHAT) chatCancel(getUserByUIN(icqEvents[i]->uin), icqEvents[i]->seq, true);

   delete icqEvents[i];                          // delete the event
   icqEvents[i] = icqEvents[icqEvents.size() - 1];     // move the last event to the position of the old one
   icqEvents.pop_back();                         // clear the last event
   e = NULL;  // make sure the pointer no longer points to the deleted memory
}


//-----ICQ::getUserByUIN---------------------------------------------------------------------------
ICQUser *ICQ::getUserByUIN(unsigned long u)
{
   for(unsigned short i = 0; i < users.size(); i++) 
      if(users[i]->uin() == u) return(users[i]);
   if (icqOwner.uin() == u) return &icqOwner;
   return NULL;
}


//-----ICQ::knownUser---------------------------------------------------------------------------
bool ICQ::knownUser(unsigned long u)
{
   for(unsigned short i = 0; i < users.size(); i++) 
      if(users[i]->uin() == u) return(true);
   if (icqOwner.uin() == u) return (true);
   return (false);
}


            
//-----ICQ::errorOn--------------------------------------------------------------------------------
bool ICQ::errorOn(const char *protocol, const char *reason)
{
   outputWindow->wprintf("%C%sError: %s.", COLOR_ERROR, protocol, reason);
   return(false);
}


//-----ICQ::setSounds------------------------------------------------------------------------------
void ICQ::setSounds(const char *p, const char *m, const char *u, const char *c, const char *f, const char *n)
{
   if (soundPlayer != NULL) delete soundPlayer;
   soundPlayer = new char[strlen(p) + 1];
   strcpy(soundPlayer, p);
   if (soundMsg != NULL) delete soundMsg;
   soundMsg = new char[strlen(m) + 1];
   strcpy(soundMsg, m);
   if (soundUrl != NULL) delete soundUrl;
   soundUrl = new char[strlen(u) + 1];
   strcpy(soundUrl, u);
   if (soundChat != NULL) delete soundChat;
   soundChat = new char[strlen(c) + 1];
   strcpy(soundChat, c);
   if (soundChat != NULL) delete soundFile;
   soundFile = new char[strlen(f) + 1];
   strcpy(soundFile, f);
   if (soundNotify != NULL) delete soundNotify;
   soundNotify = new char[strlen(n) + 1];
   strcpy(soundNotify, n);   
}


//-----ICQ::playSound------------------------------------------------------------------------------
void ICQ::playSound(char *theSound)
{
   if (soundEnabledVal == 0 || icqOwner.status() != ICQ_STATUS_ONLINE) return;
   if (soundEnabledVal == 1) 
   {
      char *soundCmd = new char[strlen(soundPlayer) + strlen(theSound) + 4];
      sprintf(soundCmd, "%s %s &", soundPlayer, theSound);
      system(soundCmd);
      delete soundCmd;
   }
   else if (soundEnabledVal == 2) 
   {
      printf("\a");
      fflush(stdout);
   }
}


//-----ICQ::nextServer-----------------------------------------------------------------------------
bool ICQ::nextServer(void)
{
   udpServer.resetSocket();
   logoff(false);
   icqServers.next();
   return(openServer());
}


//-----ICQ::openServer-----------------------------------------------------------------------------
bool ICQ::openServer()
{
   // no servers!
   if (icqServers.current() == NULL) 
   {
      outputWindow->wprintf("%sNo server set.", UDP);
      return (false);
   }
   
   // try and set the destination
   outputWindow->wprintf("%sFinding %s...", UDP, icqServers.current()->name());
   if (!udpServer.setDestination(icqServers.current()->name(), icqServers.current()->port()))
   {
      outputWindow->wprintf("      No route to host %s, port %d", icqServers.current()->name(), icqServers.current()->port());
      return (false);  // no route to host (not connected)
   }
   else
   {
      outputWindow->wprintf("      ICQ server set to %s, port %d", icqServers.current()->name(), icqServers.current()->port());
      return (true);
   }
}



//-----ICQ::logon----------------------------------------------------------------------------------
bool ICQ::logon(unsigned short logonStatus)
{
   // check to see if we have a destination
   if (!udpServer.destinationSet()) 
      if (!openServer()) return (false);   // if setting the destination fails, then return false

   outputWindow->wprintf("%sOpening socket to server.", UDP);
   udpServer.openConnection();
   connect (udpServer.sn, SIGNAL(activated(int)), this, SLOT(recvUDP(int)));
   
   // send hello packet
   Packet loginPacket;
   /* 02 00 E8 03 08 00 8F 76 20 00 34 4A 00 00 08 00 5B 63 65 50 61 62 43 00
      72 00 04 00 7F 00 00 01 04 00 00 00 00 03 00 00 00 02 00 00 00 00 00 04 
      00 72 00 */

   loginPacket << ICQ_VERSION       // 2 bytes
               << ICQ_CMDxSND_LOGON     // 2 bytes 
               << (unsigned short)icqOwner.sequence()    // 2 bytes, sequence (01 00)
               << icqOwner.uin()
               << (unsigned long)icqOwner.tcpSocket.localPort()
               << (unsigned short)(strlen(icqOwner.passwd()) + 1) // length of passwd
               << icqOwner.passwd()
               << 0x00040072ul
               << ipToIcq(udpServer.localIP())
               << (char)0x04
               << logonStatus       // 2 bytes, initial status
               << (unsigned short)0x00 << 0x03ul << 0x02ul << 0x00040000ul << (unsigned short)0x0072
   ;
   outputWindow->wprintf("%C%sRequesting logon (%d)...", COLOR_SEND, UDP, icqOwner.sequenceVal - 1);
   sendICQ(udpServer, loginPacket, ICQ_CMDxSND_LOGON, icqOwner.sequenceVal - 1);
   desiredStatus = logonStatus;
   return(true);
}


//-----ICQ::logoff---------------------------------------------------------------------------------
void ICQ::logoff(bool reconnect)
{
   // if not connected then don't both logging off
   if (udpServer.connected())
   {
      Packet logoffPacket;
      logoffPacket << ICQ_VERSION
                   << ICQ_CMDxSND_LOGOFF
                   << (unsigned short)0x0000
                   << icqOwner.uin()
                   << (unsigned short)0x0014
                   << "B_USER_DISCONNECTED"
                   << (unsigned short)0x0005
      ;
      outputWindow->wprintf("%C%sLogging off.", COLOR_SEND, UDP);
      udpServer.sendPacket(logoffPacket);
      udpServer.closeConnection();
      
      // close all open events
      unsigned short i;
      for (i = 0; i < icqEvents.size(); i++) 
         delete icqEvents[i];
      while (icqEvents.size() > 0) icqEvents.pop_back();
   }

   if (reconnect) logon(icqOwner.statusVal);
   icqOwner.statusVal = ICQ_STATUS_OFFLINE;

   emit updatedStatus();
   pingTimer.stop();
   emit doneOwnerFcn(true, ICQ_CMDxSND_LOGOFF);
}             


//-----ICQ::updateContactList----------------------------------------------------------------------
void ICQ::updateContactList(void)
{
   // create user info packet
   Packet userPacket;
   userPacket << ICQ_VERSION       // 2 bytes
              << ICQ_CMDxSND_USERxLIST  // 2 bytes 
              << (unsigned short)icqOwner.sequence()    // 2 bytes, sequence
              << icqOwner.uin()
              << (char)numUsersVal
   ;
   // add all the users to the packet, resetting their status at the same time
   for (unsigned short i = 0; i < users.size(); i++) 
   {
      userPacket << users[i]->uin();
      users[i]->statusVal = ICQ_STATUS_OFFLINE;
   }

   emit updatedUsers();
   allowUpdateUsers = false;   // do this to ease down on the update user calls when the online
                               // user comes in...gets set to true when the "end of users" packet
                               // comes in
   
   // send user info packet
   outputWindow->wprintf("%C%sSending user info request (%d)...", COLOR_SEND, UDP, icqOwner.sequenceVal - 1);
   sendICQ(udpServer, userPacket, ICQ_CMDxSND_USERxLIST, icqOwner.sequenceVal - 1);
   
}


//-----ICQ::startSearch---------------------------------------------------------------------
void ICQ::startSearch(char *nick, char *first, char *last, char *email, unsigned long s)
{
   Packet packet;
   /* 02 00 24 04 04 00 50 A5 82 00 05 00 0B 00 41 70 6F 74 68 65 6F 73 69 73
      00 07 00 47 72 61 68 61 6D 00 05 00 52 6F 66 66 00 01 00 00 */
   packet << ICQ_VERSION
          << ICQ_CMDxSND_SEARCHxSTART
          << (unsigned short)(icqOwner.sequence())
          << icqOwner.uin()
          << (unsigned short)(s)
          << (unsigned short)(strlen(nick) + 1)
          << nick
          << (unsigned short)(strlen(first) + 1)
          << first
          << (unsigned short)(strlen(last) + 1)
          << last
          << (unsigned short)(strlen(email) + 1)
          << email
   ;

   outputWindow->wprintf("%C%sStarting search for user (%d)...", COLOR_SEND, UDP, s);
   sendICQ(udpServer, packet, ICQ_CMDxSND_SEARCHxSTART, icqOwner.sequenceVal-1);
}


//-----ICQ::ping-----------------------------------------------------------------------------------
void ICQ::ping()
{
   if (icqOwner.status() == ICQ_STATUS_OFFLINE) return;

   Packet pingPacket;
   pingPacket.clearPacket();
   pingPacket << ICQ_VERSION
              << ICQ_CMDxSND_PING
              << (unsigned short)icqOwner.sequence()
              << icqOwner.uin()
   ;
   outputWindow->wprintf("%C%sPinging Mirabilis (%d)...", COLOR_SEND, UDP, icqOwner.sequenceVal - 1);
   sendICQ(udpServer, pingPacket, ICQ_CMDxSND_PING, icqOwner.sequenceVal - 1);

   /*
   outputWindow->wprintf("%C...ping...", COLOR_SEND);

   pingPacket.clearPacket();
   pingPacket << ICQ_VERSION
              << ICQ_CMDxSND_PING2
              << (unsigned short)icqOwner.sequence()
              << icqOwner.uin()
   ;
   sendICQ(udpServer, pingPacket, ICQ_CMDxSND_PING2, icqOwner.sequenceVal - 1);*/
}


//-----ICQ::createAckPacket------------------------------------------------------------------------
void ICQ::createAckPacket(Packet &ack, unsigned short theSequence)
{
   ack.clearPacket();
   ack << ICQ_VERSION
       << ICQ_CMDxSND_ACK
       << theSequence
       << icqOwner.uin();
   ;
}


//-----ICQ::unixToDos------------------------------------------------------------------------------
void ICQ::unixToDos(const char *oldStr, char *newStr)
// convert a unix style string (0x0A for returns) to a dos style string (0x0A 0x0D)
// also encodes the string if necessary
{
   unsigned long j = 0;
   for (unsigned long i = 0; i <= strlen(oldStr); i++) 
   {
      if (oldStr[i] == (char)0x0A) newStr[j++] = 0x0D;
      newStr[j++] = oldStr[i];
   }
   // translating string with Translation Table
   licqTrans->translateToServer (newStr);

}


//-----ICQ::sendMessage----------------------------------------------------------------------------
ICQEvent *ICQ::sendMessage(ICQUser *u, const char *m, bool online, unsigned long id = 0)
{
   ICQEvent *result;
   char *mDos = new char[strlen(m) * 2 + 1];
   unixToDos(m, mDos);

   if (!online) // send offline
      result = sendOfflineTcp(u, ICQ_CMDxTCP_MSG, "message", mDos, id);
   else
   {
      Packet packet;
      addTcpHeader(packet, u, ICQ_CMDxTCP_START, ICQ_CMDxTCP_MSG, mDos, id);
      result = sendTcp(packet, u, "message", ICQ_CMDxTCP_MSG, id);
   }
   
   if (result != NULL) addToHistory(u, "message", m);
   delete mDos;
   return (result);
}


//-----ICQ::sendReadAwayMsg------------------------------------------------------------------------
ICQEvent *ICQ::sendReadAwayMsg(ICQUser *u, bool online, unsigned long id = 0)
{
   ICQEvent *result;

   if (!online) // send offline
      result = sendOfflineTcp(u, ICQ_CMDxTCP_READxAWAYxMSG, "away message request", "", id);
   else
   {
      Packet packet;
      addTcpHeader(packet, u, ICQ_CMDxTCP_START, ICQ_CMDxTCP_READxAWAYxMSG, "", id);
      result = sendTcp(packet, u, "away message request", ICQ_CMDxTCP_READxAWAYxMSG, id);
   }
     
   return (result);   
}


//-----ICQ::sendUrl--------------------------------------------------------------------------------
ICQEvent *ICQ::sendUrl(ICQUser *u, const char *description, const char *url, bool online, unsigned long id = 0)
{
   // make the URL info string
   char mDosUrl[strlen(url) * 2 + 1], mDosDesc[strlen(description) * 2 + 1];
   unixToDos(url, mDosUrl);
   unixToDos(description, mDosDesc);
   char m[strlen(mDosUrl) + strlen(mDosDesc) + 2];

   unsigned short i, j;
   for (i = 0; i < strlen(url); i++) 
      m[i] = mDosUrl[i];
   m[i++] = (char)0xFE;
   for (j = 0; j <= strlen(description); j++) 
      m[i++] = mDosDesc[j];
   
   ICQEvent *result = NULL;
   //char *mDos = new char[strlen(m) * 2 + 1];
   //unixToDos(m, mDos);

   if (!online) // send offline
      result = sendOfflineTcp(u, ICQ_CMDxTCP_URL, "url", m, id);
   else
   {
      Packet packet;
      addTcpHeader(packet, u, ICQ_CMDxTCP_START, ICQ_CMDxTCP_URL, m, id);
      result = sendTcp(packet, u, "url", ICQ_CMDxTCP_URL, id);
   }
     
   if (result != NULL) addToHistory(u, "url", m);
   //delete mDos;
   return (result);
}


//-----ICQ::sendChat-------------------------------------------------------------------------------
ICQEvent *ICQ::sendChat(ICQUser *u, const char *reason, bool online, unsigned long id = 0)
{
   ICQEvent *result;
   char *mDos = new char[strlen(reason) * 2 + 1];
   unixToDos(reason, mDos);

   if (!online) // send offline
      result = sendOfflineTcp(u, ICQ_CMDxTCP_CHAT, "chat request", mDos, id);
   else
   {
      Packet packet;
      addTcpHeader(packet, u, ICQ_CMDxTCP_START, ICQ_CMDxTCP_CHAT, mDos, id);
      packet << 0x00000001ul
             << 0x00000000ul
             << (unsigned short)0x0000 
             << (char)0x00
      ;
      result = sendTcp(packet, u, "chat request", ICQ_CMDxTCP_CHAT, id);
   }
     
   delete mDos;
   return (result);
}


//-----ICQ::chatCancel-------------------------------------------------------------------------
void ICQ::chatCancel(ICQUser *u, unsigned long seq, bool online, unsigned long id = 0)
{
   // basically a fancy tcp ack packet which is sent late

   /* 50 A5 82 00 03 00 D0 07 00 00 50 A5 82 00 02 00 01 00 00 CF 60 AD D3 CF 
      60 AD D3 28 12 00 00 04 00 00 00 00 01 00 00 00 00 00 00 00 00 00 00 06 
      00 00 00 */

   Packet packet;
   addTcpHeader(packet, u, ICQ_CMDxTCP_CANCEL, ICQ_CMDxTCP_CHAT, "");
   packet << 0x00000001ul
          << 0x00000000ul
          << (unsigned short)0x0000
          << (char)0x00
   ;
   sendTcpAck(packet, u, seq);
}


//-----ICQ::chatRefuse-----------------------------------------------------------------------------
void ICQ::chatRefuse(ICQUser *u, char *reason, unsigned long theSequence)
{
   // basically a fancy tcp ack packet which is sent late

   /* 50 A5 82 00 03 00 DA 07 00 00 50 A5 82 00 02 00 03 00 6E 6F 00 CF 60 AD 
      95 CF 60 AD 95 1E 3C 00 00 04 01 00 00 00 01 00 00 00 00 00 00 00 00 00 
      00 01 00 00 00 */

   Packet packet;
   addTcpHeader(packet, u, ICQ_CMDxTCP_ACK, ICQ_CMDxTCP_CHAT, "", 0, false);
   packet << 0x00000001ul
          << 0x00000000ul
          << (unsigned short)0x0000
          << (char)0x00
   ;
   sendTcpAck(packet, u, theSequence);
}


//-----ICQ::chatAccept-----------------------------------------------------------------------------
void ICQ::chatAccept(ICQUser *u, unsigned short thePort, unsigned long theSequence)
{
   // basically a fancy tcp ack packet which is sent late

   /* 50 A5 82 00 03 00 DA 07 00 00 50 A5 82 00 02 00 01 00 00 CF 60 AD 95 CF 
      60 AD 95 1E 3C 00 00 04 00 00 00 00 01 00 00 40 78 00 00 78 40 00 00 02 
      00 00 00 */

   char backwardsPort[2];
   backwardsPort[1] = ((char *)(&thePort))[0];
   backwardsPort[0] = ((char *)(&thePort))[1];
   
   Packet packet;
   addTcpHeader(packet, u, ICQ_CMDxTCP_ACK, ICQ_CMDxTCP_CHAT, "");
   packet << (unsigned short)0x0001
          << (char)0x00
          << backwardsPort[0]
          << backwardsPort[1]
          << (unsigned short)0x0000
          << (unsigned long)thePort
   ;
   sendTcpAck(packet, u, theSequence);
}


//-----ICQ::addTcpHeader---------------------------------------------------------------------------
void ICQ::addTcpHeader(Packet &packet, ICQUser *u, unsigned short cmd, unsigned short subCmd, char *m, unsigned long fromUin = 0, bool accept = true)
{
   /* set the status indicator
      mapping: online   -> 0x0000  this is one the same at least
               away     -> 0x0004
               NA       -> 0x000E
               DND      -> 0x000A
               occupied -> 0x0009
               refusal  -> 0x0001  this gives a weird message on the receivers end
   */   
   unsigned long status;
   if (accept) 
   {
      switch (icqOwner.status()) 
      {
      case ICQ_STATUS_ONLINE: status = 0x0000; break;
      case ICQ_STATUS_AWAY: status = 0x0004; break;
      case ICQ_STATUS_NA: status = 0x000E; break;
      case ICQ_STATUS_DND: status = 0x000A; break;
      case ICQ_STATUS_OCCUPIED: status = 0x0009; break;
      case ICQ_STATUS_FREEFORCHAT: status = 0x0000; break;  // ??
      case ICQ_STATUS_PRIVATE: status = 0x0000; break;      // ??
      default: status = 0x0000; break;
      }
   }
   else status = 0x0001;

   if (fromUin == 0) fromUin = icqOwner.uin();
   packet.clearPacket();
   packet << fromUin
          << (unsigned short)0x0003  //ICQ_VERSION
          << cmd   // ICQ_CMDxTCP_ACK, ICQ_CMDxTCP_START, ICQ_CMDxTCP_CANCEL
          << (unsigned short)0x0000
          << fromUin
          << subCmd
          << (unsigned short)(strlen(m) + 1)  //  message length
          << m
          << ipToIcq(u->tcpSocket.localIP())
          << LOCALHOST
          << (unsigned long)u->tcpSocket.localPort()
          << (char)0x04
          << status
   ;
}


//-----ICQ::addToHistory---------------------------------------------------------------------------
void ICQ::addToHistory(ICQUser *u, char *cmdStr, const char *m)
{
   // add event to history file
   Message newMesg(m, 0);
   if (u->history != NULL) 
   {
      char msg[strlen(newMesg.msg()) + 256];
      sprintf(msg, "%s from %s sent %s:\n%s\n\n", cmdStr, icqOwner.alias(), newMesg.timeRec(), newMesg.msg());
      u->history->writeBlock(msg, strlen(msg));
      u->history->flush();
   }
}


//-----ICQ::sendTcp--------------------------------------------------------------------------------
ICQEvent *ICQ::sendTcp(Packet &packet, ICQUser *u, const char *cmdStr, unsigned short cmd, unsigned long id = 0)
{
   if (id == 0) id = icqOwner.uin();

   if (!u->tcpSocket.connected())
   {
      outputWindow->wprintf("%C%sConnecting to %d on port %d...", COLOR_SEND, TCP, u->uin(), u->tcpSocket.remotePort());
      u->tcpSocket.openConnection();
      
      if (!u->tcpSocket.connected()) { outputWindow->wprintf("%C      connect failed.", COLOR_ERROR); return (NULL); }
      else 
      {
         connect(u->tcpSocket.sn, SIGNAL(activated(int)), this, SLOT(recvTCP(int)));
         outputWindow->wprintf("%C      connected...%Cinitializing connection...", COLOR_RECEIVE, COLOR_SEND);
      }
      Packet helloPacket;
      helloPacket << ICQ_CMDxTCP_HANDSHAKE
                  << (unsigned short)0x0000
                  << 0x00000000ul
                  << (char)0x00
                  << id  // typically icqOwner.uin()
                  << LOCALHOST << LOCALHOST
                  << (char)0x04 << 0x0000327Cul
      ;
      if (!u->tcpSocket.sendPacket(helloPacket)) { outputWindow->wprintf("%Csend failed.", COLOR_ERROR); return(NULL); }
      outputWindow->wprintf("%C      setup completed.", COLOR_RECEIVE);
   }

   // add the extra info to the packet
   packet << u->sequence()
          << (char)0x4C
          << (unsigned short)INT_VERSION
   ;

   outputWindow->wprintf("%C%sSending %s (%d).", COLOR_SEND, TCP, cmdStr, u->sequenceVal - 1);
   return(sendICQ(u->tcpSocket, packet, ICQ_CMDxTCP_START, u->sequenceVal - 1, u->uin(), cmd));
}


//-----ICQ::sendTcpAck-----------------------------------------------------------------------------
void ICQ::sendTcpAck(Packet &packet, ICQUser *u, unsigned long theSequence)
{
   packet << theSequence;
   u->tcpSocket.sendPacket(packet);
   if (DEBUG_LEVEL >= 2) cout << "Packet sent (TCP ack):\n" << packet.print() << endl;
}



//-----ICQ::sendOfflineTcp-------------------------------------------------------------------------
ICQEvent *ICQ::sendOfflineTcp(ICQUser *user, unsigned short cmd, char *cmdStr, char *m, unsigned long id = 0)
{
   if (id == 0) id = icqOwner.uin();

   Packet packet;
   packet << ICQ_VERSION
          << ICQ_CMDxSND_THRUxSERVER
          << (unsigned short)icqOwner.sequence()
          << id
          << user->uin()
          << cmd
          << (unsigned short)(strlen(m) + 1)
          << m
   ;
   outputWindow->wprintf("%C%sSending %s through server...", COLOR_SEND, UDP, cmdStr);
   return(sendICQ(udpServer, packet, ICQ_CMDxSND_THRUxSERVER, icqOwner.sequenceVal - 1, user->uin(), cmd));
}


//-----ICQ::setStatus------------------------------------------------------------------------------
bool ICQ::setStatus(unsigned short newStatus)
{
   if (!udpServer.connected()) return(false);

   Packet packet;
   packet << ICQ_VERSION
          << ICQ_CMDxSND_SETxSTATUS
          << (unsigned short)icqOwner.sequence()
          << icqOwner.uin()
          << newStatus
   ;
   outputWindow->wprintf("%C%sSending set status packet...", COLOR_SEND, UDP);
   sendICQ(udpServer, packet, ICQ_CMDxSND_SETxSTATUS, icqOwner.sequenceVal - 1);
   
   desiredStatus = newStatus;
   return(true);
}


//-----ICQ::updateAllUsers-------------------------------------------------------------------------
void ICQ::updateAllUsers()
{
   for (unsigned short i = 0; i < users.size(); i++) 
//      if (!allUsers) 
         getUserInfo(users[i]);
//      else if (users[i]->uin() == atol(users[i]->alias()))   // only update if the alias is the uin (ie just has default user info)
//         getUserInfo(users[i]);
}


//-----ICQ::getUserInfo----------------------------------------------------------------------------
bool ICQ::getUserInfo(ICQUser *user)
{
   Packet request;
   // request packet: 02 00 60 04 B7 00 BA 95 47 00 0A 00 8F 76 20 00
   request << ICQ_VERSION
           << ICQ_CMDxSND_USERxGETINFO
           << (unsigned short)icqOwner.sequence()
           << icqOwner.uin()
           << (unsigned short)0x000A  // some kind of info sequence...increments with each call to user info, and is unique to each user
           << user->uin()
   ;
   outputWindow->wprintf("%C%sSending user info request...", COLOR_SEND, UDP);
   sendICQ(udpServer, request, ICQ_CMDxSND_USERxGETINFO, icqOwner.sequenceVal - 1);
   return(true);
   /* note that there is another use packet you can send to get further information about the user such as city and province
      of residence but we don't bother with it cuz it's pretty useless stuff */
}


//-----ICQ::authorize------------------------------------------------------------------------------
void ICQ::authorize(unsigned long uinToAuthorize)
// authorize a user to add you to their contact list...for stupid windows people
{
   Packet packet;
   /* 02 00 56 04 05 00 50 A5 82 00 A7 B8 19 00 08 00 01 00 00 */
   packet << ICQ_VERSION
          << ICQ_CMDxSND_AUTHORIZE
          << (unsigned short)icqOwner.sequence()
          << icqOwner.uin()
          << uinToAuthorize
          << (unsigned long)0x00010008   // who knows, seems to be constant
          << (char)0x00
   ;
   outputWindow->wprintf("%C%sSending authorization (%d)...", COLOR_SEND, UDP, icqOwner.sequenceVal - 1);
   sendICQ(udpServer, packet, ICQ_CMDxSND_AUTHORIZE,  icqOwner.sequenceVal - 1);
}


//-----ICQ::requestSystemMsg-----------------------------------------------------------------------
void ICQ::requestSystemMsg(void)
// request offline system messages
{
   Packet newPacket;
   /* 02 00 4C 04 02 00 50 A5 82 00 */
   newPacket << ICQ_VERSION
             << ICQ_CMDxSND_SYSxMSGxREQ
             << (unsigned short)icqOwner.sequence()
             << icqOwner.uin()
   ;
   outputWindow->wprintf("%C%sSending offline system message request (%d)...", COLOR_SEND, UDP, icqOwner.sequenceVal - 1);
   sendICQ(udpServer, newPacket, ICQ_CMDxSND_SYSxMSGxREQ,  icqOwner.sequenceVal - 1);
}


//-----ICQ::moveToTop------------------------------------------------------------------------------
void ICQ::moveToTop(ICQUser *u)
{
   unsigned short uNum = 0; 
   while (users[uNum] != u) uNum++;  // set uNum to be the index of the user
   unsigned short i;
   for (i = uNum; i > 0; i--)
   {
      if (users[uNum]->status() == ICQ_STATUS_OFFLINE && users[i - 1]->status() != ICQ_STATUS_OFFLINE) break;
      users[i] = users[i - 1];
   }
   users[i] = u;
   
   if (allowUpdateUsers) emit updatedUsers();  // update the user window if we should
}


//-----ICQ::moveOnline-----------------------------------------------------------------------------
//void ICQ::moveOnline(ICQUser *u)
//{
   // moving a user online simply means pushing them to the top of the list
//   moveToTop(u);
//}


// Order the user in the list based on the status...
void ICQ::moveOnline(ICQUser *u) 
{
   unsigned short uNum = 0;
   unsigned short newStatus = u->status();
   unsigned short i;
   switch(newStatus) 
   {
   case ICQ_STATUS_ONLINE:
   case ICQ_STATUS_AWAY:
   case ICQ_STATUS_NA:
   case ICQ_STATUS_OCCUPIED:
   case ICQ_STATUS_DND:    
     while (uNum < users.size() && users[uNum] != u) uNum++; // find the user index number .. 
     if (uNum == users.size()) return; // quit if user not found
     
     // decide which way we need to go in the list
     if(uNum != 0 && newStatus < users[uNum - 1]->status()) 
     {
       // Moving up (graphically)... 
       for(i = uNum; (i > 0) && (newStatus < users[i - 1]->status()); i--) 
 	      users[i] = users[i - 1];
       users[i] = u;
     } 
     else if (uNum != users.size() - 1 && newStatus > users[uNum + 1]->status()) 
     {
       // Moving down (graphically)...
       for(i = uNum; ((i < users.size()) && (newStatus > users[i + 1]->status())); i++) 
 	      users[i] = users[i + 1];
       users[i] = u;
     } // else stay in the same place
     break;
     
   case ICQ_STATUS_FREEFORCHAT:
   case ICQ_STATUS_PRIVATE:
     // Move user to top of list...
     moveToTop(u);
     break;
     
   default:
     break;
   };
}


//-----ICQ::moveOffline----------------------------------------------------------------------------
void ICQ::moveOffline(ICQUser *u)
{
   // assume the given user is online and push them down to the top of the offline users
   unsigned short uNum = 0; while (uNum < users.size() && users[uNum] != u) uNum++;  // now uNum is the number of the given user
   if (uNum == users.size()) return;  // user not found
   unsigned short i;
   for (i = uNum; i < users.size() - 1; i++) 
   {
      if (users[i + 1]->status() == ICQ_STATUS_OFFLINE) break;
      users[i] = users[i + 1];
   }
   users[i] = u;

   if (allowUpdateUsers) emit updatedUsers();  // update the user window if we should
}


//-----ICQ::processUdpPacket-----------------------------------------------------------------------
unsigned short ICQ::processUdpPacket(Packet &packet)
{
   unsigned short version, command, newCommand, theSequence, searchSequence, userPort, newStatus, messageLen, checkSequence, junkShort;
   unsigned long checkUin, userIP, junkLong;
   char junkChar;
   int i;
   ICQUser *u;
   Packet newPacket;

   if (DEBUG_LEVEL >= 2) cout << "Packet received (UDP):\n" << packet.print() << endl;
   
   // read in the standard UDP header info
   packet >> version 
          >> command 
          >> theSequence
   ;

   if(version != ICQ_VERSION) { outputWindow->wprintf("Bad version number %d\n", version); return(0xFFFF); }
   
   switch (command) 
   {
   case ICQ_CMDxRCV_USERxONLINE:   // initial user status packet
      /* 02 00 6E 00 0B 00 8F 76 20 00 CD CD 77 90 3F 50 00 00 7F 00 00 01 04 00
         00 00 00 03 00 00 00 */
      ackUDP(theSequence);
      packet >> checkUin;
      outputWindow->wprintf("%C%sUser %C%ld%C is online.", COLOR_RECEIVE, UDP, COLOR_DATA, checkUin, COLOR_RECEIVE);
      
      // find which user it is, verify we have them on our list
      if ((u = getUserByUIN(checkUin)) == NULL) { outputWindow->wprintf("ICQ sent unknown user, %d!", checkUin); break; }

      // read in the relevant user information
      packet >> userIP
             >> userPort
             >> junkLong >> junkShort >> junkChar  // 7 bytes of junk
             >> newStatus  // initial status of user, anything but offline
      ;
      
      userIP = icqToIp(userIP);
      u->tcpSocket.setDestination(userIP, userPort);
      u->statusVal = newStatus;
      u->setAwayMessage(NULL);
      if (u->onlineNotify()) playSound(soundNotify);
      if (newStatus != ICQ_STATUS_OFFLINE) moveOnline(u);  // put the user at the top of the list
      break;
   
   case ICQ_CMDxRCV_USERxOFFLINE:  // user just went offline packet
      /* 02 00 78 00 06 00 ED 21 4E 00 */
      ackUDP(theSequence);
      packet >> checkUin;
      outputWindow->wprintf("%C%sUser %C%ld%C went offline.", COLOR_RECEIVE, UDP, COLOR_DATA, checkUin, COLOR_RECEIVE);

      // find which user it is, verify we have them on our list
      if ((u = getUserByUIN(checkUin)) == NULL) { outputWindow->wprintf("ICQ sent unknown user, %d!", checkUin); break; }
      
      u->statusVal = ICQ_STATUS_OFFLINE;
      u->setAwayMessage(NULL);
      //u->tcpSocket.closeConnection();
      
      moveOffline(u);
      break;

   case ICQ_CMDxRCV_USERxINFO:   // user info packet
      /* 02 00 18 01 6C 00 10 00 50 A5 82 00 08 00 48 61 63 6B 49 43 51 00 04 00
         46 6F 6F 00 04 00 42 61 72 00 15 00 68 61 63 6B 65 72 73 40 75 77 61 74 
         65 72 6C 6F 6F 2E 63 61 00 00 00 00  */
      ackUDP(theSequence);
      packet >> checkSequence  // corresponds to the sequence number from the user information request packet...totally irrelevant.
             >> checkUin
      ;
      outputWindow->wprintf("%C%sUser information packet uin %C%ld:", COLOR_RECEIVE, UDP, COLOR_DATA, checkUin);
      // find which user it is, verify we have them on our list
      if ((u = getUserByUIN(checkUin)) == NULL) { outputWindow->wprintf("ICQ sent unknown user, %d!", checkUin); break; }
      
      // read in the four data fields; alias, first name, last name, and email address
      packet >> messageLen; 
      for (i = 0; i < messageLen; i++) packet >> u->aliasVal[i];
      packet >> messageLen; 
      for (i = 0; i < messageLen; i++) packet >> u->firstNameVal[i];
      packet >> messageLen; 
      for (i = 0; i < messageLen; i++) packet >> u->lastNameVal[i];
      packet >> messageLen; 
      for (i = 0; i < messageLen; i++) packet >> u->emailVal[i];


      // translating string with Translation Table
      licqTrans->translateToClient(u->aliasVal);
      licqTrans->translateToClient(u->firstNameVal);
      licqTrans->translateToClient(u->lastNameVal);

      // print out the user information
      outputWindow->wprintf("%C  %s (%s %s), %s.", COLOR_DATA, u->aliasVal, u->firstNameVal, u->lastNameVal, u->emailVal); 
      
      // save the user infomation
      u->saveInfo();

      emit doneUserInfo(true, checkUin);
      emit updatedUsers();
      break;
   
   case ICQ_CMDxRCV_USERxINVALIDxUIN:  // not a good uin
      ackUDP(theSequence);
      packet >> checkUin;
      outputWindow->wprintf("%sInvalid uin: ", UDP, checkUin);
      emit doneUserInfo(false, checkUin);
      break;

   case ICQ_CMDxRCV_USERxSTATUS:  // user changed status packet
      ackUDP(theSequence);
      packet >> checkUin;
      outputWindow->wprintf("%C%sUser %C%ld%C changed status", COLOR_RECEIVE, UDP, COLOR_DATA, checkUin, COLOR_RECEIVE);

      if ((u = getUserByUIN(checkUin)) == NULL) { outputWindow->wprintf("ICQ sent unknown user, %d!", checkUin); break; }
      
      packet >> u->statusVal;
      moveOnline(u);
      u->setAwayMessage(NULL);
      emit updatedUsers();
      break;

   case ICQ_CMDxRCV_USERxLISTxDONE:  // end of user list
      /* 02 00 1C 02 05 00 8F 76 20 00 */
      ackUDP(theSequence);
      outputWindow->wprintf("%C%sEnd of user list.", COLOR_RECEIVE, UDP);
      allowUpdateUsers = true;
      emit updatedUsers();
      break;

   case ICQ_CMDxRCV_SEARCHxFOUND:  // user found in search 
      /* 02 00 8C 00 03 00 05 00 8F 76 20 00 0B 00 41 70 6F 74 68 65 6F 73 69 73
         00 07 00 47 72 61 68 61 6D 00 05 00 52 6F 66 66 00 13 00 67 72 6F 66 66 40 75
         77 61 74 65 72 6C 6F 6F 2E 63 61 00 01 02 */
      ackUDP(theSequence);
      outputWindow->wprintf("%C%sSearch found user:", COLOR_RECEIVE, UDP);

      unsigned short i, aliasLen, firstNameLen, lastNameLen, emailLen;
      char auth;
      struct UserStatusline us;

      packet >> searchSequence
	          >> checkUin
      ;
	   sprintf(us.uin, "%ld", checkUin);

      packet >> aliasLen;
      for (i = 0; i < aliasLen; i++) packet >> us.alias[i];
      us.alias[i] = '\0';
      
      packet >> firstNameLen;
      for (i = 0; i < firstNameLen; i++) packet >> us.firstname[i];
      us.firstname[i] = '\0';
      
      packet >> lastNameLen;
      for (i = 0; i < lastNameLen; i++) packet >> us.lastname[i];
      us.lastname[i] = '\0';

      packet >> emailLen;
      for (i = 0; i < emailLen; i++) packet >> us.email[i];
      us.email[i] = '\0';

      // translating string with Translation Table
      licqTrans->translateToClient(us.alias);
      licqTrans->translateToClient(us.firstname);
      licqTrans->translateToClient(us.lastname);

      packet >> auth;
      sprintf(us.name, "%s %s", us.firstname, us.lastname);
      outputWindow->wprintf("%C      %s (%ld) <%s %s, %s>", COLOR_RECEIVE, us.alias, 
                            checkUin, us.firstname, us.lastname, us.email);
      
      emit userFound(us, searchSequence);
      break;

   case ICQ_CMDxRCV_SEARCHxDONE:  // user found in search 
      /* 02 00 A0 00 04 00 05 00 00*/
      ackUDP(theSequence);
      
      char more;
      packet >> searchSequence 
             >> more
      ;
      outputWindow->wprintf("%C%sSearch finished.", COLOR_RECEIVE, UDP);
      emit doneSearch(searchSequence, more);
      break;
   
   case ICQ_CMDxRCV_SYSxMSGxDONE:  // end of system messages
      /* 02 00 E6 00 04 00 50 A5 82 00 */

      ackUDP(theSequence);
      outputWindow->wprintf("%C%sEnd of system messages.", COLOR_RECEIVE, UDP);
      
      // send special ack for this one as well as the usual ack
      /* 02 00 42 04 04 00 50 A5 82 00 */
      newPacket << ICQ_VERSION
                << ICQ_CMDxSND_SYSxMSGxDONExACK
                << theSequence
                << icqOwner.uin()
      ;
      outputWindow->wprintf("%C%sSending end of system messages ack...", COLOR_SEND, UDP);
      sendICQ(udpServer, newPacket, ICQ_CMDxSND_SYSxMSGxDONExACK, theSequence);
      break;
        
   case ICQ_CMDxRCV_SYSxMSGxOFFLINE:  // offline system message, now have to check the sub-command
      /* 02 00 DC 00 0A 00 EC C9 45 00 CE 07 04 17 04 21 01 00 3F 00 6E 6F 2C 20 
         73 74 69 6C 6C 20 68 6F 70 69 6E 67 20 66 6F 72 20 74 68 65 20 72 65 63 
         6F 72 64 20 63 6F 6D 70 61 6E 79 2C 20 62 75 74 20 79 6F 75 20 6E 65 76 
         65 72 20 6B 6E 6F 77 2E 2E 2E 00 */
      ackUDP(theSequence);
      outputWindow->wprintf("%C%sOffline system message:", COLOR_RECEIVE, UDP);

      unsigned short yearSent;
      char monthSent, daySent, hourSent, minSent;
      packet >> checkUin
             >> yearSent
             >> monthSent
             >> daySent
             >> hourSent
             >> minSent
             >> newCommand
      ;
      // prepare a structure containing the relevant time information
      struct tm sentTime;
      sentTime.tm_sec  = 0;
      sentTime.tm_min  = minSent;
      sentTime.tm_hour = hourSent;
      sentTime.tm_mday = daySent;
      sentTime.tm_mon  = monthSent - 1;
      sentTime.tm_year = yearSent - 1900;
      
      // process the system message, sending the time it occured converted to a time_t structure
      tmpSoundEnabled = false;  // disable sound for offline system messages to avoid segfaults on multiple messages
      processSystemMessage(packet, checkUin, newCommand, mktime(&sentTime));
      break;
   
   case ICQ_CMDxRCV_SYSxMSGxONLINE:  // online system message, now have to check the sub-command
      ackUDP(theSequence);
      outputWindow->wprintf("%C%sOnline system message:", COLOR_RECEIVE, UDP);

      packet >> checkUin
             >> newCommand
      ;
      
      // process the system message, sending the current time as a time_t structure
      processSystemMessage(packet, checkUin, newCommand, time(NULL));
      break;
   
   case ICQ_CMDxRCV_SETxOFFLINE:  // we got put offline by mirabilis for some reason
      outputWindow->wprintf("%C%sKicked offline.", COLOR_RECEIVE, UDP);
      logoff(autoReconnect);
      break;
      
   case ICQ_CMDxRCV_ACK:  // icq acknowledgement
      /* 02 00 0A 00 12 00 */

      outputWindow->wprintf("%C%sack (%d)", COLOR_RECEIVE, UDP, theSequence);
      doneEvent(true, udpServer.descriptor(), theSequence);
      break;

   case ICQ_CMDxRCV_ERROR:  // icq says go away
      outputWindow->wprintf("%C%sMirabilis says bugger off.", COLOR_RECEIVE, UDP);
      break;
   
   case ICQ_CMDxRCV_HELLO: // hello packet from mirabilis received on logon
      /* 02 00 5A 00 00 00 8F 76 20 00 CD CD 76 10 02 00 01 00 05 00 00 00 00 00
         8C 00 00 00 F0 00 0A 00 0A 00 05 00 0A 00 01 */
      ackUDP(theSequence);
      outputWindow->wprintf("%C%sReceived hello from mirabilis.", COLOR_RECEIVE, UDP);
      
      requestSystemMsg();

      icqOwner.statusVal = desiredStatus;
      pingTimer.start(PING_FREQUENCY * 1000);
      emit doneOwnerFcn(true, ICQ_CMDxSND_LOGON);
      emit updatedStatus();

      updateContactList();
      break;

   case ICQ_CMDxRCV_WRONGxPASSWD:  // incorrect password sent in logon
      /* 02 00 64 00 00 00 02 00 8F 76 20 00 */
      outputWindow->wprintf("%C%sIncorrect password: %C%s", COLOR_RECEIVE, UDP, COLOR_DATA, icqOwner.passwd());
      emit doneOwnerFcn(false, ICQ_CMDxSND_LOGON);
      break;

   case ICQ_CMDxRCV_BUSY:  // server too busy to respond
      outputWindow->wprintf("%sServer busy, try again in a few minutes.", UDP);
      emit (doneOwnerFcn(false, ICQ_CMDxSND_LOGON));
      break;
  
   default:  // what the heck is this packet?  print it out
      ackUDP(theSequence);
      outputWindow->wprintf("%C%sUnknown UDP packet:\n%s", COLOR_ERROR, UDP, packet.print());
      if (errorFile != NULL) 
      {
         fprintf(errorFile, "Unknown UDP packet:\n%s", packet.print());
         fflush(errorFile);
      }
      break;
   }
   
   return(command);
}


//-----ICQ::ackUDP---------------------------------------------------------------------------------
void ICQ::ackUDP(unsigned short theSequence)
{
   // acknowledge whatever packet we received using the relevant sequence number
   Packet ack;
   createAckPacket(ack, theSequence);
   if (DEBUG_LEVEL >= 2) cout << "Packet sent (UDP):\n" << ack.print() << endl;
   udpServer.sendPacket(ack);
}


//-----ICQ::processSystemMessage-------------------------------------------------------------------
void ICQ::processSystemMessage(Packet &packet, unsigned long checkUin, unsigned short newCommand, time_t timeSent)
{
   unsigned short messageLen;
   char *message, *sysMsg;
   ICQUser *u;
   int i, j;
   
   // prepare an empty message containing the date and time the system message occured
   // use a message because proper time formatting is done in the constructor for Message
   Message sm(" ", timeSent);

   switch(newCommand)
   {
   case ICQ_CMDxTCP_MSG:  // system message: message through the server 
      outputWindow->wprintf("  %C(%s) message through server from %C%d%C.", COLOR_RECEIVE, sm.timeRec(), COLOR_DATA, checkUin, COLOR_RECEIVE);

      packet >> messageLen;
      
      // read in message from the packet one character at a time
      message = new char[messageLen + 1];
      for (i = 0; i <= messageLen; i++) packet >> message[i];
      
      // translating string with Translation Table
      licqTrans->translateToClient (message);

      if ((u = getUserByUIN(checkUin)) == NULL) 
      { 
         outputWindow->wprintf("  Message from unknown user, %ld, adding them to your list.", checkUin);
         addUser(checkUin, NULL, true);
         u = getUserByUIN(checkUin);
      }
      
      if (tmpSoundEnabled) 
         playSound(soundMsg);
      else
         tmpSoundEnabled = true;
      
      u->addMessage(message, ICQ_CMDxRCV_SYSxMSGxONLINE, ICQ_CMDxTCP_MSG, 0, timeSent);
      delete message;
      moveToTop(u);
      break;
   
   case ICQ_CMDxTCP_URL:  // system message: url through the server 
      outputWindow->wprintf("  %C(%s) url through server from %C%d%C.", COLOR_RECEIVE, sm.timeRec(), COLOR_DATA, checkUin, COLOR_RECEIVE);

      packet >> messageLen;

      // read in message from the packet one character at a time
      message = new char[messageLen + 1];
      for (i = 0; i <= messageLen; i++) packet >> message[i];

      // parse the message into url and url description
      char *urlDescription, *urlName, *url;
      urlDescription = new char[messageLen];
      urlName = new char[messageLen];
      url = new char[messageLen + 32];
      i = j = 0; 
      while (i < messageLen && message[i] != (char)0xFE) urlDescription[j++] = message[i++];
      urlDescription[j] = '\0';
      j = 0; i++;
      while (i < messageLen) urlName[j++] = message[i++];
      urlName[j] = '\0';
      
      // format the url and url description into a message and add it to the users list
      sprintf(url, "URL: %s  \n%s", urlName, urlDescription);

      // translating string with Translation Table
      licqTrans->translateToClient (url);

      if ((u = getUserByUIN(checkUin)) == NULL) 
      { 
         outputWindow->wprintf("  Message from unknown user, %ld, adding them to your list.", checkUin);
         addUser(checkUin, NULL, true);
         u = getUserByUIN(checkUin);
      }

      if (tmpSoundEnabled) 
         playSound(soundUrl);
      else
         tmpSoundEnabled = true;

      u->addMessage(url, ICQ_CMDxRCV_SYSxMSGxONLINE, ICQ_CMDxTCP_URL, 0, timeSent);
      delete message;
      delete urlDescription;
      delete urlName;
      delete url;
      moveToTop(u);
      break;
      
   case ICQ_CMDxRCV_SMxREQxAUTH:  // system message: authorisation request
      /* 02 00 04 01 08 00 8F 76 20 00 06 00 41 00 41 70 6F 74 68 65 6F 73 69 73
         FE 47 72 61 68 61 6D FE 52 6F 66 66 FE 67 72 6F 66 66 40 75 77 61 74 65
         72 6C 6F 6F 2E 63 61 FE 31 FE 50 6C 65 61 73 65 20 61 75 74 68 6F 72 69 
         7A 65 20 6D 65 2E 00 */
      outputWindow->wprintf("  (%s) authorization request from %ld.", sm.timeRec(), checkUin);
      packet >> messageLen;
      message = new char[messageLen + 1];
      for (i = 0; i <= messageLen; i++) 
      {
         packet >> message[i];
         if (message[i] == (char)0xFE) message[i] = '\n';
      }
      
      // translating string with Translation Table
      licqTrans->translateToClient (message);

      sysMsg = new char[messageLen + 128];
      sprintf(sysMsg, "(%s) authorization request from %ld:\n%s", sm.timeRec(), checkUin, message);
      icqOwner.addMessage(sysMsg, ICQ_CMDxRCV_SYSxMSGxONLINE, ICQ_CMDxRCV_SMxREQxAUTH, 0, timeSent);
      delete sysMsg;
      delete message;
      break;
   
   case ICQ_CMDxRCV_SMxADDED:  // system message: added to a contact list
      outputWindow->wprintf("  %C(%s) user %C%ld%C added you to their contact list.", COLOR_RECEIVE, sm.timeRec(), COLOR_DATA, checkUin, COLOR_RECEIVE);

      // there is a bunch of info about the given user in the packet but the read routine to get
      //   at it is totally broken right now
      unsigned short infoLen, j;
      packet >> infoLen;
      
      // declare all the strings we will need for reading in the user data
      char *userInfo, *aliasField, *firstNameField, *lastNameField, *emailField;
      userInfo =       new char[infoLen + 1];
      aliasField =     new char[infoLen];
      firstNameField = new char[infoLen];
      lastNameField =  new char[infoLen];
      emailField =     new char[infoLen];
      
      // read in the user data from the packet
      for (i = 0; i < infoLen; i++) packet >> userInfo[i];
      userInfo[infoLen] = '\0';
      
      // parse the user info string for the four fields
      i = j = 0; 
      while (i < infoLen && userInfo[i] != (char)0xFE) aliasField[j++] = userInfo[i++];
      aliasField[j] = '\0';
      j = 0; i++;
      while (i < infoLen && userInfo[i] != (char)0xFE) firstNameField[j++] = userInfo[i++];
      firstNameField[j] = '\0';
      j = 0; i++;
      while (i < infoLen && userInfo[i] != (char)0xFE) lastNameField[j++] = userInfo[i++];
      lastNameField[j] = '\0';
      j = 0; i++;
      while (i < infoLen && userInfo[i] != (char)0xFE) emailField[j++] = userInfo[i++];
      emailField[j] = '\0';
            
      // translating string with Translation Table
      licqTrans->translateToClient(aliasField);
      licqTrans->translateToClient(firstNameField);
      licqTrans->translateToClient(lastNameField);

      outputWindow->wprintf("  %s (%s %s), %s", aliasField, firstNameField, lastNameField, emailField); 
      
      sysMsg = new char[128];
      sprintf(sysMsg, "(%s) user %ld (%s, %s %s, %s) added you to their contact list.", sm.timeRec(), checkUin, aliasField, firstNameField, lastNameField, emailField);
      icqOwner.addMessage(sysMsg, ICQ_CMDxRCV_SYSxMSGxONLINE, ICQ_CMDxRCV_SMxADDED, 0, timeSent);
      delete sysMsg;
      
      delete userInfo; delete aliasField; delete firstNameField; delete lastNameField; delete emailField;
      
      break;
   
   default:
      outputWindow->wprintf("  %C(%s) unknown system message: %C%d", COLOR_ERROR, sm.timeRec(), COLOR_DATA, newCommand);
      sysMsg = new char[128];
      sprintf(sysMsg, "(%s) unknown system message: %04X", sm.timeRec(), newCommand);
      icqOwner.addMessage(sysMsg, timeSent);
      delete sysMsg;
      outputWindow->wprintf("%C%sUnknown UDP packet:\n%s", COLOR_ERROR, UDP, packet.print());
      if (errorFile != NULL) 
      {
         fprintf(errorFile, "Unknown UDP packet (system message):\n%s", packet.print());
         fflush(errorFile);
      }
      break;
   }
}




//-----ICQ::processTcpPacket-----------------------------------------------------------------------
void ICQ::processTcpPacket(Packet &packet, int sockfd)
{
   unsigned long checkUin, theSequence, senderIp, localIp, userStatus, 
                  senderPort, junkLong, thePort;
   unsigned short version, command, junkShort, newCommand, messageLen, licqVersion;
   char licqChar, junkChar;
   unsigned short i, j;
   ICQUser *u;
   static unsigned long chatUin, chatSequence;
   

   if (DEBUG_LEVEL >= 2) cout << "Packet received (TCP):\n" << packet.print() << endl;
   
   packet >> checkUin
          >> version
          >> command      // so far either message stuff or message ack
          >> junkShort    // 00 00 to fill in the MSB of the command long int which is read in as a short
          >> checkUin
          >> newCommand   // if a message then what type, message/chat/read away message/...
          >> messageLen   // length of incoming message
   ;
   
   // read in the message
   char message[messageLen + 1];
   for (i = 0; i < messageLen; i++) packet >> message[i];
   
   // read in some more stuff common to all tcp packets
   packet >> senderIp
          >> localIp
          >> senderPort
          >> junkChar      // ??
          >> userStatus
   ;
   
   senderIp = icqToIp(senderIp);
   localIp = icqToIp(localIp);
      
   // find which user was sent
   if ((u = getUserByUIN(checkUin)) == NULL) 
   { 
      outputWindow->wprintf("%C%sUnknown user, %d, adding them to your list.", COLOR_RECEIVE, TCP, checkUin);
      addUser(checkUin, NULL, true);
      u = getUserByUIN(checkUin);
   }

   switch(command)
   {
   case ICQ_CMDxTCP_START:   // incoming tcp packet containing one of many possible things
      switch(newCommand)  // do a switch on what it could be
      {
      case ICQ_CMDxTCP_MSG:  // straight message from a user
         outputWindow->wprintf("%sMessage from %s (%d).", TCP, u->alias(), checkUin);
         
         packet >> theSequence >> licqChar >> licqVersion;
         if (licqChar == 'L') outputWindow->wprintf("      Sent by Licq v0.%d.", licqVersion);
         
         ackTCP(packet, u, newCommand, theSequence);
         playSound(soundMsg);
         
         // translating string with translation table    
         licqTrans->translateToClient (message);
         
         u->addMessage(message, ICQ_CMDxTCP_START, ICQ_CMDxTCP_MSG, theSequence, 0, (licqChar == 'L'), licqVersion);
         moveToTop(u);
         break;
      case ICQ_CMDxTCP_READxAWAYxMSG:  // read away message
         /* 76 1E 3F 00 03 00 EE 07 00 00 76 1E 3F 00 E8 03 01 00 00 81 61 1D 9D 81 61
            1D 9D C9 05 00 00 04 00 00 10 00 FE FF FF FF */
         outputWindow->wprintf("%s%s (%d) requested read of away message.", TCP, u->alias(), checkUin);
         
         packet >> theSequence >> licqChar >> licqVersion;
         if (licqChar == 'L') outputWindow->wprintf("      Sent by Licq v0.%d.", licqVersion);
         
         ackTCP(packet, u, newCommand, theSequence);
         break;

      case ICQ_CMDxTCP_URL:  // url sent
         /* BA 95 47 00 03 00 EE 07 00 00 BA 95 47 00 04 00 24 00 67 6F 6F 64 20 70 6F 
            72 6E 20 73 69 74 65 FE 68 74 74 70 3A 2F 2F 63 6F 6F 6C 70 6F 72 74 6E 2E 
            63 6F 6D 00 81 61 1D 9E 7F 00 00 01 3F 07 00 00 04 00 00 10 00 03 00 00 00 */
         outputWindow->wprintf("%sURL from %s (%d).", TCP, u->alias(), checkUin);
         
         packet >> theSequence >> licqChar >> licqVersion;
         if (licqChar == 'L') outputWindow->wprintf("      Sent by Licq v0.%d.", licqVersion);
         
         ackTCP(packet, u, newCommand, theSequence);

         // parse the message into url and url description
         char *urlDescription, *urlName, *url;
         urlDescription = new char[messageLen];
         urlName = new char[messageLen];
         url = new char[messageLen + 32];
         i = j = 0; 
         while (i < messageLen && message[i] != (char)0xFE) urlDescription[j++] = message[i++];
         urlDescription[j] = '\0';
         j = 0; i++;
         while (i < messageLen) urlName[j++] = message[i++];
         urlName[j] = '\0';
         
         // translating string with Translation Table
         licqTrans->translateToClient(urlDescription);
         licqTrans->translateToClient(urlName);
         
         // format the url and url description into a message and add it to the users list
         sprintf(url, "URL: %s  \n%s", urlName, urlDescription);
         playSound(soundUrl);
         u->addMessage(url, ICQ_CMDxTCP_START, ICQ_CMDxTCP_URL, theSequence, 0, (licqChar == 'L'), licqVersion);
         moveToTop(u);

         // clean up the temporary strings
         delete urlDescription; delete urlName; delete url;
         break;

      case ICQ_CMDxTCP_CHAT:
         /* 50 A5 82 00 03 00 EE 07 00 00 50 A5 82 00 02 00 0D 00 63 68 61 74 20 72
            65 71 75 65 73 74 00 CF 60 AD D3 CF 60 AD D3 28 12 00 00 04 00 00 10 00
            01 00 00 00 00 00 00 00 00 00 00 06 00 00 00 */
         outputWindow->wprintf("%sChat request from %s (%d).", TCP, u->alias(), checkUin);
         
         packet >> junkLong >> junkLong >> junkShort >> junkChar >> theSequence >> licqChar >> licqVersion;
         if (licqChar == 'L') outputWindow->wprintf("      Sent by Licq v0.%d.", licqVersion);

         char *chatReq;
         chatReq = new char[messageLen + 32];
         
         // translating string with translation table    
         licqTrans->translateToClient (message);
         
         sprintf(chatReq, "Chat request:\n   %s", message);
         playSound(soundChat);
         u->addMessage(chatReq, ICQ_CMDxTCP_START, ICQ_CMDxTCP_CHAT, theSequence, 0, (licqChar == 'L'), licqVersion);
         delete chatReq;
         moveToTop(u);
         break;

      case ICQ_CMDxTCP_FILE:
         /* 50 A5 82 00 03 00 EE 07 00 00 50 A5 82 00 03 00 0F 00 74 68 69 73 20 69
            73 20 61 20 66 69 6C 65 00 CF 60 AD D3 CF 60 AD D3 60 12 00 00 04 00 00
            10 00 00 00 00 00 09 00 4D 61 6B 65 66 69 6C 65 00 55 0C 00 00 00 00 00
            00 04 00 00 00 */
         outputWindow->wprintf("%sFile transfer request from %s (%d).", TCP, u->alias(), checkUin);
         
         packet >> junkLong >> junkLong >> junkShort >> junkChar >> theSequence >> licqChar >> licqVersion;
         if (licqChar == 'L') outputWindow->wprintf("      Sent by Licq v0.%d.", licqVersion);

         char *fileReq ;
         fileReq = new char[messageLen + 32];
                  
         // translating string with translation table    
         licqTrans->translateToClient (message);

         sprintf(fileReq, "File description:\n   %s", message);
         playSound(soundFile);
         u->addMessage(fileReq, ICQ_CMDxTCP_START, ICQ_CMDxTCP_FILE, theSequence, 0, (licqChar == 'L'), licqVersion);
         delete fileReq;
         moveToTop(u);
         break;      

      default:
         break;
      }
      break;

   case ICQ_CMDxTCP_ACK:  // message received packet      
      
      switch (newCommand) 
      {
      case ICQ_CMDxTCP_MSG:
         /* 8F 76 20 00 03 00 DA 07 00 00 8F 76 20 00 01 00 01 00 00 CF 60 AD D3 7F 
            00 00 01 5A 12 00 00 04 00 00 00 00 14 00 00 00 */
         // read in the sequence, assume it is the last long in the packet (it should be)
         packet >> theSequence;
         break;
      case ICQ_CMDxTCP_READxAWAYxMSG:
         packet >> theSequence;
         break;
      case ICQ_CMDxTCP_URL:
         packet >> theSequence;
         break;
      case ICQ_CMDxTCP_CHAT:
         /* 50 A5 82 00 03 00 DA 07 00 00 50 A5 82 00 02 00 03 00 6E 6F 00 CF 60 AD
            95 CF 60 AD 95 1E 3C 00 00 04 01 00 00 00 01 00 00 00 00 00 00 00 00 00
            00 01 00 00 00 */
         packet >> junkShort 
                >> junkChar
                >> junkLong   // port backwards
                >> thePort    // port to connect to for chat
                >> theSequence
         ;
         if (chatSequence != theSequence || chatUin != checkUin)  // only if this is the first chat ack packet
         {                                                        // do we do anything
            chatSequence = theSequence;
            chatUin = checkUin;
            emit eventResult(u, ICQ_CMDxTCP_CHAT, userStatus == 0x0000 ? true : false, thePort);
         }
         break;
      case ICQ_CMDxTCP_FILE:
         packet >> junkLong 
                >> junkLong 
                >> junkShort 
                >> junkChar
                >> theSequence
         ;
         break;
      default: break;
      }
      
      // output the away message if there is one (ie if user status is not online)
      
      // translating string with translation table    
      licqTrans->translateToClient (message);

      if (userStatus == 0x0000)
         outputWindow->wprintf("%C%sack from %s (%d).", COLOR_RECEIVE, TCP, u->alias(), theSequence);
      else if (userStatus == 0x0001) 
      {
         outputWindow->wprintf("%srefusal from %s (%d): %s", TCP, u->alias(), theSequence, message);
      }
      else
      {
         u->setAwayMessage(message);
         outputWindow->wprintf("%C%sack from %s (%d): %s", COLOR_RECEIVE, TCP, u->alias(), theSequence, message);
      }
      
      doneEvent(true, sockfd, theSequence);
      break;
      
   case ICQ_CMDxTCP_CANCEL:
      switch (newCommand) 
      {
      case ICQ_CMDxTCP_CHAT:
         /* 50 A5 82 00 03 00 D0 07 00 00 50 A5 82 00 02 00 01 00 00 CF 60 AD D3 CF
            60 AD D3 28 12 00 00 04 00 00 00 00 01 00 00 00 00 00 00 00 00 00 00 06
            00 00 00 */
         outputWindow->wprintf("%sChat request from %s (%ld) cancelled.", TCP, u->alias(), checkUin);
         char *chatReq;
         chatReq = new char[messageLen + 32];
         sprintf(chatReq, "Chat request cancelled.");
         u->addMessage(chatReq, ICQ_CMDxTCP_CANCEL, ICQ_CMDxTCP_CHAT, 0);
         delete chatReq;
         moveToTop(u);
         break;
      
      case ICQ_CMDxTCP_FILE:
         /* 50 A5 82 00 03 00 D0 07 00 00 50 A5 82 00 02 00 01 00 00 CF 60 AD D3 CF
            60 AD D3 28 12 00 00 04 00 00 00 00 01 00 00 00 00 00 00 00 00 00 00 06
            00 00 00 */
         outputWindow->wprintf("%sFile transfer request from %s (%ld) cancelled.", TCP, u->alias(), checkUin);
         char *fileReq;
         fileReq = new char[messageLen + 32];
         sprintf(fileReq, "File transfer request cancelled.");
         u->addMessage(fileReq, ICQ_CMDxTCP_CANCEL, ICQ_CMDxTCP_FILE, 0);
         delete fileReq;
         moveToTop(u);
         break;
      
      default:
         break;
      }
      break;
   
   default:
      outputWindow->wprintf("%C%sUnknown TCP packet: %C%d", COLOR_ERROR, TCP, COLOR_DATA, command);
      packet.reset();
      outputWindow->wprintf("%s", packet.print());
      if (errorFile != NULL) 
      {
         fprintf(errorFile, "Unknown TCP packet:\n%s\n", packet.print());
         fflush(errorFile);
      }
      break;
   }
} 


//-----ICQ::ackTCP---------------------------------------------------------------------------------
void ICQ::ackTCP(Packet &packet, ICQUser *u, unsigned short newCommand, unsigned long theSequence)
{
   /* 50 A5 82 00 03 00 DA 07 00 00 50 A5 82 00 01 00 01 00 00 CF 60 AD D3 CF 
      60 AD D3 60 12 00 00 04 00 00 00 00 02 00 00 00 */
   addTcpHeader(packet, u, ICQ_CMDxTCP_ACK, newCommand, icqOwner.awayMessage());
   sendTcpAck(packet, u, theSequence);
}



//-----ICQ::recvUDP--------------------------------------------------------------------------------
void ICQ::recvUDP(int)
{
   Packet packet; 

   // mirabilis contacts us using udp on this server
   udpServer.receivePacket(packet);
   processUdpPacket(packet);
}


//-----ICQ::recvNewTCP-----------------------------------------------------------------------------
void ICQ::recvNewTCP(int)
{
   // our tcp incoming server
   TCPSocket newUserSocket;
   icqOwner.tcpSocket.receiveConnection(newUserSocket);
   
   ICQUser *newUser;
   Packet handshake;
   newUserSocket.receivePacket(handshake);

   unsigned long ulJunk, newUin, localHost;
   unsigned short command, usJunk;
   char ucJunk;
      
   handshake >> command;

   if (command != ICQ_CMDxTCP_HANDSHAKE && command != ICQ_CMDxTCP_HANDSHAKE2 && command != ICQ_CMDxTCP_HANDSHAKE3) 
   {
      outputWindow->wprintf("%sGarbage packet", TCP);
      handshake.reset();
      if (errorFile != NULL) 
      {
         fprintf(errorFile, "Garbage TCP packet:\n%s\n", handshake.print());
         fflush(errorFile);
      }
   }
   else
   {
      handshake >> ulJunk >> usJunk >> ucJunk
                >> newUin
                >> localHost >> localHost
                >> ulJunk >> ucJunk
      ;
      outputWindow->wprintf("%sConnection from uin %d.", TCP, newUin);

      if ((newUser = getUserByUIN(newUin)) != NULL) 
      { 
         newUser->tcpSocket.transferConnectionFrom(newUserSocket);
         connect(newUser->tcpSocket.sn, SIGNAL(activated(int)), this, SLOT(recvTCP(int)));
      }
      else  //fix to check for valid icq packet?
      {
         outputWindow->wprintf("Unknown user, %d, adding to list.", newUin);
         addUser(newUin, newUserSocket, true);
         emit updatedUsers();
      }
   }
}


//-----ICQ::recvTCP--------------------------------------------------------------------------------
void ICQ::recvTCP(int sockfd)
{
   Packet packet;

   for(unsigned short i = 0; i < users.size(); i++)
      if (users[i]->tcpSocket.descriptor() == sockfd)
      {
         if(!users[i]->tcpSocket.receivePacket(packet))
         {
            outputWindow->wprintf("%sConnection to %d lost.", TCP, users[i]->uin()); 
            return;
         }
         processTcpPacket(packet, sockfd);
         return;
      }
}

