// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#include <qapp.h>
#include <qfiledlg.h>
#include <qframe.h>
#include <qlabel.h>
#include <qpushbt.h>
#include <qtooltip.h>
#include "DownloadRenderer.h"
#include "ProgressBar.h"
#include "Request.h"
#include "TransferLogo.h"

DownloadRenderer::DownloadRenderer( const Url& remote )
        : _request( 0 ),
          _fp( 0 ),
          _totalSize( -1 ),
          _totalRead( 0 )
{
    QString local = QFileDialog::getSaveFileName();

    setCaption( "QWeb: Download" );
    setIconText( "QWeb: Download" );
    
    _remoteLabel = new QLabel( "Remote:", this );
    _remoteLabel->setAlignment( AlignRight | AlignVCenter );

    _remoteValue = new QLabel( remote.url(), this );
    _remoteValue->setAlignment( AlignLeft | AlignVCenter );

    _localLabel = new QLabel( "Local:", this );
    _localLabel->setAlignment( AlignRight | AlignVCenter );

    _localValue = new QLabel( local, this );
    _localValue->setAlignment( AlignLeft | AlignVCenter );

    _logo = new TransferLogo( this );
    _logo->setTransfer( TRUE );

    _progress = new ProgressBar( this );

    _stop = new QPushButton( "Stop", this );
    QToolTip::add( _stop, "Abort the Current Transfer" );

    _separator = new QFrame( this );
    _separator->setFrameStyle( QFrame::HLine | QFrame::Sunken );

    _status = new QLabel( this );

    // Request the data.
    _request = new Request( remote );
    connect( _request, SIGNAL( startOfData( Request*, QString, QString, int ) ),
             this    , SLOT( startOfData( Request*, QString, QString, int ) ) );
    connect( _request, SIGNAL( data( const char*, int ) ), this    , SLOT( data( const char*, int ) ) );
    connect( _request, SIGNAL( endOfData() )             , this    , SLOT( endOfData() ) );
    connect( _request, SIGNAL( status( QString ) )       , this    , SLOT( status( QString ) ) );
    connect( _request, SIGNAL( urlChanged( const Url& ) ), this    , SLOT( urlChanged( const Url& ) ) );
    connect( _stop   , SIGNAL( clicked() )               , _request, SLOT( abort() ) );
    if ( !_request->open() ) {
        _request->abort();
        return;
    }

    // Open the local file for writing.
    _fp = fopen( local, "w" );
    if ( !_fp ) {
        _request->abort();
        return;
    }

    setMinimumSize( 300, 108 );
    resize( 450, 108 );
}

void DownloadRenderer::resizeEvent( QResizeEvent* )
{
    const int pad        = 4;
    const int height     = 20;
    const int labelWidth = _remoteLabel->sizeHint().width();
    const int stopWidth  = _stop->sizeHint().width();

    _remoteLabel->setGeometry( pad, pad, labelWidth, height );
    _localLabel->setGeometry( pad, 2 * pad + height, labelWidth, height );

    _remoteValue->setGeometry( 2 * pad + labelWidth, pad, width() - 4 * pad - labelWidth - 100, height );
    _localValue->setGeometry( 2 * pad + labelWidth, 2 * pad + height, width() - 4 * pad - labelWidth - 100, height );

    _logo->setGeometry( width() - pad - 100, pad, 100, 40 );

    _progress->setGeometry( pad, 3 * pad + 2 * height, width() - 3 * pad - stopWidth, height );

    _stop->setGeometry( width() - pad - stopWidth, 3 * pad + 2 * height, stopWidth, height );

    _separator->setGeometry( 0, this->height() - 3 * pad - height, width(), pad );

    _status->setGeometry( pad, this->height() - pad - height, width() - 2 * pad, height );
}

void DownloadRenderer::startOfData( Request*, QString, QString, int totalSize )
{
    _totalSize = totalSize;
}

void DownloadRenderer::data( const char* bytes, int length )
{
    fwrite( bytes, sizeof( char ), length, _fp );

    _totalRead += length;
    if ( _totalSize > -1 ) {
        _progress->setValue( 100 * _totalRead / _totalSize );
    }
}

void DownloadRenderer::endOfData()
{
    fclose( _fp );
    _request = 0;
    delete this;
}

void DownloadRenderer::status( QString msg )
{
    _status->setText( msg );
}

void DownloadRenderer::urlChanged( const Url& url )
{
    _remoteValue->setText( url.url() );
}
