/*
 * GXSNMP - An snmp managment application
 * Copyright (C) 1998 Gregory McLean
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc.,  59 Temple Place - Suite 330, Cambridge, MA 02139, USA.
 *
 * Network Map panel. (New and Improved to use the gnome-canvas)
 */

#include "config.h"
#include <gnome.h>
#include <gdk/gdkkeysyms.h>
#include "main.h"
#include "net_map.h"
#include "net.h"
#include "gui.h"
#include "snmp_host.h"

#include "structs.h"
#include "protos.h"
/*
 * Forward declaration
 */
static void                init_map_grid        (void);
static void                create_map_panel     (void);
static void                delete_map_panel     (GtkWidget        *widget,
						 gpointer         data);
static void                close_panel_cb       (GtkWidget        *widget,
						 gpointer         data);
static gint                canvas_key_press     (GnomeCanvas      *widget,
						 GdkEventKey      *e,
						 gpointer         data);
static gint                canvas_item_event    (GnomeCanvasItem  *item,
						 GdkEvent         *e,
						 gpointer         data);
static gint                canvas_event         (GtkWidget        *widget,
						 GdkEvent         *e,
						 gpointer         data);
static gint                canvas_enter_event   (GtkWidget        *widget,
						 GdkEvent         *e,
						 gpointer         data);
static gint                update_mouse_stat    (GtkWidget        *widget,
						 GdkEventMotion   *e,
						 gpointer         data);
static gint                map_link_event       (GnomeCanvasItem  *widget,
						 GdkEvent         *e,
						 gpointer         data);
static void                setup_item           (GnomeCanvasItem  *item,
						 gpointer         data);
static void                free_imlib_image     (GtkObject        *object,
						 gpointer         data);
static void                host_pixmap          (GnomeCanvasGroup *group,
						 double           x,
						 double           y);

static void                map_draw_net_line    (map_node         *node);
static void                map_update_map_link  (map_links        *mlink);
/*
 * Menu callbacks 
 */
static void         quit_app_cb                 (GtkWidget        *widget,
						 gpointer         data);
/*
 * Toolbar callbacks
 */
static void         add_host_cb                 (GtkWidget        *widget,
						 gpointer         data);
static void         add_net_cb                  (GtkWidget        *widget,
						 gpointer         data);
static void         host_list_cb                (GtkWidget        *widget,
						 gpointer         data);
static void         net_list_cb                 (GtkWidget        *widget,
						 gpointer         data);
static void         mib_browser_cb              (GtkWidget        *widget,
						 gpointer         data);
static void         node_link_cb                (GtkWidget        *widget,
						 gpointer         data);
/*
 * Add label callbacks
 */
static void         add_label_ok_cb             (GtkWidget        *widget,
						 gpointer         data);
static void         add_label_can_cb            (GtkWidget        *widget,
						 gpointer         data);
/*
 * Some defines
 * Couple of modes for the canvas
 */
enum {
  ITEM_NONE     = 1 << 1,
  ITEM_DRAG     = 1 << 2,     /* dragging it around */
  ITEM_RESIZE   = 1 << 3,     /* resizing it */
  ITEM_PICK     = 1 << 4,     /* selecting it to link it */
};

/*
 * Global variables needed for this panel
 */


/*
 * Local variables for the network map panel
 */

static GtkWidget      *map_panel = NULL;

static GnomeUIInfo new_menu[] = {
  { GNOME_APP_UI_ITEM, N_("Map"), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Window"), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0, 0, NULL },
  { GNOME_APP_UI_ENDOFINFO }
};

static GnomeUIInfo map_file_menu[] = {
  { GNOME_APP_UI_SUBTREE, N_("New"), NULL, new_menu, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_NEW, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Open..."), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_OPEN, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save"), NULL, NULL, NULL, NULL, 
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save As..."), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE_AS, 0, 0, NULL },
  { GNOME_APP_UI_SEPARATOR },
  { GNOME_APP_UI_ITEM, N_("Print..."), NULL, NULL, NULL, NULL, 
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PRINT, 'P', GDK_CONTROL_MASK,
    NULL },
  { GNOME_APP_UI_SEPARATOR },
  { GNOME_APP_UI_ITEM, N_("Close"), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CLOSE, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Exit"), NULL, quit_app_cb, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_EXIT, 'X', GDK_CONTROL_MASK, 
    NULL },
  { GNOME_APP_UI_ENDOFINFO }
};

static GnomeUIInfo map_edit_menu[] = {
  { GNOME_APP_UI_ITEM, N_("Undo"), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Redo"), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0, 0, NULL },
  { GNOME_APP_UI_ENDOFINFO }
};

static GnomeUIInfo map_map_menu[] = {
  { GNOME_APP_UI_ITEM, N_("Enable Grid"), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Snap to grid"), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0, 0, NULL },
  { GNOME_APP_UI_ENDOFINFO }
};

static GnomeUIInfo map_panel_menu[] = {
  { GNOME_APP_UI_SUBTREE, N_("File"), NULL, map_file_menu, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  { GNOME_APP_UI_SUBTREE, N_("Edit"), NULL, map_edit_menu, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  { GNOME_APP_UI_SUBTREE, N_("Map"), NULL, map_map_menu, NULL, NULL, 
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  { GNOME_APP_UI_ENDOFINFO }
};

/*
 * Toolbar.. (gnome style)
 */
static GnomeUIInfo map_toolbar[] = {
  { GNOME_APP_UI_ITEM, N_("Add Host"), 
    N_("Add a new host to the database."), add_host_cb, NULL, NULL,
    GNOME_APP_PIXMAP_FILENAME, "add_host.xpm", 'A', GDK_MOD1_MASK, 
    NULL },
  { GNOME_APP_UI_ITEM, N_("Delete Host"), 
    N_("Delete a host from the database."), NULL, NULL, NULL,
    GNOME_APP_PIXMAP_FILENAME, "del_host.xpm", 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Host list"), 
    N_("Open the list of known hosts."), host_list_cb, NULL, NULL,
    GNOME_APP_PIXMAP_FILENAME, "host_list.xpm", 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Add Network"),
    N_("Add a new network to the map/database."), add_net_cb, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Delete Network"),
    N_("Delete a network from the map/database."), NULL, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Network list"), 
    N_("Open the list of known networks."), net_list_cb, NULL, NULL,
    GNOME_APP_PIXMAP_FILENAME, "net_list.xpm", 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Mib Browser"), 
    N_("Open the mib browser."), mib_browser_cb, NULL, NULL, 
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Event Browser"), NULL, NULL, NULL, NULL,
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Link Nodes"), NULL, node_link_cb, NULL, NULL, 
    GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL },
  { GNOME_APP_UI_ENDOFINFO }
};

/*
 * Add a label to the map, here for the lack of a better place to put it
 */
static GTableAttach add_label_label[] = {
  { 0, 1, 0, 1, GTK_FILL, 0, 0, 0 },
  { 0, 1, 1, 2, GTK_FILL, 0, 0, 0 },
  { 0, 1, 2, 3, GTK_FILL, 0, 0, 0 }
};

static GTableAttach add_label_widget[] = {
  { 1, 3, 0, 1, GTK_FILL, 0, 0, 0 },
  { 1, 2, 1, 2, GTK_FILL, 0, 0, 0 },
  { 1, 2, 2, 3, GTK_EXPAND | GTK_FILL, 0, 0, 0 }
};

static gchar *add_panel_tags[] = {
  "label_text",
  "color_name",
  "font_name"
};

static GTableDef add_panel_table[] = {
  { N_("Text"),
    N_("This is the text that you want to appear in the label."),
    &add_panel_tags[0], GT_ENTRY, NULL,
    &add_label_widget[0] },
  { NULL, NULL, NULL, NULL, NULL, NULL }
};

static GTCallbacks add_label_buttons[] = {
  { GT_BUTTON_OK, add_label_ok_cb, NULL, "ok_button" },
  { GT_BUTTON_CAN, add_label_can_cb, NULL, "cancel_button" },
  { GT_UI_END, NULL, NULL, NULL }
};

/*
 * Local functions 
 */

static int map_mode = ITEM_NONE;

int
map_get_mode ()
{
  return map_mode;
}

void
map_set_mode (int mode)
{
  map_mode = mode;
}

static void 
init_map_grid ()
{
  /* initilize the map grid for the autoplacement stuff */
}

static void
create_map_panel ()
{
  GtkWidget        *window_frame;
  GtkWidget        *map_table;
  GtkWidget        *separator_bar;
  GtkWidget        *logo_widget;
  GtkWidget        *mouse_stat;              /* The mouse x y on the map */
  GtkWidget        *map_stat;                /* Any stat info for the map */
  GtkWidget        *canvas;
  GtkWidget        *scrolled_window;
  GtkWidget        *frame;
  GtkWidget        *led_bar;
  GtkWidget        *s_bar;
  GnomeCanvasGroup *root;
  gchar            *logo_pixmap_filename;

  map_panel = gnome_app_new ("GXSNMP", _("Network map"));
  gtk_signal_connect (GTK_OBJECT (map_panel), "delete_event",
		      (GtkSignalFunc) delete_map_panel,
		      NULL);

  init_pixmap_d (map_panel);                   /* hmm need a better place */
  window_frame = gtk_frame_new (NULL);
  gtk_container_border_width (GTK_CONTAINER (window_frame), 4);

  map_table = gtk_table_new (8, 7, FALSE);
  gtk_object_set_data (GTK_OBJECT (map_panel), "map_table", map_table);
  gtk_widget_set_name (GTK_WIDGET (map_table), "net_map_table");
  gtk_container_border_width (GTK_CONTAINER (map_table), 2);
  gtk_container_add (GTK_CONTAINER (window_frame), map_table);

  /*
   * set up some separator bars
   */
  /*  separator_bar = gtk_hseparator_new ();
  gtk_table_attach (GTK_TABLE (map_table), separator_bar,
		    0, 7, 6, 7,
		    GTK_FILL, GTK_FILL, 3, 3); */

  separator_bar = gtk_vseparator_new ();
  gtk_table_attach (GTK_TABLE (map_table), separator_bar, 
		    1, 2, 3, 6,
		    GTK_FILL, GTK_FILL, 3, 0);

  /*  separator_bar = gtk_hseparator_new ();
  gtk_table_attach (GTK_TABLE (map_table), separator_bar,
		    0, 7, 2, 3,
		    GTK_FILL, GTK_FILL, 3, 3); */

  /*
   * The logo
   */
  logo_pixmap_filename = gnome_unconditional_pixmap_file ("logo_v.xpm");
  logo_widget = gnome_pixmap_new_from_file (logo_pixmap_filename);
  gtk_table_attach (GTK_TABLE (map_table), logo_widget,
		    0, 1, 3, 6,
		    GTK_FILL, GTK_FILL, 0, 0);
  /*
   * The canvas for the map.
   */
  gtk_widget_push_visual (gdk_imlib_get_visual ());
  gtk_widget_push_colormap (gdk_imlib_get_colormap ());
  canvas = gnome_canvas_new ();
  gtk_widget_set_name (canvas, "network_map");
  gtk_widget_pop_colormap ();
  gtk_widget_pop_visual ();
  gnome_canvas_set_size (GNOME_CANVAS (canvas), 1600, 1600);
  gnome_canvas_set_scroll_region (GNOME_CANVAS (canvas), 
				  0, 0,
				  1600, 1600);
  gtk_signal_connect (GTK_OBJECT (canvas), "key_press_event",
		      (GtkSignalFunc) canvas_key_press,
		      map_panel);
  gtk_signal_connect (GTK_OBJECT (canvas), "motion_notify_event",
		      (GtkSignalFunc) update_mouse_stat,
		      map_panel);
  gtk_signal_connect (GTK_OBJECT (canvas), "enter_notify_event",
		      (GtkSignalFunc) canvas_enter_event,
		      map_panel);
  gtk_object_set_data (GTK_OBJECT (map_panel), "map_canvas", canvas);
  gtk_widget_set_name (GTK_WIDGET (canvas), "map_canvas");
  /*
   * Scrolled area for the canvas...
   */
  scrolled_window = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (scrolled_window), GTK_SHADOW_IN);
  gtk_table_attach (GTK_TABLE (map_table), scrolled_window, 
		    3, 6, 4, 5,
		    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_widget_set_usize (scrolled_window, 600, 400);
  s_bar = gtk_hscrollbar_new (GTK_LAYOUT (canvas)->hadjustment);
  gtk_table_attach (GTK_TABLE (map_table), s_bar,
		    2, 6, 5, 6,
		    GTK_FILL, GTK_FILL, 0, 3);
  s_bar = gtk_hruler_new ();
  gtk_widget_set_name (GTK_WIDGET (s_bar), "net_map_ruler");
  gtk_signal_connect_object (GTK_OBJECT (canvas), "motion_notify_event",
			     (GtkSignalFunc) GTK_WIDGET_CLASS (GTK_OBJECT (s_bar)->klass)->motion_notify_event,
			     GTK_OBJECT (s_bar));
  gtk_table_attach (GTK_TABLE (map_table), s_bar,
		    3, 6, 3, 4,
		    GTK_FILL, GTK_FILL, 0, 0);
  gtk_ruler_set_range (GTK_RULER (s_bar), 0, 10, 5, 10);
  s_bar = gtk_vscrollbar_new (GTK_LAYOUT (canvas)->vadjustment);
  gtk_table_attach (GTK_TABLE (map_table), s_bar,
		    6, 7, 3, 5,
		    GTK_FILL, GTK_FILL, 3, 0);
  s_bar = gtk_vruler_new ();
  gtk_widget_set_name (GTK_WIDGET (s_bar), "net_map_ruler");
  gtk_signal_connect_object (GTK_OBJECT (canvas), "motion_notify_event",
			     (GtkSignalFunc) GTK_WIDGET_CLASS (GTK_OBJECT (s_bar)->klass)->motion_notify_event,
			     GTK_OBJECT (s_bar));
  gtk_ruler_set_range (GTK_RULER (s_bar), 0, 10, 5, 10);
  gtk_table_attach (GTK_TABLE (map_table), s_bar,
		    2, 3, 4, 5,
		    GTK_FILL, GTK_FILL, 0, 0);
  gtk_container_add (GTK_CONTAINER (scrolled_window), canvas);
  root = GNOME_CANVAS_GROUP (gnome_canvas_root (GNOME_CANVAS (canvas)));
  gtk_object_set_data (GTK_OBJECT (map_panel), "map_canvas_root", root);
  GTK_WIDGET_SET_FLAGS (canvas, GTK_CAN_FOCUS);
  gtk_widget_grab_focus (canvas);
  /*
   * Status areas
   */
  mouse_stat = gtk_statusbar_new ();
  gtk_table_attach (GTK_TABLE (map_table), mouse_stat,
		    0, 1, 7, 8,
		    GTK_FILL, GTK_FILL, 2, 0);
  gtk_object_set_data (GTK_OBJECT (map_panel), "mouse_stat", mouse_stat);
  gtk_object_set_data (GTK_OBJECT (map_panel), "mouse_stat_id", (int *)
		       gtk_statusbar_get_context_id (GTK_STATUSBAR 
						     (mouse_stat), 
						     "mouse_stat"));
						     
  map_stat = gtk_statusbar_new ();
  gtk_table_attach (GTK_TABLE (map_table), map_stat,
		    3, 6, 7, 8,
		    GTK_FILL, GTK_FILL, 2, 0);  
  gtk_object_set_data (GTK_OBJECT (map_panel), "map_stat", map_stat);
  gtk_object_set_data (GTK_OBJECT (map_panel), "map_stat_id", (gint *)
		       gtk_statusbar_get_context_id (GTK_STATUSBAR 
						     (map_stat), 
						     "map_stat"));
  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
  gtk_table_attach (GTK_TABLE (map_table), frame, 
		    1, 3, 7, 8,
		    GTK_FILL, GTK_FILL, 2, 0);
  led_bar = (GtkWidget *)led_bar_new (5);
  gtk_container_add (GTK_CONTAINER (frame), led_bar);
  
  /*
   * Set the gnome_app contents...
   */
  gnome_app_set_contents (GNOME_APP (map_panel), window_frame);
  gnome_app_create_menus (GNOME_APP (map_panel), map_panel_menu);
  gnome_app_create_toolbar (GNOME_APP (map_panel), map_toolbar);
  gtk_widget_show_all (map_panel);
}


/*
 * Callback functions
 */

/*
 * Window Manager delete signal
 */
static void
delete_map_panel (GtkWidget *widget, gpointer data)
{
}

/*
 * Close button callback
 */
static void
close_panel_cb (GtkWidget *widget, gpointer data)
{
  destroy_map_panel ();
}

/*
 * Keypress on the canvas
 */
static gint
canvas_key_press (GnomeCanvas *widget, GdkEventKey *e, gpointer data)
{
  int    x, y;
  double x1, y1, x2, y2;

  /* TODO: update the zoom in/out cases to handle portable stuff...
   */
  gnome_canvas_get_scroll_offsets (widget, &x, &y);
  switch (e->keyval)
    {
    case 0x3d:                  /* Ick! is there a symbol for the '+' key? */
      map_zoom_in (widget);
      break;
    case 0x2d:                  /* Ick! is there a symbol for the '-' key? */
      map_zoom_out (widget);
      break;
    case GDK_Up:
    case GDK_KP_Up:
      if (e->state && GDK_SHIFT_MASK)
	gnome_canvas_scroll_to (widget, x, y - 40);
      else
	gnome_canvas_scroll_to (widget, x, y - 20);
      break;
    case GDK_Down:
    case GDK_KP_Down:
      if (e->state && GDK_SHIFT_MASK)
	gnome_canvas_scroll_to (widget, x, y + 40);
      else
	gnome_canvas_scroll_to (widget, x, y + 20);
      break;
    case GDK_Left:
    case GDK_KP_Left:
      if (e->state && GDK_SHIFT_MASK)
	gnome_canvas_scroll_to (widget, x - 20, y);
      else
	gnome_canvas_scroll_to (widget, x - 10, y);
      break;
    case GDK_Right:
    case GDK_KP_Right:
      if (e->state && GDK_SHIFT_MASK)
	gnome_canvas_scroll_to (widget, x + 20, y);
      else
	gnome_canvas_scroll_to (widget, x + 10, y);
      break;
    case GDK_Page_Up:
    case GDK_KP_Page_Up:
      gnome_canvas_scroll_to (widget, x, y - widget->height);
      break;
    case GDK_Page_Down:
    case GDK_KP_Page_Down:
      gnome_canvas_scroll_to (widget, x, y + widget->height);
      break;
    case GDK_Home:
    case GDK_KP_Home:
      gnome_canvas_scroll_to (widget, 0, 0);
      break;
    case GDK_End:
    case GDK_KP_End:
      gnome_canvas_get_scroll_region (widget, &x1, &y1, &x2, &y2);
      gnome_canvas_scroll_to (widget, x2, y1);
      break;
    default:
      break;
    }
  return FALSE;
}

/*
 * Events on the items in the canvas
 */
static gint
canvas_item_event (GnomeCanvasItem *item, GdkEvent *event, gpointer data)
{
  static double     x, y;
  double            new_x, new_y;
  GdkCursor         *fleur;
  int               mode;
  hosts             *host;
  GList             *node_links;
  map_node          *node;

  node = (map_node *)data;
  mode = map_get_mode ();
  switch (event->type) 
    {
    case GDK_BUTTON_PRESS:
      switch (event->button.button)
	{
	case 1:
	  if (mode == ITEM_PICK)
	    {
	      map_select_node (node);
	    }
	  else
	    {
	      x = event->button.x;
	      y = event->button.y;
	      fleur = gdk_cursor_new (GDK_FLEUR);
	      gnome_canvas_item_grab (item,
				      GDK_POINTER_MOTION_MASK |
				      GDK_BUTTON_RELEASE_MASK,
				      fleur,
				      event->button.time);
	      gdk_cursor_destroy (fleur);
	      map_select_node (item, node->mn_cgroup);
	      map_set_mode (ITEM_DRAG);
	    }
	  break;
	case 2:
	  x = event->button.x;
	  y = event->button.y;
	  fleur = gdk_cursor_new (GDK_FLEUR);
	  gnome_canvas_item_grab (item,
				  GDK_POINTER_MOTION_MASK |
				  GDK_BUTTON_RELEASE_MASK,
				  fleur,
				  event->button.time);
	  gdk_cursor_destroy (fleur);
	  map_set_mode (ITEM_RESIZE);
	  break;
	case 3:
	  switch (node->mn_type)
	    {
	    case MAP_NODE_HOST:
	      host = (hosts *)node->mn_data;
	      gtk_object_set_data (GTK_OBJECT 
				   (GNOME_CANVAS_ITEM (item)->canvas),
				   "selected_host", host);
	      host_menu_right_popup ((int)&event->button, event->button.time);
	      return TRUE;
	      break;
	    case MAP_NODE_NETWORK:
	      do_network_popup_menu ((GdkEventButton *)event, node->mn_data);
	      return TRUE;
	      break;
	    default:
	      do_net_map_popup_menu ((GdkEventButton *)event, NULL);
	      break;
	    }
	  break;
	default:
	  break;
	}
    case GDK_MOTION_NOTIFY:
      if ( (mode == ITEM_DRAG) && (event->motion.state & GDK_BUTTON1_MASK))
	{
	  new_x = event->motion.x;
	  new_y = event->motion.y;
	  /* 
	   * TODO: Auto scroll the canvas when moving beyond the bounds of 
	   * visible window? And the BIG CAD style cross-hairs.
	   * Also need to adjust this code to deal with network drags.
	   */
	  gnome_canvas_item_move (item, new_x - x, new_y - y);
	  if (node->mn_links)
	    {
	      node_links = node->mn_links;
	      while (node_links)
		{
		  /* update this link */
		  map_update_map_link (node_links->data);
		  node_links = g_list_next (node_links);
		}
	    }
	  x = new_x;
	  y = new_y;
	}
      else
	if ( (mode == ITEM_RESIZE)  && (event->motion.state & GDK_BUTTON1_MASK))
	  {
	    g_print ("Do item resize\n");
	  }
      break;
    case GDK_BUTTON_RELEASE:
      gnome_canvas_item_ungrab (item, event->button.time);
      if (mode != ITEM_PICK)
	{
	  map_set_mode (ITEM_NONE);
	}
      break;
    case GDK_KEY_PRESS:
      /* g_print ("Item event got a key press..\n"); */
      break;
    case GDK_KEY_RELEASE:
      /* g_print ("Item event got a key release...\n"); */
    default:
      break;
    }
  return FALSE;
}

/*
 * General canvas events _not_ over a connected item.
 */
static gint 
canvas_event (GtkWidget *widget, GdkEvent *e, gpointer data)
{
  switch (e->type)
    {
    case GDK_BUTTON_PRESS:
      switch (e->button.button)
	{
	case 3:
	  g_print ("Main Menu Popup?\n");
	  do_net_map_popup_menu ((GdkEventButton *)e, NULL);
	  break;
	default:
	  break;
	}
    default:
      break;
    }
  return FALSE;
}

/*
 * This is done so the key bindings work consistantly. (without having to 
 * click on the canvas all the time.. Or after moving an item around
 * The canvas looses focus. This appears to work though there may be some
 * ugly side effects.
 */
static gint 
canvas_enter_event (GtkWidget *widget, GdkEvent *e, gpointer data)
{
  gtk_widget_grab_focus (widget);
  return FALSE;
}
  
static gint
update_mouse_stat (GtkWidget *widget, GdkEventMotion *e, gpointer data)
{
  gint              x, y, statusid;
  gchar             buf[80];
  GdkModifierType   state;
  GtkWidget         *stat_widget;

  if (e->is_hint)
    gdk_window_get_pointer (e->window, &x, &y, &state);
  else
    {
      x     = e->x;
      y     = e->y;
      state = e->state;
    }
  snprintf (buf, sizeof (buf), "%d x %d", x, y);
  stat_widget = get_widget (map_panel, "mouse_stat");
  if (stat_widget)
    {
      statusid = gtk_statusbar_get_context_id (GTK_STATUSBAR (stat_widget),
					       "mouse_stat");
      gtk_statusbar_push (GTK_STATUSBAR (stat_widget), statusid, buf);
      /*      gtk_statusbar_pop (GTK_STATUSBAR (stat_widget), statusid); */
    }
  return FALSE;
}

static gint
map_link_event (GnomeCanvasItem *item, GdkEvent *e, gpointer data)
{
  static gdouble      x,y;
  static gint         new_x, new_y;
  static int          mode = 0;
  GdkCursor           *fleur;

  switch (e->type)
    {
    case GDK_BUTTON_PRESS:
      switch (e->button.button)
	{
	case 1:
	  x = e->button.x;
	  y = e->button.y;
	  fleur = gdk_cursor_new (GDK_FLEUR);
	  gnome_canvas_item_grab (item,
				  GDK_POINTER_MOTION_MASK |
				  GDK_BUTTON_RELEASE_MASK,
				  fleur,
				  e->button.time);
	  gdk_cursor_destroy (fleur);
	  mode = mode ^ITEM_DRAG;
	  break;
	case 2:
	  g_print ("Add a point to the link?\n");
	  break;
	case 3:
	  g_print ("do a link menu?\n");
	default:
	break;
	}
    case GDK_MOTION_NOTIFY:
      if ( (mode & ITEM_DRAG) && (e->motion.state & GDK_BUTTON1_MASK))
	{
	  new_x = e->motion.x;
	  new_y = e->motion.y;
	  gnome_canvas_item_move (item, new_x - x, new_y - y);
	  x = new_x;
	  y = new_y;
	}
      break;
    case GDK_BUTTON_RELEASE:
      gnome_canvas_item_ungrab (item, e->button.time);
      if (mode & ITEM_DRAG)
	mode = mode ^ ITEM_DRAG;
      break;
    default:
      break;
    }
  return FALSE;
}
/* 
 * Menu callback functions
 */
static void
quit_app_cb (GtkWidget *widget, gpointer data)
{
  quit_app ();
}

/*
 * Toolbar callback functions
 */
static void
add_host_cb (GtkWidget *widget, gpointer data)
{
  open_add_host_panel ();
}
static void 
add_net_cb (GtkWidget *widghet, gpointer data)
{
  open_add_network_panel ();
}

static void
host_list_cb (GtkWidget *widget, gpointer data)
{
  hlist_panel_open ();
}

static void
net_list_cb (GtkWidget *widget, gpointer data)
{
  open_network_panel ();
}

static void
mib_browser_cb (GtkWidget *widget, gpointer data)
{
  open_browser_panel ();
}
static void
node_link_cb (GtkWidget *widget, gpointer data)
{
  map_set_mode (ITEM_PICK);
  gtk_object_set_data (GTK_OBJECT (map_panel), "number_selected", (int *)0);
  map_update_stat (_("Select the first node to link from."));
}

/*
 * Add label callbacks
 */

static void
add_label_ok_cb (GtkWidget *widget, gpointer data)
{
}

static void
add_label_can_cb (GtkWidget *widget, gpointer data)
{
}

/*
 * Exported global functions
 * (Standard panel funcs)
 *
 */

/*
 * Open
 */
/*
 * Testing the canvas mapping/dragging code..
 */
/*
net_entry *test = NULL;
net_entry *test2;
*/
void
open_map_panel ()
{

  if (map_panel)
    {
      if (!GTK_WIDGET_VISIBLE (map_panel))
 	gtk_widget_show (map_panel);
      return;
    }
  else
    {
      init_map_grid ();
      create_map_panel ();
      reset_map_panel ();
      /*      test = (net_entry *) g_malloc (sizeof (net_entry));
      test->nl_name = g_strdup ("Test Network");
      inet_aton ("207.15.208.0", &test->nl_net);
      inet_aton ("255.255.255.0", &test->nl_mask);
      test->nl_mnode = NULL;
      nl_add_network (test);
      test2 = (net_entry *) g_malloc (sizeof (net_entry));
      test2->nl_name = g_strdup ("Second test network");
      inet_aton ("208.21.108.0", &test2->nl_net);
      inet_aton ("255.255.255.0", &test2->nl_mask);
      map_add_network (test, 100, 50);
      map_add_network (test2, 150, 60); */
    }
}

/*
 * Close/destroy
 */
void
destroy_map_panel ()
{
  g_return_if_fail (map_panel != NULL);

  if (map_panel)
    {
      gtk_widget_destroy (map_panel);
      map_panel = NULL;
    }
}

/*
 * Hide
 */
void
hide_map_panel ()
{
  g_return_if_fail (map_panel != NULL);

  if (GTK_WIDGET_VISIBLE (map_panel))
    gtk_widget_hide (map_panel);
}

/*
 * reset
 */
void 
reset_map_panel ()
{
  g_return_if_fail (map_panel != NULL);
}

/*
 * Adding labels to the map
 */
void
open_add_label_panel ()
{
}

void 
open_edit_label_panel ()
{
}

/*
 * Support functions for the map
 */

static void
setup_item (GnomeCanvasItem *item, gpointer data)
{
  g_return_if_fail (item != NULL);
  gtk_signal_connect (GTK_OBJECT (item), "event",
		      (GtkSignalFunc) canvas_item_event,
		      data);
}
static void
free_imlib_image (GtkObject *object, gpointer data)
{
  gdk_imlib_destroy_image (data);
}

static void
host_pixmap (GnomeCanvasGroup *group, double x, double y)
{
  GdkImlibImage    *im;
  GnomeCanvasItem  *image;
  char            *fn;

  fn = gnome_unconditional_pixmap_file ("desktop.xpm");
  im = gdk_imlib_load_image (fn);
  y -= (im->rgb_height + 3);
  image = gnome_canvas_item_new (group,
				 gnome_canvas_image_get_type (),
				 "image", im,
				 "x", x,
				 "y", y,
				 "width", (double) im->rgb_width,
				 "height", (double) im->rgb_height,
				 "anchor", GTK_ANCHOR_N,
				 NULL);
  gtk_object_set_data (GTK_OBJECT (group), "image_item", image);
  gtk_signal_connect (GTK_OBJECT (image), "destroy",
		      (GtkSignalFunc) free_imlib_image,
		      im);
}
/*
 * Draw the actual network line. 
 * TODO: Update to represent diffrent network types and speeds when 
 *       we get that far.
 */
static void
map_draw_net_line (map_node *node)
{
  GnomeCanvasPoints   *points;
  GnomeCanvasItem     *item;

  points = gnome_canvas_points_new (2);
  points->coords[0] = -90.0;
  points->coords[1] = -2.0;
  points->coords[2] = 90.0;
  points->coords[3] = -2.0;

  item = gnome_canvas_item_new (node->mn_cgroup,
				gnome_canvas_line_get_type (),
				"points", points,
				"fill_color", "green",
				"width_units", 2.0,
				NULL);
  gtk_object_set_data (GTK_OBJECT (node->mn_cgroup), "line_item", item);
  gnome_canvas_points_free (points);
}

/*
 * Internal Function : map_update_map_link
 * Description       : This function is resposnible for updating any node
 *                     links on the map when nodes are dragged around.
 */
static void
map_update_map_link (map_links *mlink)
{
  GnomeCanvasPoints    *points;

  points = gnome_canvas_points_new (2);

  points->coords[0] = mlink->ml_from->xpos;
  points->coords[1] = mlink->ml_from->ypos;
  points->coords[2] = mlink->ml_to->xpos;
  points->coords[3] = mlink->ml_to->ypos;
  gnome_canvas_item_set (mlink->ml_item, 
			 "points", points,
			 NULL);
  gnome_canvas_points_free (points);

}
/*
 * Function    : map_add_node
 * Description : This function will do the actual placement and creation
 *               of a new node on the map. This function will also connect
 *               up any host type nodes to the respective network if its
 *               added in the map. NOTE:: When completeing the network
 *               functions and reloading from the database. Load networks
 *               first.
 */
gboolean
map_add_node (NodeType type, gpointer node_data)
{
  GnomeCanvasGroup   *root;
  map_node           *new_node;        /* The node we are adding */
  map_node           *old_node;        /* The node to connect to if any */
  hosts              *host_node;
  net_entry          *net_node;

  old_node = NULL;
  new_node = NULL;
  root = (GnomeCanvasGroup *)gtk_object_get_data (GTK_OBJECT (map_panel),
						  "map_canvas_root");
  if (root)
    {
      new_node = (map_node *)g_malloc (sizeof (map_node));
      new_node->mn_cgroup = GNOME_CANVAS_GROUP (gnome_canvas_item_new 
						(root,
						 gnome_canvas_group_get_type(),
						 NULL));
      new_node->mn_data = node_data;
      new_node->mn_type = type;
      new_node->mn_links = NULL;
      setup_item (GNOME_CANVAS_ITEM (new_node->mn_cgroup), new_node); 
      switch (type)
	{
	case MAP_NODE_HOST:
	  host_node = (hosts *)node_data;
	  host_node->hl_mnode = new_node;
	  gnome_canvas_item_move (GNOME_CANVAS_ITEM (new_node->mn_cgroup), 
				  host_node->hl_location.x,
				  host_node->hl_location.y);
	  host_pixmap (new_node->mn_cgroup, 0.0, 0.0);
	  map_set_node_label (new_node, host_node->hl_disp);
	  g_print ("Hunting down %s\n", inet_ntoa (host_node->hl_addr));
	  old_node = (map_node *)nl_get_host_net (host_node);
	  if (old_node)
	    {
	      map_link_nodes (old_node, new_node);
	    }
	  break;
	case MAP_NODE_NETWORK:
	  net_node   = (net_entry *)node_data;
	  new_node->mn_name = g_strdup (net_node->nl_name);
	  net_node->nl_mnode = new_node;
	  gnome_canvas_item_move (GNOME_CANVAS_ITEM (new_node->mn_cgroup),
				  net_node->nl_location.x,
				  net_node->nl_location.y);
	  map_draw_net_line (new_node);
	  map_set_node_label (new_node, net_node->nl_name);
	  break;
	case MAP_NODE_LINK:
	  g_print ("Add a link node to the map.\n");
	  break;
	}
      return TRUE;
    }
  return FALSE;
}
/*
 * Function   : map_add_host 
 * Description: This will add a host to the network map at a given x/y 
 *              position. This is just a convience function for the
 *              real node add function.
 * Arguments  : host  -- The hosts structure of the host to add.
 *              x     -- The X position on the map to add it to.
 *              y     -- The Y position on the map to add it to.
 * Returns    : TRUE  -- Host was suscessfully added to the map.
 *              FALSE -- Host was not added to the map. Check app_errno 
 *                       for further info.
 */
gboolean
map_add_host (hosts *host, gdouble x, gdouble y)
{
  g_return_val_if_fail (map_panel != NULL, FALSE);

  host->hl_location.x = x;
  host->hl_location.y = y;
  map_add_node (MAP_NODE_HOST, host);
  return TRUE;
}

/*
 * Function    : map_link_nodes
 * Description : This function will create the link between two nodes
 *               on the map. See the map_links structure in structs.h
 *               to examine what constitutes a link in the context of the 
 *               map.
 */
void
map_link_nodes (map_node *node1, map_node *node2)
{
  /*
   * node2  == the node to connect to. 
   * node1  == the node to connect FROM.
   */
  GnomeCanvasGroup   *to_group;
  GnomeCanvasGroup   *from_group;
  GnomeCanvasItem    *link_item;
  GnomeCanvasPoints  *points;
  map_links          *map_link;

  g_return_if_fail (node1 != NULL);
  g_return_if_fail (node2 != NULL);
  to_group   = node2->mn_cgroup;
  from_group = node1->mn_cgroup;
  map_link = (map_links *)g_malloc (sizeof (map_links));

  points = gnome_canvas_points_new (2);
  points->coords[0] = from_group->xpos;
  points->coords[1] = from_group->ypos;
  points->coords[2] = to_group->xpos;
  points->coords[3] = to_group->ypos - 2;

  link_item = gnome_canvas_item_new (GNOME_CANVAS_GROUP
				 (GNOME_CANVAS_ITEM (to_group)->canvas->root),
				  gnome_canvas_line_get_type (),
				  "points", points,
				  "fill_color", "black",
				  "width_units", 0.0,
				  NULL);
  gtk_signal_connect (GTK_OBJECT (link_item), "event",
		      (GtkSignalFunc) map_link_event,
		      NULL);
  map_link->ml_from = node1->mn_cgroup;
  map_link->ml_to   = node2->mn_cgroup;
  map_link->ml_item = link_item;
  node1->mn_links = g_list_append (node1->mn_links, map_link);
  map_link->ml_from = node2->mn_cgroup;
  map_link->ml_to   = node1->mn_cgroup;
  node2->mn_links = g_list_append (node2->mn_links, map_link);
  gnome_canvas_points_free (points);
}

/*
 * Function   : map_del_host
 * Description: This function will delete a host from the map.
 * Arguments  : host  -- The hosts structure describing the host to 
 *                       remove from the map.
 * Returns    : TRUE  -- Host was deleted from the map.
 *              FALSE -- Host was not deleted from the map. Check app_errno
 *                       for further info.
 */
gboolean
map_del_host (hosts *host)
{
  g_return_val_if_fail (host != NULL, FALSE);
  return FALSE;
}

/*
 * Function   : map_add_network
 * Description: This function shall add a network to the map at a given
 *              x/y position. This is just a convience wrapper for the
 *              real node add function.
 * Arguments  : network  -- The network structure of the network to add
 *                          to the map.
 *              X        -- The X position on the map.
 *              Y        -- The Y position on the map.
 * Returns    : TRUE     -- Network was added to the map.
 *              FALSE    -- Network was not added to the map. Check app_errno
 *                          for further info.
 *              
 */
gboolean
map_add_network (net_entry *network, gdouble x, gdouble y)
{
  g_return_val_if_fail (map_panel != NULL, FALSE);

  network->nl_location.x = x;
  network->nl_location.y = y;
  map_add_node (MAP_NODE_NETWORK, network);
  return TRUE;
}

/*
 * Function   : map_del_network
 * Description: This function shall delete a network from the map's display.
 * Arguments  : network  -- The network structure of the network to delete
 *                          from the map.
 * Returns    : TRUE     -- Network was deleted from the map.
 *              FALSE    -- Network was not deleted from the map. Check
 *                          app_errno for further info.
 */
gboolean
map_del_network (net_entry *network)
{
  g_return_val_if_fail (network != NULL, FALSE);
  return FALSE;
}

/*
 * Function    : map_set_node_label
 * Description : This function will change a given nodes label on the map.
 * Arguments   : node  -- The node pointer of the node.
 *               text  -- A string pointer to the new label text.
 * Returns     : Nothing.
 */
void
map_set_node_label (map_node *node, gchar *text)
{
  GnomeCanvasItem   *item;

  if ( (item = gtk_object_get_data (GTK_OBJECT (node->mn_cgroup),
				    "text_item")) )
    {
      gnome_canvas_item_set (item, "text", text, NULL);
      return;
    }
	
  switch (node->mn_type)
    {
    case MAP_NODE_HOST:
      item = gnome_canvas_item_new (node->mn_cgroup,
				    gnome_canvas_text_get_type (),
				    "text", text,
				    "x", 0.0,
				    "y", 0.0,
				    "fill_color", "firebrick",
				    "anchor", GTK_ANCHOR_N,
				    NULL);
      break;
    case MAP_NODE_NETWORK:
      item = gnome_canvas_item_new (node->mn_cgroup,
				    gnome_canvas_text_get_type (),
				    "text", text,
				    "x", 0.0,
				    "y", 0.0,
				    "fill_color", "red",
				    "anchor", GTK_ANCHOR_N,
				    NULL);
      break;
    default:
      item = gnome_canvas_item_new (node->mn_cgroup,
				    gnome_canvas_text_get_type (),
				    "text", text,
				    "x", 0.0,
				    "y", 0.0,
				    "fill_color", "black",
				    "anchor", GTK_ANCHOR_N,
				    NULL);
      break;
    }
  gtk_object_set_data (GTK_OBJECT (node->mn_cgroup), "text_item", item);

}


/*
 * Function    : map_zoom_in
 * Description : This function will zoom the map in.
 * Arguments   : canvas -- The GNOME canvas pointer of the map.
 * Returns     : Nothing.
 */
void
map_zoom_in (GnomeCanvas *canvas)
{
  double pix = canvas->pixels_per_unit;

  if (pix < 10.0)
    {
      pix += 0.5;
      gnome_canvas_set_pixels_per_unit (canvas, pix);
    }
}
/*
 * Function    : map_zoom_out
 * Description : This function will zoom the map out.
 * Arguments   : canvas -- The GNOME canvas pointer of the map.
 * Returns     : Nothing.
 */
void
map_zoom_out (GnomeCanvas *canvas)
{
  double pix = canvas->pixels_per_unit;
  
  if (pix > 1.0)
    {
      pix -= 0.5;
      gnome_canvas_set_pixels_per_unit (canvas, pix);
    }
}

/*
 * Function    : map_highlight_host
 * Description : This function will highlight a given host on the map.
 * Arguments   : host -- The hosts entry structure pointer to the host.
 * Returns     ; None.
 */
void
map_highlight_host (hosts *host)
{
  GnomeCanvasItem  *item;
  GtkArg           my_arg;
  gboolean         is_active;
  GdkColor         *color;

  g_return_if_fail (host != NULL);
  item = gtk_object_get_data (GTK_OBJECT (host->hl_mnode), "text_item");
  if (item)
    {
      is_active = (gboolean)gtk_object_get_data (GTK_OBJECT (item), 
						   "is_active");
      if (!is_active)
	{
	  my_arg.name = "fill_color_gdk";
	  gtk_object_getv (GTK_OBJECT (item), 1, &my_arg);
	  color = GTK_VALUE_BOXED (my_arg);
	  gtk_object_set_data (GTK_OBJECT (item), "start_color",
			       (GdkColor *)color);
	  gnome_canvas_item_set (item,
				 "fill_color", "blue",
				 NULL);
	  gtk_object_set_data (GTK_OBJECT (item), "is_active", 
			       (gboolean *)TRUE);
	}
    }
}

/*
 * Function    : map_unhighlight_host
 * Description : This will return a given host on the map to its un highlit
 *               look.
 * Arguments   : host    -- The hosts entry structure pointer of the host 
 *                          highlight.
 * Returns     : None.
 */
void
map_unhighlight_host (hosts *host)
{
  GnomeCanvasItem   *item;
  GdkColor           *color;

  g_return_if_fail (host != NULL);

  item = gtk_object_get_data (GTK_OBJECT (host->hl_mnode), "text_item");
  if (item)
    {
      color = (GdkColor *)gtk_object_get_data (GTK_OBJECT (item), 
					      "start_color");
      gnome_canvas_item_set (item,
			     "fill_color_gdk", color,
			     NULL);
      gtk_object_remove_data (GTK_OBJECT (item), "start_color");
      gtk_object_remove_data (GTK_OBJECT (item), "is_active");
    }
}

/*
 * Function    : map_get_canvas_widget 
 * Description : This function will return the canvas widget of the map.
 * Arguments   : None.
 * Returns     : canvas gtk widget pointer or null if the map has been 
 *               destroyed.
 */
GtkWidget *
map_get_canvas_widget ()
{
  g_return_val_if_fail (map_panel != NULL, NULL);

  return (GtkWidget *)get_widget (map_panel, "map_canvas");
}

/*
 * Function    : map_get_selected_host 
 * Arguments   : None
 * Returns     : The hosts entry structure of the selected host or, 
 *               NULL on failure. This will return the last selected
 *               host.
 */
hosts *
map_get_selected_host ()
{
  GtkWidget   *canvas;
  g_return_val_if_fail (map_panel != NULL, NULL);

  canvas = map_get_canvas_widget ();
  return (hosts *)(gtk_object_get_data (GTK_OBJECT (canvas), "selected_host"));
}

/*
 * Function    : map_update_stat 
 * Description : This will update the status in the map_stat area,
 * Arguments   : varargs format.
 * Returns     : Nothing.
 */
void
map_update_stat (gchar *msg, ...)
{
  va_list   ap;
  gchar     buf[2048];
  GtkWidget *stat_bar;
  gint      statusid;

  va_start (ap, msg);
  vsnprintf (buf, sizeof (buf), msg, ap);
  va_end (ap);
  
  stat_bar = GTK_WIDGET (get_widget (map_panel, "map_stat"));
  if (stat_bar)
    {
      
      statusid = gtk_statusbar_get_context_id (GTK_STATUSBAR (stat_bar), 
					       "map_stat");
      gtk_statusbar_pop (GTK_STATUSBAR (stat_bar), statusid);
      gtk_statusbar_push (GTK_STATUSBAR (stat_bar), statusid, buf);
    }
}
/*
 * Function    : map_select_item 
 * Description : This function will draw a select box around a given item.
 *               A secondary function of this is that it handles the manual
 *               linking of nodes.
 */
void
map_select_node (map_node *node)
{
  map_node    *node1;
  int         num_selected;

  if (map_get_mode () == ITEM_PICK)
    {
      num_selected = (int)gtk_object_get_data (GTK_OBJECT (map_panel), 
					  "number_selected");
      switch (num_selected)
	{
	case 0:
	  gtk_object_set_data (GTK_OBJECT (map_panel), "node_1", node);
	  map_update_stat (_("Select the second node to link to."));
	  num_selected++;
	  gtk_object_set_data (GTK_OBJECT (map_panel), "number_selected",
			       (int *)num_selected);
	  break;
	case 1:
	  node1 = (map_node *)gtk_object_get_data (GTK_OBJECT (map_panel), 
						   "node_1");
	  if (node1)
	    {
	      map_link_nodes (node1, node);
	      gtk_object_remove_data (GTK_OBJECT (map_panel), "node_1");
	      gtk_object_remove_data (GTK_OBJECT (map_panel), 
				      "number_selected");
	      map_update_stat (_("Nodes linked."));
	      map_set_mode (ITEM_NONE);
	    }
	  break;
	default:
	  g_print ("num_selected %d",num_selected);
	  break;
	}
    }
  /* Not picking a node. Selecting it for some other purpose.
   */
 /*
   * -- This needs code in the canvas to be completed...
   
  GnomeCanvasPoints    *points;
  GnomeCanvasItem      *outline;


  points = gnome_canvas_points_new (2);

  points->coords[0] = item->x1;
  points->coords[1] = item->y1 - 20;
  points->coords[2] = item->x1 + 10;
  points->coords[3] = item->y1 - 20;

  outline = gnome_canvas_item_new (group,
				   gnome_canvas_line_get_type (),
				   "points", points,
				   "fill_color", "black",
				   "width_units", 2.0,
				   NULL);
  gnome_canvas_points_free (points);
  */
}

/* EOF */
