/*
 *	cook - file construction tool
 *	Copyright (C) 1994, 1997, 1998 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to do lexical analysis on the fingerprint cache file
 */

#include <ac/ctype.h>
#include <errno.h>
#include <ac/stdio.h>

#include <error_intl.h>
#include <fngrprnt_lex.h>
#include <str.h>
#include <fngrprnt.gen.h> /* after str.h */

static char	*fn;
static FILE	*fp;
static int	nerr;
static long	linum;


void
fngrprnt_lex_open(s)
	char	*s;
{
	fn = s;
	fp = fopen(s, "r");
	if (!fp)
	{
		if (errno != ENOENT)
		{
			sub_context_ty	*scp;

			scp = sub_context_new();
			sub_errno_set(scp);
			sub_var_set(scp, "File_Name", "%s", fn);
			fatal_intl(scp, i18n("open $filename: $errno"));
			/* NOTREACHED */
		}
	}
	linum = 1;
	nerr = 0;
}


void
fngrprnt_lex_close()
{
	if (nerr)
	{
		sub_context_ty	*scp;

		scp = sub_context_new();
		sub_var_set(scp, "File_Name", "%s", fn);
		sub_var_set(scp, "Number", "%d", nerr);
		sub_var_optional(scp, "Number");
		fatal_intl(scp, i18n("$filename: found $number fatal errors"));
		/* NOTREACHED */
	}
	if (fp)
		fclose(fp);
	nerr = 0;
	fn = 0;
	fp = 0;
	linum = 0;
}


static int lex_getc _((void));

static int
lex_getc()
{
	int		c;

	c = fgetc(fp);
	switch (c)
	{
	case EOF:
		if (ferror(fp))
		{
			sub_context_ty	*scp;

			scp = sub_context_new();
			sub_errno_set(scp);
			sub_var_set(scp, "File_Name", "%s", fn);
			fatal_intl(scp, i18n("read $filename: $errno"));
			/* NOTREACHED */
		}
		break;
	
	case '\n':
		linum++;
		break;
	}
	return c;
}


static void lex_getc_undo _((int));

static void
lex_getc_undo(c)
	int		c;
{
	switch (c)
	{
	case EOF:
		break;

	case '\n':
		--linum;
		/* fall through... */

	default:
		ungetc(c, fp);
		break;
	}
}


static void fngrprnt_lex_error _((sub_context_ty *, char *));

static void
fngrprnt_lex_error(scp, s)
	sub_context_ty	*scp;
	char		*s;
{
	string_ty	*buffer;
	int		len;
	int		need_to_delete;

	if (scp)
		need_to_delete = 0;
	else
	{
		scp = sub_context_new();
		need_to_delete = 1;
	}

	buffer = subst_intl(scp, s);
	len = buffer->str_length;
	while (len > 0 && isspace(buffer->str_text[len - 1]))
		--len;

	/* re-use substitution context */
	sub_var_set(scp, "File_Name", "%s", fn);
	sub_var_set(scp, "Number", "%ld", linum);
	sub_var_set(scp, "MeSsaGe", "%.*S", len, buffer);
	error_intl(scp, i18n("$filename: $number: $message"));
	str_free(buffer);
	if (++nerr >= 20)
	{
		/* re-use substitution context */
		sub_var_set(scp, "File_Name", "%s", fn);
		fatal_intl(scp, i18n("$filename: too many fatal errors"));
	}

	if (need_to_delete)
		sub_context_delete(scp);
}


int
fngrprnt_lex()
{
	int		c;
	char		buffer[2000];
	char		*cp;
	long		n;

	if (!fp)
		return 0;
	for (;;)
	{
		c = lex_getc();
		switch (c)
		{
		case EOF:
			return 0;

		case '"':
			cp = buffer;
			for (;;)
			{
				c = lex_getc();
				if (c == EOF || c == '\n')
				{
					unterm:
					fngrprnt_error("unterminated string");
					break;
				}
				if (c == '"')
					break;
				if (c == '\\')
				{
					c = lex_getc();
					if (c == EOF || c == '\n')
						goto unterm;
					if (c != '"' && c != '\\')
					{
						sub_context_ty	*scp;

						scp = sub_context_new();
						sub_var_set(scp, "Name", "\\%c", c);
						fngrprnt_lex_error
						(
							scp,
						  i18n("unknown '$name' escape")
						);
						sub_context_delete(scp);
					}
				}
				if (cp < ENDOF(buffer))
					*cp++ = c;
			}
			fngrprnt_lval.lv_string = str_n_from_c(buffer, cp - buffer);
			return STRING;

		case '0': case '1': case '2': case '3': case '4':
		case '5': case '6': case '7': case '8': case '9': 
			n = 0;
			for (;;)
			{
				n = n * 10 + c - '0';
				c = lex_getc();
				switch (c)
				{
				case '0': case '1': case '2': case '3':
				case '4': case '5': case '6': case '7':
				case '8': case '9': 
					continue;
				
				default:
					break;
				}
				break;
			}
			lex_getc_undo(c);
			fngrprnt_lval.lv_number = n;
			return NUMBER;

		case ' ':
		case '\t':
		case '\n':
			break;

		case '=':
			return EQ;

		case '{':
			return LB;
		
		case '}':
			return RB;

		default:
			return JUNK;
		}
	}
}


void
fngrprnt_error(s)
	char		*s;
{
	fngrprnt_lex_error(0, s);
}
