/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#ifndef __GBWIDGET_H__
#define __GBWIDGET_H__

#include <stdio.h>
#include <stdlib.h>
#include "project.h"

#define GB_WIDGET_DATA_KEY	"GB_WIDGET_DATA"
#define GB_IS_GB_WIDGET(w) \
	(gtk_object_get_data(GTK_OBJECT(w), GB_WIDGET_DATA_KEY) != NULL)

/* This key has a non-null value if a widget is a placeholder. */
#define GB_PLACEHOLDER_KEY	"GB_PLACEHOLDER"
/* Macro to see if a widget is a placeholder. */
#define GB_IS_PLACEHOLDER(w) \
	(gtk_object_get_data(GTK_OBJECT(w), GB_PLACEHOLDER_KEY) != NULL)


/* This holds a string describing any special relationship a widget has with
   its parent, e.g. "CList:title" or "Dialog:ok_button". These widgets may
   also have different behaviours, e.g. they can't be deleted. */
#define GB_CHILD_NAME_KEY	"GB_CHILD_NAME"

/*
 * The hash table associates a Gtk widget class name with a table of functions
 *  used by the builder, e.g. creating a new widget, saving the widget etc.
 */
extern GHashTable *gb_widget_table;

/* The id hash stores the last id used for each widget class. */
extern GHashTable *gb_widget_id_hash;


#define GB_STYLE_UNNAMED	"<none>"
#define GB_STYLE_DEFAULT	"Default"
/* This is the default GTK font spec., from gtkstyle.c */
#define GB_DEFAULT_XLFD_FONTNAME \
	"-adobe-helvetica-medium-r-normal--*-120-*-*-*-*-*-*"

#define GB_NUM_STYLE_STATES 5
extern gchar*  GbStateNames[];
#define GB_NUM_STYLE_COLORS 4
extern gchar*  GbColorNames[];
extern gchar*  GbBgPixmapName;

/* This contains all extra info needed to recreate a style.
   NOTE: A GtkStyle may be used by more than one GbStyle. */
typedef struct _GbStyle GbStyle;
struct _GbStyle
{
  gchar		*name;
  gchar		*xlfd_fontname;
  gchar		*bg_pixmap_filenames[GB_NUM_STYLE_STATES];
  GtkStyle	*style;
  gint		 ref_count;
};


/* The style data hash stores named GbStyles with the name as the key.
   Note that the GbStyle name field is also used as the key so be careful when
   freeing or updating it. */
extern GHashTable *gb_style_hash;

/* The default GbStyle of created widgets. We need to use a separate default
   style for the widgets created so that the Glade UI doesn't change.
   Remember to push the GtkStyle before creating any gbwidgets. */
extern GbStyle *gb_widget_default_gb_style;


/*************************************************************************
 * Extra data kept for each widget.
 *************************************************************************/

/* Flags in GbWidgetData */
enum
{
  GB_VISIBLE		= 1 << 0, /* If the widget is initially visible */
  GB_SENSITIVE		= 1 << 1, /* If the widget is initially sensitive */
  GB_GRAB_DEFAULT	= 1 << 2, /* If it grabs the default */
  GB_GRAB_FOCUS		= 1 << 3, /* If it grabs the focus */
  GB_STYLE_IS_UNNAMED	= 1 << 4, /* If it's using its own unnamed style */
  GB_STYLE_PROPAGATE	= 1 << 5, /* If it propgates style to its children */
  GB_ACTIVE		= 1 << 6, /* If it is initially active (for toggles) */
  GB_WIDTH_SET		= 1 << 7, /* If the width is set explicitly */
  GB_HEIGHT_SET		= 1 << 8, /* If the height is set explicitly */

  GB_INITIAL_EXPOSE	= 1 << 9  /* Internally used - set when created, so
				     that we display the size & position
				     properties when first exposed. */
};

typedef struct _GbWidgetData  GbWidgetData;
struct _GbWidgetData
{
  guint16	 flags;
  gint16	 x;
  gint16	 y;
  gint16	 width;
  gint16	 height;
  gint		 events;
  gchar		*tooltip;
  GList		*signals;
  GList		*accelerators;
  GbStyle	*gbstyle;
};

typedef struct _GbSignal GbSignal;
struct _GbSignal
{
  gchar		*name;
  gchar		*handler;
  gchar		*object;
  gboolean	 after;
  gchar		*data;
};

typedef struct _GbAccelerator GbAccelerator;
struct _GbAccelerator
{
  guint8	 modifiers;
  gchar		*key;
  gchar		*signal;
};



/*************************************************************************
 * Data structures passed to the GbWidget functions
 *************************************************************************/

/* GbWidgetNewData - used when new widgets are created.
   name is the name of the new widget. type is its Gtk type id.
   callback is the function to be called when the widget has been created.
   It is used after a dialog has been shown to let the user configure a
   widget (e.g. setting the number of rows in a new vbox).
   widget is the placeholder to be replaced or the fixed container to add
   the widget to. x, y, width & height specify the position and size of the
   new widget, when adding to a fixed container.
   data1 & data2 are used to hold pointers to entry fields in the dialog boxes,
   so we can get at the values easily.
   */

typedef struct _GbWidgetNewData		GbWidgetNewData;
typedef struct _GbWidgetCreateArgData	GbWidgetCreateArgData;
typedef struct _GbWidgetGetArgData	GbWidgetGetArgData;
typedef struct _GbWidgetSetArgData	GbWidgetSetArgData;
typedef struct _GbWidgetCreateMenuData	GbWidgetCreateMenuData;
typedef struct _GbWidgetWriteSourceData	GbWidgetWriteSourceData;

typedef void (*GbWidgetNewCallback)	(GtkWidget *widget,
					 struct _GbWidgetNewData *data);

typedef enum
{
  GB_CREATING,
  GB_SHOWING,
  GB_APPLYING,
  GB_LOADING,
  GB_SAVING
} GbWidgetAction;

struct _GbWidgetNewData
{
  GbWidgetAction action;
  gchar		*name;
  guint		 type;
  GbWidgetNewCallback callback;
  GtkWidget	*parent;
  GtkWidget	*current_child;
  gint		 x, y;
  GbWidgetData	*widget_data;

  /* This is only used when loading. */
  GbWidgetSetArgData *loading_data;
};

struct _GbWidgetCreateArgData
{
  gchar		 dummy;		/* This is not used. It is only here to avoid
				   compiler problems (on Alpha). */
};


struct _GbWidgetGetArgData
{
  GbWidgetAction action;
  GbWidgetData	*widget_data;

  /* These are only used when saving. */
  gchar		*filename;
  FILE		*fp;
  gchar		*buffer;
  gint		 buffer_pos;
  gint		 buffer_space;
  gint		 indent;
  GbStatusCode	 status;
};


typedef enum
{
  GB_TOKEN_START_TAG,
  GB_TOKEN_END_TAG,
  GB_TOKEN_DATA
} GbTokenType;

typedef struct _GbLoadedProperty  GbLoadedProperty;
struct _GbLoadedProperty
{
  gint tag_index;
  gint cdata_index;
  gint line_number;
};

#define GB_LOADED_PROPERTIES_INC 16

typedef enum
{
  GB_STANDARD_PROPERTIES,
  GB_CHILD_PROPERTIES
} GbPropertyType;

struct _GbWidgetSetArgData {
  GbWidgetAction action;
  GbWidgetData	*widget_data;
  GtkWidget	*property_to_apply;
  gboolean	 apply;

  /* These are only used when loading. */
  gchar		*filename;
  FILE		*fp;
  gint		 line_number;
  gchar		*buffer;
  gint		 buffer_pos;
  gint		 buffer_space;
  GbTokenType	 token_type;
  gint		 token;
  GbStatusCode	 status;
  GbLoadedProperty *properties;
  gint		 nproperties;
  gint		 properties_space;
  GbLoadedProperty *child_properties;
  gint		 nchild_properties;
  gint		 child_properties_space;
  GbPropertyType loading_type;
  GList		*signals;
  GList		*accelerators;
  GbStyle	*gbstyle;
  GList		*error_messages;
};


struct _GbWidgetCreateMenuData
{
  GtkWidget	*menu;
  GtkWidget	*child;
};


struct _GbWidgetWriteSourceData
{
  gchar		*project_name;
  gchar		*main_c_filename;
  gchar		*main_h_filename;
  gchar		*signal_c_filename;
  gchar		*signal_h_filename;
  gboolean	 set_widget_names;
  gboolean	 use_widget_hash;

  GbStatusCode	 status;
  FILE		*cfp;
  FILE		*hfp;
  FILE		*sigcfp;
  FILE		*sighfp;
  gboolean	 need_tooltips;
  gboolean	 need_accel_group;
  gchar		*wname;
  GtkWidget	*component;
  gchar		*component_name;
  GbWidgetData	*widget_data;
  GtkWidget	*parent;
  GHashTable	*standard_widgets;	/* A hash of GTK widgets used to
					   determine if values differ from
					   their default values. */
  GtkWidget	*standard_widget;	/* This is the standard widget
					   corresponding to the widget being
					   written. */
  gboolean	 create_widget;
  gboolean	 write_children;
  gchar		*buffer;
  gint		 buffer_pos;
  gint		 buffer_space;
  gchar		*decl_buffer;
  gint		 decl_buffer_pos;
  gint		 decl_buffer_space;
};


/*************************************************************************
 * The GbWidget struct which contains info on each widget class.
 *************************************************************************/

/* These are used in the properties_page_number of the GbWidget struct to
   specify whether the GbWidget's properties page has been created or if it
   doesn't need one (i.e. it has no properties of its own). */
#define GB_PROPERTIES_NOT_CREATED	-1
#define GB_PROPERTIES_NOT_NEEDED	-2

/* The GbWidget struct, which consists of pointers to the functions
   implemented by each GbWidget. The properties_page_number is the page in
   the widget properties notebook corresponding to the GbWidget */
typedef struct _GbWidget  GbWidget;
struct _GbWidget
{
  gchar		**pixmap_struct;
  GdkPixmap	 *gdkpixmap;
  GdkBitmap	 *mask;
  gchar		 *tooltip;
  gint		  properties_page_number;	/* Used internally */

  GtkWidget* (* gb_widget_new)		(GbWidgetNewData             *data);

  void (* gb_widget_create_properties)	(GtkWidget		     *widget,
					 GbWidgetCreateArgData       *data);
  void (* gb_widget_get_properties)	(GtkWidget                   *widget,
					 GbWidgetGetArgData	     *data);
  void (* gb_widget_set_properties)	(GtkWidget                   *widget,
					 GbWidgetSetArgData	     *data);
  void (* gb_widget_create_popup_menu)	(GtkWidget                   *widget,
					 GbWidgetCreateMenuData      *data);
  void (* gb_widget_write_source)	(GtkWidget                   *widget,
					 GbWidgetWriteSourceData     *data);
  GtkWidget* (* gb_widget_get_child)	(GtkWidget                   *widget,
					 gchar			     *child_name);
};



/*************************************************************************
 * Macros to support showing properties and saving.  Note that since they
 * are macros you need to be careful how you use them. I may make them
 * functions later.
 *************************************************************************/

#define gb_widget_output_string(data, property, value) \
	if (data->action == GB_SAVING) save_string(data, property, value); \
	else property_set_string(property, value);
#define gb_widget_output_text(data, property, value) \
	if (data->action == GB_SAVING) save_text(data, property, value); \
	else property_set_text(property, value);
#define gb_widget_output_int(data, property, value) \
	if (data->action == GB_SAVING) save_int(data, property, value); \
	else property_set_int(property, value);
#define gb_widget_output_float(data, property, value) \
	if (data->action == GB_SAVING) save_float(data, property, value); \
	else property_set_float(property, value);
#define gb_widget_output_bool(data, property, value) \
	if (data->action == GB_SAVING) save_bool(data, property, value); \
	else property_set_bool(property, value);
#define gb_widget_output_choice(data, property, value, symbol) \
	if (data->action == GB_SAVING) save_choice(data, property, symbol); \
	else property_set_choice(property, value);
#define gb_widget_output_combo(data, property, value) \
	if (data->action == GB_SAVING) save_combo(data, property, value); \
	else property_set_combo(property, value);
#define gb_widget_output_color(data, property, value) \
	if (data->action == GB_SAVING) save_color(data, property, value); \
	else property_set_color(property, value);
#define gb_widget_output_bgpixmap(data, property, value, filename) \
	if (data->action == GB_SAVING) save_bgpixmap(data, property, filename); \
	else property_set_bgpixmap(property, value, filename);
#define gb_widget_output_dialog(data, property, string, value) \
	if (data->action == GB_SAVING) save_dialog(data, property, value); \
	else property_set_dialog(property, string, value);
#define gb_widget_output_filename(data, property, value) \
	if (data->action == GB_SAVING) save_filename(data, property, value); \
	else property_set_filename(property, value);
#define gb_widget_output_font(data, property, value, xlfd_fontname) \
	if (data->action == GB_SAVING) save_font(data, property, xlfd_fontname); \
	else property_set_font(property, value, xlfd_fontname);


/* Macros to support applying properties and loading. */
#define gb_widget_input_string(data, property) \
	(data->action == GB_LOADING) ? load_string(data, property) \
	 : property_get_string(property, data->property_to_apply, &data->apply)
#define gb_widget_input_text(data, property) \
	(data->action == GB_LOADING) ? load_text(data, property) \
	 : property_get_text(property, data->property_to_apply, &data->apply)
#define gb_widget_input_int(data, property) \
	(data->action == GB_LOADING) ? load_int(data, property) \
	 : property_get_int(property, data->property_to_apply, &data->apply)
#define gb_widget_input_float(data, property) \
	(data->action == GB_LOADING) ? load_float(data, property) \
	 : property_get_float(property, data->property_to_apply, &data->apply)
#define gb_widget_input_bool(data, property) \
	(data->action == GB_LOADING) ? load_bool(data, property) \
	 : property_get_bool(property, data->property_to_apply, &data->apply)
#define gb_widget_input_choice(data, property) \
	(data->action == GB_LOADING) ? load_choice(data, property) \
	 : property_get_choice(property, data->property_to_apply, &data->apply)
#define gb_widget_input_combo(data, property) \
	(data->action == GB_LOADING) ? load_combo(data, property) \
	 : property_get_combo(property, data->property_to_apply, &data->apply)
#define gb_widget_input_color(data, property) \
	(data->action == GB_LOADING) ? load_color(data, property) \
	 : property_get_color(property, data->property_to_apply, &data->apply)
#define gb_widget_input_bgpixmap(data, property, filename) \
	(data->action == GB_LOADING) ? load_bgpixmap(data, property, filename) \
	 : property_get_bgpixmap(property, data->property_to_apply, &data->apply, filename)
#define gb_widget_input_dialog(data, property) \
	(data->action == GB_LOADING) ? load_dialog(data, property) \
	 : property_get_dialog(property, data->property_to_apply, &data->apply)

#define gb_widget_input_filename(data, property) \
	(data->action == GB_LOADING) ? load_filename(data, property) \
	 : property_get_filename(property, data->property_to_apply, &data->apply)
#define gb_widget_input_font(data, property, xlfd_fontname) \
	(data->action == GB_LOADING) ? load_font(data, property, xlfd_fontname) \
	 : property_get_font(property, data->property_to_apply, &data->apply, xlfd_fontname)


/*************************************************************************
 * Public Functions
 *************************************************************************/

/* Call this first, to initialize all GtkWidget & GbWidget types */
void	    gb_widgets_init();

void	    gb_widget_init_struct	(GbWidget	       *gbwidget);


GtkWidget*  gb_widget_new		(gchar		       *class_name,
					 GtkWidget	       *parent);
GtkWidget*  gb_widget_new_full		(gchar		       *class_name,
					 GtkWidget	       *parent,
					 GtkWidget	       *current_child,
					 gint			x,
					 gint			y,
					 GbWidgetNewCallback	callback,
					 GbWidgetAction		action,
					 GbWidgetSetArgData    *loading_data);
GbWidgetData* gb_widget_new_widget_data (void);
gint	    gb_widget_get_last_id	(gchar		       *name);
void	    gb_widget_set_last_id	(gchar		       *name,
					 gint			last_id);
gchar*	    gb_widget_new_name		(gchar		       *class_name);
void	    gb_widget_create_from	(GtkWidget	       *widget,
					 gchar		       *name);
gboolean    gb_widget_can_finish_new	(GbWidgetNewData       *data);
void        gb_widget_real_initialize   (GtkWidget	       *widget,
					 GbWidgetData	       *wdata);
void        gb_widget_initialize	(GtkWidget	       *widget,
					 GbWidgetNewData       *data);
void	    gb_widget_free_new_data	(GbWidgetNewData       *data);
void	    gb_widget_reset_ids		(void);

void	    gb_widget_set_usize		(GtkWidget	       *widget,
					 gint			w,
					 gint			h);
gint	    gb_widget_create_properties	(GtkWidget	       *widget);

void	    gb_widget_show_properties	(GtkWidget	       *widget);
void	    gb_widget_show_position_properties
					(GtkWidget	       *widget);
void	    gb_widget_show_style	(GtkWidget	       *widget);

void	    gb_widget_apply_properties	(GtkWidget	       *widget,
					 GtkWidget	       *property);

void	    gb_widget_show_popup_menu	(GtkWidget	       *widget,
					 GdkEventButton	       *event);
void	    gb_widget_load		(GtkWidget	       *widget,
					 GbWidgetSetArgData    *data,
					 GtkWidget	       *parent);
void	    gb_widget_load_style	(GbWidgetSetArgData    *data);
void	    gb_widget_add_child		(GtkWidget	       *parent,
					 GbWidgetSetArgData    *data,
					 GtkWidget	       *child);

void	    gb_widget_save		(GtkWidget	       *widget,
					 GbWidgetGetArgData    *data);
void	    gb_widget_write_source	(GtkWidget	       *widget,
					 GbWidgetWriteSourceData *data);
void	    gb_widget_write_standard_source
					(GtkWidget	       *widget,
					 GbWidgetWriteSourceData *data);
void	    gb_widget_write_add_child_source
					(GtkWidget	       *widget,
					 GbWidgetWriteSourceData *data);

void	    gb_widget_replace_child	(GtkWidget	       *widget,
					 GtkWidget	       *current_child,
					 GtkWidget	       *new_child);

/* Tooltips functions. */
gboolean    gb_widget_get_show_tooltips	(void);
void	    gb_widget_toggle_show_tooltips
					(GtkWidget	       *widget,
					 gpointer	        data);
void	    gb_widget_reset_tooltips	(void);

/* Functions for handling GbStyles. */
void	    gb_widget_reset_gb_styles	(void);
GbStyle*    gb_widget_new_gb_style	(void);
GbStyle*    gb_widget_copy_gb_style	(GbStyle	       *gbstyle);
void	    gb_widget_ref_gb_style	(GbStyle	       *gbstyle);
void	    gb_widget_unref_gb_style	(GbStyle	       *gbstyle);
void	    gb_widget_destroy_gb_style	(GbStyle	       *gbstyle,
					 gboolean		remove_from_hash);

void	    gb_widget_set_gb_style	(GtkWidget	       *widget,
					 GbStyle	       *gbstyle);
void	    gb_widget_update_gb_styles	(GbStyle	       *old_gbstyle,
					 GbStyle	       *new_gbstyle);
void	    gb_widget_save_style	(GbStyle	       *gbstyle,
					 GbWidgetGetArgData    *data,
					 gboolean	        save_all);

/* Convenience functions for handling all 6 properties of adjustments. */
void	    gb_widget_output_adjustment	(GbWidgetGetArgData *data,
					 gchar		    *Values[],
					 GtkAdjustment	    *adjustment);
gboolean    gb_widget_input_adjustment	(GbWidgetSetArgData *data, 
					 gchar		    *Values[],
					 GtkAdjustment	    *adjustment);


/* Common functions for gbwidgets that have a child label - button/menuitem
   etc. */
void	    gb_widget_remove_label	(GtkWidget	       *menuitem,
					 GtkWidget	       *widget);

void	    gb_widget_output_child_label(GtkWidget	       *widget,
					 GbWidgetGetArgData    *data,
					 gchar		       *Label);
void	    gb_widget_input_child_label	(GtkWidget	       *widget,
					 GbWidgetSetArgData    *data,
					 gchar		       *Label);

void	    gb_widget_update_table_placeholders
					(GtkWidget	       *table,
					 gint			rows,
					 gint			cols);

void	    gb_widget_update_box_size	(GtkWidget	       *widget,
					 gint			size);

#endif	/* __GBWIDGET_H__ */
