/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include "../gb.h"

#ifdef GLD_HAVE_GTK_1_1

/* Include the 21x21 icon pixmap for this widget, to be used in the palette */
/* This widget has no icon as it doesn't appear in the palette */
/*#include "../graphics/"*/

/*
 * This is the GbWidget struct for this widget (see ../gbwidget.h).
 * It is initialized in the init() function at the end of this file
 */
static GbWidget gbwidget;

/* These are the special children of the widget. */
static gchar *FontSelOKButton = "FontSel:ok_button";
static gchar *FontSelCancelButton = "FontSel:cancel_button";
static gchar *FontSelApplyButton = "FontSel:apply_button";

static gchar *Title = "FontSelDialog|GtkWindow::title";
static gchar *Type = "FontSelDialog|GtkWindow::type";
static gchar *Position = "FontSelDialog|GtkWindow::position";
static gchar *Shrink = "FontSelDialog|GtkWindow::allow_shrink";
static gchar *Grow = "FontSelDialog|GtkWindow::allow_grow";
static gchar *AutoShrink = "FontSelDialog|GtkWindow::auto_shrink";
static gchar *WMName = "FontSelDialog|GtkWindow::wmclass_name";
static gchar *WMClass = "FontSelDialog|GtkWindow::wmclass_class";

static gchar *GbTypeChoices[] =
{"Top Level", "Dialog", "Popup", NULL};
static gint GbTypeValues[] =
{
  GTK_WINDOW_TOPLEVEL,
  GTK_WINDOW_DIALOG,
  GTK_WINDOW_POPUP
};
static gchar *GbTypeSymbols[] =
{
  "GTK_WINDOW_TOPLEVEL",
  "GTK_WINDOW_DIALOG",
  "GTK_WINDOW_POPUP"
};

static gchar *GbPositionChoices[] =
{"None", "Center", "Mouse", NULL};
static gint GbPositionValues[] =
{
  GTK_WIN_POS_NONE,
  GTK_WIN_POS_CENTER,
  GTK_WIN_POS_MOUSE
};
static gchar *GbPositionSymbols[] =
{
  "GTK_WIN_POS_NONE",
  "GTK_WIN_POS_CENTER",
  "GTK_WIN_POS_MOUSE"
};

/******
 * NOTE: To use these functions you need to uncomment them AND add a pointer
 * to the funtion in the GbWidget struct at the end of this file.
 ******/

/*
 * Creates a new GtkWidget of class GtkFontSelectionDialog, performing any specialized
 * initialization needed for the widget to work correctly in this environment.
 * If a dialog box is used to initialize the widget, return NULL from this
 * function, and call data->callback with your new widget when it is done.
 * If the widget needs a special destroy handler, add a signal here.
 */
GtkWidget*
gb_font_selection_dialog_new (GbWidgetNewData *data)
{
  GtkWidget *new_widget = gtk_font_selection_dialog_new (_("Select Font"));
  gtk_signal_connect (GTK_OBJECT (new_widget), "delete_event",
		      GTK_SIGNAL_FUNC (editor_close_window), NULL);

  gb_widget_create_from (GTK_FONT_SELECTION_DIALOG (new_widget)->ok_button,
			 "ok_button");
  gtk_object_set_data (GTK_OBJECT (GTK_FONT_SELECTION_DIALOG (new_widget)->ok_button),
		       GB_CHILD_NAME_KEY, g_strdup (FontSelOKButton));

  gb_widget_create_from (GTK_FONT_SELECTION_DIALOG (new_widget)->cancel_button,
			 "cancel_button");
  gtk_object_set_data (GTK_OBJECT (GTK_FONT_SELECTION_DIALOG (new_widget)->cancel_button),
		       GB_CHILD_NAME_KEY, g_strdup (FontSelCancelButton));

  gb_widget_create_from (GTK_FONT_SELECTION_DIALOG (new_widget)->apply_button,
			 "apply_button");
  gtk_object_set_data (GTK_OBJECT (GTK_FONT_SELECTION_DIALOG (new_widget)->apply_button),
		       GB_CHILD_NAME_KEY, g_strdup (FontSelApplyButton));

  return new_widget;
}



/*
 * Creates the components needed to edit the extra properties of this widget.
 */
static void
gb_font_selection_dialog_create_properties (GtkWidget * widget, GbWidgetCreateArgData * data)
{
  property_add_string (Title, _("Title:"), _("The title of the window"));
  property_add_choice (Type, _("Type:"),
		       _("The type of the window"),
		       GbTypeChoices);
  property_add_choice (Position, _("Position:"),
		       _("The initial position of the dialog"),
		       GbPositionChoices);
  property_add_bool (Shrink, _("Shrink:"), _("If the window can be shrunk"));
  property_add_bool (Grow, _("Grow:"), _("If the window can be enlarged"));
  property_add_bool (AutoShrink, _("Auto Shrink:"),
		     _("If the window shrinks automatically"));
  property_add_string (WMName, _("WM Name:"), _("The name to pass to the window manager"));
  property_add_string (WMClass, _("WM Class:"), _("The class name to pass to the window manager"));
}



/*
 * Gets the properties of the widget. This is used for both displaying the
 * properties in the property editor, and also for saving the properties.
 */
static void
gb_font_selection_dialog_get_properties (GtkWidget *widget, GbWidgetGetArgData * data)
{
  gint i, type;

  gb_widget_output_string (data, Title, GTK_WINDOW (widget)->title);

  /* The type is stored in the object hash as we can't change it after the
     window is realized. It will default to 0, which is OK. */
  type = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (widget), Type));
  gb_widget_output_choice (data, Type, type, GbTypeSymbols[type]);

  for (i = 0; i < sizeof (GbPositionValues) / sizeof (GbPositionValues[0]); i
       ++)
    {
      if (GbPositionValues[i] == GTK_WINDOW (widget)->position)
	gb_widget_output_choice (data, Position, i, GbPositionSymbols[i]);
    }
  gb_widget_output_bool (data, Shrink, GTK_WINDOW (widget)->allow_shrink);
  gb_widget_output_bool (data, Grow, GTK_WINDOW (widget)->allow_grow);
  gb_widget_output_bool (data, AutoShrink, GTK_WINDOW (widget)->auto_shrink);

  /* These are stored in the object hash since we can't set them after the
     window is realized. */
  gb_widget_output_string (data, WMName,
			 gtk_object_get_data (GTK_OBJECT (widget), WMName));
  gb_widget_output_string (data, WMClass,
			gtk_object_get_data (GTK_OBJECT (widget), WMClass));
}



/*
 * Sets the properties of the widget. This is used for both applying the
 * properties changed in the property editor, and also for loading.
 */
static void
gb_font_selection_dialog_set_properties (GtkWidget * widget, GbWidgetSetArgData * data)
{
  gint i;
  gchar *title, *type, *position, *wmname, *wmclass;
  gboolean shrink, grow, auto_shrink, myApply;

  title = gb_widget_input_string (data, Title);
  if (data->apply)
    gtk_window_set_title (GTK_WINDOW (widget), title);

  type = gb_widget_input_choice (data, Type);
  if (data->apply)
    {
      for (i = 0; i < sizeof (GbTypeValues) / sizeof (GbTypeValues[0]); i++)
	{
	  if (!strcmp (type, GbTypeChoices[i])
	      || !strcmp (type, GbTypeSymbols[i]))
	    {
	      gtk_object_set_data (GTK_OBJECT (widget), Type, GINT_TO_POINTER
				   (i));
	      break;
	    }
	}
    }

  position = gb_widget_input_choice (data, Position);
  if (data->apply)
    {
      for (i = 0; i < sizeof (GbPositionValues) / sizeof (GbPositionValues[0]);
	   i++)
	{
	  if (!strcmp (position, GbPositionChoices[i])
	      || !strcmp (position, GbPositionSymbols[i]))
	    {
	      gtk_window_position (GTK_WINDOW (widget), GbPositionValues[i]);
	      break;
	    }
	}
    }

  shrink = gb_widget_input_bool (data, Shrink);
  myApply = data->apply;
  grow = gb_widget_input_bool (data, Grow);
  myApply |= data->apply;
  auto_shrink = gb_widget_input_bool (data, AutoShrink);
  if (data->apply || myApply)
    gtk_window_set_policy (GTK_WINDOW (widget), shrink, grow, auto_shrink);

  wmname = gb_widget_input_string (data, WMName);
  if (wmname && wmname[0] == '\0')
    wmname = NULL;
  if (data->apply)
    {
      g_free (gtk_object_get_data (GTK_OBJECT (widget), WMName));
      gtk_object_set_data (GTK_OBJECT (widget), WMName, g_strdup (wmname));
    }

  wmclass = gb_widget_input_string (data, WMClass);
  if (wmclass && wmclass[0] == '\0')
    wmclass = NULL;
  if (data->apply)
    {
      g_free (gtk_object_get_data (GTK_OBJECT (widget), WMClass));
      gtk_object_set_data (GTK_OBJECT (widget), WMClass, g_strdup (wmclass));
    }
}



/*
 * Adds menu items to a context menu which is just about to appear!
 * Add commands to aid in editing a GtkFontSelectionDialog, with signals pointing to
 * other functions in this file.
 */
/*
static void
gb_font_selection_dialog_create_popup_menu (GtkWidget * widget, GbWidgetCreateMenuData * data)
{

}
*/



/*
 * Writes the source code needed to create this widget.
 * You have to output everything necessary to create the widget here, though
 * there are some convenience functions to help.
 */
static void
gb_font_selection_dialog_write_source (GtkWidget * widget, GbWidgetWriteSourceData * data)
{
  gint type, i;
  gchar *wmname, *wmclass;
  gchar *wname, *child_name;

  if (data->create_widget)
    {
      source_add (data, "  %s = gtk_font_selection_dialog_new (\"%s\");\n",
	      data->wname, source_make_string (GTK_WINDOW (widget)->title));
    }

  gb_widget_write_standard_source (widget, data);

  type = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (widget), Type));
  if (type != GTK_WINDOW_TOPLEVEL)
    {
      /* Note: there is no gtk_window_set_type () */
      source_add (data, "  GTK_WINDOW (%s)->type = %s;\n",
		  data->wname, GbTypeSymbols[type]);
    }

  if (GTK_WINDOW (widget)->position != GTK_WIN_POS_NONE)
    {
      for (i = 0; i < sizeof (GbPositionValues) / sizeof (GbPositionValues[0]);
	   i++)
	{
	  if (GbPositionValues[i] == GTK_WINDOW (widget)->position)
	    source_add (data, "  gtk_window_position (GTK_WINDOW (%s), %s);\n",
			data->wname, GbPositionSymbols[i]);
	}
    }

  if (GTK_WINDOW (widget)->allow_shrink != FALSE
      || GTK_WINDOW (widget)->allow_grow != TRUE
      || GTK_WINDOW (widget)->auto_shrink != FALSE)
    source_add (data,
		"  gtk_window_set_policy (GTK_WINDOW (%s), %s, %s, %s);\n",
		data->wname,
		GTK_WINDOW (widget)->allow_shrink ? "TRUE" : "FALSE",
		GTK_WINDOW (widget)->allow_grow ? "TRUE" : "FALSE",
		GTK_WINDOW (widget)->auto_shrink ? "TRUE" : "FALSE");

  wmname = gtk_object_get_data (GTK_OBJECT (widget), WMName);
  wmclass = gtk_object_get_data (GTK_OBJECT (widget), WMClass);
  if (wmname || wmclass)
    source_add (data,
	    "  gtk_window_set_wmclass (GTK_WINDOW (%s), \"%s\", \"%s\");\n",
		data->wname, wmname ? source_make_string (wmname) : "",
		wmclass ? source_make_string (wmclass) : "");

  /* We output the source code for the buttons here, but we don't want them
     to be created. We need to remember the dialog's name since data->wname
     will be overwritten. */
  wname = g_strdup (data->wname);

  source_add (data, "\n");
  child_name = gtk_widget_get_name (GTK_FONT_SELECTION_DIALOG (widget)->ok_button);
  child_name = source_create_valid_identifier (child_name);
  source_add (data, "  %s = GTK_FONT_SELECTION_DIALOG (%s)->ok_button;\n",
	      child_name, wname);
  g_free (child_name);
  data->create_widget = FALSE;
  gb_widget_write_source (GTK_FONT_SELECTION_DIALOG (widget)->ok_button,
			  data);

  child_name = gtk_widget_get_name (GTK_FONT_SELECTION_DIALOG (widget)->cancel_button);
  child_name = source_create_valid_identifier (child_name);
  source_add (data, "  %s = GTK_FONT_SELECTION_DIALOG (%s)->cancel_button;\n",
	      child_name, wname);
  g_free (child_name);
  data->create_widget = FALSE;
  gb_widget_write_source (GTK_FONT_SELECTION_DIALOG (widget)->cancel_button,
			  data);
  child_name = gtk_widget_get_name (GTK_FONT_SELECTION_DIALOG (widget)->apply_button);
  child_name = source_create_valid_identifier (child_name);
  source_add (data, "  %s = GTK_FONT_SELECTION_DIALOG (%s)->apply_button;\n",
	      child_name, wname);
  g_free (child_name);
  data->create_widget = FALSE;
  gb_widget_write_source (GTK_FONT_SELECTION_DIALOG (widget)->apply_button,
			  data);

  g_free (wname);

  data->write_children = FALSE;
}



/*
 * Initializes the GbWidget structure.
 * I've placed this at the end of the file so we don't have to include
 * declarations of all the functions.
 */
GbWidget*
gb_font_selection_dialog_init ()
{
  /* Initialise the GTK type */
  gtk_font_selection_dialog_get_type();

  /* Initialize the GbWidget structure */
  gb_widget_init_struct(&gbwidget);

  /* Fill in the pixmap struct & tooltip */
  gbwidget.pixmap_struct = NULL;
  gbwidget.tooltip = NULL;

  /* Fill in any functions that this GbWidget has */
  gbwidget.gb_widget_new		= gb_font_selection_dialog_new;
  gbwidget.gb_widget_create_properties	= gb_font_selection_dialog_create_properties;
  gbwidget.gb_widget_get_properties	= gb_font_selection_dialog_get_properties;
  gbwidget.gb_widget_set_properties	= gb_font_selection_dialog_set_properties;
  gbwidget.gb_widget_write_source	= gb_font_selection_dialog_write_source;
/*
  gbwidget.gb_widget_create_popup_menu	= gb_font_selection_dialog_create_popup_menu;
*/

  return &gbwidget;
}

#endif /* GLD_HAVE_GTK_1_1 */
