/* 
   NSException.h

   Copyright (C) 1995, 1996, 1997 Ovidiu Predescu and Mircea Oancea.
   All rights reserved.

   Author: Ovidiu Predescu <ovidiu@bx.logicnet.ro>

   This file is part of the Foundation Extensions Library.

   Permission to use, copy, modify, and distribute this software and its
   documentation for any purpose and without fee is hereby granted, provided
   that the above copyright notice appear in all copies and that both that
   copyright notice and this permission notice appear in supporting
   documentation.

   We disclaim all warranties with regard to this software, including all
   implied warranties of merchantability and fitness, in no event shall
   we be liable for any special, indirect or consequential damages or any
   damages whatsoever resulting from loss of use, data or profits, whether in
   an action of contract, negligence or other tortious action, arising out of
   or in connection with the use or performance of this software.
*/

#ifndef __NSException_h__
#define __NSException_h__

#import <Foundation/NSException.h>
#import <Foundation/NSArray.h>

extern NSString *NSMallocException;

@interface NSException (Extensions)
- (BOOL)exceptionIsKindOfClass:(Class)class;
				/* return [self isKindOfClass:class] */
- (BOOL)exceptionIsIn:(NSArray*)exceptions;
- (NSString*)errorString;
- initWithFormat:(NSString*)format, ...;
- setName:(NSString*)name;
- setReason:(NSString*)reason;
- setUserInfo:(NSDictionary*)userInfo;
@end /* NSException (Extension) */


@interface NSException (MissingMethods)
- (NSString*)name;
- (NSString*)reason;
- (NSDictionary*)userInfo;
@end

/*
 * The new macros for handling exceptions.
 */

#define TRY \
{ \
    NS_DURING {

#define END_TRY \
_quit: ; \
    } \
    NS_HANDLER { \
	BOOL _caught = NO; \
	if (0) {

#define CATCH(exception_class) \
    } else if ([localException isKindOfClass:[exception_class class]]) { \
	_caught = YES;

#define MULTICATCH(exception_classes) \
    } else if([localException exceptionIsIn: \
	    [NSArray arrayWithObjects:exception_classes, nil]]) { \
	_caught = YES;

#define OTHERWISE \
    } else { \
	_caught = YES;

#define CLEANUP \
# error "CLEANUP not supported"

#define FINALLY \
# error "FINALLY not supported"

#define END_CATCH \
	} \
	if (!_caught) \
	    [localException raise]; \
    } \
    NS_ENDHANDLER \
}

    /*  Use BREAK inside a TRY block to get out of it */
#define BREAK	({_NSRemoveHandler(&_localHandler); goto _quit;})

    /*  If you want to generate an exception issue a THROW with the exception
	an object derived from the NSException class. */
#define THROW(exception)		[exception raise]

    /*  If you want to reraise an exception inside an exception handler
	just say RERAISE. */
#define RERAISE                 THROW(localException)


/*
 * Assertions.
 */

#ifndef __FoundationException_definition__
#define __FoundationException_definition__

@interface FoundationException : NSException
@end

#endif /* __FoundationException_definition__ */

@interface AssertException : FoundationException
@end


#ifndef PRECOMP

#define Assert(condition) \
    ({if(!(condition)) {\
	NSLog([@"Assertion failed: " stringByAppendingString:(@#condition)]); \
	THROW([AssertException new]); \
    } \
    0;})

#endif /* PRECOMP */


#endif /* __NSException_h__ */
