/*
 *	Ohio Trollius
 *	Copyright 1995 The Ohio State University
 *	GDB
 *
 *	$Log:	bfstate.c,v $
 * Revision 6.1  96/11/23  17:52:31  nevin
 * Ohio Release
 * 
 * Revision 6.0  96/02/29  13:17:21  gdburns
 * Ohio Release
 * 
 * Revision 5.2.1.1  96/01/19  15:59:32  gdburns
 * Remove user space limit.
 * Don't collect packets; use nh_data[0] for total length.
 * 
 * Revision 5.2  94/08/22  13:40:46  gdburns
 * Ohio Release
 * 
 * Revision 5.1  94/05/18  10:38:03  gdburns
 * Ohio Release
 * 
 * Revision 2.3.1.1  94/05/18  10:02:45  gdburns
 * Print nh_type for MPI and PVM messages.
 * 
 * Revision 2.3  94/04/22  12:23:32  gdburns
 * Ohio Release
 *
 *	Function:	- prints Trollius buffer status
 */

#include <stdio.h>
#include <stdlib.h>

#include <args.h>
#include <bfreq.h>
#include <events.h>
#include <net.h>
#include <priority.h>
#include <terror.h>

/*
 * local functions
 */
static void		help();
static void		bfstate();

/*
 * local variables
 */
static struct bfstatus	*bftable;		/* buffer state descriptor */
static int		bfmax;			/* max entries */

/*
 * syntax for bfstate command
 */
static char		*usage =
	"bfstate [-ghlp] [-B <maxmsgs>] <nodes>\n";


main(argc, argv)

int   			argc;    
char   			*argv[];

{
	int		n_index;		/* index of node entry */
	int		nodeid;			/* value of node entry */
	int		n_flags;		/* extra node entry info */
/*
 * Parse the command line.
 */
	validopts("ghlpB");
	followed("B");

	if (do_args(&argc, argv)) {
		fprintf(stderr, usage);
		exit(errno);
	}

	if (opt_taken('h')) {
		help();
		exit(0);
	}
/*
 * Allocate buffer state table.
 */
	if (opt_taken('B')) {
		intparam('B', &bfmax);
	} else {
		bfmax = BFSMAX;
	}

	if (bfmax > 0) {
		bftable = (struct bfstatus *) malloc((unsigned) (bfmax *
				sizeof(struct bfstatus)));

		if (! bftable) {
			perror("bfstate (malloc)");
			exit(errno);
		}
	}
/*
 * Attach to kernel.
 */
	if (kinit(PRCMD)) {
		terror("bfstate (kinit)");
		exit(errno);
	}

	if (nid_parse(&argc, argv) || (errno = (argc == 1) ? 0 : EUSAGE)) {
		fprintf(stderr, usage);
		kexit(errno);
	}
/*
 * Loop through all specified nodes.
 */
	nid_get(&n_index, &nodeid, &n_flags);

	if (n_index < 0) {
		errno = EUSAGE;
		fprintf(stderr, usage);
		kexit(errno);
	}

	printf("NODE      DEST      EVENT    TYPE     LENGTH\n");

	do {
		bfstate(nodeid);
		nid_get(&n_index, &nodeid, &n_flags);
	} while (n_index);

	kexit(0);
	return(0);
}

/*
 *	bfstate
 *
 *	Function:	- finds and prints the buffer state of a remote node
 *	Accepts:	- node identifier
 */
static void
bfstate(nodeid)

int			nodeid;

{
	int		size;		/* # of return blocks */
	int		limit;		/* # of blocks to print */
	int		i;		/* favourite index */
	int4		totlength;	/* total message length */
	struct bfparms	parms;		/* buffer parameters */
	struct bfstatus	*pi;		/* current packet ptr */

	size = rbfstate(nodeid, bftable, bfmax, &parms);
	if (size < 0) lamfail("bfstate (rbfstate)");

	limit = (size < bfmax) ? size : bfmax;

	for (i = 0; i < limit; ++i) {
		pi = bftable + i;

		if ((!opt_taken('p') && (pi->bfs_flags & N2ND)) ||
				(!opt_taken('g') && (pi->bfs_event < 0)))
				continue;
/*
 * Collect all packets from long messages.
 */
		if (!opt_taken('p') && !(pi->bfs_flags & N2ND) &&
				(pi->bfs_flags & NMORE)) {
			totlength = pi->bfs_data[0];
		} else {
			totlength = pi->bfs_length;
		}

		if (totlength >= 0) {
			printf("%-9.9s", mnemonic(nodeid));
			printf(" %-9.9s", mnemonic(pi->bfs_node));

			if ((pi->bfs_flags & KSYNCSQL) ||
					(((pi->bfs_event & 0xFFFF0000) != 0)
					&& ((pi->bfs_event & 0xFFFF0000) !=
					0xFFFF0000))) {
				printf(" %-8X", pi->bfs_event);
			} else {
				printf(" %-8d", pi->bfs_event);
			}

			printf(" %-8X", pi->bfs_type);
			printf(" %-7d\n", totlength);
		}
	}

	if (size > bfmax) {
		printf("...and more buffered messages...\n");
	}

	if (opt_taken('l')) {
		printf("%-9.9s", mnemonic(nodeid));
		printf(" max space = %d, used space = %d\n",
				parms.bfp_maxspace, parms.bfp_nspace);
	}
}

/*
 *	help
 *
 *	Function:	- prints helpful information on the bfstate command
 */
static void
help()

{
	printf("\nSynopsis:	bfstate [options] <nodes>\n");
	printf("\nDescription:	Print information on message buffers.\n");
	printf("\nOptions:	-g	Include system messages.\n");
	printf("\t\t-h	Print this help message.\n");
	printf("\t\t-l	Also print buffer levels and maximums.\n");
	printf("\t\t-p	Print each packet.\n");
	printf("\t\t-B <#>	Set maximum # of messages to print.\n");
	mnusage();
	printf("\t\th (local), o (origin), N (all)\n");
	printf("\nExample:	bfstate h n3-9,11\n");
	printf("\t\t\t\"Get buffer status information on the local\n");
	printf("\t\t\t node and nodes 3 thru 9 and 11.\"\n");
}
