/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	NJN
 *
 *	$Id: shm.sgi.c,v 6.1 96/11/22 13:35:14 nevin Rel $
 *
 *	Function:	- SGI shared memory low-level routines
 */

#include <errno.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/uio.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>
#include <terror.h>
#include <typical.h>
#include <t_types.h>

/*
 * public functions
 */
int			_shm_serverinit();
int			_shm_clientinit();
int			_shm_cleanup();

/*
 * external functions
 */
extern char		*killname();

/*
 * private functions
 */
static char		*getarenaname();

/*
 * private variables
 */
static int		arena_count = 0;

/*
 *	_shm_clientinit
 *
 *	Function:	- client side shared memory initialization
 *	Accepts:	- process
 *			- LAM msg to fill with info for client
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_clientinit(ps, msg)

struct c2c_proc		*ps;
struct nmsg		*msg;

{
	ulock_t		*p;
	char		*arena_name;		/* name of shared arena file */
/*
 * Initialize with info from server.
 */
	ps->cp_shmbuf = *((char **) msg->nh_msg);
	ps->cp_shmin = ps->cp_shmbuf;
	ps->cp_shmout = ps->cp_shmbuf + SHMBUFSIZE;

	p = (ulock_t *) (msg->nh_msg + sizeof(char *));
	ps->cp_iwtlock = *p;
	ps->cp_irdlock = *(p+1);
	ps->cp_owtlock = *(p+2);
	ps->cp_ordlock = *(p+3);
/*
 * Join the arena.
 */
	arena_name = getarenaname();
	if (arena_name == 0) return(LAMERROR);

	if (usconfig(CONF_INITSIZE, ARENASIZE) == (ptrdiff_t) -1) {
		return(LAMERROR);
	}
	if (usconfig(CONF_INITUSERS, SHMMAXP) == (ptrdiff_t) -1) {
		return(LAMERROR);
	}
	
	ps->cp_arena = usinit(arena_name);
	if (ps->cp_arena == 0) return(LAMERROR);

	++arena_count;
	
	free(arena_name);
	return(0);
}

/*
 *	_shm_serverinit
 *
 *	Function:	- server side shared memory initialization
 *	Accepts:	- process
 *			- LAM msg to fill with info for client
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_serverinit(ps, msg)

struct c2c_proc		*ps;
struct nmsg		*msg;

{
	ulock_t		*p;
	char		*arena_name;		/* name of shared arena file */

	if (usconfig(CONF_INITSIZE, ARENASIZE) == (ptrdiff_t) -1) {
		return(LAMERROR);
	}
	if (usconfig(CONF_INITUSERS, SHMMAXP) == (ptrdiff_t) -1) {
		return(LAMERROR);
	}

	arena_name = getarenaname();
	if (arena_name == 0) return(LAMERROR);
    
	ps->cp_arena = usinit(arena_name);
	if (ps->cp_arena == 0) return(LAMERROR);

	++arena_count;
/*
 * Create the shared memory area.
 */
	ps->cp_shmbuf = (char *) usmalloc(SHMBLKSIZE, ps->cp_arena);
	if (ps->cp_shmbuf == 0) return(LAMERROR);

	ps->cp_shmout = ps->cp_shmbuf;
	ps->cp_shmin = ps->cp_shmbuf + SHMBUFSIZE;
/*
 * Initialize the locks.
 */
	ps->cp_owtlock = usnewlock(ps->cp_arena);
	if (ps->cp_owtlock == 0) return(LAMERROR);
	ps->cp_ordlock = usnewlock(ps->cp_arena);
	if (ps->cp_ordlock == 0) return(LAMERROR);
	ps->cp_iwtlock = usnewlock(ps->cp_arena);
	if (ps->cp_iwtlock == 0) return(LAMERROR);
	ps->cp_irdlock = usnewlock(ps->cp_arena);
	if (ps->cp_irdlock == 0) return(LAMERROR);

	if (ussetlock(ps->cp_ordlock) == -1) return(LAMERROR);
	if (ussetlock(ps->cp_irdlock) == -1) return(LAMERROR);
/*
 * Set information to pass to client.
 */
	*((char **) msg->nh_msg) = ps->cp_shmbuf;
	p = (ulock_t *) (msg->nh_msg + sizeof(char *));
	*p = ps->cp_owtlock;
	*(p+1) = ps->cp_ordlock;
	*(p+2) = ps->cp_iwtlock;
	*(p+3) = ps->cp_irdlock;

	free(arena_name);
	return(0);
}

/*
 *	_shm_cleanup
 *
 *	Function:	- clean up a process's shared memory structures
 *	Accepts:	- process
 *	Returns:	- 0 or LAMERROR
 */
int
_shm_cleanup(ps)

struct c2c_proc		*ps;

{
	char		*arena_name;		/* name of shared arena file */

	if (ps->cp_arena) {
		if (--arena_count <= 0) {
			usdetach(ps->cp_arena);
			arena_name = getarenaname();
/*
 * Since spawned/client processes do not use shared memory we can safely
 * unlink the arena file.  If we ever change to let such processes use
 * shared memory we will have to be careful not to unlink the arena file
 * until all are finished with it.  This will probably involve a barrier
 * somewhere in MPI_Finalize().  
 */
			if (arena_name) {
				unlink(arena_name);
				free(arena_name);
			}
		}
	}

	ps->cp_arena = 0;
	return(0);
}

/*
 *	getarenaname
 *
 *	Function:	- forms shared arena filename
 *			- allocates space
 *	Returns:	- arena filename
 */
char *
getarenaname()

{
	char		*kill;			/* kill filename */
	char		*arena;			/* arena filename */
    
	kill = killname();
	if (kill == 0) return(0);

	arena = (char *) malloc(strlen(kill) + 32);
	if (arena == 0) {
		free(kill);
		return(0);
	}
/*
 * suffix has form "pid part of jobid"."node part of jobid"
 */
	sprintf(arena, "%s-a%d.%d", kill, lam_jobid >> 16, lam_jobid & 0xFFFF);

	free(kill);
	return(arena);
}
