/*      $Id: Exception.c,v 1.6 1997/09/26 07:48:01 acken Exp $    */
/*  Provides facilities to raise and handle exceptions.
    Copyright (C) 1997  Michael van Acken

    This file is part of OOC.

    OOC is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.  

    OOC is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details. 

    You should have received a copy of the GNU General Public License
    along with OOC. If not, write to the Free Software Foundation, 59
    Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include <stdlib.h>
#include <stddef.h>
#include <setjmp.h>

#include "__oo2c.h"
#include "__mini_gc.h"
#include "__config.h"

/* --- begin #include "Exception.d" */
#include "Exception.h"

/* local definitions */
Exception_Source Exception_halt;
Exception_Source Exception_assert;
Exception_Source Exception_signal;
Exception_Source Exception_runtime;

/* function prototypes */

/* module and type descriptors */
static const struct {
  int length;
  void* pad;
  const char name[10];
} _n0 = {10, NULL, {"Exception"}};
static struct _MD Exception__md = {
  NULL, 
  &Kernel_ModuleDesc__td.td, 
  {
    NULL, 
    (const unsigned char*)_n0.name, 
    -1, 
    NULL
  }
};

static const struct {
  int length;
  void* pad;
  const char name[20];
} _n1 = {20, NULL, {"SourceDesc"}};
static const struct {
  int length;
  void* pad;
  _Type btypes[1];
} Exception_SourceDesc__tdb = {
  1, 
  NULL, 
  {
    &Exception_SourceDesc__td.td
  }
};
static const struct {
  int length;
  void* pad;
  const void* tbprocs[1];
} _tb0 = {0, NULL, {
  NULL
}};
struct _TD Exception_SourceDesc__td = {
  NULL,
  &Types_TypeDesc__td.td,
  {
    Exception_SourceDesc__tdb.btypes,
    _tb0.tbprocs,
    (const unsigned char*)_n1.name,
    &Exception__md.md,
    0, 
    '0', '1',
    sizeof(Exception_SourceDesc),
    NULL
  }
};

/* local strings */

/* --- end #include "Exception.d" */


/* boolean variable to distinguish between normal and exception execution 
   state; the program starts in the state of normal execution, of course */
static int is_exception = 0;

/* current top of context stack; NULL means the stack is empty, the default
   exception handler should be used */
static _ExecutionContext top_context = NULL;

/* this variable is used to pass the exception source to the reactivation
   of PUSHCONTEXT: */
void* _exception_source = NULL;

/* exception source associated with exceptions raised by this module */
static Exception_Source eexcept;
#define NORMAL_EXECUTION 1
#define EXCEPTIONAL_EXECUTION 2
#define STACK_EMPTY 3
#define INVALID_RAISE 4

/* the following variables describe together with _exception_source the raised
   exception if the program is in the exceptional execution state: */
#define MESSAGE_LENGTH 128
static _ExecutionContext _exception_context = NULL;
static LONGINT _exception_number = 0;
static CHAR _exception_message[MESSAGE_LENGTH];


#define STR_PARAM(text) (CHAR*)text,sizeof(text)


static _ModId _mid;

void _push_exception_context(_ExecutionContext c) {
  c->next = top_context;
  top_context = c;
}

void Exception_POPCONTEXT(void) {
  if(top_context) {     /* stack isn't empty */
    if(is_exception) {
      /* pass exception along to next higher exception handler; since we are
         in an exception, RAISE will reactivate the second stack element */
      Exception_RAISE(_exception_source, _exception_number, 
                      _exception_message,
                      strlen((char*)_exception_message));
    } else {                    /* just pop top of stack */
      top_context = top_context->next;
    }
  } else {
    Exception_RAISE(eexcept, STACK_EMPTY,
                    STR_PARAM("[Exception] POPCONTEXT called while stack is empty"));
  }
}

void Exception_RETRY(void) {
  _exception_pos = 0;           /* clear file position */
  if(is_exception) {
    longjmp(*(jmp_buf*)_exception_context->jmpbuf, -1);
  } else {
    Exception_RAISE(eexcept, NORMAL_EXECUTION, 
                    STR_PARAM("[Exception] RETRY called in normal execution state"));
  }
}

void Exception_ACKNOWLEDGE(void) {
  _exception_pos = 0;           /* clear file position */
  if(is_exception) {
    is_exception = 0;
  } else {
    Exception_RAISE(eexcept, NORMAL_EXECUTION, 
                    STR_PARAM("[Exception] ACKNOWLEDGE called in normal execution state"));
  }
}

void Exception_AllocateSource(Exception_Source *newSource) {
  NEW_REC(*newSource, Exception_SourceDesc);
}

void Exception_RAISE(Exception_Source source, LONGINT number, 
                     const CHAR* message__ref, LONGINT message_0d) {
  if (!source) {
    Exception_RAISE(eexcept, INVALID_RAISE, 
                    STR_PARAM("[Exception] First parameter of RAISE is NIL"));
  }
    
  if(is_exception) {
    /* if an exception was raised while handling an exception, pop the topmost
       context that caused the second exception and reactivate the context 
       below it to deal with the new exception */
    top_context = top_context->next;
  } else {
    is_exception = 1;
  }
  /* store information describing the current exception in global variables */
  _exception_source = source;
  _exception_number = number;
  _exception_context = top_context;
  _string_copy(_exception_message, message__ref, MESSAGE_LENGTH);
  if (_exception_context) {
    /* reactivate topmost context */
    longjmp(*(jmp_buf*)_exception_context->jmpbuf, 1);
  } else {
    /* oops, stack is empty; use default exception handler from __oo2c.c --
       it'll write a nice message and abort */
    _default_exception_handler(source, number, message__ref);
  }
}

LONGINT Exception_CurrentNumber(Exception_Source source) {
  if(is_exception) {
    return _exception_number;
  } else {
    Exception_RAISE(eexcept, NORMAL_EXECUTION, 
                    STR_PARAM("[Exception] CurrentNumber called in normal execution state"));
    return 0;  /* never reached */
  }
}

void Exception_GetMessage(CHAR* text, LONGINT text_0d) {
  if(is_exception) {
    _string_copy(text, _exception_message, text_0d);
  } else {
    Exception_RAISE(eexcept, NORMAL_EXECUTION, 
                    STR_PARAM("[Exception] GetMessage called in normal execution state"));
  }
}

BOOLEAN Exception_IsExceptionalExecution(void) {
  return (BOOLEAN)(is_exception != 0);
}

void Exception__init(void) {
  _mid = _register_module(&Exception__md.md, &Exception_SourceDesc__td.td);
  Exception_AllocateSource(&eexcept);
  Exception_AllocateSource(&Exception_halt);
  Exception_AllocateSource(&Exception_assert);
  Exception_AllocateSource(&Exception_runtime);
  Exception_AllocateSource(&Exception_signal);
  _exception_halt = Exception_halt;
  _exception_assert = Exception_assert;
  _exception_raise = (void (*) (void*, LONGINT, const CHAR*, LONGINT)) Exception_RAISE;
}
