/* 
 *  orbit-naming-server : a CORBA CosNaming server
 *
 *  Copyright (C) 1998 Sebastian Wilhelmi; University of Karlsruhe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "name.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define THROW(name,arg,ev) CORBA_exception_set( ev, CORBA_USER_EXCEPTION, \
						ex_##name, arg ), \
  fprintf( stderr, "%s thrown.\n", ex_##name );

#define RETHROW(ev, val ) if ((ev)->_major != CORBA_NO_EXCEPTION ) return val;

typedef struct _NamingContext_Servant NamingContext_Servant;
typedef struct _NameList NameList;

typedef enum
  {
    result_ok, result_lower_level, result_exception,
    result_not_found
  }
ActionReturnType;

struct _NamingContext_Servant
  {
    POA_CosNaming_NamingContext base;
    GPtrArray *list;
    PortableServer_POA root_poa;
    CORBA_boolean is_activator;
  };

struct _NameList
  {
    CosNaming_NameComponent name;
    CosNaming_BindingType type;
    CORBA_Object object;
  };

static void NamingContext_bind (NamingContext_Servant *, 
				CosNaming_Name *,
				CORBA_Object, 
				CORBA_Environment *);
static void NamingContext_rebind (NamingContext_Servant *, 
				  CosNaming_Name *,
				  CORBA_Object, 
				  CORBA_Environment *);
static void NamingContext_bind_context (NamingContext_Servant *,
					CosNaming_Name *,
					CosNaming_NamingContext,
					CORBA_Environment *);
static void NamingContext_rebind_context (NamingContext_Servant *,
					  CosNaming_Name *,
					  CosNaming_NamingContext,
					  CORBA_Environment *);
static CORBA_Object NamingContext_resolve (NamingContext_Servant *,
					   CosNaming_Name *,
					   CORBA_Environment *);
static void NamingContext_unbind (NamingContext_Servant *,
				  CosNaming_Name *,
				  CORBA_Environment *);
static CORBA_Object NamingContext_new_context (NamingContext_Servant *,
					       CORBA_Environment *);
static CORBA_Object NamingContext_bind_new_context (NamingContext_Servant *,
						    CosNaming_Name *,
						    CORBA_Environment *);
static void NamingContext_destroy (NamingContext_Servant *,
				   CORBA_Environment *);
static void NamingContext_list(NamingContext_Servant *, 
			       CORBA_unsigned_long, 
			       CosNaming_BindingList **, 
			       CosNaming_BindingIterator *, 
			       CORBA_Environment *);

static PortableServer_ServantBase__epv base_epv =
{
  NULL,				/* _private */
  NULL,				/* finalize */
  NULL,				/* use base default_POA function */
};

static POA_CosNaming_NamingContext__epv CosNaming_NamingContext_epv =
{
  NULL,				/* private data */
  (gpointer) & NamingContext_bind,
  (gpointer) & NamingContext_rebind,
  (gpointer) & NamingContext_bind_context,
  (gpointer) & NamingContext_rebind_context,
  (gpointer) & NamingContext_resolve,
  (gpointer) & NamingContext_unbind,
  (gpointer) & NamingContext_new_context,
  (gpointer) & NamingContext_bind_new_context,
  (gpointer) & NamingContext_destroy,
  (gpointer) & NamingContext_list
};

static POA_CosNaming_NamingContext__vepv vepv =
{
  &base_epv,
  &CosNaming_NamingContext_epv
};

static
ActionReturnType
lookup_name (GPtrArray * list, CosNaming_Name * n, int *pos,
	     CORBA_Environment * ev)
{
  if (n->_length == 0)		/* no name given. */
    {
      THROW (CosNaming_NamingContext_InvalidName, NULL, ev);
      return result_exception;
    }

  for (*pos = 0; *pos < list->len; (*pos)++)
    {
      NameList *current = g_ptr_array_index (list, *pos);
      if (strcmp (current->name.id, n->_buffer[0].id) == 0 &&
	  strcmp (current->name.kind, n->_buffer[0].kind) == 0)
	/* FIXME: really test for both id and kind to be equal????? */
	{
	  if (n->_length > 1)	/* in a lower-level context */
	    return result_lower_level;	/* in current context */
	  return result_ok;
	}
    }
  return result_not_found;
}

static
void
insert_name (GPtrArray * list, CosNaming_NameComponent * name,
	     CosNaming_BindingType type,
	     CORBA_Object obj,
	     CORBA_Environment * ev)
{
  NameList *insert = g_new (NameList, 1);
  insert->name.id = g_strdup (name->id);
  insert->name.kind = g_strdup (name->kind);
  insert->type = type;
  insert->object = CORBA_Object_duplicate (obj, ev);
  g_ptr_array_add (list, insert);
}

static
void
remove_name (GPtrArray * list, int pos, CORBA_Environment * ev)
{
  NameList *found = g_ptr_array_index (list, pos);
  g_free (found->name.id);
  g_free (found->name.kind);
  CORBA_Object_release (found->object, ev);
  g_ptr_array_remove_index (list, pos);
  g_free (found);
}

void
NameComponent__copy( CosNaming_NameComponent *dest, 
		     CosNaming_NameComponent *src)
{
  dest->id = CORBA_string_alloc( strlen( src->id ) );
  dest->kind = CORBA_string_alloc( strlen( src->kind ) );
  strcpy( dest->id, src->id );
  strcpy( dest->kind, src->kind );
}

static CORBA_Object
prepare_resending (GPtrArray * list, int pos, CosNaming_Name * old,
		   CosNaming_Name ** new)
{
  NameList *found = g_ptr_array_index (list, pos);
  int i;

  *new = CosNaming_Name__alloc ();
  (*new)->_length = old->_length - 1;
  (*new)->_buffer =
    CORBA_sequence_CosNaming_NameComponent_allocbuf ((*new)->_length);
  for (i = 0; i < (*new)->_length; i++)
    {
      NameComponent__copy( (*new)->_buffer + i, old->_buffer + i + 1 );
    }
  /* FIXME: narrowing to CosNaming_NamingContext */
  return found->object;
}

static void
do_re_bind_no_context (GPtrArray * list,
		       CosNaming_Name * n,
		       CORBA_Object obj,
		       gboolean is_rebind,
		       gboolean is_context, 
		       CORBA_Environment * ev)
{
  int pos;
  ActionReturnType res = lookup_name (list, n, &pos, ev);

  if (res == result_exception)
    return;

  if (res == result_lower_level)
    {
      CosNaming_Name *new;
      CORBA_Object context = prepare_resending (list, pos, n, &new);
      if (is_rebind)
	{
	  if (is_context)
	    CosNaming_NamingContext_rebind_context (context, new, obj, ev);
	  else
	    CosNaming_NamingContext_rebind (context, new, obj, ev);
	}
      else
	{
	  if (is_context)
	    CosNaming_NamingContext_bind_context (context, new, obj, ev);
	  else
	    CosNaming_NamingContext_bind (context, new, obj, ev);
	}
      CORBA_free (new);
      return;
    }

  if (res == result_not_found && is_rebind)
    {
      THROW (CosNaming_NamingContext_NotFound, NULL, ev);
      return;
    }

  if (res == result_ok && !is_rebind)
    {
      THROW (CosNaming_NamingContext_AlreadyBound, NULL, ev);
      return;
    }

  if (is_rebind)
    remove_name (list, pos, ev);

  insert_name (list, n->_buffer,
	       is_context ? CosNaming_ncontext : CosNaming_nobject, obj, ev);
}

static void
NamingContext_bind (NamingContext_Servant * servant,
		    CosNaming_Name * n,
		    CORBA_Object obj,
		    CORBA_Environment * ev)
{
  do_re_bind_no_context (servant->list, n, obj, FALSE, FALSE, ev);
}

static void
NamingContext_rebind (NamingContext_Servant * servant,
		      CosNaming_Name * n,
		      CORBA_Object obj,
		      CORBA_Environment * ev)
{
  do_re_bind_no_context (servant->list, n, obj, TRUE, FALSE, ev);
}

static void
NamingContext_bind_context (NamingContext_Servant * servant,
			    CosNaming_Name * n,
			    CosNaming_NamingContext nc,
			    CORBA_Environment * ev)
{
  do_re_bind_no_context (servant->list, n, nc, FALSE, TRUE, ev);
}

static void
NamingContext_rebind_context (NamingContext_Servant * servant,
			      CosNaming_Name * n,
			      CosNaming_NamingContext nc,
			      CORBA_Environment * ev)
{
  do_re_bind_no_context (servant->list, n, nc, TRUE, TRUE, ev );
}

static CORBA_Object
NamingContext_resolve (NamingContext_Servant * servant,
		       CosNaming_Name * n,
		       CORBA_Environment * ev)
{
  int pos;
  NameList *found;
  ActionReturnType res = lookup_name (servant->list, n, &pos, ev);

  if (res == result_exception)
    return CORBA_OBJECT_NIL;

  if (res == result_lower_level)
    {
      CosNaming_Name *new;
      CORBA_Object context = prepare_resending (servant->list, pos, n, &new);
      CORBA_Object result = CosNaming_NamingContext_resolve (context, new, ev);
      CORBA_free (new);
      return result;
    }

  if (res == result_not_found)
    {
      THROW (CosNaming_NamingContext_NotFound, NULL, ev);
      return CORBA_OBJECT_NIL;
    }

  found = g_ptr_array_index (servant->list, pos);
  return CORBA_Object_duplicate(found->object, ev);
}

static void
NamingContext_unbind (NamingContext_Servant * servant,
		      CosNaming_Name * n,
		      CORBA_Environment * ev)
{
  int pos;
  ActionReturnType res = lookup_name (servant->list, n, &pos, ev);

  if (res == result_exception)
    return;

  if (res == result_lower_level)
    {
      CosNaming_Name *new;
      CORBA_Object context = prepare_resending (servant->list, pos, n, &new);
      CosNaming_NamingContext_unbind (context, new, ev);
      CORBA_free (new);
      return;
    }

  if (res == result_not_found)
    {
      THROW (CosNaming_NamingContext_NotFound, NULL, ev);
      return;
    }

  remove_name (servant->list, pos, ev);
}

static CosNaming_NamingContext
get_new_context( PortableServer_POA root_poa, CORBA_Environment * ev )
{
  CosNaming_NamingContext context;
  NamingContext_Servant* servant = g_new0( NamingContext_Servant, 1 );
  PortableServer_ObjectId *oid;

  servant->base._private = NULL;
  servant->base.vepv = &vepv;
  servant->list = g_ptr_array_new ();
  servant->root_poa = root_poa;

  POA_CosNaming_NamingContext__init ((PortableServer_Servant)servant, ev);
  RETHROW( ev, NULL );

  oid = PortableServer_POA_activate_object(root_poa, servant, ev);
  RETHROW( ev, NULL );

  g_print("oid = %s\n", oid->_buffer);

  CORBA_free(oid);

  context = PortableServer_POA_servant_to_reference (root_poa, 
						     (PortableServer_Servant)servant, ev);
  RETHROW( ev, NULL );

  return context;
}


static CORBA_Object
NamingContext_new_context (NamingContext_Servant * servant,
			   CORBA_Environment * ev)
{
  g_message("NamingContext_new_context called");
  return get_new_context( servant->root_poa , ev );
}

static CORBA_Object
NamingContext_bind_new_context (NamingContext_Servant * servant,
				CosNaming_Name * n,
				CORBA_Environment * ev)
{
  CosNaming_NamingContext context =  get_new_context( servant->root_poa , ev );
  RETHROW( ev, NULL );
  do_re_bind_no_context (servant->list, n, context, FALSE, TRUE, ev);
  return( context );
}

static void
NamingContext_destroy (NamingContext_Servant * servant,
		       CORBA_Environment * ev)
{
  if( servant->list->len > 0 )
    {
      THROW(CosNaming_NamingContext_NotEmpty, NULL, ev);
      return;
    }
  g_ptr_array_free ( servant->list, TRUE );
  // FIXME: g_free( servant ) ??
  // FIXME: CORBA_Object_release myself ??
  return;
}

static void
NamingContext_list(NamingContext_Servant * servant, 
		   CORBA_unsigned_long how_many, 
		   CosNaming_BindingList ** bl, 
		   CosNaming_BindingIterator * bi, 
		   CORBA_Environment * ev)
{
  int i;
  *bl = CosNaming_BindingList__alloc();
  (*bl)->_length = servant->list->len;
  (*bl)->_buffer = CORBA_sequence_CosNaming_Binding_allocbuf( (*bl)->_length );
  for (i = 0; i < servant->list->len; i++)
    {
      NameList *current = g_ptr_array_index (servant->list, i);
      (*bl)->_buffer[ i ].binding_name._length = 1;
      (*bl)->_buffer[ i ].binding_name._buffer = 
	CORBA_sequence_CosNaming_NameComponent_allocbuf( 1 );
      NameComponent__copy( (*bl)->_buffer[ i ].binding_name._buffer, 
			   &current->name );
      (*bl)->_buffer[ i ].binding_type = current->type;
    }
  *bi = CORBA_OBJECT_NIL;
  return;
}

int
main (int argc, char *argv[])
{
  CORBA_ORB orb;
  CORBA_Environment ev;
  CosNaming_NamingContext context;
  CORBA_char *objref;
  FILE *outfile;
  PortableServer_POA root_poa;
  PortableServer_POAManager pm;

  CORBA_exception_init (&ev);

  orb = CORBA_ORB_init (&argc, argv, "orbit-local-orb", &ev);

  root_poa = (PortableServer_POA) 
	  CORBA_ORB_resolve_initial_references (orb, "RootPOA", &ev); 

  context = get_new_context( root_poa, &ev );

  objref = CORBA_ORB_object_to_string (orb, context, &ev);

  outfile = fopen ("/tmp/namingserviceid", "w");
  fprintf (outfile, "%s", objref);
  fclose (outfile);

  CORBA_free (objref);

  CORBA_Object_release (context, &ev);

  pm = PortableServer_POA__get_the_POAManager(root_poa, &ev);
  PortableServer_POAManager_activate(pm, &ev);

  CORBA_ORB_run (orb, &ev);

  return 0;
}
