"""Routines to collect and store data that is received from the server."""

#    Copyright (C) 1998 Kevin O'Connor
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import sys
import string
import re
import cPickle
import operator
import time

import empQueue

# Routines to parse text messages from server,
#  and convert to meaningfull data - YUCK

###########################################################################
#############################  dictDB class   #############################
class dictDB:
    """Store information for server databases.

    This class mimics a standard dictionary type in many ways.  (It defines
    the getitem, get, keys, values, items methods.)  However, it is
    specifically tailored for the server databases.  (IE. data from a
    dump/pdump/sdump/etc. - the sector and unit databases.)  This tailoring
    is done in two ways.

    First, there is a seamless tracking of secondary indexes.  Any series
    of keys may be used to designate any number of secondary index at the
    creation time of this class.  When an update is made, not only is the
    item inserted into the primary database, but a reference is updated in
    the secondary databases.  (EG. The plane/ship/land units use a
    secondary database that is indexed by x and y.)

    Second, there is an automatic detection of updated items.  Any time an
    update is made to the database, an entry is filed in a special 'update'
    database.  This is used by the Tk graphics routines to ensure that only
    updated information is redrawn.  Since Tk redraws tend to be very time
    consuming, partial redraws become extremely important.
    """
    def __getstate__(self):
	"""Called by the pickle module - determines what will be saved."""
	return (self.primary_keytype, self.primary, self.timestamp,
		self.secondary.keys())
    def __setstate__(self, v):
	"""Initialize the class - called by the pickle module."""
	(self.primary_keytype, self.primary, self.timestamp, seckeys) = v

	# List of items that have recently changed
	self.uDB = {}

	# rebuild secondary indexes
	self.secondary = {}
	for sec_type in seckeys:
	    self.secondary[sec_type] = t = {}
	    for pri_key, pri_val in self.primary.items():
		try:
		    sec_key = tuple(map(operator.getitem,
					[pri_val]*len(sec_type), sec_type))
		    try: t[sec_key][pri_key] = pri_val
		    except KeyError: t[sec_key] = {pri_key:pri_val}
		except KeyError:
		    pass

	# Initialize some handlers
	self.get = self.primary.get
	self.items = self.primary.items
	self.values = self.primary.values
	self.keys = self.primary.keys
    def __init__(self, key, *seckeys):
	"""Initialization the class - called at creation time only."""
	self.__setstate__((key, {}, 0, seckeys))
    def __getitem__(self, k):
	return self.primary[k]
    def updates(self, list, returnRemaining=0):
	"""Update the database with the items stored in LIST.

	Given a list of dictionary types, extract the primary key from each
	dict, and add the item to the primary, secondary, and update
	databases.
	"""
	# Python optimization - copy frequently used variables into
	# local namespace.
	(self__primary, self__primary_keytype, self__secondary__items,
	 self__uDB, __tuple, __map, __len, operator__getitem) = (
	     self.primary, self.primary_keytype, self.secondary.items(),
	     self.uDB, tuple, map, len, operator.getitem)

	if returnRemaining:
	    # This flag instructs the routine to return all items in the
	    # database that were _not_ updated during this call.
	    remainingList = self__primary.copy()
	for dict in list:
	    # find the key
	    pri_key = __tuple(__map(operator__getitem,
				    [dict]*__len(self__primary_keytype),
				    self__primary_keytype))
	    # find dictionary corresponding to key, and place
	    # in local variable d
	    try:
		d = self__primary[pri_key]
	    except KeyError:
		# New key
		d = self__primary[pri_key] = {}
	    else:
		# Key already present

		# Remove this item from the returnRemaining list
		if returnRemaining:
		    del remainingList[pri_key]

		# Dont update entries if no changes are made.  Although
		# this may be an expensive operation, the most expensive
		# operations (by far) are screen redraws - any code that
		# reduces redraws will improve performance.
		for key, value in dict.items():
		    try:
			if d[key] != value:
			    break
		    except KeyError:
			break
		else:
		    # The 'for loop' completed without an exception,
		    # and without any differences encountered - no
		    # changes present.
		    continue

		# Remove key from the secondary indexes
		for sec_type, sec_db in self__secondary__items:
		    try:
			sec_key = __tuple(__map(operator__getitem,
						[d]*__len(sec_type), sec_type))
			del sec_db[sec_key][pri_key]
			if not sec_db[sec_key]:
			    del sec_db[sec_key]
		    except KeyError:
			pass

	    # Add item to the primary database
	    self__primary[pri_key].update(dict)
	    # Add item to the updateDB
	    self__uDB[pri_key] = d
	    # Add key to the secondary indexes
	    for sec_type, sec_db in self__secondary__items:
		try:
		    sec_key = __tuple(__map(operator__getitem,
					    [d]*__len(sec_type), sec_type))
		    try: sec_db[sec_key][pri_key] = d
		    except KeyError: sec_db[sec_key] = {pri_key:d}
		except KeyError:
		    pass
	if returnRemaining:
	    return remainingList
    def getSec(self, sec_type):
	"""Return the secondary index (a dictionary) for SEC_TYPE."""
	return self.secondary[sec_type]
    def __repr__(self):
	return repr(self.primary)
    def __str__(self):
	return string.join(map(str, self.primary.items()), "\n")

###########################################################################
#############################  Country class  #############################
CN_OWNED = -1
CN_ENEMY = -2
CN_UNOWNED = -3

class Countries:
    """Track empire countries by name or number.

    Note: There will generally be only one instance of this class -
    megaDB['countries'].

    This class is used to solve one basic problem of tracking empire
    countries - sometimes the server returns a country name, other times it
    returns a country number, and other times it will returns both.  To
    solve this, the local databases store owners by country number; code
    can call back to this class when the name is desired.  When a country
    is identified by the server solely as a name, this class will attempt
    to resolve it to a number, or if that is not possible it will track the
    name until a number can be resolved for it.
    """
    uDB = {}

    def __init__(self):
	self.unresolved = {}
	self.nameList = {}
##	self.idList = {}
	self.player = CN_OWNED

    def resolveName(self, name, dbname, key):
	"""Return a token for the country represented by name."""
	try:
	    return self.nameList[name]
	except KeyError:
	    # A country name with no currently available country Id.
	    try: list = self.unresolved[name]
	    except KeyError: list = self.unresolved[name] = []
	    list.append((dbname, key))
	    return name

    def resolveId(self, id):
	"""Return a token for the country represented by id."""
	# Numeric type
	if id == self.player:
	    return CN_OWNED
	return id

    def resolveNameId(self, name, id):
	"""Return a token for the country; resolve any outstanding names."""
	if id == self.player:
	    self.foundResolotion(name, CN_OWNED)
	    return CN_OWNED
	self.foundResolotion(name, id)
	return id

    def resolvePlayer(self, name, id):
	"""Note the current player's id and name."""
	self.player = id
	self.foundResolotion(name, CN_OWNED)
	return CN_OWNED

    def foundResolotion(self, name, id):
	"""Attempt to resolve a name/id pair."""
	if self.unresolved.has_key(name):
	    list = self.unresolved[name]
	    for dbname, key in list:
		db = megaDB[dbname][key]
		# HACK++
		if db['owner'] == name:
		    db['owner'] = id
	    del self.unresolved[name]
	self.nameList[name] = id
##	self.idList[id] = name

###########################################################################
#############################  Time class     #############################
s_time = (r"(?P<day>\S\S\S) (?P<month>\S\S\S) +(?P<date>\d+) (?P<hour>\d+)"
	  +r":(?P<minute>\d+):(?P<second>\d+)(?: (?P<year>\d\d\d\d))?")
class EmpTime:
    """Track the empire server time.

    Note: There will generally be only one instance of this class -
    megaDB['time'].

    This code is used to guess at what time it is at the server.  It is
    mainly used to try and approximate when the next update will trigger.
    Times are noted by the various parsers, and sent to this class (via
    megaDB['time']) to note the time.
    """
    uDB = {}
    Months = {"Jan":1, "Feb":2, "Mar":3, "Apr":4, "May":5, "Jun":6,
	      "Jul":7, "Aug":8, "Sep":9, "Oct":10, "Nov":11, "Dec":12}
    Days = {"Mon":0, "Tue":1, "Wed":2, "Thu":3, "Fri":4, "Sat":5, "Sun":6}
    def __init__(self):
## 	self.timeDiff = 0.0
	self.nextUpdate = 0.0

    observedDrift = [None]
    guessYear = [time.localtime(time.time())[0]]

    def translateTime(self, match, raw=0):
	"""Convert a regular expression processed time string to local time."""
	year = match.group('year')
	if year == None:
	    year = self.guessYear[0]
	else:
	    year = self.guessYear[0] = string.atoi(year)
	tt = tuple(map(string.atoi,
		       match.group('date', 'hour', 'minute', 'second')))
	tt = (year, self.Months[match.group('month')]) + tt + (
	    self.Days[match.group('day')], 0, 0)
	tt = time.mktime(tt)
	return tt
    def noteTime(self, match):
	"""Note the time from an empire server string."""
## 	tt = self.translateTime(match) + self.timeDiff
	tt = self.translateTime(match)
	ct = time.time()
	drift = ct - tt
	if self.observedDrift[0] == None:
	    self.observedDrift[0] = drift
	else:
	    variance = self.observedDrift[0] - drift
	    if variance < -300.0 or variance > 300.0:
		# Something odd is happenining - reset the drift
		viewer.flash("PTkEI - time oddity..")
		self.observedDrift[0] = drift
	    elif self.observedDrift[0] > drift:
		self.observedDrift[0] = drift
## 	    print "local:%s server:%s drift:%s offset:%s" % (
## 		ct, tt, self.observedDrift[0], self.timeDiff)

##     def noteTimestamp(self, match, ts):
## 	tt = self.translateTime(match)
## 	self.timeDiff = ts-tt
## 	self.noteTime(match)
    def noteNextUpdate(self, match):
	"""Make a note of the next update."""
## 	self.nextUpdate = self.translateTime(match) + self.timeDiff
	self.nextUpdate = self.translateTime(match)
    def getCountDown(self):
	"""Return a tuple containing the time until the update."""
	drift = self.observedDrift[0]
	if drift == None:
	    drift = 0.0
	count = self.nextUpdate - time.time() + drift
	count = divmod(count, (megaDB['version']['ETUTime']
			       * megaDB['version']['UpdateTime']))[1]
	r, seconds = divmod(count, 60)
	hours, minutes = divmod(r, 60)
	return (hours, minutes, seconds)

###########################################################################
#############################  Map Functions  #############################
def updateDesignations(lst, mapType):
    """Update map designations from a list of (x,y,designation) triples.

    This function is complex, because it must determine when a designation
    change should be honored, and when it should be ignored.  Depending on
    mapType, it may be necessary to distrust the information.
    """
##     print lst
    DB = megaDB['SECTOR']
    changes = []
    for col, row, t in lst:
	if t == '0' or string.lower(t) != t:
	    # Letters in map with uppercase letters are
	    # unit designators, and dont reveal any useful info.
	    continue
	ldict = DB.get((col, row), {})
	oldown = ldict.get('owner')
	olddes = ldict.get('des')
	ndict = {}
	if mapType == 'b':
	    # bmap
	    if t == ' ':
		continue
	    if t == '?':
		if not olddes or olddes == '-':
		    ndict['des'] = t
	    elif not olddes or oldown != CN_OWNED:
		ndict['des'] = t
	else:
	    if t == ' ':
		if oldown == CN_OWNED:
		    ndict['owner'] = CN_UNOWNED
		else:
		    continue
	    elif t == '?':
		if not oldown or oldown == CN_OWNED:
		    ndict['owner'] = CN_ENEMY
		if not olddes:
		    ndict['des'] = t
	    elif (t == '.' or t == '\\'):
		if oldown != 0:
		    ndict['owner'] = 0
		ndict['des'] = t
	    elif (t == '^'):
		ndict['des'] = t
	    elif mapType == 'n':
		# future designations
		if t != '-':
		    ndict['owner'] = CN_OWNED
		if olddes == t:
		    ndict['sdes'] = '_'
		else:
		    ndict['sdes'] = t
	    elif mapType == 'r':
		# radar maps
		ndict['des'] = t
	    else:
		# normal maps
		if t != '-':
		    ndict['owner'] = CN_OWNED
		ndict['des'] = t
	ndict['x'] = col
	ndict['y'] = row
	changes.append(ndict)
    DB.updates(changes)

def parseStarMap(sects, coord, mapType):
    """Handle a map that is returned from a radar, or prompt."""
    worldx, worldy = megaDB['version']['worldsize']
    l = len(sects)
    hl = l/2
    tl = l*2

    lst = []
    for i in range(l):
	y = str((coord[1]-hl+i + worldy/2) % worldy - worldy/2)
	for j in range(abs(hl-i), tl-abs(hl-i), 2):
	    x = str((coord[0]-l+j+1 + worldx/2) % worldx - worldx/2)
	    lst.append((x, y, sects[i][j]))
    updateDesignations(lst, mapType)

###########################################################################
###########################  Header functions   ###########################
def sectToCoords(str):
    """Convert a string of the form 'x,y' to its x and y values."""
    try:
	idx = string.index(str, ',')
	return {'x':str(string.atoi(str[:idx])),
		'y':str(string.atoi(str[idx+1:]))}
    except ValueError:
	return {}

def newdesToDes(str):
    """Convert a 'xy' string to designation x, and newdesignation y."""
    if len(str) == 2:
	return {'des':str[0], 'sdes':str[1]}
    return {'des':str[0], 'sdes':'_'}

# Translations from header names to dump names:
headerConvert = {
    'sect': sectToCoords, 'de': newdesToDes,
    'own': (lambda str:
	    {'owner':megaDB['countries'].resolveId(string.atoi(str))}),
#    'old\nown': 'oldowner',
    'sct\neff': 'eff', 'rd\neff': 'road',
    'rl\neff': 'rail', 'def\neff': 'defense',
    'civ': 'civ', 'mil': 'mil', 'shl': 'shell', 'gun': 'gun',
    'pet': 'pet', 'food': 'food', 'bars': 'bar',
#    'lnd': xxx, 'pln': xxx,
    }

# Translate full sector names to their abbreviations:
sectorNameConvert = {
    'sea': '.', 'defense plant': 'd', 'technical center': 't',
    'mountain': '^', 'shell industry': 'i', 'fortress': 'f',
    'sanctuary': 's', 'mine': 'm', 'research lab': 'r',
    'wasteland': '/', 'gold mine': 'g', 'nuclear plant': 'n',
    'wilderness': '-', 'harbor': 'h', 'library/school': 'l',
    'gold mine': 'g', 'nuclear plant': 'n',
    'capital': 'c', 'warehouse': 'w', 'enlistment center': 'e',
    'park': 'p', 'uranium mine': 'u', 'headquarters': '!',
    'airfield': '*',
    'highway': '+', 'agribusiness': 'a',
    'radar installation': ')', 'oil field': 'o', 'bank': 'b',
    'light manufacturing': 'j', 'trading post': 'v',
    'bridge head': '#', 'heavy manufacturing': 'k',
    'bridge span': '=', 'refinery': '%',
    }

def composeHeader(*args):
    """Convert a series of string headers into a list of named qualifiers."""
    num = len(args)
    lst = string.split(args[-1], ' ')
    pos = 0
    new = []

    for i in lst:
	p2 = pos + len(i)
	new.append(string.lstrip(string.join(
	    map(string.strip,
		map(operator.getslice, args, [pos] * num, [p2] * num)),
	    "\n")))
	pos = p2 + 1
    new = filter(None, new)
##     print new
    return new

def composePreamble(mtch, str):
    """Convert info of the form 'type val, type val, ...' to a list."""
    lst = {}
    while str:
	mt = mtch.match(str)
	lst[mt.group('comd')] = mt.group('val')
	str = mt.group('next')
    return lst

###########################################################################
#############################  Parse classes  #############################
curtimeFormat = re.compile("^"+s_time+"$")
class ParseDump(empQueue.baseDisp):
    """Parse output from dump command."""
    dumpcommand = re.compile(r"^\s*\S+\s+(\S+)(?:\s+(\S+)\s*)?$")
##     timestamp = re.compile(r"^(?:\?timestamp>(\d+))?$")
    def Begin(self, cmd):
	self.out.Begin(cmd)

	self.getheader = 0
	self.Hlist = None
	self.updateList = []
	self.sett = self.full = 0
	self.DB = None

	# Determine if the timestamp should be set for this dump
	mm = self.dumpcommand.match(cmd)
	if (mm and mm.group(1) == '*'):
	    if not mm.group(2):
		# This is a total dump
		self.full = 1
		self.sett = 1
	    elif mm.group(2)[:3] == '?ti':
		# This is a timestamp dump
		self.sett = 1
    dumpheader = re.compile(
	r"^DUMP (?P<dumpName>.*) (?P<timeStamp>\d+)$")
    def data(self, msg):
	self.out.data(msg)
	if self.Hlist == None:
	    # check time
	    mtch = curtimeFormat.match(msg)
	    if mtch:
##		   self.ctime = mtch
		megaDB['time'].noteTime(mtch)
		return
	    # parse header
	    mtch = self.dumpheader.match(msg)
	    if mtch:
		tp = mtch.group('dumpName')
		self.lost = (tp == 'LOST ITEMS')
		self.DB = megaDB[tp]
		ts = string.atoi(mtch.group('timeStamp'))
##		   megaDB['time'].noteTimestamp(self.ctime, float(ts))
		if self.sett:
		    # It is possible for the server to process a command
		    # during the same second it calculates the current
		    # timestamp.  This results in a timestamp that wont
		    # cover the last command.  Therefore, it is safest to
		    # assume the timestamp is always one second less than
		    # the value received from the server.
		    self.DB.timestamp = ts-1
		self.getheader = 1
	    elif self.getheader == 1:
		self.Hlist = string.split(msg)
		self.getheader = 0
	    return

	# parse each line
	Dlist = string.split(msg)
	l = len(Dlist)
	if l == len(self.Hlist):
	    # Normal line
	    DDict = {'owner':CN_OWNED}
	    map(operator.setitem, [DDict]*l, self.Hlist, Dlist)
##	    self.DB.update(DDict)
	    self.updateList.append(DDict)
	else:
	    # End of dump
	    self.Hlist = None

    def End(self, cmd):
	self.out.End(cmd)
	if self.DB == None:
	    # Something odd happened - no dump lines at all.
	    return
	# Update the database
	if not self.full:
	    # simple update
	    self.DB.updates(self.updateList)
	else:
	    # A full dump
	    others = self.DB.updates(self.updateList, 1)
	    list = []
	    for i in others.values():
		if i.get('owner') == CN_OWNED:
		    dict = i.copy()
		    dict.update({'owner':CN_UNOWNED})
		    list.append(dict)
	    self.DB.updates(list)
	# Merge lost database with normal databases.
	if self.lost:
	    for i in megaDB['updated']['LOST ITEMS'].values():
		subDB = (('SECTOR', 'x', 'y'), ('SHIPS', 'id'),
			 ('PLANES', 'id'), ('LAND UNITS', 'id'),
			 ('NUKES', 'x', 'y'))[string.atoi(i['type'])]
		key = []
		d = {}
		for j in subDB[1:]:
		    key.append(i[j])
		    d[j] = i[j]
		if megaDB[subDB[0]].get(tuple(key),
					{}).get('owner') == CN_OWNED:
		    d['owner'] = CN_UNOWNED
		    megaDB[subDB[0]].updates([d])
	    megaDB['updated']['LOST ITEMS'].clear()

	self.out.updateDB()

class ParseMap(empQueue.baseDisp):
    """Parse output from various map commands."""
    mapcommand = re.compile(r"^\s*(\S?)map")
    def Begin(self, cmd):
	self.out.Begin(cmd)
	self.Mpos = 0

	# Determine the type of map
	mm = self.mapcommand.match(cmd)
	if mm:
	    self.mapType = mm.group(1)
	else:
	    self.mapType = ''

    def data(self, msg):
	self.out.data(msg)
	# parse header
	row = string.strip(msg[:5])
	msg_b = msg[5:]
	if self.Mpos == 0 and row == "":
	    self.Mpos = 1
	    self.Mhead = msg_b
	elif self.Mpos == 1:
	    self.Mpos = 2
	    cols = []
	    cols = map(string.atoi, map(operator.add, self.Mhead, msg_b))
	    if len(cols) == 1:
		if cols[0] > 9:
		    # Cant accurately parse maps with 1 column
		    self.Mpos = -1
	    else:
		# Fix negative values
		if cols[-1] > 9 and cols[-1] < cols[-2]:
		    cols[-1] = - cols[-1]
		for i in range(len(cols)-1):
		    if cols[i] > 9 and cols[i] > cols[i+1]:
			cols[i] = - cols[i]
	    self.oddcol = (cols[0] & 1)
	    self.lst = []
	    self.Mhead = map(str, cols)
	elif self.Mpos == 2:
	    if row == "":
		# End of map
		self.Mpos = -1
		updateDesignations(self.lst, self.mapType)
		self.out.updateDB()
		return
	    # Parse map
	    oddstart = (string.atoi(row) & 1) ^ self.oddcol
	    for i in range(oddstart, len(self.Mhead), 2):
		self.lst.append((self.Mhead[i], row, msg_b[i]))

ss_sect = r"-?\d+"
s_sector = r"(?P<sectorX>"+ss_sect+"),(?P<sectorY>"+ss_sect+")"
s_sector2 = r"(?P<sector2X>"+ss_sect+"),(?P<sector2Y>"+ss_sect+")"
s_comm = r"(?P<comm>\S+)"
class ParseMove(empQueue.baseDisp):
    """Parse an explore prompt."""
    def __init__(self, disp):
	empQueue.baseDisp.__init__(self, disp)
	self.map = []

    ownSector = re.compile("^Sector "+s_sector+" is now yours\.$")
    def data(self, msg):
	mm = self.ownSector.match(msg)
	if mm:
	    megaDB['SECTOR'].updates([{'x':mm.group('sectorX'),
				       'y':mm.group('sectorY'),
				       'owner':CN_OWNED}])
	    self.out.updateDB()
	self.map.append(msg)
	self.out.data(msg)

    s_mob = r"(?P<mob>\d+\.\d+)"
    s_des = r"(?P<des>.)"
    promptFormat = re.compile("^<"+s_mob+": "+s_des+" "+s_sector+"> $")
    def flush(self, msg):
	self.out.flush(msg)
	mm = self.promptFormat.match(msg)
	if mm:
	    # Extract map from last three lines of data
	    sects = []
	    for i in self.map[-3:]:
		sects.append(i[3:8])
	    coord = map(string.atoi, mm.group('sectorX', 'sectorY'))
	    parseStarMap(sects, coord, '')
	    viewer.markSectors([coord], 'prompt')
	    self.out.updateDB()
    def End(self, cmd):
	self.out.End(cmd)
	viewer.markSectors([], 'prompt')

s_landIdent = r"(?P<landType>\S+)(?:.+)? #(?P<landId>\d+)"
s_shipIdent = r"(?P<shipType>\S+)(?:.+)? \(#(?P<shipId>\d+)\)"
s_counName = r"(?P<counName>.*)"
s_counId = r"\(#(?P<counId>\d+)\)"
s_counIdent = s_counName + " " + s_counId
class ParseUnits(empQueue.baseDisp):
    """Parse info from a variety of unit type commands."""
    def Begin(self, cmd):
	self.out.Begin(cmd)
	self.num = None
	self.Map = []
    s_eff = r"(?P<eff>\d+)%"
    s_dist = r"(?P<dist>\d+)"
    s_shipLandSector = r"(?:"+s_shipIdent+" at |"+s_landIdent+" at |)"
    start_radar = re.compile(r"^"+s_shipLandSector
			     +s_sector+" efficiency "+s_eff
			     +", max range "+s_dist+"$")
    s_sectorName = r"(?P<sectorName>.*)"
    s_sectorStats = r"(?P<sectorStats>(?: with (?:approx )?\d+ \S+)*)"
    s_sectorShip = ("(?:"+s_sectorName+" "+s_eff+" efficient"
		    +s_sectorStats+"|"+s_shipIdent+")")
    view_info = re.compile(r"^(?:Your|"+s_counIdent+") "+s_sectorShip
			   +" @ "+s_sector)
    view_stats = re.compile(
	r"^ with (?:approx )?(?P<val>\d+) (?P<comd>\S+)(?P<next>.*)$")
    unit_stop = re.compile(
	"^(?:"+s_shipIdent+"|"+s_landIdent+
	") (?:stopped at|is out of mobility & stays in) "
	+s_sector+"$")
    def data(self, msg):
	self.out.data(msg)
	if self.num != None:
	    # In a pre-established radar
	    self.Map.append(msg)
	    if len(self.Map) == self.num:
		parseStarMap(self.Map, self.coord, 'r')
		self.out.updateDB()
		del self.coord
		self.Map = []
		self.num = None
	    return
	# Check for start of radar
	mm = self.start_radar.match(msg)
	if mm:
	    self.coord = map(string.atoi, mm.group('sectorX', 'sectorY'))
	    self.num = string.atoi(mm.group('dist'))*2+1
	    self.Map = []
	    return
	# Check for view
	mm = self.view_info.match(msg)
	if mm:
	    own = mm.group('counId')
	    if own == None:
		own = CN_OWNED
	    else:
		own = megaDB['countries'].resolveNameId(
		    mm.group('counName'), string.atoi(own))
	    if mm.group('shipId'):
		# A ship
		megaDB['SHIPS'].updates([
		    {'id':mm.group('shipId'), 'type':mm.group('shipType'),
		     'x': mm.group('sectorX'), 'y': mm.group('sectorY'),
		     'owner':own}])
	    else:
		lst = composePreamble(self.view_stats, mm.group('sectorStats'))
		lst.update({'des': sectorNameConvert[mm.group('sectorName')],
			    'eff': mm.group('eff'), 'owner': own,
			    'x': mm.group('sectorX'),
			    'y': mm.group('sectorY')})
		megaDB['SECTOR'].updates([lst])
		self.Map = []
	    return
	# Check for unit stop
	mm = self.unit_stop.match(msg)
	if mm:
	    if mm.group('shipId'):
		# A ship
		megaDB['SHIPS'].updates([{'id':mm.group('shipId'),
					  'type':mm.group('shipType'),
					  'owner':CN_OWNED,
					  'x':mm.group('sectorX'),
					  'y':mm.group('sectorY')}])
	    else:
		# A land unit
		megaDB['LAND UNITS'].updates([{'id':mm.group('landId'),
					       'owner':CN_OWNED,
					       'x':mm.group('sectorX'),
					       'y':mm.group('sectorY')}])
	    self.Map = []
	    return
	# Add to generic map prompt queue
	self.Map.append(msg)
    ss_mob = r"-?\d+\.\d"
    s_minMob = r"(?P<minMob>"+ss_mob+")"
    s_maxMob = r"(?P<maxMob>"+ss_mob+")"
    nav_prompt = re.compile(r"^<"+s_minMob+":"+s_maxMob+": "+s_sector+"> $")
    def flush(self, msg):
	self.out.flush(msg)
	if len(self.Map) > 2:
	    mm = self.nav_prompt.match(msg)
	    if mm:
		coord = map(string.atoi, mm.group('sectorX', 'sectorY'))
		parseStarMap(self.Map[-3:], coord, 'r')
		viewer.markSectors([coord], 'prompt')
		self.Map = []
		self.out.updateDB()
    def End(self, cmd):
	self.out.End(cmd)
	viewer.markSectors([], 'prompt')
	self.out.updateDB()

class ParseSpy(empQueue.baseDisp):
    """Handle spy reports."""
    def __init__(self, disp):
	empQueue.baseDisp.__init__(self, disp)
	self.pos = 0
	self.changes = []

    s_unitStats = r"(?P<unitStats>\S+ \d+(?:, \S+ \d+)*)"
    unitInfo = re.compile("^(?:Spies report e|E)nemy \("+s_counName
			  +"\) unit in "+s_sector+":  "+s_landIdent
			  +"(?: \("+s_unitStats+"\))?$")
    unitStats = re.compile(
	r"(?P<comd>\S+) (?P<val>\d+)(?:, (?P<next>.*))?")
    def data(self, msg):
	self.out.data(msg)
	if self.pos == 0:
	    if msg == 'SPY report':
		self.pos = 1
	elif self.pos == 1:
	    # Note date
	    mm = curtimeFormat.match(msg)
	    if mm:
		megaDB['time'].noteTime(mm)
		self.pos = 2
	elif self.pos == 2:
	    # first header
	    self.hdr = msg
	    self.pos = 3
	elif self.pos == 3:
	    # second line of header
	    self.hdr = composeHeader(self.hdr, msg)
	    self.pos = 4
	elif self.pos == 4:
	    # Meat of message
	    mt = self.unitInfo.match(msg)
	    if mt:
		lst = composePreamble(self.unitStats, mt.group('unitStats'))
		id = mt.group('landId')
		lst['id'] = id
		lst['type'], lst['x'], lst['y'] = (
		    mt.group('landType', 'sectorX', 'sectorY'))
		lst['owner'] = megaDB['countries'].resolveName(
		    mt.group('counName'), 'LAND UNITS', (id,))
		megaDB['LAND UNITS'].updates([lst])
	    else:
		strs = string.split(msg)
		l = len(self.hdr)
		if len(strs) == l:
		    # Standard table format
		    lst = {}
		    map(operator.setitem, [lst]*l, self.hdr, strs)
		    new = {}
		    for i, j in lst.items():
			if callable(j):
			    j = j(j)
			try:
			    idx = headerConvert[i]
			except KeyError:
			    pass
			else:
			    if callable(idx):
				new.update(idx(j))
			    else:
				new[idx] = j
##		    print new
		    self.changes.append(new)
    def End(self, cmd):
	self.out.End(cmd)
	megaDB['SECTOR'].updates(self.changes)
	self.out.updateDB()

class ParseCoastWatch(empQueue.baseDisp):
    """Handle the coastwatch command."""
##   ptkei-0.21 (#  1) bb   battleship (#12) @ 0,-4
    s_counIdent = s_counName + r" \(#\s*(?P<counId>\d+)\)"
    line = re.compile("^\s*"+s_counIdent+" "+s_shipIdent+" @ "+s_sector)
    def data(self, msg):
	self.out.data(msg)
	mm = self.line.match(msg)
	if mm:
	    megaDB['SHIPS'].updates([
		{'id':mm.group('shipId'), 'type':mm.group('shipType'),
		 'x': mm.group('sectorX'), 'y': mm.group('sectorY'),
		 'owner':megaDB['countries'].resolveNameId(
		     mm.group('counName'), string.atoi(mm.group('counId')))}])
	    self.out.updateDB()

class ParseBuild(empQueue.baseDisp):
    """Parse build command."""
## Bridge span built over -4,0
    bridge = re.compile(r"^Bridge span built over "+s_sector+"$")
    def data(self, msg):
	self.out.data(msg)
	mm = self.bridge.match(msg)
	if mm:
	    megaDB['SECTOR'].updates([{'owner':0,
				       'x':mm.group('sectorX'),
				       'y':mm.group('sectorY'),
				       'des':'='}])
	    self.out.updateDB()

class ParseNation(empQueue.baseDisp):
    """Parse nation command."""
    nationLine = re.compile("^"+s_counId+" "+s_counName
			    +" Nation Report\t"+s_time)
    def data(self, msg):
	self.out.data(msg)
	mt = re.match(self.nationLine.pattern, msg)
	if mt:
	    megaDB['time'].noteTime(mt)
	    megaDB['countries'].resolvePlayer(
		mt.group('counName'), string.atoi(mt.group('counId')))
##	    megaDB['nation']['id'] = string.atoi(mt.group('counId'))
##	    megaDB['nation']['name'] = mt.group('counName')

class ParseReport(empQueue.baseDisp):
    """Parse output from report command."""
    def data(self, msg):
	mm = curtimeFormat.match(msg)
	if mm:
	    megaDB['time'].noteTime(mm)
	    return
	self.out.data(msg)
	line = string.split(msg)
	try:
	    id = string.atoi(line[0])
	except ValueError:
	    pass
	else:
	    megaDB['countries'].resolveNameId(line[1], id)

class ParseRelations(empQueue.baseDisp):
    """Parse output from relations."""
    s_counName = r"(?P<counName>.*?)"
    s_yourRelation = r"(?P<your>\S+)"
    s_theirRelation = r"(?P<their>\S+)"
    header = re.compile("^\s*"+s_counName
			+" Diplomatic Relations Report\t"+s_time+"$")
    line = re.compile("^\s*"+s_counId+"\) "+s_counName+"\s+"
		      +s_yourRelation+"\s+"+s_theirRelation+"$")
    def data(self, msg):
	self.out.data(msg)
	mm = self.line.match(msg)
	if mm:
	    megaDB['countries'].resolveNameId(
		string.atoi(mm.group('counId')), mm.group('counName'))

class ParseRealm(empQueue.baseDisp):
    """Parse output from realm command."""
    s_range = (r"(?P<minX>"+ss_sect+"):(?P<maxX>"+ss_sect
	       +"),(?P<minY>"+ss_sect+"):(?P<maxY>"+ss_sect+")")
    s_realm = r"#(?P<realm>\d+)"
    rm = re.compile(r"^Realm " + s_realm + " is " + s_range + "$")
    def data(self, msg):
	self.out.data(msg)
	mtch = self.rm.match(msg)
	if mtch:
	    vals = tuple(map(string.atoi, mtch.groups()))
	    realm = vals[0]
	    vals = vals[1:]
	    checkUpdated('realm', realm, vals)

class ParseRead(empQueue.baseDisp):
    """Parse and store telegrams and announcements."""
    def Begin(self, cmd):
	self.out.Begin(cmd)

	self.stor = None
	self.tlist = []

	c = string.lstrip(cmd)
	if c[:3] == 'wir':
	    dbname = 'announcements'
	else:
	    dbname = 'telegrams'
	self.archive = megaDB[dbname]
	self.uarchive = megaDB['updated'][dbname]

    def data(self, msg):
	self.out.data(msg)
	if msg[:1] == '>':
	    # A message header
	    self.stor = []
	    self.tlist[0:0] = [self.stor]
	    self.stor.append(msg)
	elif self.stor != None:
	    # message body
	    self.stor.append(msg)

    def End(self, cmd):
	self.out.End(cmd)
	pos = len(self.archive)
	if pos == 0:
	    last = None
	else:
	    last = self.archive[-1][0]
	for i in self.tlist:
	    if i[0] == last:
		# Encountered duplicate message.
		# (Note: This is a pretty good check, but it might
		# get fooled by countries that change names..)
		break
	    self.archive[pos:pos] = [i]
	    self.uarchive[:0] = [i]
	self.out.updateDB()

class ParseVersion(empQueue.baseDisp):
    """Parse the version info."""
    s_maxX = r"(?P<maxX>\d+)"
    s_maxY = r"(?P<maxY>\d+)"
## An Empire time unit is 75 seconds long.
## Use the 'update' command to find out the time of the next update.
## The current time is Sun Sep	6 18:15:49.
## An update consists of 48 empire time units.
    worldsize = re.compile("World size is "+s_maxX+" by "+s_maxY+"\.$")
    curtime = re.compile(r"^The current time is "+s_time+"\.$")
    etu = re.compile(r"^An Empire time unit is (?P<etu>\d+) seconds long\.$")
    updt = re.compile(
	r"^An update consists of (?P<updt>\d+) empire time units\.$")
    def data(self, msg):
	self.out.data(msg)
	mm = self.worldsize.match(msg)
	if mm:
	    wsize = tuple(map(string.atoi, mm.groups()))
	    checkUpdated('version', 'worldsize', wsize)
	    return
	mm = self.curtime.match(msg)
	if mm:
	    megaDB['time'].noteTime(mm)
	    return
	mm = self.etu.match(msg)
	if mm:
	    etu = string.atoi(mm.group('etu'))
	    checkUpdated('version', 'ETUTime', etu)
	    return
	mm = self.updt.match(msg)
	if mm:
	    updt = string.atoi(mm.group('updt'))
	    checkUpdated('version', 'UpdateTime', updt)
    def End(self, cmd):
	self.out.End(cmd)
	self.out.updateDB()

class ParseUpdate(empQueue.baseDisp):
    """Parser for the update command.

    Grab info that will allow the client to calculate when the next update
    will occur.
    """
## The next update is at Sun Sep  6 20:00:00.
## The current time is	 Sun Sep  6 19:07:38.
    next = re.compile(r"^The next update is at "+s_time+r"\.$")
    curtime = re.compile(r"^The current time is	  "+s_time+r"\.$")
    def data(self, msg):
	self.out.data(msg)
	mm = self.next.match(msg)
	if mm:
	    self.nextTime = mm
	    megaDB['time'].noteNextUpdate(mm)
	    return
	mm = self.curtime.match(msg)
	if mm:
	    self.curTime = mm
	    megaDB['time'].noteTime(mm)

standardParsers = {}
for i in (
    (ParseRead,
     'read', 'wire', 'wir'),
    (ParseDump,
     'dump', 'pdump', 'ldump', 'sdump', 'ndump', 'lost'),
    (ParseMap,
     'map', 'nmap', 'bmap'),
    (ParseRealm,
     'realm'),
    (ParseMove,
     'explore', 'explor', 'explo', 'expl', 'exp',
     'move', 'mov',
     'transport', 'transpor', 'transpo', 'transp', 'trans', 'tran'),
    (ParseVersion,
     'version', 'versio', 'versi', 'vers', 'ver', 've', 'v'),
    (ParseUpdate,
     'update', 'updat', 'upda', 'upd'),
    (ParseNation,
     'nation', 'natio', 'nati', 'nat'),
    (ParseSpy,
     'spy', 'sp'),
    (ParseUnits,
     'radar', 'rada', 'rad',
     'lradar', 'lrada', 'lrad',
     'lookout', 'lookou', 'looko', 'look', 'loo',
     'llookout', 'llookou', 'llooko', 'llook', 'lloo',
     'navigate', 'navigat', 'naviga', 'navig', 'navi', 'nav',
     'march', 'marc'),
    (ParseReport,
     'report', 'repor', 'repo'),
    (ParseRelations,
     'relations', 'relation', 'relatio', 'relati', 'relat', 'rela', 'rel'),
    (ParseCoastWatch,
     'coastwatch', 'coastwatc', 'coastwat', 'coastwa', 'coastw', 'coast',
     'coas', 'coa'),
    (ParseBuild,
     'build', 'buil', 'bui',),
    ):
    for j in i[1:]:
	standardParsers[j] = i[0]

###########################################################################
#############################  Empire Selectors ###########################

# Conversion of empire selectors to local database format.
# Note: This was taken from the html info pages.

commodityConversion = [
    ('civil', 2, 'civ'), ('milit', 3, 'mil'), ('shell', 2), ('gun', 2),
    ##	('petrol',),  # Yuck, sometimes this is 'pet', othertimes 'petrol'
    ('iron', 2), ('dust', 2), ('bar', 2), ('food', 2), ('oil', 2), ('lcm', 2),
    ('hcm', 2), ('uw', 2), ('rad', 3),
    ]

sectorConversion = [
    ('xloc', 2, 'x'), ('yloc', 2, 'y'), ('owner', 2), ('des', 2),
    ('effic', 2, 'eff'), ('mobil', 2, 'mob'),
    ('terr', 2),
    ##	('timestamp',),
    ('road', 2), ('rail', 2), ('dfense', 2, 'defense'),
    ('work', 2), ('coastal', 2, 'coast'),
##     ('newdes', (lambda ldb:
##		(ldb['sdes'] == '_' and ldb['des'] or ldb['sdes']))),
    ('newdes', 2, "(sdes != '_' and sdes or des)"),
    ('min', 2), ('gold', 2),
    ('fert', 2), ('ocontent', 2), ('uran', 2),
    ##	('oldown',),
    ('off', 2), ('xdist', 2, 'dist_x'), ('ydist', 2, 'dist_y'), ('avail', 2),

    ('petrol', 2, 'pet'),

    ('c_dist', 4), ('m_dist', 4), ('u_dist', 4), ('s_dist', 4), ('g_dist', 4),
    ('p_dist', 4), ('i_dist', 4), ('d_dist', 4), ('b_dist', 4), ('f_dist', 4),
    ('o_dist', 4), ('l_dist', 4), ('h_dist', 4), ('r_dist', 4),
    ('c_del', 4), ('m_del', 4), ('u_del', 4), ('s_del', 4), ('g_del', 4),
    ('p_del', 4), ('i_del', 4), ('d_del', 4), ('b_del', 4), ('f_del', 4),
    ('o_del', 4), ('l_del', 4), ('h_del', 4), ('r_del', 4),
    ]

unitsConversion = [
    ('xloc', 2, 'x'), ('yloc', 2, 'y'),
    ##	('owner',),
    ('type', 2), ('effic', 2, 'eff'), ('mobil', 2, 'mob'),
    ##	('timestamp',), ('sell',),
    ('tech', 2), ('uid', 2, 'id'),

    ##	('group',) # Hrmm. this really should be in each subgroup

    ##	('opx',), ('opy',), ('mission'),
    ]

shipConversion = [
    ('fleet', 2, 'flt'), ('nplane', 2, 'pln'), ('fuel', 2),
    ('nxlight', 2,'xl'), ('nchoppers', 3, 'he'),
    ##	('autonav'),

    ('group', 2, 'flt'),
    ('petrol', 2),
    ]

planeConversion = [
    ('wing', 2), ('range', 2), ('ship', 2), ('att', 2),
    ('def', 2), ('harden', 2, 'hard'), ('nuketype', 2, 'nuke'),
    ##	('flags',),
    ('land', 2),

    ('group', 2, 'wing'),
    ]

landConversion = [
    ('att', 2), ('def', 2), ('army', 4), ('ship', 2), ('harden', 2, 'fort'),
    ('retreat', 2, 'retr'), ('fuel', 2), ('land', 2), ('nxlight', 2, 'xl'),

    ('group', 2, 'army'),
    ('petrol', 2),
    ]

nukeConversion = [
    ('xloc', 2, 'x'), ('yloc', 2, 'y'), ('number', 2, 'num'),
    ##	('ship',), ('trade',), ('timestamp'),
    ]

def createConversionDB(list):
    dict = {}
    for i in list:
	if len(i) < 3:
	    val = i[0]
	else:
	    val = i[2]
	for j in range(i[1], len(i[0])+1):
	    dict[i[0][:j]] = val
    return dict

selectors = {
    'SECTOR': createConversionDB(commodityConversion + sectorConversion),
    'SHIPS': createConversionDB(commodityConversion + unitsConversion
				+ shipConversion),
    'PLANES': createConversionDB(unitsConversion + planeConversion),
    'LAND UNITS': createConversionDB(commodityConversion + unitsConversion
				     + landConversion),
    'NUKES': createConversionDB(nukeConversion),
    }

del commodityConversion, sectorConversion, unitsConversion
del shipConversion, planeConversion, landConversion, nukeConversion
del createConversionDB

###########################################################################
#############################  Useful functions ###########################

# Check for broken string.atoi function
try:
    string.atoi('-')
except ValueError:
    # Working atoi
    fixedAtoI = string.atoi
else:
    # Broken atoi
    def fixedAtoI(str):
	"""Fixed string.atoi with +/- checking."""
	if str == '+' or str == '-':
	    raise ValueError
	return string.atoi(str)

def checkUpdated(dbname, item, val):
    """Update the value in the main/update databases iff a change is made."""
    if megaDB[dbname].get(item) != val:
	megaDB[dbname][item] = megaDB['updated'][dbname][item] = val

def GetPrompt():
    """Return a string with the main prompt."""
    ndb = megaDB['prompt']
    inform, minutes, btus = ndb['inform'], ndb['minutes'], ndb['BTU']
    if inform:
	return "(%s) [%d:%d] Command : " % (inform, minutes, btus)
    return "[%d:%d] Command : " % (minutes, btus)

def loadDB(file):
    """Load database from FILE.

    If the file doesn't exist, create a default database.
    """
    global megaDB
    DBVersion = 19
    try:
	fl = open(file, "rb")
    except IOError:
	megaDB = {
	    'DB_Version':DBVersion,
	    'SECTOR':dictDB(('x', 'y')),
	    'SHIPS':dictDB(('id',), ('x', 'y')),
	    'PLANES':dictDB(('id',), ('x', 'y')),
	    'LAND UNITS':dictDB(('id',), ('x', 'y')),
	    'NUKES':dictDB(('x', 'y', 'type'), ('x', 'y')),
	    'LOST ITEMS':dictDB(('type', 'id', 'x', 'y')),
	    'login':{'host':"empire.idlpaper.com", 'port':5678,
		     'coun':"visitor", 'repr':"visitor"},
	    'version':{'worldsize':(256,256), 'ETUTime':1024,
		       'UpdateTime':1024},
	    'realm':{},
	    'announcements':[],
	    'telegrams':[],
	    'countries':Countries(),
	    'prompt':{'minutes':0, 'BTU':0, 'inform':""},
	    'time':EmpTime(),
	    }
	first = 1
    else:
	megaDB = cPickle.load(fl)
	if megaDB['DB_Version'] != DBVersion:
	    print "Database has an incorrect version number."
	    sys.exit()
	fl.close()
	first = 0
    megaDB['updated'] = {'SECTOR':megaDB['SECTOR'].uDB,
			 'SHIPS':megaDB['SHIPS'].uDB,
			 'PLANES':megaDB['PLANES'].uDB,
			 'LAND UNITS':megaDB['LAND UNITS'].uDB,
			 'NUKES':megaDB['NUKES'].uDB,
			 'LOST ITEMS':megaDB['LOST ITEMS'].uDB,
			 'version':{},
			 'realm':{},
			 'announcements':[],
			 'telegrams':[],
			 'countries':megaDB['countries'].uDB,
			 'prompt':{},
			 'time':megaDB['time'].uDB}
    sys.exitfunc = (lambda file=file:
		    saveDB(file))
    return first

def saveDB(filename):
    """Write the database back to disk."""
    print 'PTkEI: Saving DB..'
    del megaDB['updated']
    fl = open(filename, 'wb')
    cPickle.dump(megaDB, fl, 1)
    fl.close()
