/* $Id: input.c,v 1.3 1998/10/30 05:17:59 ajapted Exp $
***************************************************************************

   Linux_joy: input

   Copyright (C) 1998 Andrew Apted     [andrew@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdlib.h>
#include <unistd.h>
#include <termios.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <linux/joystick.h>

#include <ggi/internal/gii.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif


#define MAX_NR_JAXES		8
#define MAX_NR_JBUTTONS		32 


typedef struct joystick_hook
{
	int fd;

	unsigned char num_axes;
	unsigned char num_buttons;

	int axes[MAX_NR_JAXES];
	char buttons[MAX_NR_JBUTTONS];

} JoystickHook;

#define JOYSTICK_HOOK(inp)	((JoystickHook *) inp->priv)


/* ---------------------------------------------------------------------- */


static int GII_joystick_init(gii_input *inp, char *filename)
{
	JoystickHook *jj;

	int version;

	char name[128];


	/* allocate joystick hook */

	jj = inp->priv = _gii_malloc(sizeof(JoystickHook));

	/* open the device file */

	jj->fd = open(filename, O_RDONLY);

	if (jj->fd < 0) {
		perror("Linux_joy: Couldn't open joystick device");
		return -1;
	}

	/* get version and get name */
	
	if (ioctl(jj->fd, JSIOCGVERSION, &version) < 0) {
		perror("Linux_joy: Couldn't read version:");
		version=0;
	}

	DPRINT("Linux_joy: Joystick driver version %d.%d.%d\n",
		(version >> 16) & 0xff, (version >> 8) & 0xff,
		version & 0xff);

	if (version < 0x010000) {
		fprintf(stderr, "Linux_joy: Sorry, only driver versions "
			">= 1.0.0 supported.\n");
		close(jj->fd);
		return -1;
	}

	if (ioctl(jj->fd, JSIOCGNAME(sizeof(name)), name) < 0) {
		strcpy(name, "Unknown");
	}
	
	DPRINT("Linux_joy: Joystick driver name `%s'.\n", name);

	/* get number of axes and buttons */

	if (ioctl(jj->fd, JSIOCGAXES, &jj->num_axes) ||
	    ioctl(jj->fd, JSIOCGBUTTONS, &jj->num_buttons)) {
		perror("Linux_joy: error getting axes/buttons");
		close(jj->fd);
		return -1;
	}

	DPRINT("Linux_joy: Joystick has %d axes.\n", jj->num_axes);
	DPRINT("Linux_joy: Joystick has %d buttons.\n", jj->num_buttons);

	if (jj->num_axes > MAX_NR_JAXES) {
		jj->num_axes = MAX_NR_JAXES;
	}
	if (jj->num_buttons > MAX_NR_JBUTTONS) {
		jj->num_buttons = MAX_NR_JBUTTONS;
	}
	
	DPRINT("Linux_joy: init OK.\n");

	return 0;
}

static void GII_joystick_exit(gii_input *inp)
{
	JoystickHook *jj = JOYSTICK_HOOK(inp);

	close(jj->fd);
	jj->fd = -1;

	free(jj);
	inp->priv = NULL;

	DPRINT("Linux_joy: exit OK.\n");
}

static gii_event_mask GII_joystick_handle_data(gii_input *inp)
{
	JoystickHook *jj = JOYSTICK_HOOK(inp);

	struct js_event js;

	gii_event ev;
	gii_event_mask result=0;

	int i;


	/* read the joystick packet */

	if (read(jj->fd, &js, sizeof(js)) != sizeof(js)) {
		perror("Linux_joy: Error reading joystick");
		return 0;
	}

	switch (js.type & ~JS_EVENT_INIT) {

	  case JS_EVENT_AXIS:

		if (js.number > jj->num_axes)
			break;
		if (jj->axes[js.number] == js.value)
			break;

		jj->axes[js.number] = js.value;

		DPRINT_EVENTS("JOY-AXIS[%d] -> %d.\n", js.number,
				js.value);

		_giiEventBlank(&ev);

		ev.any.type   = evValAbsolute;
		ev.any.size   = sizeof(gii_val_event);
		ev.any.origin = inp->origin;
		ev.any.target = 0;
		ev.val.first  = 0;
		ev.val.count  = jj->num_axes;

		for (i=0; i < ev.val.count; i++) {
			ev.val.value[i] = jj->axes[i];
		}

		_giiEvQueueAdd(inp, &ev);

		result |= emValAbsolute;
		break;
	
	  case JS_EVENT_BUTTON:

		if (js.number > jj->num_buttons)
			break;
		if (jj->buttons[js.number] == js.value)
			break;

		jj->buttons[js.number] = js.value;

		DPRINT_EVENTS("JOY-BUTTON[%d] -> %d.\n", js.number,
				js.value);

		_giiEventBlank(&ev);

		ev.any.type   = js.value ? evKeyPress : evKeyRelease;
		ev.any.size   = sizeof(gii_key_event);
		ev.any.origin = inp->origin;
		ev.any.target = 0;
		ev.key.effect = 0;
		ev.key.button = js.number;
		ev.key.sym    = GIIK_VOID;
		ev.key.label  = GIIK_VOID;

		_giiEvQueueAdd(inp, &ev);

		result |= (1 << ev.any.type);
		break;

	  default:
		DPRINT_EVENTS("JOY: unknown event from driver "
				"(0x%02x)\n", js.type);
		break;
	}
	
	return result;
}

static gii_event_mask GII_joystick_poll(gii_input *inp)
{
	JoystickHook *jj = JOYSTICK_HOOK(inp);
	
	gii_event_mask result = 0;


	DPRINT_MISC("linux_joy: poll(%p)\n", inp);
	
	for (;;) {

		fd_set readset;

		struct timeval t = {0,0};
		int rc;

		FD_ZERO(&readset);
		FD_SET(jj->fd, &readset);

		/* FIXME !!! doesn't handle -EINTR */
		rc = select(inp->maxfd, &readset, NULL, NULL, &t);

		if (rc <= 0)
			break;

		result |= GII_joystick_handle_data(inp);
	}

	return result;
}


/* ---------------------------------------------------------------------- */


static int GII_linux_joy_close(gii_input *inp)
{
	DPRINT_MISC("Linux_joy close\n");

	if (JOYSTICK_HOOK(inp)) {
		GII_joystick_exit(inp);
	}

	return 0;
}

int GIIdlinit(void *vis, const char *args)  /* !!! */
{
	gii_input *inp = (gii_input *) vis;   /* FIXME ! HACKHACKHACK !*/
	char *filename = "/dev/js0";

	
	DPRINT_MISC("linux_joy starting.\n");

	/* Initialize */

	if (args && *args) {
		filename = (char *) args;
	}

	if (GII_joystick_init(inp, filename) < 0) {
		return -1;
	}
	
	/* We leave these on the default handlers
	 *	inp->GIIseteventmask = _GIIstdseteventmask;
	 *	inp->GIIgeteventmask = _GIIstdgeteventmask;
	 *	inp->GIIgetselectfdset = _GIIstdgetselectfd;
	 */
	inp->GIIeventpoll = GII_joystick_poll;
	inp->GIIclose = GII_linux_joy_close;

	inp->targetcan = emKey | emValuator;
	inp->GIIseteventmask(inp, emKey | emValuator);

	inp->maxfd = JOYSTICK_HOOK(inp)->fd + 1;
	FD_SET(JOYSTICK_HOOK(inp)->fd, &inp->fdset);

	DPRINT_MISC("linux_joy fully up\n");

	return 0;
}
