#! /bin/sh
#    $Id: PubStruct_control.sh,v 6.5 1998/05/14 16:11:15 kimelman Exp $
#  ===========================================================================
# 
#                             PUBLIC DOMAIN NOTICE
#             National Center for Biotechnology Information (NCBI)
# 
#   This software/database is a "United States Government Work" under the
#   terms of the United States Copyright Act.  It was written as part of
#   the author's official duties as a United States Government employee and
#   thus cannot be copyrighted.  This software/database is freely available
#   to the public for use. The National Library of Medicine and the U.S.
#   Government do not place any restriction on its use or reproduction.
#   We would, however, appreciate having the NCBI and the author cited in
#   any work or product based on this material
# 
#   Although all reasonable efforts have been taken to ensure the accuracy
#   and reliability of the software and data, the NLM and the U.S.
#   Government do not and cannot warrant the performance or results that
#   may be obtained by using this software or data. The NLM and the U.S.
#   Government disclaim all warranties, express or implied, including
#   warranties of performance, merchantability or fitness for any particular
#   purpose.
# 
#  ===========================================================================
# 
#  Author:  Michael Kimelman
# 
#  PubStruct maintainance script 
#                    
#  Modifications:  
#  --------------------------------------------------------------------------
#  $Log: PubStruct_control.sh,v $
#  Revision 6.5  1998/05/14 16:11:15  kimelman
#  Compression stuff added in debug mode.
#  few bugs fixed, related to OpenServer/SQL Server switching
#
#  Revision 6.4  1998/05/08 18:12:35  kimelman
#  minibugs & style
#
#  Revision 6.3  1998/05/08 03:00:43  kimelman
#  uniform configuration addaed
#
#  Revision 6.2  1998/04/15 15:58:29  kimelman
#  grammar
#
#  Revision 6.1  1998/04/15 14:53:57  kimelman
#  1. Make retrieval uniform from open server and sql server.
#  2. mmdbsrv retrival performance tuning:
#  	- size of read-in buffers
#  	- direct pdb2mmdb DB lookup instead of full loading pdb to mmdb translataion
#  	  table
#  3. cleaning mmdblocl.* and make.mmdbsrv in order to remove obsolete code
#
# 
#  ==========================================================================
# 

progname="$0"
if [ ! -f $progname ] ; then
    progname="`which $progname`"
fi
if [ ! -f $progname ] ; then
    echo "ERROR: can't determine $0 home directory"
    exit 1
fi
progdir=`dirname $progname`
    
options="$*"

#default PubStruct production DB server
DBserver=BACH10 
DBname=PubStruct
DBuser=kimelman
DBpasswrd=kmlmNu
FILEpath=/net/vaster/mmdb/mmdb/data
tempname=/tmp/$$

reload_schema=no
reload_proc=no
load_list=empty
download_list=empty
delete_list=empty

verbose=no

usage() {

cat <<EOF

  Usage : $progname [ options ]

options:
    --reload_schema - drops tables and data and create new tables from PubStruct.scr
    --reload_proc   - drops all of the stored procs and create proc from PubStruct_proc.scr

    --delete_list <filename> 
                    - delete Structures with mmdb_id listed in <filename>
    --path <path_to_gzipped_files>   (default $FILEpath)
                    - set path to directory contained gzipped asns
    --load_list <filename>    
                    - load to database structures with mmdb_id listed in <filename>
                      special <filename>='all' leads to loading every structure found
                      in <--path> directory
    --download_list <filename>    
                    - download structures with mmdb_id listed in <filename>
                      special <filename>='all' leads to dumping all active structures
                      in <--path> directory
    --DBserver      - SQL server name          ( default $DBserver )
    --DBname        - Database   name          ( default $DBname   )
    --DBuser        - SQL server user name     ( default $DBuser   )
    --DBpasswd      - SQL server user password ( default $DBpasswd )
    --SYBASE        - SYBASE environment variable ( default $SYBASE )
    --verbose       - turn on debugging mode

Examples:

 creating data and loading full structure data from /net/vaster/mmdb/mmdb/data:
  agate{kml}pubstruct> $progname --reload_schema --load_list all ;
 
 The same but loading data from saturn :
  agate{kml}pubstruct> $progname --reload_schema --load_list all --path /net/saturn/pubmed2/MMDB
  
 Montly update (/net/vaster/mmdb/mmdb/data has already been updated):
  agate{kml}pubstruct> $progname --delete_list ../deleted --load_list ../updated

EOF
  
}

while [ $# -gt 0 ]; do
  case "$1" in
    ## remove everything in DB and createw it from scrath
    "--reload_schema" )
        reload_schema=yes
        reload_proc=yes
    ;;
    "--reload_proc" )
        reload_proc=yes
    ;;
    "--load_list" )
        load_list="$2"
        shift
    ;;
    "--download_list" )
        download_list="$2"
        shift
    ;;
    "--delete_list" )
        delete_list="$2"
        shift
    ;;
    "--path" ) ## path to directory with *.val.gz files
        FILEpath="$2"
        shift
    ;;
    "--DBserver" )
        DBserver="$2"
        shift
    ;;
    "--DBname" )
        DBname="$2"
        shift
    ;;
    "--DBuser" )
        DBuser="$2"
        shift
    ;;
    "--DBpasswd" )
        DBpasswrd="$2"
        shift
    ;;

    "--SYBASE" )
        SYBASE="$2"
        export SYBASE
        shift
    ;;

    ## turn on verbose mode
    "--verbose" | "-v" )
        verbose=yes
    ;;
    * )
        echo "${progname}: Unrecognized argument: $1" 
        usage
        exit 1
    ;;
  esac
  shift
done

# QA : check options for correctness
#------------------------------------
if [ $reload_proc != yes -a $reload_schema != yes -a $load_list = empty -a \
     $download_list = empty -a  $delete_list = empty ] ; then
  echo "WARNING: No action specified!!!"
  echo "${progname}: ${options}" 
  usage
  exit 0
fi

if [  $reload_schema = yes -a $delete_list != empty ] ; then
  echo "WARNING: 'delete_list' option is useless with 'reload_schema', exit - NO ACTION TAKEN"
  echo "${progname}: ${options}"
  usage
  exit 1
fi

# QA passed
#------------------------------------
if [ $verbose = yes ] ; then
  set -x
fi

. ${progdir}/st_configure.sh

ISQL="$SYBASE_isql -S$DBserver -U$DBuser -P$DBpasswrd"

drops() { #--------<< drop objects >>------------------------------
    ## prepare list of objects to drop.
    cat >clean1 <<EOF
use $DBname
go
select 'drop $1    '+name from sysobjects where type = '$2'
go
EOF
   ## prepared dropping script
   cat >clean2 <<EOF
use $DBname
go
`$ISQL <clean1 | grep "drop" `
go
EOF
   ## run it
   $ISQL < clean2
   rm -f clean1 clean2
} #--------<< drop objects >>--------------------------------------

#==================================================================
# Start Actions
#

if [ $reload_proc = yes ] ; then
    drops proc P
    drops trigger TR
fi

if [ $reload_schema = yes ] ; then
    drops table U
   ## recreate DB schema
   $ISQL <PubStruct.scr
fi

if [ $reload_proc = yes ] ; then
   ## recreate DB stored procedures
   $ISQL <PubStruct_proc.scr
fi

if [ $load_list = all ] ; then
  ## prepare list of files to load
  ls $FILEpath | grep gz | sed 's/[.]val[.]gz$//g' | sort -n >$tempname
  load_list=$tempname
fi
if [ $download_list = all ] ; then
  cat >clean1 <<EOF
use $DBname
go
select distinct 'id ', mmdb_id from Struct where state = 0 and suppressed = 0
go
EOF
  $ISQL < clean1 | grep 'id ' | awk ' { print $2 }' >$tempname.down
  rm clean1
  download_list=$tempname.down
fi

OPTS="--dbpath=$DBserver:$DBname=$DBuser,$DBpasswrd --path=$FILEpath"
[ $verbose != yes ] || OPTS="-v $OPTS"

if [ $delete_list != empty ] ; then
  ${progdir}/loader --remove  --state=0 $OPTS <$delete_list
fi

if [ $load_list != empty ] ; then
  if [ -r $load_list ] ; then
    ${progdir}/loader --load              --state=1 $OPTS <$load_list
    ${progdir}/loader --download --checks --state=1 $OPTS <$load_list
  else
    echo "input file <$load_list> doesnt exist - loading failed"
  fi
fi
if [ $download_list != empty ] ; then
  if [ -r $download_list ] ; then
    ${progdir}/loader --download $OPTS <$download_list
  else
    echo "input file <$download_list> doesnt exist - downloading failed"
  fi
fi


if [ -f cmp_log ] ; then
  if [ `cat cmp_log | wc -l ` -ne 0 ] ; then
    echo "FAILED!!!!!!!!"
    mail kimelman -s"pdb loading failed" <<EOF

$0 : $*

`cat cmp_log`
EOF
    exit
  else
    rm cmp_log
  fi
fi
[ ! -d mmdb1 ] || rm -rf mmdb1
[ ! -d mmdb  ] || [ -r $download_list ] || rm -rf mmdb

[ ! -f $tempname ] || rm $tempname 
[ ! -f $tempname.down ] || rm $tempname.down

if [ $reload_schema = yes -a $load_list != empty ] ; then
   cat >clean2 <<EOF
use $DBname
go
exec initial_timestamps
go
EOF
  $ISQL < clean2
  rm -f clean2
fi

exit 0

#
# The end
#
