/*								-*- C++ -*-
 * $Id: GDI_colour.cpp,v 1.2 1997/07/14 09:43:50 wmglo Exp $
 *
 * Purpose: classes to cover colours and colourmaps
 *
 * Authors: Markus Holzem and Julian Smart
 *
 * Copyright: (C) 1995, AIAI, University of Edinburgh (Julian)
 * Copyright: (C) 1995, GNU (Markus)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Additionally everyone using this library has to announce it with:
 *
 *   This software uses the wxWindows-Xt GUI library
 *   (C) Markus Holzem, available via
 *       ftp://ftp.aiai.ed.ac.uk/pub/packages/wxwin/ports/xt
 */

#ifdef __GNUG__
#pragma implementation "GDI_colour.h"
#endif

#define  Uses_XLib
#define  Uses_wxColour
#include "wx.h"

IMPLEMENT_DYNAMIC_CLASS(wxColour, wxObject)
IMPLEMENT_DYNAMIC_CLASS(wxColourDatabase, wxList)
IMPLEMENT_DYNAMIC_CLASS(wxColourMap, wxObject)

// shift between wxWindows RGB- and XColor RGB-values
// (necessary because the values specify an intensity)
#define SHIFT (8*(sizeof(short int)-sizeof(char)))

//-----------------------------------------------------------------------------
// private data of wxColour and wxColourMap
//-----------------------------------------------------------------------------

class wxColourRep : public wxObject {
DECLARE_DYNAMIC_CLASS(wxColourRep)
public:
    XColor   xcolor;
    Bool     have_pixel;
    Colormap xcolormap;
    int      ref_count;
#if WXDEBUG
    void Dump(ostream& str);
#endif
};

#if WXDEBUG
void wxColourRep::Dump(ostream& str)
{
    str << "wxColourRep ref = " << ref_count;
}
#endif // WXDEBUG

IMPLEMENT_ABSTRACT_CLASS(wxColourRep, wxObject);

class wxColourMapRep : public wxObject {
DECLARE_DYNAMIC_CLASS(wxColourMapRep)
public:
    Colormap       xcolormap;
    Bool           priv;
    unsigned long* pix_array;
    int            pix_array_n;
    int            ref_count;
#if WXDEBUG
    void Dump(ostream& str);
#endif
};

IMPLEMENT_ABSTRACT_CLASS(wxColourMapRep, wxObject);

#if WXDEBUG
void wxColourMapRep::Dump(ostream& str)
{
    str << "wxColourMapRep ref = " << ref_count;
}
#endif // WXDEBUG

//-----------------------------------------------------------------------------
// wxColour
//-----------------------------------------------------------------------------

wxColour::wxColour(void)
{
    __type = wxTYPE_COLOUR;

    rep = NULL; // not Ok
}

wxColour::wxColour(wxColour& col) // copy constructor
{
    __type = wxTYPE_COLOUR;

    rep = col.rep;
    if (rep) rep->ref_count++;
}

wxColour::wxColour(const char *col) // find colour by name
{
    __type = wxTYPE_COLOUR;

    wxNode *db_node;
    if ( (db_node = wxTheColourDatabase->Find(col)) ) {
	// colour defined in wxWindows database
	rep = ((wxColour*)(db_node->Data()))->rep;
	if (rep) ++rep->ref_count;
    } else {
	rep = wxNEW wxColourRep;
	rep->xcolormap = wxAPP_COLOURMAP->GetColormap();
	if (XParseColor(wxAPP_DISPLAY, rep->xcolormap, col, &(rep->xcolor))) {
	    // colour defined in X database
	    rep->have_pixel = FALSE;
	    rep->ref_count  = 1;
	} else {
	    // colour not defined
	    delete rep; rep = NULL;
	    wxError(col, "wxColour: could not find colour");
	}
    }
}

wxColour::wxColour(unsigned char r, unsigned char g, unsigned char b)
{
    __type = wxTYPE_COLOUR;

    rep = NULL; Set(r, g, b); // set RGB-values
}

wxColour::~wxColour(void)
{
    FreePixel(FALSE);
}

//--- assignment -------------------------------------------------------------

wxColour& wxColour::operator = (const wxColour& col)
{
    FreePixel(FALSE); // free pixel before assignment
    rep = col.rep;
    if (rep) rep->ref_count++;
    // return this for further assignments
    return (*this);
}

void wxColour::operator = (const wxColour* p_col)
{
    FreePixel(FALSE); // free pixel before assignment
    // new_colour has to ok! otherwise THIS colour remains empty
    if (p_col) {
	rep = p_col->rep;
	if (rep) rep->ref_count++;
    }
}

wxColour& wxColour::operator = (const char *col)
{
    FreePixel(FALSE); // free pixel before assignment

    wxNode *db_node;
    if ( (db_node = wxTheColourDatabase->Find(col)) ) {
	// colour defined in wxWindows database
	rep = ((wxColour*)(db_node->Data()))->rep;
	if (rep) ++rep->ref_count;
    } else {
	rep = wxNEW wxColourRep;
	rep->xcolormap = wxAPP_COLOURMAP->GetColormap();
	if (XParseColor(wxAPP_DISPLAY, rep->xcolormap, col, &(rep->xcolor))) {
	    // colour defined in X database
	    rep->have_pixel = FALSE;
	    rep->ref_count  = 1;
	} else {
	    // colour not defined
	    delete rep; rep = NULL;
	    wxError(col, "wxColour: could not find colour");
	}
    }
    return (*this);
}

//--- get and set RGB values --------------------------------------------------

void wxColour::Set(unsigned char r, unsigned char g, unsigned char b)
{
    FreePixel(FALSE);
    rep = wxNEW wxColourRep; // create new X representation
    rep->xcolormap = wxAPP_COLOURMAP->GetColormap();
    rep->ref_count  = 1;

    rep->xcolor.red   = ((unsigned short)r) << SHIFT; // init XColor structure
    rep->xcolor.green = ((unsigned short)g) << SHIFT;
    rep->xcolor.blue  = ((unsigned short)b) << SHIFT;
    rep->xcolor.flags = DoRed | DoGreen | DoBlue;
    rep->have_pixel   = FALSE; // no pixel value assigned
}

void wxColour::Get(unsigned char *r, unsigned char *g, unsigned char *b)
{
    if (rep) {
	*r = (unsigned char)(rep->xcolor.red   >> SHIFT);
	*g = (unsigned char)(rep->xcolor.green >> SHIFT);
	*b = (unsigned char)(rep->xcolor.blue  >> SHIFT);
    } else {
	*r = *g = *b = 0;
    }
}

unsigned char wxColour::Red(void)
{
    return ( rep ? (unsigned char)(rep->xcolor.red >> SHIFT) : 0 );
}

unsigned char wxColour::Green(void)
{
    return ( rep ? (unsigned char)(rep->xcolor.green >> SHIFT) : 0 );
}

unsigned char wxColour::Blue(void)
{
    return ( rep ? (unsigned char)(rep->xcolor.blue >> SHIFT) : 0 );
}

//--- allocate and free X pixel values ----------------------------------------

unsigned long wxColour::GetPixel(wxColourMap *cmap)
{
    if (!rep) {
	// use something as a default value
	return(WhitePixelOfScreen(wxAPP_SCREEN));
    }
    if (rep->have_pixel && rep->xcolormap==cmap->GetColormap()) {
	// I have the pixelvalue for this colormap
	return (rep->xcolor.pixel);
    }
    // now there are two possibilities:
    // 1. this is the first computation of the pixel value
    // 2. the colour map has changed
    FreePixel(TRUE);			  // free pixel value if any
    rep->xcolormap = cmap->GetColormap(); // colourmap to use
    // allocate pixel
    if(!cmap->IsPrivate() ) {
	if (!XAllocColor(wxAPP_DISPLAY, rep->xcolormap, &(rep->xcolor))) {
	    // failed => use black as default
	    wxError("alloc new colour failed, using black.", "wxColour");
	    return BlackPixelOfScreen(wxAPP_SCREEN);
	}
	rep->have_pixel = TRUE; // allocation successful
    } else {
	XColor cols[256]; // size is arbitrary!
	int i;
	unsigned long pix_min = 0, diff, diff_min = 3*65536;

	for(i=0; i<256; i++)
	    cols[i].pixel = i;
	XQueryColors(wxAPP_DISPLAY, rep->xcolormap, cols, 256);
	for(i=0; i<256; i++) {
	    diff = wxAbs(rep->xcolor.red - cols[i].red) +
		wxAbs(rep->xcolor.green - cols[i].green) +
		wxAbs(rep->xcolor.blue - cols[i].blue);
	    if(diff < diff_min) {
		diff_min = diff;
		pix_min = cols[i].pixel;
	    }
	}
	rep->xcolor.pixel = pix_min;
	rep->have_pixel = FALSE; // the color values may change
    }
    return (rep->xcolor.pixel);
}

void wxColour::FreePixel(Bool free_pixel_only)
{
    // free the pixel value.
    // -- this will contradict to the cheeky usage of colour creation
    // -- in applications like the colours demo

    if (rep) {
	// free pixel (with or without memory for wxColourRep
	if (free_pixel_only) {
	    if (rep->have_pixel) {
		XFreeColors(wxAPP_DISPLAY, rep->xcolormap, &(rep->xcolor.pixel), 1, 0);
	    }
	    rep->have_pixel = FALSE;
	} else {
	    // unreference to old colour and destroy it if necessary
	    --rep->ref_count;
	    if (rep->ref_count==0) {
		if (rep->have_pixel)
		    XFreeColors(wxAPP_DISPLAY, rep->xcolormap, &(rep->xcolor.pixel), 1,0);
		delete rep; // destroy X representation;
	    }
	    rep = NULL; // not Ok
	}
    }
}

//-----------------------------------------------------------------------------
// wxColourDatabase
//-----------------------------------------------------------------------------

wxColourDatabase::wxColourDatabase (void) : wxList(wxKEY_STRING)
{
    // there is one representation of wxColourDatabase
    wxTheColourDatabase = this;
    // standard preloaded colours
    wxBLACK      = FindColour("BLACK");
    wxWHITE      = FindColour("WHITE");
    wxGREY       = FindColour("GREY");
    wxRED        = FindColour("RED");
    wxBLUE       = FindColour("BLUE");
    wxGREEN      = FindColour("GREEN");
    wxCYAN       = FindColour("CYAN");
    wxLIGHT_GREY = FindColour("LIGHT GREY");
    // find colours not defined in database (take from Windows 3.1 code)
    FindColour(234, 234, 172, "MEDIUM GOLDENROD");
    FindColour(107, 142,  35, "MEDIUM FOREST GREEN");
    FindColour(255,   0, 255, "LIGHT MAGENTA");
    FindColour(100, 100, 100, "MEDIUM GREY");
}

wxColourDatabase::~wxColourDatabase (void)
{
    wxNode *node = First();
    while (node) {
	wxColour *colour = (wxColour*)node->Data();
	wxNode *next = node->Next();
	delete colour; delete node;
	node = next;
    }
}

wxColour *wxColourDatabase::FindColour(const char *colour)
{
    wxNode *node;

    if ( (node = Find(colour)) )
	return (wxColour*)node->Data(); // colour already defined

    // create new colour and add it to list
    wxColour *col = wxNEW wxColour(colour);
    if (!(col->Ok())) {
	// colour not found
	delete col;
	return NULL;
    }
    Append(colour, col);
    return col;
}

wxColour *wxColourDatabase::FindColour(unsigned char r, unsigned char g, unsigned char b,
				       const char *colour)
{
    for (wxNode *node = First(); node; node = node->Next()) {
	wxColour *col = (wxColour*)node->Data ();
	if (col->Ok() && (col->Red()==r && col->Green()==g && col->Blue()==b))
	    // I've found a matching colour
	    return col;
    }
    // create new colour and add it to list
    wxColour *col = wxNEW wxColour(r, g, b);
    if (!(col->Ok())) {
	// colour not found
	delete col;
	return NULL;
    }
    Append(colour, col);
    return col;
}

char *wxColourDatabase::FindName(wxColour& colour)
{
    if (colour.Ok()) {
	unsigned char red   = colour.Red();
	unsigned char green = colour.Green();
	unsigned char blue  = colour.Blue();

	for (wxNode *node = First(); node; node = node->Next()) {
	    wxColour *col = (wxColour*)node->Data ();
	    if (col->Ok()
            && (col->Red()==red && col->Green()==green && col->Blue()==blue)) {
		char *found = node->key.string;
		if (found)
		    return found;
	    }
	}
    }
    return NULL;
}

//-----------------------------------------------------------------------------
// wxColourMap
//-----------------------------------------------------------------------------

wxColourMap::wxColourMap(Bool priv)
{
     __type = wxTYPE_COLOURMAP;
 
     rep = wxNEW wxColourMapRep; // create new X representation

     if ((rep->priv = priv)) {
	 rep->xcolormap = XCreateColormap(wxAPP_DISPLAY,
					  RootWindow(wxAPP_DISPLAY, 0),
					  DefaultVisual(wxAPP_DISPLAY, 0),
					  AllocAll);
    } else
	rep->xcolormap = DefaultColormapOfScreen(wxAPP_SCREEN);
    rep->pix_array_n = 0;
    rep->ref_count   = 1;
}

wxColourMap::wxColourMap(wxColourMap& new_cmap)
{
    __type = wxTYPE_BRUSH;

    rep = new_cmap.rep;
    if (rep) rep->ref_count++;
}

wxColourMap::~wxColourMap(void)
{
    // delete if representation has no more references
    DeleteColormap();
}

wxColorMap& wxColourMap::operator = (wxColourMap& new_cmap)
{
    // delete if representation has no more references
    DeleteColormap();
    // reference representation
    rep = new_cmap.rep;
    if (rep) rep->ref_count++;

    return (*this);
}

Colormap wxColourMap::GetColormap(void)
{
    if (rep)
	return (rep->xcolormap);
    return (wxAPP_COLOURMAP->rep->xcolormap);
}

void wxColourMap::DeleteColormap(void)
{
    // delete if representation has no more references
    if (rep && --rep->ref_count==0) {
 	if (rep->priv) {
	    // free colourmap
	    XFreeColormap(wxAPP_DISPLAY, rep->xcolormap);
	} else {
	    // free colours
	    if (rep->pix_array_n > 0) {
		int i, j;
		for (i = j = 0; i < rep->pix_array_n; i = j) {
		    while (j < rep->pix_array_n && rep->pix_array[j] != 0)
			j++;
		    if (j > i)
			XFreeColors(wxAPP_DISPLAY, rep->xcolormap,
				    &rep->pix_array[i], j-i, 0);
		    while (j < rep->pix_array_n && rep->pix_array[j] == 0)
			j++;
		}
		delete[] rep->pix_array;
		rep->pix_array_n = 0;
	    }
 	}
 	delete rep;
    }
}

Bool wxColourMap::IsPrivate()
{
    return (rep && rep->priv);
}

Bool wxColourMap::Create(const int n,
			 const unsigned char *red,
			 const unsigned char *green,
			 const unsigned char *blue,
			 Bool priv)
{
    Display *display = wxAPP_DISPLAY;

    if (!rep)
	return FALSE;
    if (rep->pix_array_n > 0) {
	int i, j;
	for (i=j=0; i<rep->pix_array_n; i=j) {
	    while (j<rep->pix_array_n && rep->pix_array[j]!=0)
		j++;
	    if (j > i)
		XFreeColors(display, rep->xcolormap, &rep->pix_array[i], j-i, 0);
	    while (j<rep->pix_array_n && rep->pix_array[j]==0)
		j++;
	}
	delete[] rep->pix_array;
    }
    rep->pix_array_n = 0;

    if (!n) {
	rep->pix_array = NULL;
	return FALSE;
    }

    if (priv) {
	rep->pix_array = 0;
	if (!rep->priv) {
	    rep->xcolormap = XCreateColormap(display,
					     RootWindow(display, 0),
					     DefaultVisual(display, 0),
					     AllocAll);
	    if (!rep->xcolormap) return FALSE;
	    rep->priv = TRUE;
	}
	XColor *xcols = new XColor[n];
	int i;
	for (i=0; i<n; i++) {
	    xcols[i].flags = DoRed | DoGreen | DoBlue;
	    xcols[i].red = (unsigned short)red[i] << 8;
	    xcols[i].green = (unsigned short)green[i] << 8;
	    xcols[i].blue = (unsigned short)blue[i] << 8;
	    xcols[i].pixel = i;
	}
	XStoreColors(display, rep->xcolormap, xcols, n);
	delete[] xcols;
    } else {
	if(rep->priv) {
	    XFreeColormap(display, rep->xcolormap);
	    rep->xcolormap = DefaultColormapOfScreen(wxAPP_SCREEN);
	    rep->priv = FALSE;
	}
	rep->pix_array = new unsigned long[n];
	if (!rep->pix_array) return FALSE;
	rep->pix_array_n = n;
	XColor xcol;
	xcol.flags = DoRed | DoGreen | DoBlue;
	for(int i = 0; i < n; i++) {
	    xcol.red = (unsigned short)red[i] << 8;
	    xcol.green = (unsigned short)green[i] << 8;
	    xcol.blue = (unsigned short)blue[i] << 8;
	    rep->pix_array[i] = (XAllocColor(display, rep->xcolormap, &xcol) == 0) ?
		0 : xcol.pixel;
	}
    }

    return TRUE;
}

Bool wxColourMap::TransferBitmap8(unsigned char *data, unsigned long size,
				  void *dest, unsigned int bpp)
{
    switch (bpp) {
    case 8: {
	unsigned char *dptr = (unsigned char *)dest;
	while (size-- > 0) {
	    if ((int)*data < rep->pix_array_n)
		*dptr = (unsigned char)rep->pix_array[*data];
	    data++;
	    dptr++;
	}
	break;
    }
    case 16: {
	unsigned short *dptr = (unsigned short *)dest;
	while(size-- > 0) {
	    if((int)*data < rep->pix_array_n)
		*dptr = (unsigned short)rep->pix_array[*data];
	    data++;
	    dptr++;
	}
	break;
    }
    case 24: {
	struct rgb24 { unsigned char r, g, b; } *dptr = (struct rgb24 *)dest;
	while(size-- > 0) {
	    if((int)*data < rep->pix_array_n) {
		dptr->r = rep->pix_array[*data] & 0xFF;
		dptr->g = (rep->pix_array[*data] >> 8) & 0xFF;
		dptr->b = (rep->pix_array[*data] >> 16) & 0xFF;
	    }
	    data++;
	    dptr++;
	}
	break;
    }
    case 32: {
	unsigned long *dptr = (unsigned long *)dest;
	while(size-- > 0) {
	    if((int)*data < rep->pix_array_n)
		*dptr = rep->pix_array[*data];
	    data++;
	    dptr++;
	}
	break;
    }
    default:
	return FALSE;
    }
    return TRUE;
}
