#include <GL/gl.h>
#include <GL/glut.h>

#include "draw.h"
#include "glbiff.h"


///// globals
float flag_position=0;
float door_position=0;
float xform_factor=0;

bool fDoorOpen = false;
bool fXformOn = false;



/*
 * set_colour()
 *
 * generic routine to set current colour; uses some reasonable default
 * values. Requires that you pass in the RGB of the desired colour.
 */
void set_colour(float r, float g, float b)
{
  float ambient = 0.2;
  float diffuse = 0.7;
  float specular = 0.4;
  GLfloat mat[4];
  /**** set ambient lighting parameters ****/
  mat[0] = ambient*r;
  mat[1] = ambient*g;
  mat[2] = ambient*b;
  mat[3] = 1.0;
  glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT, mat);

  /**** set diffuse lighting parameters ******/
  mat[0] = diffuse*r;
  mat[1] = diffuse*g;
  mat[2] = diffuse*b;
  mat[3] = 1.0;
  glMaterialfv (GL_FRONT_AND_BACK, GL_DIFFUSE, mat);

  /**** set specular lighting parameters *****/
  mat[0] = specular*r;
  mat[1] = specular*g;
  mat[2] = specular*b;
  mat[3] = 1.0;
  glMaterialfv (GL_FRONT_AND_BACK, GL_SPECULAR, mat);
  glMaterialf (GL_FRONT_AND_BACK, GL_SHININESS, 0.5);
}

/*
 * draw_mbox_top()
 *
 * draws the Bezier patch which makes the top of mailbox
 */
void draw_mbox_top() {
  glEnable(GL_MAP2_VERTEX_3);
  glEvalMesh2(GL_FILL, 0, DOME_SEGS, 0, DOME_SEGS);
  glDisable(GL_MAP2_VERTEX_3);
}

/*
 * draw_mbox_side()
 *
 * draw a single side of a mailbox; (nx,ny,nz) is the normal vector for
 * the side.
 */
void draw_mbox_side( float nx, float ny, float nz ) {
  glBegin(GL_POLYGON);
    glNormal3f(nx,ny,nz);
    glVertex3f(0,0,0);
    glVertex3f(0,0,SIDE_HEIGHT);
    glVertex3f(0,1,SIDE_HEIGHT);
    glVertex3f(0,1,0);
  glEnd();
}

/*
 * draw_mbox_door()
 *
 * draws the door, using a Bezier curve, among other things. This is used 
 * both for the front, and the back doors.
 */
void draw_mbox_door() {
  glEnable(GL_MAP1_VERTEX_3);
  glPushMatrix();
  glTranslatef(0,0,SIDE_HEIGHT);
  glBegin(GL_POLYGON);
    glNormal3f(0,-1,0);
    glVertex3f(0,0,-SIDE_HEIGHT);
    for( int i=0; i<=DOME_SEGS; i++ )
      glEvalCoord1f(((float)i)/((GLfloat)DOME_SEGS));
    glVertex3f(1,0,-SIDE_HEIGHT);
  glEnd();
  glPopMatrix();
  glDisable(GL_MAP1_VERTEX_3);
}

/*
 * draw_mbox_flag()
 *
 * draws the flag, taking 'flag_position' into account.
 */
void draw_mbox_flag() {
  glPushMatrix();
  glScalef(FLAG_SCALE,FLAG_SCALE,FLAG_SCALE*1.3);
  glRotatef(90-flag_position,0,1,0);
  glTranslatef(0.05,0,0.5);

  glPushMatrix();
  glScalef(0.1,0.05,1);
  glutSolidCube(1.0);
  glPopMatrix();

  glPushMatrix();
  glTranslatef(0.2,0,0.35);
  glScalef(0.3,0.05,0.3);
  glutSolidCube(1.0);
  glPopMatrix();

  glPopMatrix();
}

/*
 * draw_env()
 *
 * draws a single small envelope: stands for 1 email
 */
void draw_env() {
  set_colour(1,1,1);
  glPushMatrix();
  glRotatef(-10,0,1,0);
  glTranslatef(0.01,0.5+0.1,0.2);
  glScalef(0.02,1,0.4);
  glutSolidCube(1.0);
  glPopMatrix();
}

/*
 * draw_big_env()
 *
 * draws a bigger envelope: stands for 5 emails
 */
void draw_big_env() {
  set_colour(0.8,0.6,0);
  glPushMatrix();
  glRotatef(-5,0,1,0);
  glTranslatef(0.01,0.5+0.1,0.25);
  glScalef(0.02,1,0.5);
  glutSolidCube(1.0);
  glPopMatrix();
};

/*
 * draw_box()
 *
 * draws a small package: stands for 25 emails
 */
void draw_box() {
  set_colour(0.4,0.3,0.1);
  glPushMatrix();
  glTranslatef(0.2,0.5+0.1,0.25);
  glScalef(0.4,1,0.5);
  glutSolidCube(1.0);
  glPopMatrix();
};

/*
 * draw_big_box()
 *
 * draws a big package: stands for 50 emails
 */
void draw_big_box() {
  set_colour(0.5,0.5,0.1);
  glPushMatrix();
  glTranslatef(0.2,0.5+0.1,0.37);
  glScalef(0.4,1,0.74);
  glutSolidCube(1.0);
  glPopMatrix();
};

/*
 * draw_mail()
 *
 * draws the representation of 'mail_count' emails in the mailbox
 */
void draw_mail() {
  // determine number of each object
  int bbox, box, benv, env;

  if( mail_count>=MAX_MAIL_DRAWABLE ) {
    bbox=2;
    box=benv=env=0;
  } else {
    int mc=mail_count;

    bbox=mc/50;
    mc-=bbox*50;

    box=mc/25;
    mc-=box*25;

    benv=mc/5;
    mc-=benv*5;

    env=mc;
  }

  bool fLL, fLR, fUL, fUR;
  fLL=fLR=fUL=fUR=true;

  if( bbox ) {
    glPushMatrix();
    glTranslatef(0.09,0,0);
    draw_big_box();
    glPopMatrix();
    fLL=fUL=false;

    if( bbox==2 ) {
      glPushMatrix();
      glTranslatef(0.51,0,0);
      draw_big_box();
      glPopMatrix();
      fLR=fUR=false;
    }
  }

  if( box ) {
    glPushMatrix();
    if( fLL ) {
      glTranslatef(0.02,0,0);
      fLL=false;
    } else {
      glTranslatef(0.52,0,0);
      fLR=false;
    }
    draw_box();
    glPopMatrix();
  }

  if( benv || env ) {
    glPushMatrix();
    if( fLL ) {
      fLL=false;
    } else if( fLR ) {
      glTranslatef(0.5,0,0);
      fLR=false;
    } else if( fUR ) {
      glTranslatef(0.5,0,0.5);
      fUR=false;
    }

    for( int i=0; i<benv; i++ ) {
      glTranslatef( 0.05,0,0 );
      draw_big_env();
    }
    for( int i=0; i<env; i++ ) {
      glTranslatef( 0.05,0,0 );
      draw_env();
    }

    glPopMatrix();
  }
}

/*
 * draw_mbox()
 *
 * draws the complete mailbox; post, flag and contained email included
 */
void draw_mbox() {
  set_colour(0.5,0.5,0.5);

  /* scale the whole thing */
  glPushMatrix();
  glScalef(1,1.6,1);

  /* draw the top */
  glPushMatrix();
  glTranslatef(0,0,SIDE_HEIGHT);
  draw_mbox_top();
  glPopMatrix();

  draw_mbox_side( -1, 0, 0 );

  glPushMatrix();
  glTranslatef(1,0,0);
  draw_mbox_side( 1, 0, 0 );
  glPopMatrix();

  /* draw the bottom */
  glBegin(GL_POLYGON);
    glNormal3f(0,0,1);
    glVertex3f(0,0,0);
    glVertex3f(1,0,0);
    glVertex3f(1,1,0);
    glVertex3f(0,1,0);
  glEnd();

  // draw the back door
  glPushMatrix();
  glTranslatef(0,1,0);
  draw_mbox_door();
  glPopMatrix();

  glPopMatrix();
    
  // draw the front door
  glPushMatrix();
  glRotatef(door_position,1,0,0);
  draw_mbox_door();
  glPopMatrix();

  // draw the flag in red
  set_colour(1.0,0,0);
  glPushMatrix();
  glScalef(1.0,1.2,1.0);
  glTranslatef(1.1,0.2,0.5);
  glRotatef(90,0,0,1);
  draw_mbox_flag();
  glPopMatrix();

  // draw the mail count
  draw_mail();
}


/*
 * draw_land()
 *
 * draws the square which represents the ground
 */
void draw_land() {
  set_colour(0.0,0.6,0.0);
  glBegin(GL_POLYGON);
    glNormal3f(0,0,1);
    glVertex3f(-LAND_WIDTH/2,-LAND_WIDTH/2,-3);
    glVertex3f(-LAND_WIDTH/2,LAND_WIDTH/2,-3);
    glVertex3f(LAND_WIDTH/2,LAND_WIDTH/2,-3);
    glVertex3f(LAND_WIDTH/2,-LAND_WIDTH/2,-3);
  glEnd();
}

/*
 * draw_post()
 *
 * draws the post on which the mailbox is sitting
 */
void draw_post() {
  set_colour(0.4,0.4,0);

  glPushMatrix();
  glTranslatef(0.5,0.5*1.6,-1.51);
  glScalef(0.2,0.2,3);
  glutSolidCube(1.0);
  glPopMatrix();
}

/*
 * display()
 *
 * this is the heart of the displaying routines; this function is called
 * whenver the physical status of the mailbox changes, or whenever the
 * window is to be refreshed (say, after a resize operation)
 *
 * NOTE: if it was called because of a physical status change (flag or
 * door opening or closing), this routine will not return until all
 * things come to rest again.
 */
void display(void)
{
  static float rot=0;
  static bool incr=true;

  bool fAnotherDisplay = false;

  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  glPushMatrix();
  glRotatef(-45+xform_factor/2, 0.0, 0.0, 1.0);
  glTranslatef( -(90-xform_factor)/90, -xform_factor/22.5, xform_factor/720);
  glScalef(2,2,2);

  if( unreadmail && flag_position<90 ) {
    flag_position+=2;
    fAnotherDisplay = true;
  }
  if( !unreadmail && flag_position>0 ) {
    flag_position-=2;
    fAnotherDisplay = true;
  }

  if( fDoorOpen && door_position<90 ) {
    door_position+=2;
    fAnotherDisplay = true;
  }
  if( !fDoorOpen && door_position>0 ) {
    door_position-=2;
    fAnotherDisplay = true;
  }

  if( fXformOn && xform_factor<90 ) {
    xform_factor+=2;
    fAnotherDisplay = true;
  }
  if( !fXformOn && xform_factor>0 ) {
    xform_factor-=2;
    fAnotherDisplay = true;
  }

  draw_land();
  draw_post();
  draw_mbox();

  glPopMatrix();
  glFinish();
  glutSwapBuffers();

  if( fAnotherDisplay )
    display();
}

