/*
 * ratDbFolder.c --
 *
 *      This file contains code which implements standard c-client folders.
 *
 * TkRat software and its included text is Copyright 1996,1997,1998
 * by Martin Forssn
 *
 * Postilion software and its included text and images
 * Copyright (C) 1998 Nic Bernstein
 *
 * The full text of the legal notices is contained in the files called
 * COPYING and COPYRIGHT.TkRat, included with this distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <time.h>
#include "ratFolder.h"


/*
 * This is the private part of a Db folder info structure.
 */

typedef struct DbFolderInfo {
    int *listPtr;		/* List of messages in this folder */
    char *searchExpr;		/* The search expression used to create
				 * this folder. */
    char *keywords;		/* Keywords to add to inserted messages */
    char *exDate;		/* Expiration date of inserted messages */
    char *exType;		/* Expiration type of new messages */
} DbFolderInfo;

typedef enum {Db_Name, Db_Mail} DbAdrInfo;

/*
 * Procedures private to this module.
 */
static RatCloseProc Db_CloseProc;
static RatUpdateProc Db_UpdateProc;
static RatInsertProc Db_InsertProc;
static RatSetFlagProc Db_SetFlagProc;
static RatGetFlagProc Db_GetFlagProc;
static RatCreateProc Db_CreateProc;
static int GetAddressInfo(Tcl_Interp *interp, Tcl_DString *dsPtr, char *adr,
	DbAdrInfo info);


/*
 *----------------------------------------------------------------------
 *
 * RatDbFolderInit --
 *
 *      Initializes the dbase folder data.
 *
 * Results:
 *      The return value is normally TCL_OK; if something goes wrong
 *	TCL_ERROR is returned and an error message will be left in
 *	interp->result.
 *
 * Side effects:
 *	The C-client library is initialized and the apropriate mail drivers
 *	are linked.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatDbFolderInit(Tcl_Interp *interp)
{
    Tcl_CreateCommand(interp, "RatInsert", RatInsertCmd, (ClientData) NULL,
	    (Tcl_CmdDeleteProc *) NULL);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RatDbFolderCreate --
 *
 *      Creates a db folder entity.
 *
 * Results:
 *      The return value is normally TCL_OK; if something goes wrong
 *	TCL_ERROR is returned and an error message will be left in
 *	interp->result.
 *
 * Side effects:
 *	A db folder is created.
 *
 *
 *----------------------------------------------------------------------
 */

RatFolderInfo*
RatDbFolderCreate(Tcl_Interp *interp, int argc, char *argv[])
{
    RatFolderInfo *infoPtr;
    DbFolderInfo *dbPtr;
    int *listPtr, number, i;
    Tcl_DString expression;

    if (argc < 6) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" ", argv[1], " keywords exDate exType search_expression\"",
		(char *) NULL);
	return (RatFolderInfo *) NULL;
    }
    Tcl_DStringInit(&expression);
    for (i=5; i < argc; i++) {
	Tcl_DStringAppendElement(&expression, argv[i]);
    }
    if (TCL_OK != RatDbSearch(interp, Tcl_DStringValue(&expression),
	    &number, &listPtr)) {
	Tcl_VarEval(interp, "RatLog 4 \"$t(dbase_error): ", interp->result,
		"\" time", (char*)NULL);
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "Failed to search dbase \"",
		Tcl_DStringValue(&expression), "\"", (char *) NULL);
	Tcl_DStringFree(&expression);
	return (RatFolderInfo *) NULL;
    }

    infoPtr = (RatFolderInfo *) ckalloc(sizeof(*infoPtr));
    dbPtr = (DbFolderInfo *) ckalloc(sizeof(*dbPtr));

    infoPtr->name = cpystr("Database search");
    infoPtr->type = "dbase";
    infoPtr->number = number;
    infoPtr->size = 0;
    for (i=0; i<number; i++) {
	infoPtr->size += atoi(RatDbGetEntry(listPtr[i])->content[RSIZE]);
    }
    infoPtr->closeProc = Db_CloseProc;
    infoPtr->updateProc = Db_UpdateProc;
    infoPtr->insertProc = Db_InsertProc;
    infoPtr->setFlagProc = Db_SetFlagProc;
    infoPtr->getFlagProc = Db_GetFlagProc;
    infoPtr->infoProc = Db_InfoProc;
    infoPtr->createProc = Db_CreateProc;
    infoPtr->private = (ClientData) dbPtr;
    dbPtr->listPtr = listPtr;
    dbPtr->searchExpr = cpystr(Tcl_DStringValue(&expression));
    dbPtr->keywords = cpystr(argv[2]);
    dbPtr->exDate = cpystr(argv[3]);
    dbPtr->exType = cpystr(argv[4]);
    Tcl_DStringFree(&expression);

    return infoPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_CloseProc --
 *
 *      See the documentation for closeProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for closeProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Db_CloseProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp)
{
    DbFolderInfo *dbPtr = (DbFolderInfo *) infoPtr->private;
    free(dbPtr->listPtr);
    free(dbPtr->searchExpr);
    free(dbPtr->keywords);
    free(dbPtr->exDate);
    free(dbPtr->exType);
    free(dbPtr);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_UpdateProc --
 *
 *      See the documentation for updateProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for updateProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Db_UpdateProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int expunge)
{
    DbFolderInfo *dbPtr = (DbFolderInfo *) infoPtr->private;
    int *listPtr, number;

    if (expunge) {
	RatDbEntry *entryPtr;
	int i, dst;

	if (TCL_OK != RatDbExpunge(interp)) {
	    return -1;
	}
	infoPtr->size = 0;
	for (i=dst=0; i<infoPtr->number; i++) {
	    if ((entryPtr = RatDbGetEntry(dbPtr->listPtr[i]))) {
		dbPtr->listPtr[dst] = dbPtr->listPtr[i];
		infoPtr->msgCmdPtr[dst++] = infoPtr->msgCmdPtr[i];
		infoPtr->size += atoi(entryPtr->content[RSIZE]);
	    } else {
		if (infoPtr->msgCmdPtr[i]) {
		    RatMessageDelete(interp, infoPtr->msgCmdPtr[i]);
		}
	    }
	}
	infoPtr->number = dst;
    }
    if (TCL_OK != RatDbSearch(interp, dbPtr->searchExpr, &number, &listPtr)) {
	Tcl_VarEval(interp, "RatLog 4 \"$t(dbase_error): ", interp->result,
		"\" time", (char*)NULL);
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "Failed to search dbase \"",
		dbPtr->searchExpr, "\"", (char *) NULL);
	return -1;
    }
    if (number >= infoPtr->number) {
	ckfree(dbPtr->listPtr);
	dbPtr->listPtr = listPtr;
    }
    return infoPtr->number;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_InsertProc --
 *
 *      See the documentation for insertProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for insertProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Db_InsertProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int argc,
	char *argv[])
{
    DbFolderInfo *dbPtr = (DbFolderInfo *) infoPtr->private;
    Tcl_CmdInfo cmdInfo;
    int i;

    for (i=0; i<argc; i++) {
	if (0 == Tcl_GetCommandInfo(interp, argv[i], &cmdInfo)) {
	    Tcl_AppendResult(interp, "No such message: ", argv[i], NULL);
	    return TCL_ERROR;
	}
	RatInsertMsg(interp, (MessageInfo*)cmdInfo.clientData,
		dbPtr->keywords, dbPtr->exDate, dbPtr->exType);
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_SetFlagProc --
 *
 *      See the documentation for setFlagProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for setFlagProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Db_SetFlagProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int index,
	RatFlag flag, int value)
{
    DbFolderInfo *dbPtr = (DbFolderInfo *) infoPtr->private;
    RatDbEntry *entryPtr = RatDbGetEntry(dbPtr->listPtr[index]);
    char newStatus[5];
    int flagArray[4] = {0,0,0,0};
    int dst, i;

    for (i=0; entryPtr->content[STATUS][i]; i++) {
	switch(entryPtr->content[STATUS][i]) {
	    case 'R':	flagArray[RAT_SEEN] = 1; break;
	    case 'D':	flagArray[RAT_DELETED] = 1; break;
	    case 'F':	flagArray[RAT_FLAGGED] = 1; break;
	    case 'A':	flagArray[RAT_ANSWERED] = 1; break;
	}
    }
    flagArray[flag] = value;
    dst = 0;
    if (flagArray[RAT_SEEN]) { newStatus[dst++] = 'R'; }
    if (flagArray[RAT_DELETED]) { newStatus[dst++] = 'D'; }
    if (flagArray[RAT_FLAGGED]) { newStatus[dst++] = 'F'; }
    if (flagArray[RAT_ANSWERED]) { newStatus[dst++] = 'A'; }
    newStatus[dst] = '\0';
    return RatDbSetStatus(interp, dbPtr->listPtr[index], newStatus);
}


/*
 *----------------------------------------------------------------------
 *
 * Db_GetFlagProc --
 *
 *      See the documentation for getFlagProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for getFlagProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Db_GetFlagProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int index,
	RatFlag flag)
{
    DbFolderInfo *dbPtr = (DbFolderInfo *) infoPtr->private;
    RatDbEntry *entryPtr = RatDbGetEntry(dbPtr->listPtr[index]);
    char flagChar;
    int i;

    switch(flag) {
	case RAT_SEEN:		flagChar = 'R'; break;
	case RAT_DELETED:	flagChar = 'D'; break;
	case RAT_FLAGGED:	flagChar = 'F'; break;
	case RAT_ANSWERED:	flagChar = 'A'; break;
	default:		return 0;
    }
    for (i=0; entryPtr->content[STATUS][i]; i++) {
	if (entryPtr->content[STATUS][i] == flagChar) {
	    return 1;
	}
    }
    return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * GetAddressInfo --
 *
 *      Gets info from an address. The info argument decides what we
 *	wants to extract. The requested data will be appeded to dsPtr.
 *
 * Results:
 *	Returns true if this address points to me.
 *
 * Side effects:
 *	The dsPtr DString will be modified
 *
 *
 *----------------------------------------------------------------------
 */

static int
GetAddressInfo(Tcl_Interp *interp, Tcl_DString *dsPtr, char *adr,
	DbAdrInfo info)
{
    ADDRESS *addressPtr = NULL;
    char *s = cpystr(adr);
    int ret;

    rfc822_parse_adrlist(&addressPtr, s, currentHost);
    free(s);
    if (!addressPtr) {
	return 0;
    }
    ret = RatAddressIsMe(interp, addressPtr, 1);

    if (Db_Name == info && addressPtr->personal) {
	Tcl_DStringAppend(dsPtr, addressPtr->personal, -1);
    } else {
	Tcl_DStringAppend(dsPtr, RatAddressMail(addressPtr), -1);
    }
    mail_free_address(&addressPtr);
    return ret;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_InfoProc --
 *
 *      See the documentation for infoProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for infoProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */

char*
Db_InfoProc(Tcl_Interp *interp, ClientData clientData, RatFolderInfoType type,
	int index)
{
    static Tcl_DString ds;
    static int initialized = 0;
    static char buf[1024];
    int i, seen, deleted, marked, answered, me, dbIndex, zone;
    RatFolderInfo *infoPtr = (RatFolderInfo*)clientData;
    DbFolderInfo *dbPtr = (DbFolderInfo*)infoPtr->private;
    ADDRESS *addressPtr, *address2Ptr;
    MESSAGECACHE elt;
    RatDbEntry *entryPtr;
    struct tm *tmPtr;
    Tcl_CmdInfo info;
    time_t time;

    if (index >= 0) {
	dbIndex = dbPtr->listPtr[infoPtr->presentationOrder[index]];
    } else {
	dbIndex = -index;
    }
    entryPtr = RatDbGetEntry(dbIndex);

    if (!initialized) {
	Tcl_DStringInit(&ds);
	initialized = 1;
    }

    switch (type) {
	case RAT_FOLDER_SUBJECT:
	    return entryPtr->content[SUBJECT];
	case RAT_FOLDER_NAME:
	    Tcl_DStringSetLength(&ds, 0);
	    if (GetAddressInfo(interp, &ds, entryPtr->content[FROM], Db_Name)) {
		Tcl_DStringSetLength(&ds, 0);
		Tcl_DStringAppend(&ds,
			Tcl_GetVar2(interp, "t", "to", TCL_GLOBAL_ONLY), -1);
		Tcl_DStringAppend(&ds, ": ", 2);
		GetAddressInfo(interp, &ds, entryPtr->content[TO], Db_Name);
	    }
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_MAIL_REAL:
	    Tcl_DStringSetLength(&ds, 0);
	    GetAddressInfo(interp, &ds, entryPtr->content[FROM], Db_Mail);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_MAIL:
	    Tcl_DStringSetLength(&ds, 0);
	    if (GetAddressInfo(interp, &ds, entryPtr->content[FROM], Db_Mail)) {
		Tcl_DStringSetLength(&ds, 0);
		Tcl_DStringAppend(&ds,
			Tcl_GetVar2(interp, "t", "to", TCL_GLOBAL_ONLY), -1);
		Tcl_DStringAppend(&ds, ": ", 2);
		GetAddressInfo(interp, &ds, entryPtr->content[TO], Db_Mail);
	    }
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_NAME_RECIPIENT:
	    if (RatIsEmpty(entryPtr->content[TO])) {
		return entryPtr->content[TO];
	    }
	    Tcl_DStringSetLength(&ds, 0);
	    if (GetAddressInfo(interp, &ds, entryPtr->content[TO], Db_Name)) {
		Tcl_DStringSetLength(&ds, 0);
		Tcl_DStringAppend(&ds,
			Tcl_GetVar2(interp, "t", "from", TCL_GLOBAL_ONLY), -1);
		Tcl_DStringAppend(&ds, ": ", 2);
		GetAddressInfo(interp, &ds, entryPtr->content[FROM], Db_Name);
	    }
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_MAIL_RECIPIENT:
	    if (RatIsEmpty(entryPtr->content[TO])) {
		return entryPtr->content[TO];
	    }
	    Tcl_DStringSetLength(&ds, 0);
	    if (GetAddressInfo(interp, &ds, entryPtr->content[TO], Db_Mail)) {
		Tcl_DStringSetLength(&ds, 0);
		Tcl_DStringAppend(&ds,
			Tcl_GetVar2(interp, "t", "from", TCL_GLOBAL_ONLY), -1);
		Tcl_DStringAppend(&ds, ": ", 2);
		GetAddressInfo(interp, &ds, entryPtr->content[FROM], Db_Mail);
	    }
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_SIZE:
	    return entryPtr->content[RSIZE];
	case RAT_FOLDER_SIZE_F:
	    return RatMangleNumber(atoi(entryPtr->content[RSIZE]));
	case RAT_FOLDER_DATE_F:
	    time = atoi(entryPtr->content[DATE]);
	    tmPtr = localtime(&time);
	    return RatFormatDate(interp, tmPtr->tm_mon, tmPtr->tm_mday);
	case RAT_FOLDER_DATE_N:
	    return entryPtr->content[DATE];
	case RAT_FOLDER_DATE_IMAP4:
	    Tcl_DStringSetLength(&ds, 256);
	    time = atoi(entryPtr->content[DATE]);
	    tmPtr = localtime(&time);
	    elt.day = tmPtr->tm_mday;
	    elt.month = tmPtr->tm_mon+1;
	    elt.year = tmPtr->tm_year+1900-BASEYEAR;
	    elt.hours = tmPtr->tm_hour;
	    elt.minutes = tmPtr->tm_min;
	    elt.seconds = tmPtr->tm_sec;
	    zone = RatGetTimeZone(time);
	    if (zone >= 0) {
		elt.zoccident = 1;
		elt.zhours = zone/(60*60);
		elt.zminutes = (zone%(60*60))/60;
	    } else {
		elt.zoccident = 0;
		elt.zhours = (-1*zone)/(60*60);
		elt.zminutes = ((-1*zone)%(60*60))/60;
	    }
	    return mail_date(Tcl_DStringValue(&ds), &elt);
	case RAT_FOLDER_STATUS:
	    seen = deleted = marked = answered = me = 0;
	    for (i=0; entryPtr->content[STATUS][i]; i++) {
		switch (entryPtr->content[STATUS][i]) {
		    case 'R': seen = 1;		break;
		    case 'D': deleted = 1;	break;
		    case 'F': marked = 1;	break;
		    case 'A': answered = 1;	break;
		}
	    }
	    addressPtr = NULL;
	    if (!RatIsEmpty(entryPtr->content[TO])) {
		Tcl_DStringSetLength(&ds, 0);
		Tcl_DStringAppend(&ds, entryPtr->content[TO], -1);
		rfc822_parse_adrlist(&addressPtr, Tcl_DStringValue(&ds),
			currentHost);
		for (address2Ptr = addressPtr; !me && address2Ptr;
			address2Ptr = address2Ptr->next) {
		    if (RatAddressIsMe(interp, address2Ptr, 1)) {
			me = 1;
		    }
		}
	    }
	    mail_free_address(&addressPtr);
	    i = 0;
	    if (!seen) {
		buf[i++] = 'N';
	    }
	    if (deleted) {
		buf[i++] = 'D';
	    }
	    if (marked) {
		buf[i++] = 'F';
	    }
	    if (answered) {
		buf[i++] = 'A';
	    }
	    if (me) {
		buf[i++] = '+';
	    } else {
		buf[i++] = ' ';
	    }
	    buf[i] = '\0';
	    return buf;
	case RAT_FOLDER_TYPE:		
	    return NULL;
	case RAT_FOLDER_PARAMETERS:
	    return NULL;
	case RAT_FOLDER_INDEX:
	    for (i=0; i < infoPtr->number; i++) {
		if (dbIndex == dbPtr->listPtr[infoPtr->presentationOrder[i]]) {
		    sprintf(buf, "%d", i+1);
		    return buf;
		}
	    }
	    return "1";
	case RAT_FOLDER_TO:
	    return entryPtr->content[TO];
	case RAT_FOLDER_FROM:
	    return entryPtr->content[FROM];
	case RAT_FOLDER_SENDER:		/* fallthrough */
	case RAT_FOLDER_CC:		/* fallthrough */
	case RAT_FOLDER_REPLY_TO:
	    if (NULL == infoPtr->msgCmdPtr[index]) {
		infoPtr->msgCmdPtr[index] = Db_CreateProc(infoPtr,interp,index);
	    }
	    Tcl_GetCommandInfo(interp, infoPtr->msgCmdPtr[index], &info);
	    return (*infoPtr->infoProc)(interp, info.clientData, type, index);
	case RAT_FOLDER_FLAGS:
	    Tcl_DStringSetLength(&ds, 0);
	    for (i=0; entryPtr->content[STATUS][i]; i++) {
		switch (entryPtr->content[STATUS][i]) {
		    case 'R':
			Tcl_DStringAppend(&ds, " " RAT_SEEN_STR, -1); break;
		    case 'D':
			Tcl_DStringAppend(&ds, " " RAT_DELETED_STR, -1); break;
		    case 'F':
			Tcl_DStringAppend(&ds, " " RAT_FLAGGED_STR, -1); break;
		    case 'A':
			Tcl_DStringAppend(&ds, " " RAT_ANSWERED_STR, -1); break;
		}
	    }
	    if (Tcl_DStringLength(&ds)) {
		return Tcl_DStringValue(&ds)+1;
	    } else {
		return "";
	    }
	case RAT_FOLDER_END:
	    break;
    }
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Db_CreateProc --
 *
 *      See the documentation for createProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for createProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static char*
Db_CreateProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int index)
{
    DbFolderInfo *dbPtr = (DbFolderInfo *) infoPtr->private;
    return RatDbMessageCreate(interp, infoPtr, dbPtr->listPtr[index]);
}
