/*
   Egon Animator
   Copyright (C) 1997  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * buffer.c
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/xpm.h>

#include "../xcommon/Animator.h"
#include "egon.h"
#include "../common/fonts.h"
#include "../xcommon/embed.h"
#include "../common/cmalloc.h"

/*
   Module name: buffer.c

   This module handles creating and deleting buffers.  It also contains
   a function for finding a certain buffer, given its name.
*/

buffer *b_list;

static int unique_name(char *p)
{
	buffer *b = b_list;

	if (b) {
		do {
			b = b->next;
			if (!strcmp(p, b->name)) return 0;
		} while (b != b_list);
	}
	return 1;
}

char *buffer_name(char *fn)
{
	static char name[256];
	int i;

	char *p = strrchr(fn, '/');
	if (p != NULL) p++;
	if (p == NULL || *p == '\0') p = fn;
	if (unique_name(p)) {
		strcpy(name, p);
		return name;
	}
	for (i = 2; i < 100; i++) {
		sprintf(name, "%s<%d>", p, i);
		if (unique_name(name))
			return name;
	}
	return name;	/* unlikely and no problem */
}

/*
   buffer *new_buffer(char *name, char *path)
   Creates a new buffer with name as name.  It is the responsability of
   the caller to make sure that the name is unique.  The new buffer is
   inserted in the buffer list.
*/
buffer *new_buffer(char *name, char *path)
{
	buffer *b;

	b = (buffer *)cmalloc(sizeof(buffer));

	if (b_list == NULL)
		b_list = b;	/* first buffer ever */
	else
		b->next = b_list->next;		/* add to list */
	b_list->next = b;

	b->cast = NULL;
	strcpy(b->name, name);
	strcpy(b->path, path);
	b->delta = 100;
	b->duration = 1000;
	b->width = 600;
	b->height = 400;
	b->bg = NULL;
	b->change = FALSE;
	b->now = 0;
	b->state = ANI_STOP;
	b->plugin = NULL;
	b->nplugin = 0;
	return b;
}

/*
   buffer *free_buffer(buffer *b)
   Removes b from the buffer list and frees the memory.
   If b_list points to b, it is set to point to the next buffer.
   If the buffer list is empty, b_list is set to NULL.
   Returns the next buffer in the buffer list, or NULL if it is empty.
*/
buffer *free_buffer(buffer *b)
{
	buffer *pb, *next;

	next = b->next;
	if (next == b)
		next = NULL;

	/* unlink from buffer list */
	for (pb = b_list;
	     pb->next != b && pb->next != pb;
	     pb = pb->next);
	pb->next = b->next;

	/* make sure b_list does not point to a deleted buffer */
	if (b_list == b)
		b_list = b_list->next;

	/* no buffers in the list => b_list = NULL */
	if (b_list == b)
		b_list = NULL;
	cfree((char *) b);

	return next;
}

static int plugin_name_taken(char *name)
{
        buffer *b = b_list;
        int i;

        do {
                for (i = 0; i < b->nplugin; i++)
                        if (!strcmp(b->plugin[i].name, name)) return 1;
                b = b->next;
        } while (b != b_list);
        return 0;
}

/* make a unique plugin name */
void plugin_unique_name(char *from, char *to)
{
        int i;

        strcpy(to, from);
        for (i = 2; i < 100 && plugin_name_taken(to); i++)
                sprintf(to, "%d_%s", i, from);
}

/* figure out where to put the plugins */
char *plugin_basedir(buffer *buf, char *dir)
{
        int i;
        static char b[1024];
        pid_t pid = getpid();

        if (!dir) dir = b;
        sprintf(dir, "%s/%ld/%s", siag_basedir, (long)pid, buf->name);
        /* now, we don't want to keep any ugly characters */
        for (i = strlen(dir)-strlen(buf->name); dir[i]; i++)
                if (!isalnum(dir[i])) dir[i] = '_';
        return dir;
}

/*
   buffer *find_buffer_by_name(char *name)
   Returns the buffer with the name name, or NULL if it can't be found.
*/
buffer *find_buffer_by_name(char *name)
{
	buffer *b;

	if ((b = b_list) == NULL)
		return NULL;

	do {
		if (!strcmp(name, b->name))
			return b;
		b = b->next;
	} while (b != b_list);

	return NULL;
}

int buffer_plugin2index(buffer *buf, int ph)
{
        int n;

        if (!buf) return -1;

        for (n = 0; n < buf->nplugin; n++)
                if (ph == buf->plugin[n].ph) return n;

        return -1;
}

