/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for control of GF1 chip (synthesizer things)
 */

#include "driver.h"
#include "tables.h"

#define MAX_INSTRUMENT_MODES	GUS_INSTR_COUNT

#define GF1_SYNTH_VOICES()	(card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].voices)
#define GF1_SYNTH_VOICE_MIN()	(card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].min)
#define GF1_SYNTH_VOICE_MAX()	(card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].max)

static struct GUS_STRU_INSTRUMENT_VOICE_COMMANDS *gus_instrument_voice_commands[ MAX_INSTRUMENT_MODES ];
static struct GUS_STRU_INSTRUMENT_NOTE_COMMANDS *gus_instrument_note_commands[ MAX_INSTRUMENT_MODES ];
static struct GUS_STRU_INSTRUMENT_CHANNEL_COMMANDS *gus_instrument_channel_commands[ MAX_INSTRUMENT_MODES ];

void gus_engine_init( gus_card_t *card )
{
  int i;

  card -> gf1.mode = GF1_MODE_NONE;
  gus_set_default_handlers( card, GF1_HANDLER_SYNTH_DMA_WRITE | GF1_HANDLER_RANGE | GF1_VOICE_RANGE_SYNTH );
  for ( i = 0; i < MAX_INSTRUMENT_MODES; i++ )
    {
      gus_instrument_voice_commands[ i ] = NULL;
      gus_instrument_note_commands[ i ] = NULL;
      gus_instrument_channel_commands[ i ] = NULL;
    }
  gus_gf1_simple_init( card );
  gus_gf1_patch_init( card );
  gus_gf1_interwave_init( card );
}

void gus_engine_instrument_register(
		unsigned short mode,
		struct GUS_STRU_INSTRUMENT_VOICE_COMMANDS *voice_cmds,
		struct GUS_STRU_INSTRUMENT_NOTE_COMMANDS *note_cmds,
		struct GUS_STRU_INSTRUMENT_CHANNEL_COMMANDS *channel_cmds )
{
  gus_instrument_voice_commands[ mode ] = voice_cmds;
  gus_instrument_note_commands[ mode ] = note_cmds;
  gus_instrument_channel_commands[ mode ] = channel_cmds;
}

int gus_engine_instrument_register_ask( unsigned short mode )
{
  if ( mode >= GUS_INSTR_COUNT ) return 0;
  return gus_instrument_voice_commands[ mode ] != NULL;
}

void gus_engine_voice_program( gus_card_t *card, gus_voice_t *voice, unsigned int program )
{
  gus_instrument_t *instrument;
  
  if ( voice -> instrument &&
       voice -> instrument -> number.instrument == program ) return;	/* already */
  while ( 1 )
    {
      instrument = gus_instrument_look( &card -> gf1.mem_alloc, program );
      if ( !instrument || (instrument -> flags & GUS_INSTR_F_NOT_FOUND) )
        {
          voice -> instrument = NULL;
          voice -> commands = NULL;
          return;		/* grrr. this instrument isn't loaded */
        }
      if ( !(instrument -> flags & GUS_INSTR_F_ALIAS) ) break;
      program = instrument -> info.alias;
    }
  voice -> commands = gus_instrument_voice_commands[ instrument -> mode ];
  voice -> instrument = voice -> commands ? instrument : NULL;
}

void gus_engine_channel_program( gus_card_t *card, gus_channel_t *channel, unsigned int program )
{
  gus_instrument_t *instrument;
  
  program &= 0xffff;
  program |= (unsigned int)channel -> bank << 16;
  if ( channel -> instrument &&
       channel -> instrument -> number.instrument == program ) return;	/* already */
  while ( 1 )
    {
      instrument = gus_instrument_look( &card -> gf1.mem_alloc, program );
      if ( !instrument || (instrument -> flags & GUS_INSTR_F_NOT_FOUND) )
        {
          if ( !instrument && ( program & 0xffff0000 ) )	/* some bank? - try use first */
            {
#if defined( GUSCFG_DEBUG_INSTRUMENTS ) && 0
              PRINTK( "gus: bank not found %i/%i\n", program >> 16, program & 0xffff );
#endif
              program &= 0x0000ffff;
              continue;
            }
          channel -> instrument = NULL;
          channel -> commands = NULL;
          return; 	/* grrr. this instrument isn't loaded */
        }
      if ( !(instrument -> flags & GUS_INSTR_F_ALIAS) ) break;
      program = instrument -> info.alias;
    }
  channel -> commands = gus_instrument_channel_commands[ instrument -> mode ];
  channel -> instrument = channel -> commands ? instrument : NULL;
}

void gus_engine_channel_program_drum( gus_card_t *card, gus_channel_t *channel, unsigned int program )
{
  if ( channel -> number == GUS_DRUM_CHANNEL )
    {
      if ( card -> gf1.midi_emul == GUS_MIDI_EMUL_GS )
        channel -> bank = program & 0x7f;
      return;
    }
  return gus_engine_channel_program( card, channel, program );
}

/*
 *  channel specific commands
 */

#ifdef GUSCFG_DEBUG
void gus_show_voice_allocation( gus_card_t *card )
{
  unsigned long flags;
  gus_channel_t *channel;
  gus_note_t *note = channel -> notes;
  gus_voice_t *voice;

  CLI( &flags );
  for ( channel = card -> gf1.syn_channels; channel; channel = channel -> next )
    {
      for ( note = channel -> notes; note; note = note -> next )
        {
          printk( "<%i> note = 0x%lx\n", channel -> number, (long)note );            
          printk( "<%i> note -> voices = 0x%lx\n", channel -> number, (long)note -> voices );
          for ( voice = note -> voices; voice; voice = voice -> next )
            {
              printk( "  -%i- voice -> flags = 0x%x\n", voice -> number, voice -> flags );
              printk( "  -%i- voice -> instrument = 0x%lx\n", voice -> number, (long)voice -> instrument );
              if ( voice -> flags & VFLG_USED )
                {
                  gf1_select_voice( card, voice -> number );
                  gus_print_voice_registers( card );
                }
            }
          printk( "<%i> note -> channel = 0x%lx\n", channel -> number, (long)note -> channel );
          printk( "<%i> note -> commands = 0x%lx\n", channel -> number, (long)note -> commands );
        }
    }
  STI( &flags );
}
#endif

gus_voice_t *gus_voice_alloc( gus_card_t *card, gus_note_t *note, unsigned char priority, void (*steal_notify)( gus_card_t *card, gus_voice_t *voice ) )
{
  unsigned int maxpriority = 1;
  unsigned int minage = ~0;
  gus_voice_t *voice, *steal_voice = NULL;
  unsigned long flags;
    
#if 0
  for ( voice = card -> gf1.syn_voice_first; voice; voice = voice -> anext )
    if ( voice -> flags & VFLG_USED ) 
      printk( "%i,", voice -> instrument -> number.instrument );
     else
    if ( voice -> flags & VFLG_DYNAMIC )
      printk( "-," );
     else
      printk( "S," );
  printk( "\n" );
#endif
  CLI( &flags );
  for ( voice = card -> gf1.syn_voice_first; voice; voice = voice -> anext )
    {
      if ( !(voice -> flags & VFLG_DYNAMIC) ) continue;
      if ( !(voice -> flags & VFLG_USED) ) goto __found;
      if ( voice -> priority >= maxpriority )
        {
          if ( voice -> priority > maxpriority || voice -> age < minage )
            {
              minage = voice -> age;
              steal_voice = voice;
            }
          maxpriority = voice -> priority;
        }
    }
  if ( steal_voice )
    {
#if 0
      printk( "steal_voice = %i, age = %i, priority = %i\n", steal_voice -> number, steal_voice -> age, steal_voice -> priority );
#endif
      if ( steal_voice -> steal_notify )
        steal_voice -> steal_notify( card, steal_voice );
      voice = steal_voice;
      goto __found;
    }
#if defined( GUSCFG_DEBUG_INSTRUMENTS ) && 1
   PRINTK( "gus_voice_alloc: no free voice?\n" );
#endif
  STI( &flags );
  return NULL;

__found:
  voice -> flags &= VFLG_DYNAMIC;
  if ( !note )
    {
      STI( &flags );
      return voice;	/* ok.. this call is only for free of one voice */
    }
  voice -> flags |= VFLG_USED;
  voice -> instrument = note -> instrument;
  voice -> note = note;
  voice -> steal_notify = steal_notify;
  if ( card -> gf1.timer_slave )
    voice -> age = card -> gf1.timer_master_card -> gf1.timer_abs_tick;
   else
    voice -> age = card -> gf1.timer_abs_tick;
  voice -> priority = priority;
  voice -> next = note -> voices;
  note -> voices = voice;
  STI( &flags );
#if 0
  printk( "voice alloc = 0x%lx, %i, flags = 0x%x, note = 0x%lx\n", (long)voice, voice -> number, voice -> flags, (long)voice -> note );
#endif
  return voice;
}

void gus_voice_free( gus_card_t *card, gus_voice_t *voice )
{
  gus_voice_t *pvoice;
  gus_note_t *note = voice -> note;
  unsigned long flags;
  
#if 0
  printk( "voice free = 0x%lx, voice -> flags = 0x%x, voice -> note = 0x%lx\n", (long)voice, voice -> flags, (long)voice -> note );
#endif
  CLI( &flags );
  if ( !note || !voice )
    {
      STI( &flags );
      return;
    }
  pvoice = note -> voices;
  if ( !pvoice )
    {
      STI( &flags );
      return;
    }
  while ( pvoice && pvoice != voice && pvoice -> next != voice )
    pvoice = pvoice -> next;
  if ( !pvoice )
    {
      STI( &flags );
      return;
    }
  if ( pvoice == voice )
    note -> voices = voice -> next;
   else
    pvoice -> next = voice -> next;
  voice -> flags &= VFLG_DYNAMIC;
  voice -> note = NULL;
  voice -> instrument = NULL;
  voice -> layer = NULL;
  voice -> wave = NULL;
  voice -> next = NULL;
  STI( &flags );
}

gus_note_t *gus_note_alloc( gus_card_t *card, gus_channel_t *channel )
{
  int idx, loop;
  gus_note_t *nnote;
  unsigned long flags;
  
  if ( !channel ) return NULL;
#if 0
  for ( idx = 0; idx < GF1_SYNTH_VOICES(); idx++ )
    {
      nnote = card -> gf1.syn_notes + idx;
      if ( nnote -> flags & NFLG_USED ) 
        printk( "%i,", nnote -> channel -> number );
       else
        printk( "-," );
    }
  printk( "\n" );
#endif
  loop = 0;
  while ( 1 ) {
    CLI( &flags );
    for ( idx = 0, nnote = card -> gf1.syn_notes;
          idx < GF1_SYNTH_VOICES() && ( nnote -> flags & NFLG_USED );
          idx++, nnote++ );
    if ( idx >= GF1_SYNTH_VOICES() )
      {
        STI( &flags );
        if ( !gus_voice_alloc( card, NULL, 0, NULL ) )
          {
#if defined( GUSCFG_DEBUG_INSTRUMENTS ) && 1
            printk( "gus_note_alloc: no free notes?\n" );
#endif
            return NULL;
          }
        if ( ++loop > 32 )
          {
#if defined( GUSCFG_DEBUG_INSTRUMENTS ) && 1
            PRINTK( "gus_note_alloc: note free problem?\n" );
#endif
            return NULL;
          }
        continue;
      }
    break;
  }
  nnote -> flags = NFLG_USED;
  nnote -> channel = channel;
  nnote -> next = channel -> notes;
  channel -> notes = nnote;
  STI( &flags );
#if 0
  printk( "note alloc = 0x%lx\n", (long)nnote );
#endif
  return nnote;
}

void gus_note_free( gus_card_t *card, gus_note_t *note )
{
  gus_note_t *pnote;
  gus_channel_t *channel;
  unsigned long flags;
  
#if 0
  printk( "note free = 0x%lx\n", (long)note );
#endif
  CLI( &flags );
  if ( !note )
    {
      STI( &flags );
      return;
    }
  channel = note -> channel;
  if ( !channel )
    {
      STI( &flags );
      return;
    }
  if ( note -> voices )
    PRINTK( "gus_note_free: some voices is used!!!\n" );
  pnote = channel -> notes;
  if ( !pnote )
    {
#if defined( GUSCFG_DEBUG_INSTRUMENTS ) && 0
      printk( "gus_note_free: invalid channel?\n" );
#endif
      STI( &flags );
      return;
    }
  if ( channel -> notes == note )
    {
      channel -> notes = note -> next;
      goto __free;
    }
  while ( pnote && pnote -> next != note )
    pnote = pnote -> next;
  if ( !pnote )
    {
#ifdef GUSCFG_DEBUG_INSTRUMENTS
      printk( "gus_note_free: note 0x%lx not found..\n", (long)note );
#endif
      STI( &flags );
      return;
    }
  pnote -> next = note -> next;
  __free:
  note -> flags = 0;
  note -> channel = NULL;
  note -> next = NULL;
  STI( &flags );
}

void gus_voice_and_note_free( gus_card_t *card, gus_voice_t *voice )
{
  gus_note_t *note;
  
  if ( voice )
    {
      note = voice -> note;
      gus_voice_free( card, voice );
      if ( note )
        {
          if ( !note -> voices )
            gus_note_free( card, note );
        }
    }
}

void gus_engine_midi_go_note_stop( gus_card_t *card, gus_note_t *note, int wait )
{
  gus_voice_t *voice, *voice_next;
  
  gus_int_disable();
  if ( !note || !note -> commands ) goto __end;
  for ( voice = note -> voices; voice; voice = voice_next )
    {
      voice_next = voice -> next;
      note -> commands -> note_stop( card, voice, !wait );
    }
  if ( wait )
    {
      for ( voice = note -> voices; voice; voice = voice_next )
        {
          voice_next = voice -> next;
          note -> commands -> note_wait( card, voice );
        }
    } 
  __end:
  gus_int_enable();
}

void gus_engine_midi_go_note_off( gus_card_t *card, gus_note_t *note )
{
  gus_voice_t *voice, *next_voice;
  
  gus_int_disable();
  if ( !note || !note -> instrument || !note -> commands ) goto __end;
  if ( note -> instrument -> layer != GUS_INSTR_L_NONE )
    {
      if ( note -> channel && note -> channel -> commands )
        note -> channel -> commands -> chn_trigger_up( card, note );
    }
  for ( voice = note -> voices; voice; voice = next_voice )
    {
      next_voice = voice -> next;
      note -> commands -> note_off( card, voice );
    }
  __end:
  gus_int_enable();
}

void gus_engine_midi_recompute_volume( gus_card_t *card, gus_channel_t *channel )
{
  gus_note_t *note, *note_next;
  gus_voice_t *voice, *voice_next;
  
  gus_int_disable();
  for ( note = channel -> notes; note; note = note_next )
    {
      note_next = note -> next;
      if ( note -> commands )
        for ( voice = note -> voices; voice; voice = voice_next )
          {
            voice_next = voice -> next;
            note -> commands -> note_volume( card, voice );  
          }
    }
  gus_int_enable();
}

void gus_engine_midi_note_on( gus_card_t *card, gus_channel_t *channel, unsigned char note, unsigned char velocity, unsigned char priority )
{
  gus_instrument_t *pinstrument;
  gus_note_t *pnote, *pnote_next;

#if 0
  printk( "note on - chn %i, instr %i, note %i, velocity = %i\n", channel -> number, channel -> instrument ? channel -> instrument -> number.instrument : -1, note, velocity );
#endif
  if ( !velocity )
    {
      gus_engine_midi_note_off( card, channel, note, velocity );
      return;
    }
  if ( channel -> number == GUS_DRUM_CHANNEL )
    gus_engine_channel_program( card, channel, note + 128 );
  gus_int_disable();
  pinstrument = channel -> instrument;
  if ( !pinstrument || !channel -> commands ) goto __end;
  if ( pinstrument -> exclusion != GUS_INSTR_E_NONE )
    for ( pnote = channel -> notes; pnote; pnote = pnote_next )
      {
        pnote_next = pnote -> next;
        if ( pnote -> instrument -> exclusion_group == pinstrument -> exclusion_group )
          {
            if ( pinstrument -> exclusion == GUS_INSTR_E_SINGLE )
              gus_engine_midi_go_note_stop( card, pnote, 0 );
             else
              {
                if ( pnote -> number == note )
                  gus_engine_midi_go_note_stop( card, pnote, 0 );
              }
          }
      }
  if ( channel -> commands )
    channel -> commands -> chn_trigger_down( card, channel, pinstrument, note, velocity, priority );
  if ( channel -> number == GUS_DRUM_CHANNEL )
    {
      channel -> commands = NULL;
      channel -> instrument = NULL;
    }
  __end:
  gus_int_enable();
#if 0
  printk( "note on - end\n" );
#endif
}

void gus_engine_midi_note_off( gus_card_t *card, gus_channel_t *channel, unsigned char note, unsigned char velocity )
{
  gus_note_t *pnote, *pnote_next;

#if 0
  printk( "note off - %i\n", note );
#endif
  gus_int_disable();
  if ( channel -> number == GUS_DRUM_CHANNEL ) goto __end;	/* play instrument to end */
  if ( !channel -> instrument || !channel -> commands ) goto __end;
  if ( channel -> flags & CFLG_SUSTAIN )
    {
      for ( pnote = channel -> notes; pnote; pnote = pnote_next )
        {
          pnote_next = pnote -> next;
          if ( pnote -> number == note && !(pnote -> flags & NFLG_RELEASING) )
            pnote -> flags |= NFLG_SUSTAIN;
        }
      goto __end;
    }
  for ( pnote = channel -> notes; pnote; pnote = pnote_next )
    {
      pnote_next = pnote -> next;
      if ( pnote -> number == note && !(pnote -> flags & NFLG_RELEASING) ) break;
    }
  if ( pnote )
    gus_engine_midi_go_note_off( card, pnote );
  __end:
  gus_int_enable();
}

static void gus_engine_midi_volume( gus_card_t *card, gus_channel_t *channel, unsigned char volume )
{
  channel -> volume = volume < 128 ? volume : 127;
  gus_engine_midi_recompute_volume( card, channel );
}

static void gus_engine_midi_expression( gus_card_t *card, gus_channel_t *channel, unsigned char expression )
{
  channel -> expression = expression < 128 ? expression : 127;
  gus_engine_midi_recompute_volume( card, channel );
}

void gus_engine_midi_pitchbend( gus_card_t *card, gus_channel_t *channel, unsigned short pitchbend )
{  
  gus_note_t *note, *note_next;
  gus_voice_t *voice, *voice_next;

  channel -> pitchbend = pitchbend;
  channel -> gf1_pitchbend = gus_compute_pitchbend( pitchbend, channel -> pitchbend_range );
#if 0
  printk( "pitchbend = %i, range = %i, value = %i\n", pitchbend, channel -> pitchbend_range, channel -> gf1_pitchbend );
#endif
  gus_int_disable();
  for ( note = channel -> notes; note; note = note_next )
    {
      note_next = note -> next;
      if ( note -> commands )
        for ( voice = note -> voices; voice; voice = voice_next )
          {
            voice_next = voice -> next;
            note -> commands -> note_pitchbend( card, voice );
          }
    }
  gus_int_enable();
}

static void gus_engine_midi_sustain( gus_card_t *card, gus_channel_t *channel, unsigned char sustain )
{
  gus_note_t *note, *note_next;
    
  if ( sustain )
    channel -> flags |= CFLG_SUSTAIN;
   else
    {
      gus_int_disable();
      channel -> flags &= ~CFLG_SUSTAIN;
      for ( note = channel -> notes; note; note = note_next )
        {
          note_next = note -> next;
          if ( note -> channel == channel && (note -> flags & NFLG_SUSTAIN) )
            gus_engine_midi_go_note_off( card, note );
        }
      gus_int_enable();
    }
}

static void gus_engine_midi_balance( gus_card_t *card, gus_channel_t *channel, unsigned char balance )
{
  channel -> pan = balance < 128 ? balance : 127;
  gus_engine_midi_recompute_volume( card, channel );
}

static void gus_engine_midi_vibrato( gus_card_t *card, gus_channel_t *channel, unsigned char vibrato )
{
  gus_note_t *note, *note_next;
  gus_voice_t *voice, *voice_next;

  gus_int_disable();
  channel -> vib_depth = vibrato < 128 ? vibrato : 127;
  for ( note = channel -> notes; note; note = note_next )
    {
      note_next = note -> next;
      if ( note -> commands )
        for ( voice = note -> voices; voice; voice = voice_next )
          {
            voice_next = voice -> next;
            note -> commands -> note_vibrato( card, voice );
          }
    }
  gus_int_enable();
}

static void gus_engine_midi_tremolo( gus_card_t *card, gus_channel_t *channel, unsigned char tremolo )
{
  gus_note_t *note, *note_next;
  gus_voice_t *voice, *voice_next;

  gus_int_disable();
  channel -> trem_depth = tremolo < 128 ? tremolo : 127;
  for ( note = channel -> notes; note; note = note_next )
    {
      note_next = note -> next;
      if ( note -> commands )
        for ( voice = note -> voices; voice; voice = voice_next )
          {
            voice_next = voice -> next;
            note -> commands -> note_tremolo( card, voice );
          }
    }
  gus_int_enable();
}

static void gus_engine_midi_legato( gus_card_t *card, gus_channel_t *channel, unsigned char legato )
{
  if ( legato )
    channel -> flags |= CFLG_LEGATO;
   else
    channel -> flags &= ~CFLG_LEGATO;
}

static void gus_engine_midi_notes_off( gus_card_t *card, gus_channel_t *channel )
{
  gus_note_t *note, *note_next;
  
  gus_int_disable();
  for ( note = channel -> notes; note; note = note_next )
    {
      note_next = note -> next;
      if ( !(note -> flags & NFLG_RELEASING) )
        {
          if ( channel -> flags & CFLG_SUSTAIN )
            note -> flags |= NFLG_SUSTAIN;
           else
            gus_engine_midi_go_note_off( card, note );
        }
    }
  gus_int_enable();
}

static void gus_engine_midi_sounds_off( gus_card_t *card, gus_channel_t *channel )
{
  gus_note_t *note, *note_next;
  
  gus_int_disable();
  for ( note = channel -> notes; note; note = note_next )
    {
      note_next = note -> next;
      gus_engine_midi_go_note_stop( card, note, 0 );
    }
  gus_int_enable();
}

void gus_engine_midi_control( gus_card_t *card, gus_channel_t *channel, unsigned char p1, unsigned char p2 )
{
#if 0
  printk( "engine midi control : (%i) 0x%x, 0x%x\n", channel -> number, p1, p2 );
#endif
  gus_int_disable();
  switch ( p1 ) {
    case GUS_MCTL_MSB_BANK:
      if ( card -> gf1.midi_emul == GUS_MIDI_EMUL_GS )
        channel -> bank = p2;
       else
        {
          channel -> bank &= 0x7f;
          channel -> bank |= p2 << 7;
        }
      break;
    case GUS_MCTL_LSB_BANK:
      if ( card -> gf1.midi_emul != GUS_MIDI_EMUL_GS )
        {
          channel -> bank &= ~0x7f;
          channel -> bank |= p2;
        }
      break;
    case GUS_MCTL_MSB_MODWHEEL:
      gus_engine_midi_tremolo( card, channel, p2 );
      gus_engine_midi_vibrato( card, channel, p2 );
      break;
    case GUS_MCTL_MSB_DATA_ENTRY:
      if ( card -> gf1.rpn_or_nrpn == 0 )
        {
          if ( channel -> rpn == 0 )
            {
              if ( p2 > 24 ) p2 = 24;
              channel -> pitchbend_range = ( channel -> pitchbend_range & 0x7f ) | ( ( p2 & 0x7f ) << 7 );
            }
        }
      break;
    case GUS_MCTL_MSB_MAIN_VOLUME:
      gus_engine_midi_volume( card, channel, p2 );
      break;
    case GUS_MCTL_MSB_PAN:
      gus_engine_midi_balance( card, channel, p2 );
      break;
    case GUS_MCTL_MSB_EXPRESSION:
      gus_engine_midi_expression( card, channel, p2 );
      break;
    case GUS_MCTL_LSB_DATA_ENTRY:
      if ( card -> gf1.rpn_or_nrpn == 0 )
        {
          if ( channel -> rpn == 0 )
            channel -> pitchbend_range = ( channel -> pitchbend_range & ~0x7f ) | ( p2 & 0x7f );
        }      
      break;
    case GUS_MCTL_SUSTAIN:
      gus_engine_midi_sustain( card, channel, p2 >= 64 );
      break;
    case GUS_MCTL_DATA_INCREMENT:
      if ( card -> gf1.rpn_or_nrpn == 0 )
        {
          if ( channel -> rpn == 0 )
            {
              channel -> pitchbend_range++;
              if ( channel -> pitchbend_range > ( 24 * 128 ) + 127 )
                channel -> pitchbend_range = ( 24 * 128 ) + 127;
            }
        }
      break;
    case GUS_MCTL_DATA_DECREMENT:
      if ( card -> gf1.rpn_or_nrpn == 0 )
        {
          if ( channel -> rpn == 0 )
            {
              channel -> pitchbend_range--;
              if ( channel -> pitchbend_range > ( 24 * 128 ) + 127 )
                channel -> pitchbend_range = 0;
            }
        }
      break;
    case GUS_MCTL_NONREG_PARM_NUM_LSB:
      card -> gf1.rpn_or_nrpn = 1;
      channel -> nrpn = ( channel -> nrpn & ~0x7f ) | ( p2 & 0x7f );
      break;
    case GUS_MCTL_NONREG_PARM_NUM_MSB:
      card -> gf1.rpn_or_nrpn = 1;
      channel -> nrpn = ( channel -> nrpn & 0x7f ) | ( ( p2 & 0x7f ) << 7 );
      break;
    case GUS_MCTL_REGIST_PARM_NUM_LSB:
      card -> gf1.rpn_or_nrpn = 0;
      channel -> rpn = ( channel -> rpn & ~0x7f ) | ( p2 & 0x7f );
      break;
    case GUS_MCTL_REGIST_PARM_NUM_MSB:
      card -> gf1.rpn_or_nrpn = 0;
      channel -> rpn = ( channel -> rpn & 0x7f ) | ( ( p2 & 0x7f ) << 7 );
      break;
    case GUS_MCTL_ALL_SOUNDS_OFF:
      gus_engine_midi_sounds_off( card, channel );
      break;
    case GUS_MCTL_RESET_CONTROLLERS:
      gus_engine_midi_sustain( card, channel, 0 );
      gus_engine_midi_vibrato( card, channel, 0 );
      gus_engine_midi_tremolo( card, channel, 0 );
      gus_engine_midi_pitchbend( card, channel, 8192 );
      gus_engine_midi_volume( card, channel, 100 );
      gus_engine_midi_expression( card, channel, 127 );
      gus_engine_midi_balance( card, channel, 64 );
      if ( channel -> commands )
        {
          channel -> commands -> chn_control( card, channel, GUS_MCTL_E1_REVERB_DEPTH, 40 );
          channel -> commands -> chn_control( card, channel, GUS_MCTL_E3_CHORUS_DEPTH, 0 );
        }
      break;
    case GUS_MCTL_ALL_NOTES_OFF:
    case GUS_MCTL_OMNI_OFF:
    case GUS_MCTL_OMNI_ON:
    case GUS_MCTL_MONO1:
    case GUS_MCTL_MONO2:
      gus_engine_midi_notes_off( card, channel );
      break;
    case GUS_MCTL_TREMOLO:
      gus_engine_midi_tremolo( card, channel, p2 );
      break;
    case GUS_MCTL_VIBRATO:
      gus_engine_midi_vibrato( card, channel, p2 );
      break;
    case GUS_MCTL_LEGATO:
      gus_engine_midi_legato( card, channel, p2 >= 64 );
      break;
  }
  if ( channel -> commands )
    channel -> commands -> chn_control( card, channel, p1, p2 );
  gus_int_enable();
#if 0
  printk( "engine midi control : end\n" );
#endif
}

/*
 *
 */

void gus_engine_process_effects( gus_card_t *card )
{
  gus_voice_t *voice;
  
  if ( card -> gf1.syn_voices_change ) return;
  for ( voice = card -> gf1.syn_voice_first; voice; voice = voice -> anext )
    {
      if ( card -> gf1.sw_lfo )
        gus_lfo_effect_interrupt( card, voice );
      if ( (voice -> flags & VFLG_USED) &&
           (voice -> flags & VFLG_EFFECT_TIMER) &&
           voice -> commands )
        voice -> commands -> voice_interrupt_effect( card, voice );
    }
}

void gus_engine_process_voices( gus_card_t *card )
{
#if 0
  printk( "gus_engine_process_voices - 0x%x (%i)\n", card, card -> number );
#endif
}

/*
 *  interrupt handlers
 */

static void gus_interrupt_synth_wave( gus_card_t *card, int voice )
{
  struct GUS_STRU_VOICE *pvoice;
  
  pvoice = card -> gf1.syn_voices + voice;
#if 0
  printk( "card - %i, wave - %i (flags = 0x%x, commands = 0x%x)\n", card -> number, voice, pvoice -> flags, pvoice -> commands );
#endif
  if ( !card -> gf1.syn_voices_change && 
       pvoice -> commands && (pvoice -> flags & VFLG_IRQ) )
    pvoice -> commands -> voice_interrupt_wave( card, pvoice );
   else
    gus_gf1_smart_stop_voice( card, voice );
}

static void gus_interrupt_synth_volume( gus_card_t *card, int voice )
{
  struct GUS_STRU_VOICE *pvoice;
  
#if 0
  printk( "card - %i, volume - %i\n", card -> number, voice );
#endif
  pvoice = card -> gf1.syn_voices + voice;
  if ( !card -> gf1.syn_voices_change &&
       pvoice -> commands && (pvoice -> flags & VFLG_IRQ) )
    pvoice -> commands -> voice_interrupt_volume( card, pvoice );
   else
    gus_gf1_smart_stop_voice( card, voice );
}

static void gus_interrupt_synth_dma_write( gus_card_t *card )
{
  gus_gf1_done_dma_transfer( card );
  card -> dmas[ GUS_DMA_GPLAY ] -> lock |= WK_WAKEUP;
  if ( GETLOCK( card, dma1 ) & WK_SLEEP )
    {
      GETLOCK( card, dma1 ) &= ~WK_SLEEP;
      WAKEUP( card, dma1 );
    }
}

/*
 *
 */

void gus_engine_voices_init( gus_card_t *card, int soft )
{
  int i, min, max;
  gus_voice_t *voice, *pvoice;
  gus_note_t *note;
  gus_channel_t *channel;
  unsigned int dynamic = card -> gf1.syn_voice_dynamic;

#if 0
  printk( "voices_init!!!\n" );
#endif
  if ( !card -> gf1.syn_voices ) return;
  min = GF1_SYNTH_VOICE_MIN();
  max = GF1_SYNTH_VOICE_MAX();
  gus_gf1_stop_voices( card, min, max );
  pvoice = card -> gf1.syn_voice_first = NULL;
  for ( i = 0; i < GUS_VOICES; i++ )
    {
      voice = &card -> gf1.syn_voices[ i ];
      MEMSET( voice, 0, sizeof( gus_voice_t ) );
      voice -> number = i;
      note = &card -> gf1.syn_notes[ i ];
      MEMSET( note, 0, sizeof( gus_note_t ) );
      if ( min <= i && i <= max )
        {
          if ( dynamic & ( 1 << ( i - min ) ) )
            voice -> flags |= VFLG_DYNAMIC;
          if ( !card -> gf1.syn_voice_first ) card -> gf1.syn_voice_first = voice;
          if ( pvoice ) pvoice -> anext = voice;
          pvoice = voice;
        }
    }
  for ( i = 0; i < GUS_CHANNELS; i++ )
    {
      channel = &card -> gf1.syn_channels[ i ];
      switch ( soft ) {
        case 0:
          MEMSET( channel, 0, sizeof( gus_channel_t ) );
          channel -> number = i;
          channel -> volume = 100;
          channel -> expression = 127;
          channel -> pitchbend = 8192;
          channel -> gf1_pitchbend = 1024;
          channel -> pan = 64;
          channel -> pitchbend_range = 2 << 7;
          channel -> effect1_level = VOLUME( 40 );
          channel -> effect3_level = 0; 
          channel -> next = i + 1 < GUS_CHANNELS ? channel + 1 : NULL;
          channel -> nrpn = channel -> rpn = 0x3fff;
          break;
        case 1:
          channel -> notes = NULL;
          break;
        case 2:
          channel -> notes = NULL;
          channel -> instrument = NULL;
          break;
      }
    }
  if ( !soft )
    card -> gf1.rpn_or_nrpn = 0;
#if 0
  PRINTK( "voices init done..\n" );
#endif
}

int gus_engine_reset( gus_card_t *card, int voices, unsigned int dynamic )
{
  int res;

  if ( ( res = gus_engine_reset_voices( card, voices, dynamic, 2 ) ) < 0 )
    return res;
  gus_memory_reset( card, &card -> gf1.mem_alloc, GUS_ALLOC_MODE_NORMAL, 0 );
  return 0;
}

int gus_engine_reset_voices( gus_card_t *card, int voices, unsigned int dynamic, int soft )
{
  if ( voices < 1 || voices > 32 ) return -EINVAL;
  card -> gf1.syn_voices_change++;
  if ( voices != card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].rvoices )
    {
      card -> gf1.syn_voice_dynamic = dynamic;
      card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].rvoices = voices;
      gus_reselect_active_voices( card );
    }
   else
    {
      if ( card -> gf1.syn_voice_dynamic != dynamic )
        {
          card -> gf1.syn_voice_dynamic = dynamic;
          gus_engine_voices_init( card, soft );
        }
    }
  card -> gf1.syn_voices_change--;
  return 0;
}

void gus_engine_invalidate_instrument( gus_card_t *card, gus_instrument_t *instrument )
{
  unsigned long flags;
  int voice_number;
  gus_voice_t *voice;
  gus_note_t *note;
  gus_channel_t *channel;
  
  if ( !(card -> gf1.mode & GF1_MODE_ENGINE) ) return;
  if ( card -> gf1.syn_voices_change ) return;
  if ( gus_intr_count )
    {
      PRINTK( "gus: invalidate_instrument called in interrupt!!!\n" );
      return;
    }
#if 0
  PRINTK( "invalidate_instrument %i - start\n", instrument -> number.instrument );
#endif
  gus_int_disable();
  for ( voice_number = 0; voice_number < GUS_VOICES; voice_number++ )
    {
      voice = card -> gf1.syn_voices + voice_number;
      if ( voice -> instrument != instrument ) continue;
      gus_gf1_stop_voice( card, voice -> number );
      if ( voice -> flags & VFLG_DYNAMIC )
        gus_voice_and_note_free( card, voice );
      CLI( &flags );
      voice -> flags &= VFLG_DYNAMIC;
      voice -> commands = NULL;
      voice -> instrument = NULL;
      voice -> layer = NULL;
      voice -> wave = NULL;
      voice -> note = NULL;
      voice -> next = NULL;
      STI( &flags );
    }
  for ( channel = card -> gf1.syn_channels; channel; channel = channel -> next )
    if ( channel -> instrument == instrument )
      {
        if ( channel -> notes )
          {
#if defined( GUSCFG_DEBUG ) && 0
            PRINTK( "gus: invalidate_instrument channel %i free problem...\n", channel -> number );
            gus_show_voice_allocation( card );
#endif
            for ( note = channel -> notes; channel -> notes; note = channel -> notes )
              for ( voice = note -> voices; voice; voice = note -> voices )
                {
                  gus_gf1_stop_voice( card, voice -> number );
                  gus_voice_and_note_free( card, voice );
                }
#if defined( GUSCFG_DEBUG ) && 0
            PRINTK( "gus: invalidate_instrument verify...\n" );
            gus_show_voice_allocation( card );
#endif
          }
        CLI( &flags );
        channel -> commands = NULL;
        channel -> instrument = NULL;
        STI( &flags );
      }
  gus_int_enable();
#if 0
  PRINTK( "invalidate_instrument - stop\n" );
#endif
}

static void gus_engine_voices_change_start( gus_card_t *card )
{
  card -> gf1.syn_voices_change++;
  gus_gf1_stop_voices( card, GF1_SYNTH_VOICE_MIN(), GF1_SYNTH_VOICE_MAX() );
}

static void gus_engine_voices_change_stop( gus_card_t *card )
{
  gus_engine_voices_init( card, 1 );
  card -> gf1.syn_voices_change--;
}

static void gus_engine_volume_change( gus_card_t *card )
{
}

int gus_engine_set_ultraclick( gus_card_t *card, gus_ultraclick_t *ultraclick, int space )
{
  gus_ultraclick_t sultraclick;
  
  if ( space == SP_USER )
    {
      if ( VERIFY_AREA( VERIFY_READ, ultraclick, sizeof( *ultraclick ) ) ) return -EIO;
      MEMCPY_FROMFS( &sultraclick, ultraclick, sizeof( sultraclick ) );
      ultraclick = &sultraclick;
    }
  card -> gf1.smooth_pan = ultraclick -> smooth_pan;
  card -> gf1.full_range_pan = ultraclick -> full_range_pan;
  card -> gf1.volume_ramp = ultraclick -> volume_ramp;
  return 0;
}

int gus_engine_get_ultraclick( gus_card_t *card, gus_ultraclick_t *ultraclick, int space )
{
  gus_ultraclick_t sultraclick;

  sultraclick.smooth_pan = card -> gf1.smooth_pan;
  sultraclick.full_range_pan = card -> gf1.full_range_pan;
  sultraclick.volume_ramp = card -> gf1.volume_ramp;
  if ( space == SP_USER )
    {
      if ( VERIFY_AREA( VERIFY_WRITE, ultraclick, sizeof( *ultraclick ) ) ) return -EIO;
      MEMCPY_TOFS( ultraclick, &sultraclick, sizeof( sultraclick ) );
    }
   else
    MEMCPY( ultraclick, &sultraclick, sizeof( sultraclick ) );
  return 0;
}

/*
 *
 */
 
int gus_engine_open( gus_card_t *card )
{
  int res;

  if ( card -> gf1.mode & GF1_MODE_ENGINE ) return -EBUSY;
  if ( ( res = gus_dma_malloc( card, GUS_DMA_GPLAY, "gf1 engine", 0 ) ) < 0 ) return res;
  card -> gf1.syn_voices = (gus_voice_t *)gus_malloc( sizeof( gus_voice_t ) * GUS_VOICES );
  if ( !card -> gf1.syn_voices )
    {
      gus_dma_free( card, GUS_DMA_GPLAY, 0 );
      return -ENOMEM;
    }
  card -> gf1.syn_notes = (gus_note_t *)gus_malloc( sizeof( gus_note_t ) * GUS_VOICES );
  if ( !card -> gf1.syn_notes )
    {
      gus_free( card -> gf1.syn_voices, sizeof( gus_channel_t ) * GUS_VOICES );
      gus_dma_free( card, GUS_DMA_GPLAY, 0 );
      return -ENOMEM;
    }  
  card -> gf1.syn_channels = (gus_channel_t *)gus_malloc( sizeof( gus_channel_t ) * GUS_CHANNELS );
  if ( !card -> gf1.syn_channels )
    {
      gus_free( card -> gf1.syn_notes, sizeof( gus_note_t ) * GUS_VOICES );
      gus_free( card -> gf1.syn_voices, sizeof( gus_channel_t ) * GUS_CHANNELS );
      gus_dma_free( card, GUS_DMA_GPLAY, 0 );
      return -ENOMEM;
    }  
  card -> gf1.syn_voice_dynamic = 0;
  gus_engine_voices_init( card, 0 );
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].interrupt_handler_wave = gus_interrupt_synth_wave;
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].interrupt_handler_volume = gus_interrupt_synth_volume;
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].voices_change_start = gus_engine_voices_change_start;
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].voices_change_stop = gus_engine_voices_change_stop;
  card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].volume_change = gus_engine_volume_change;
  card -> gf1.interrupt_handler_synth_dma_write = gus_interrupt_synth_dma_write;
  card -> gf1.smooth_pan = card -> gf1.default_smooth_pan;
  card -> gf1.full_range_pan = card -> gf1.default_full_range_pan;
  card -> gf1.volume_ramp = card -> gf1.default_volume_ramp;
  gus_gf1_open( card, GF1_MODE_ENGINE );
  gus_memory_reset( card, &card -> gf1.mem_alloc, GUS_ALLOC_MODE_NORMAL, 0 );
  gus_gf1_daemon_wait( card );
  gus_effects_reset( card );
  return 0;
}

void gus_engine_close( gus_card_t *card )
{
  if ( !(card -> gf1.mode & GF1_MODE_ENGINE) ) return;
  gus_gf1_daemon_wait( card );
  gus_gf1_close( card, GF1_MODE_ENGINE );
  gus_set_default_handlers( card, GF1_HANDLER_SYNTH_DMA_WRITE | GF1_HANDLER_RANGE | GF1_VOICE_RANGE_SYNTH );
  if ( card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].rvoices > 0 )
    {
      card -> gf1.voice_ranges[ GF1_VOICE_RANGE_SYNTH ].rvoices = 0;
      gus_reselect_active_voices( card );
    }
  gus_memory_reset( card, &card -> gf1.mem_alloc, GUS_ALLOC_MODE_NORMAL, 1 );
  if ( card -> gf1.syn_channels )
    {
      gus_free( card -> gf1.syn_channels, sizeof( gus_channel_t ) * GUS_CHANNELS );
      card -> gf1.syn_channels = NULL;
    }
  if ( card -> gf1.syn_notes )
    {
      gus_free( card -> gf1.syn_notes, sizeof( gus_note_t ) * GUS_VOICES );
      card -> gf1.syn_notes = NULL;
    }
  if ( card -> gf1.syn_voices )
    {
      card -> gf1.syn_voice_first = NULL;
      gus_free( card -> gf1.syn_voices, sizeof( gus_voice_t ) * GUS_VOICES );
      card -> gf1.syn_voices = NULL;
    }
  gus_dma_free( card, GUS_DMA_GPLAY, 0 );
  gus_effects_reset( card );
}
