/*
 *  error.c:		Error handling
 *
 *  Written by:		Stefan Frank
 *			Ullrich Hafner
 *  
 *  Credits:	Modelled after variable argument routines from Jef
 *		Poskanzer's pbmplus package. 
 *  
 *  Copyright (C) 1998 Ullrich Hafner <hafner@informatik.uni-wuerzburg.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 */

/*
 *  $Date: 1998/07/10 10:24:48 $
 *  $Author: hafner $
 *  $Revision: 1.4 $
 *  $State: Exp $
 */

#include "config.h"

#include <stdio.h>

#if STDC_HEADERS
#	include <stdarg.h>
#	define VA_START(args, lastarg) va_start(args, lastarg)
#else  /* not STDC_HEADERS */
#	include <varargs.h>
#	define VA_START(args, lastarg) va_start(args)
#endif /* not STDC_HEADERS */

#include "misc.h"

#include "error.h"

/*******************************************************************************

			     global variables
  
*******************************************************************************/

int		error_line  = 0;
char		*error_file = NULL;
enum verbosity verboselevel = SOMEVERBOSITY;

/*******************************************************************************

			     local variables
  
*******************************************************************************/

static char	*executable = "(name not initialized)";

/*******************************************************************************

			       public code
  
*******************************************************************************/

void
init_error_handling (const char *name)
/*
 *  Initialize filename of executable.
 *
 *  No return value.
 */
{
   executable = strdup (name);
}

void
_error (const char *format, ...)
/*
 *  Print error message and exit.
 *
 *  No return value.
 */
{
   va_list	args;

   VA_START (args, format);

   fprintf (stderr, "%s: %s: line %d:\nError: ",
	    executable, error_file, error_line);
#if HAVE_VPRINTF
   vfprintf (stderr, format, args);
#elif HAVE_DOPRNT
   _doprnt (format, args, stderr);
#endif /* HAVE_DOPRNT */
   fputc ('\n', stderr);
   va_end(args);

   exit (1);
}

void
_file_error (const char *filename)
/*
 *  Print file error message and exit.
 *
 *  No return value.
 */
{
   fprintf (stderr, "%s: %s: line %d:\nError: ",
	    executable, error_file, error_line);
   perror (filename);

   exit (2);
}

void 
_warning (const char *format, ...)
/*
 *  Issue a warning and continue execution.
 *
 *  No return value.
 */
{
   va_list	args;

   VA_START (args, format);

   if (verboselevel == NOVERBOSITY)
      return;
	
   fprintf (stderr, "%s: %s: line %d:\nWarning: ",
	    executable, error_file, error_line);
#if HAVE_VPRINTF
   vfprintf (stderr, format, args);
#elif HAVE_DOPRNT
   _doprnt (format, args, stderr);
#endif /* HAVE_DOPRNT */
   fputc ('\n', stderr);

   va_end (args);
}

void 
message (const char *format, ...)
/*
 *  Print a message to stderr.
 */
{
   va_list	args;

   VA_START (args, format);

   if (verboselevel == NOVERBOSITY)
      return;

#if HAVE_VPRINTF
   vfprintf (stderr, format, args);
#elif HAVE_DOPRNT
   _doprnt (format, args, stderr);
#endif /* HAVE_DOPRNT */
   fputc ('\n', stderr);
   va_end (args);
}

void 
debug_message (const char *format, ...)
/*
 *  Print a message to stderr.
 */
{
   va_list	args;

   VA_START (args, format);

   if (verboselevel < ULTIMATEVERBOSITY)
      return;

   fprintf (stderr, "*** ");
#if HAVE_VPRINTF
   vfprintf (stderr, format, args);
#elif HAVE_DOPRNT
   _doprnt (format, args, stderr);
#endif /* HAVE_DOPRNT */
   fputc ('\n', stderr);
   va_end (args);
}

void
info (const char *format, ...)
/*
 *  Print a message to stderr. Do not append a newline.
 */
{
   va_list	args;

   VA_START (args, format);

   if (verboselevel == NOVERBOSITY)
      return;

#if HAVE_VPRINTF
   vfprintf (stderr, format, args);
#elif HAVE_DOPRNT
   _doprnt (format, args, stderr);
#endif /* HAVE_DOPRNT */
   fflush (stderr);
   va_end (args);
}

