// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: gpmdev.cc,v 1.9 1998/01/21 01:06:37 jgg Exp $
/* ######################################################################

   GPMDev - GPM based mouse input handler
   
   The GPM mouse handler accepts mouse input from the Linux GPM driver
   and translates it into our internal event structure.
   
   ###################################################################### */
									/*}}}*/
// Include files							/*{{{*/
#include <config.h>

#ifdef HAVE_LIBGPM

#include <unistd.h>
#include <stdlib.h>
#include <gpm.h>               // GPM >= 1.12

#include <deity/gpmdev.h>
#include <deity/widget.h>
									/*}}}*/

// GPMFd::GPMFd - Constructor						/*{{{*/
// ---------------------------------------------------------------------
/* We open a connection and indicate that we want GPM to draw the cursor and
   that we wish to get all mouse events. */
GPMFd::GPMFd(Widget *Root) : SelectLoop::Fd(-1,true), Root(Root)
{
   /* First check that we are not in XTerm mode. KeyboardFd handles
      Xterm's mouse, we don't use or need gpm for that. If gpm attempts
      to init the xterm for mouse mode it will bugger up our init. */
   char *TermType = getenv("TERM");
   if (TermType != 0 && strncmp(TermType,"xterm",5) == 0)
      return;
   
   Gpm_Connect ConnectInfo;
   memset(&ConnectInfo,0,sizeof(ConnectInfo));
   ConnectInfo.eventMask = ~0;
   ConnectInfo.defaultMask = GPM_MOVE | GPM_HARD;

   // SelectLoop will ignore us if FD < 0
   Gpm_Open(&ConnectInfo,0);
   FileDesc = gpm_fd;
   
   OldButtons = 0;
}
									/*}}}*/
// GPMFd::~GPMFd - Destructor						/*{{{*/
// ---------------------------------------------------------------------
/* Close the mouse connection and reset */
GPMFd::~GPMFd()
{
   Gpm_Close();
}
									/*}}}*/
// GPMFd::Read - Read a mouse event					/*{{{*/
// ---------------------------------------------------------------------
/* This accepts and then translates a mouse event from GPM to something
   we can handle. */
bool GPMFd::Read(SelectLoop &)
{
   GrabMutex WidgetLock(Widget::WidgetLock);
   
   Gpm_Event Event;
   memset(&Event,0,sizeof(Event));
   Gpm_GetEvent(&Event);

   // Recode the event
   MouseEvent E;
   memset(&E,0,sizeof(E));
   if ((Event.type & GPM_MOVE) == GPM_MOVE)
      E.Type = MouseEvent::Motion;
   if ((Event.type & GPM_DOWN) == GPM_DOWN)
      E.Type = MouseEvent::ButtonDn;
   if ((Event.type & GPM_UP) == GPM_UP)
      E.Type = MouseEvent::ButtonUp;
   if ((Event.type & GPM_DRAG) == GPM_DRAG)
      E.Type = MouseEvent::Motion;

   // Recode the click counter
   if ((Event.type & GPM_MFLAG) == GPM_MFLAG)
   {
      if ((Event.type & GPM_SINGLE) == GPM_SINGLE)
	 E.Clicks = 1;
      if ((Event.type & GPM_DOUBLE) == GPM_DOUBLE)
	 E.Clicks = 2;
      if ((Event.type & GPM_TRIPLE) == GPM_TRIPLE)
	 E.Clicks = 3;
   }
   
   // Recode position
   E.Pos = Point(Event.x - 1,Event.y - 1);

   /* GPM doesnt send event information for a specific button and it leaves 
      but button masked on release. This means it is impossible to determine
      what button was pressed and what button was released! We simply advoid
      this and don't support complex multi-button things */
   if ((Event.buttons & GPM_B_MIDDLE) != 0)
      E.Button = MouseEvent::Aux;
   if ((Event.buttons & GPM_B_RIGHT) != 0)
      E.Button = MouseEvent::Menu;
   if ((Event.buttons & GPM_B_LEFT) != 0)
      E.Button = MouseEvent::Select;
   OldButtons = Event.buttons;

   // No buttons when just moving
   if (E.Type == MouseEvent::Motion)
      E.Button = 0;
   
   /* Recode the current button state. XWin doesnt encode the down button 
      click into the state mask so we dont either. */
   unsigned long But = E.Button;
   if (E.Type != MouseEvent::ButtonDn)
      But = 0;
   if ((Event.buttons & GPM_B_LEFT) == GPM_B_LEFT && But != MouseEvent::Select)
      E.ButtonState = MouseEvent::Select;
   if ((Event.buttons & GPM_B_RIGHT) == GPM_B_RIGHT && But != MouseEvent::Menu)
      E.ButtonState |= MouseEvent::Menu;
   if ((Event.buttons & GPM_B_MIDDLE) == GPM_B_MIDDLE && But != MouseEvent::Aux)
      E.ButtonState |= MouseEvent::Aux;

   // Determine if this event is a click event
   if ((Event.type & GPM_MFLAG) == 0 && (Event.type & GPM_UP) == GPM_UP)
      E.Type |= MouseEvent::Click;

   // Drag?
   if ((E.Type & MouseEvent::Motion) == MouseEvent::Motion && 
       E.ButtonState != 0 && (Event.type & GPM_MFLAG) != 0)
      E.Type |= MouseEvent::Drag;
       
   // Inject the event into the widget tree
   Root->RouteMouse(E);
   Root->DoActions();
   
   GenGC::GC->Flush();
   GPM_DRAWPOINTER(&Event);
   
   return true;
};
									/*}}}*/

#endif /* HAVE_LIBGPM */
