/*
 * X-Mame generic video code
 *
 */
#define __VIDEO_C_
#include "xmame.h"
#include <math.h>

static unsigned char current_palette[256*3];
static unsigned char color_dirty[256];
static int palette_dirty = FALSE;

/* Create a bitmap. Also calls osd_clearbitmap() to appropriately initialize */
/* it to the background color. */
/* VERY IMPORTANT: the function must allocate also a "safety area" 8 pixels wide all */
/* around the bitmap. This is required because, for performance reasons, some graphic */
/* routines don't clip at boundaries of the bitmap. */
struct osd_bitmap *osd_new_bitmap(int width,int height,int depth)       /* ASG 980209 */
{
	struct osd_bitmap *bitmap;

	if (!width) width++;
	if (!height) height++;

	if (Machine->orientation & ORIENTATION_SWAP_XY)
	{
		int temp;

		temp = width;
		width = height;
		height = temp;
	}

	if ((bitmap = malloc(sizeof(struct osd_bitmap))) != 0)
	{
		int i,rowlen,rdwidth;
		unsigned char *bm;
		int safety;


		if (width > 32) safety = 8;
		else safety = 0;        /* don't create the safety area for GfxElement bitmaps */

		if (depth != 8 && depth != 16) depth = 8;

		bitmap->depth = depth;
		bitmap->width = width;
		bitmap->height = height;

		rdwidth = (width + 7) & ~7;     /* round width to a quadword */
		if (depth == 16)
			rowlen = 2 * (rdwidth + 2 * safety) * sizeof(unsigned char);
		else
			rowlen =     (rdwidth + 2 * safety) * sizeof(unsigned char);

		if ((bm = malloc((height + 2 * safety) * rowlen)) == 0)
		{
			free(bitmap);
			return 0;
		}

		if ((bitmap->line = malloc(height * sizeof(unsigned char *))) == 0)
		{
			free(bm);
			free(bitmap);
			return 0;
		}

		for (i = 0;i < height;i++)
			bitmap->line[i] = &bm[(i + safety) * rowlen + safety];

		bitmap->_private = bm;

		osd_clearbitmap(bitmap);
	}

	return bitmap;
}

void osd_free_bitmap(struct osd_bitmap *bitmap)
{
	if (bitmap)
	{
		free(bitmap->line);
		free(bitmap->_private);
		free(bitmap);
		bitmap = NULL;
	}
}

/* set the bitmap to black */
void osd_clearbitmap(struct osd_bitmap *bitmap)
{
	int i;

	for (i = 0;i < bitmap->height;i++)
	{
		if (bitmap->depth == 16)
			memset(bitmap->line[i], 0, 2*bitmap->width);
		else
			memset(bitmap->line[i], 0, bitmap->width);
	}

	if (bitmap == Machine->scrbitmap)
	{
		osd_mark_dirty (0,0,bitmap->width-1,bitmap->height-1,1);

		/* signal the layer system that the screenneeds a complete refresh */
		layer_mark_full_screen_dirty();
	}
}

struct osd_bitmap *osd_create_display(int width, int height, int attributes)
{
   int a;

   bitmap = osd_create_bitmap(width,height);
   if (bitmap==NULL) return NULL;
   
   if (Machine->drv->video_attributes & VIDEO_TYPE_VECTOR)
   {
      visual.min_x = 0;
      visual.min_y = 0;
      visual.max_x = width - 1;
      visual.max_y = height - 1;
   }
   else
     visual = Machine->drv->visible_area;
   
   if( Machine->orientation & ORIENTATION_SWAP_XY) {
      a=visual.max_x; visual.max_x=visual.max_y; visual.max_y=a;
      a=visual.min_x; visual.min_x=visual.min_y; visual.min_y=a;
      a=width; width=height; height=a;
   } 

   if (Machine->orientation & ORIENTATION_FLIP_X)
   {
      a            = width - visual.min_x - 1;
      visual.min_x = width - visual.max_x - 1;
      visual.max_x = a;
   }
   
   if (Machine->orientation & ORIENTATION_FLIP_Y)
   {
      a            = height - visual.min_y - 1;
      visual.min_y = height - visual.max_y - 1;
      visual.max_y = a;
   }
   
   visual_width = visual.max_x - visual.min_x + 1;
   visual_height = visual.max_y - visual.min_y + 1;

   /* Can we do dirty? First check if the driver uses the GfxLayer system */
   if (Machine->drv->layer)
   {
      use_layer = TRUE;
   }
   else if ( (Machine->drv->video_attributes & VIDEO_SUPPORTS_DIRTY) == 0 )
   {
      use_dirty = FALSE;
   }
   
   if (sysdep_create_display()==OSD_NOT_OK)
   {
      /* better check if the bitmap isn't freed by the sysdep code when
         cleaning up, atleast the X code frees it itself */
      if (bitmap) { osd_free_bitmap(bitmap); bitmap = NULL; }
      return bitmap;
   }
   
   set_ui_visarea (visual.min_x, visual.min_y, visual.max_x, visual.max_y);
   
   /* Clear the current palette */
   memset((void *)current_palette, 0, 3*256*sizeof(unsigned char));
   memset((void *)color_dirty, FALSE,   256*sizeof(unsigned char));
   
   /* for debugging only */
   fprintf(stderr_file, "height = %d, widt= %d, visheight = %d, viswidth = %d,"
           "visstarty= %d, visstartx= %d\n",
            height, width, visual_height, visual_width, visual.min_y,
            visual.min_x);
   
   return bitmap;
}   

void osd_allocate_colors(unsigned int _totalcolors,const unsigned char *palette,unsigned short *pens)
{
   int a, color_start, color_end;
   
   totalcolors = _totalcolors;

   if (totalcolors>256)
   {
      fprintf(stderr_file, "Warning: More than 256 colors (%d) are needed for this emulation,\n"
         "some parts of the screen may be corrupted\n", totalcolors);
      /* fill the remainder of the pens array with 0's to make sure */
      /* nothing strange happens                                    */
      for (a=256;a<totalcolors;a++) pens[a]=0;
      totalcolors=256;
   }
   else
      fprintf(stderr_file, "Game uses %d colors\n", totalcolors);
      
   /* can we use color 0 & 255 for our own purposes ? */
   if (totalcolors <= 254)
   {
      color_start  = 1;
      color_end    = totalcolors+1;
      totalcolors += 2;
   }
   else
   {
      color_start  = 0;
      color_end    = totalcolors;
   }
   
   /* alloc the palette */
   sysdep_alloc_palette();
   
   /* init the palette */
   for (a=color_start; a<color_end; a++)
   {
      current_palette[a*3   ] = palette[(a-color_start)*3  ];
      current_palette[a*3 +1] = palette[(a-color_start)*3+1];
      current_palette[a*3 +2] = palette[(a-color_start)*3+2];
      pens[a-color_start]     = a;
      color_dirty[a]          = TRUE;
   }
   
   /* set the ui-colors */
   a = totalcolors - 1;
   
   Machine->uifont->colortable[0] = 0;
   Machine->uifont->colortable[1] = a;
   Machine->uifont->colortable[2] = a;
   Machine->uifont->colortable[3] = 0;

   /* if we use color 0 & 255 initialise them */
   if (color_start)
   {
      current_palette[0]   = current_palette[1]       = current_palette[2]       = 0x00;
      current_palette[a*3] = current_palette[a*3 + 1] = current_palette[a*3 + 2] = 0xFF;
      color_dirty[0] = color_dirty[a] = TRUE;
   }
      
   palette_dirty = TRUE;
}

void osd_get_pen(int pen,unsigned char *red, unsigned char *green, unsigned char *blue)
{
    *red   = current_palette[pen*3   ];
    *green = current_palette[pen*3 +1];
    *blue  = current_palette[pen*3 +2];
}

void osd_modify_pen(int pen, unsigned char red,unsigned char green,unsigned char blue) 
{
	/* Optimise out operation if no palette change is required */
	if (current_palette[pen*3   ] == red &&
	    current_palette[pen*3 +1] == green &&
	    current_palette[pen*3 +2] == blue)
	    return;
	    
	current_palette[pen*3   ] = red;
	current_palette[pen*3 +1] = green;
	current_palette[pen*3 +2] = blue;
	
	color_dirty[pen] = TRUE;
	palette_dirty    = TRUE;
}

#ifdef HAVE_GETTIMEOFDAY
/* Standard UNIX clock() is based on CPU time, not real time.
   Here is a real-time drop in replacement for UNIX systems that have the
   gettimeofday() routine.  This results in much more accurate timing for
   throttled emulation.
*/
clock_t uclock()
{
  static long init_sec = 0;

  struct timeval tv;
  gettimeofday(&tv, 0);
  if (init_sec == 0) init_sec = tv.tv_sec;
  return (tv.tv_sec - init_sec) * 1000000 + tv.tv_usec;
}
#endif

void osd_update_display(void)
{
	static int showfps, showfpstemp, showprofile;
	clock_t curr;
	#define MEMORY 10
	static clock_t prev[MEMORY];
	static int memory,speed;
	extern int frameskip;
	int i, need_to_clear_bitmap = 0;

	if (osd_key_pressed_memory(OSD_KEY_FRAMESKIP))
	{
		frameskip = (frameskip + 1) % 4;
		showfpstemp = 50;
	}

	if (osd_key_pressed_memory(OSD_KEY_THROTTLE))
	{
		throttle ^= 1;
	}

	if (use_profiler && osd_key_pressed_memory(OSD_KEY_SHOW_PROFILE))
	{
		showprofile ^= 1;
		if (showprofile == 0)
		{
			need_to_clear_bitmap = 1;
		}
		else osd_profiler_init(); /* reset the counters */
	}

	if (osd_key_pressed_memory(OSD_KEY_SHOW_FPS))
	{
		showfps ^= 1;
		if (showfps == 0)
		{
			need_to_clear_bitmap = 1;
		}
	}

	if (showfpstemp)         /* MAURY_BEGIN: nuove opzioni */
	{
		showfpstemp--;
		if ((showfps == 0) && (showfpstemp == 0))
		{
			need_to_clear_bitmap = 1;
		}
	}

	/* now wait until it's time to update the screen */
	if (throttle)
	{
		osd_profiler(OSD_PROFILE_IDLE);
		do
		{
			curr = uclock();
		} while ((curr - prev[memory]) < (frameskip+1) * UCLOCKS_PER_SEC/Machine->drv->frames_per_second);
		osd_profiler(OSD_PROFILE_END);
	}
	else curr = uclock();

	memory = (memory+1) % MEMORY;

	if (curr - prev[memory])
	{
		int divdr;


		divdr = Machine->drv->frames_per_second * (curr - prev[memory]) / (100 * MEMORY);
		speed = (UCLOCKS_PER_SEC * (frameskip+1) + divdr/2) / divdr;
	}

	prev[memory] = curr;

	if (showfps || showfpstemp) /* MAURY: nuove opzioni */
	{
		int trueorientation;
		int fps,i,l;
		char buf[30];

		/* hack: force the display into standard orientation to avoid */
		/* rotating the text */
		trueorientation = Machine->orientation;
		Machine->orientation = ORIENTATION_DEFAULT;

		fps = (Machine->drv->frames_per_second / (frameskip+1) * speed + 50) / 100;
		sprintf(buf," %3d%%(%3d/%d fps)",speed,fps,Machine->drv->frames_per_second);
		l = strlen(buf);
		for (i = 0;i < l;i++)
			drawgfx(Machine->scrbitmap,Machine->uifont,buf[i],DT_COLOR_WHITE,0,0,visual.max_x-(l-i)*Machine->uifont->width,visual.min_y,0,TRANSPARENCY_NONE,0);
		Machine->orientation = trueorientation;
	}
	
	if (showprofile) osd_profiler_display();
	
	if (palette_dirty)
	{
	   for (i=0; i<256; i++)
	   {
	      if(color_dirty[i])
	      {
	         int r, g, b;
                 r = 255 * brightness * pow(current_palette[i*3  ] / 255.0, 1 / gamma_correction) / 100;
                 g = 255 * brightness * pow(current_palette[i*3+1] / 255.0, 1 / gamma_correction) / 100;
                 b = 255 * brightness * pow(current_palette[i*3+2] / 255.0, 1 / gamma_correction) / 100;
                 sysdep_modify_pen(i, r, g ,b);
	         color_dirty[i] = FALSE;
	      }
	   }
	   palette_dirty = FALSE;
	}
	
	osd_profiler(OSD_PROFILE_BLIT);
	sysdep_update_display();
	osd_profiler(OSD_PROFILE_END);
	
	if (need_to_clear_bitmap)
		osd_clearbitmap(Machine->scrbitmap);
}

int osd_skip_this_frame(int recommend)
{
	return recommend;
}

void osd_set_gamma(float _gamma)
{
	gamma_correction = _gamma;
	memset((void *)color_dirty, TRUE, totalcolors*sizeof(unsigned char));
	palette_dirty = TRUE;
}

float osd_get_gamma(void)
{
	return gamma_correction;
}

/* brightess = percentage 0-100% */
void osd_set_brightness(int _brightness)
{
	brightness = _brightness;
	memset((void *)color_dirty, TRUE, totalcolors*sizeof(unsigned char));
	palette_dirty = TRUE;
}

int osd_get_brightness(void)
{
	return brightness;
}

#ifndef x11
void osd_save_snapshot(void)
{
}
#endif
