/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                   Author :  Paul Taylor                               */
/*                   Date   :  January 1995                              */
/*-----------------------------------------------------------------------*/
/*              Key/Value list template class                            */
/*                                                                       */
/*=======================================================================*/
#ifndef __EST_KV_H__
#define __EST_KV_H__

#include <math.h>
#include "EST_TList.h"
#include "EST_String.h"
#include "instantiate/EST_KVLI.h"


/** Templated Key-Value Item. Serves as the items in the list of the
KVL class.  */
template<class K, class V> class KVI {
 public:
    K k;
    V v;

    inline bool operator==(const KVI<K,V> &i){
	return( (i.k == k) && (i.v == v) );
    }

    friend  ostream& operator << (ostream& s, KVI<K,V> const &i);
};


/** Templated Key-Value list. Objects of type KVL contain lists which
are accessed by a key of type {\bf K}, which returns a value of type
{\bf V}. */
template<class K, class V> class KVL {
 private:
    EST_Litem *find_pair_key(const K &key) const;
 public:
    /**@name Constructor functions */
    //@{
    /// default constructor
    KVL() {;}
    /// copy constructor
    KVL(const KVL<K, V> &kv);
    //@}

    /// Linked list of key-val pairs. Don't use
    /// this as it will be made private in the future
    EST_TList< KVI<K,V> > list;	

    /// number of key value pairs in list
    const int length() const {return list.length();} 

    /// Return First key value pair in list
    EST_Litem * head() const {return list.head();};

    /// Empty list.
    void clear();
    
    /**@name Access functions.
     */
    //@{
    /// return value according to key (const)
    const V &val(const K &rkey, bool m=0) const;
    /// return value according to key (non-const)
    V &val(const K &rkey, bool m=0);
    /// return value according to ptr
    const V &val(EST_Litem *ptr, bool m=0) const;     
    /// value or default 
    const V &val_def(const K &rkey, const V &def) const;
    /// find key, reference by ptr 
    const K &key(EST_Litem *ptr, int m=1) const;	
    /** change key-val pair. If no corresponding entry is present, add
      to end of list.
      */
    int change_val(const K &rkey,const V &rval); 
    /** change key-val pair. If no corresponding entry is present, add
      to end of list.*/
    int change_val(EST_TBI *ptr,const V &rval); // change key-val pair.
    /// change name of key pair.
    int change_key(EST_TBI *ptr,const K &rkey);

    /// add key-val pair to list
    int add_item(const K &rkey,const V &rval, int no_search = 0);

    /// remove key and val pair from list
    int remove_item(const K &rkey, int quiet = 0);

    //@}
    
    /// Returns true if key is present.
    const int present(const K &rkey) const;

    /// apply function to each pair
    void map(void (*func)(K&, V&));
    
    friend ostream& operator << (ostream& s, KVL<K,V> const &kv);
    
    /// full copy of KV list.
    KVL<K, V> & operator =  (const KVL<K,V> &kv);
    /// add kv after existing list.
    KVL<K, V> & operator += (const KVL<K,V> &kv); 
    /// make new concatenated list
    KVL<K, V>   operator +  (const KVL<K,V> &kv); 
};

template<class K, class V>
extern ostream& operator << (ostream& s, KVI<K,V> const &i);

template<class K, class V>
extern ostream& operator << (ostream& s,  KVL<K,V> const &l);

#endif				// __KVL_H__
