/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                    Copyright (c) 1994,1995,1996                       */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                      Author :  Paul Taylor                            */
/*                      Date   :  June 1994                              */
/*-----------------------------------------------------------------------*/
/*                  EST_Track file manipulation program                  */   
/*                                                                       */
/*=======================================================================*/

#include "EST.h"
#include "EST_cmd_line_options.h"

#define DEFAULT_TIME_SCALE 0.001

int StrListtoIList(EST_StrList &s, EST_IList &il);
void extract_channel(EST_Track &orig, EST_Track &nt, EST_IList &ch_list);

EST_write_status save_snns_pat(const EST_String filename, 
			       EST_TrackList &inpat, EST_TrackList &outpat);

EST_read_status read_TrackList(EST_TrackList &tlist, EST_StrList &files, 
			       EST_Option &al);

void extract(EST_Track &tr, EST_Option &al);

int main(int argc, char *argv[])
{
    EST_String in_file("-"), out_file("-");
    EST_Option al, settings;
    EST_String fname, ftmp;
    EST_StrList files;
    EST_Track tr;
    EST_TrackList trlist;
    EST_TBI *p;

    parse_command_line(argc, argv, 
       EST_String("Usage:   ")+
       "ch_track <input file> -o <output file> <options>\n"+
       "Summary: change/copy track files\n"+
       "use \"-\" to make input and output files stdin/out\n"+
       "-h               Options help\n"+
       options_track_input()+
       "\n"+
       options_track_output()+
       "\n"+
       "-info            Describe track file\n"+
       "-time_channel <string>\n"+
       "		 Which track in track file holds pitchmark times\n"+
       "-time_scale <float>    \n"+
       "		 Scale of pitchmarks (default 0.001 = milliseconds)\n"+
       "-extract <string> Extract channels give by int list\n" +
       "-f <int>         Sample rate in Hz (for headerless data)\n"+
       "-N               Do NOT read environment operations file\n"+
       "-diff            Differentiatie contour\n"+
       "-sm <float>      amount of smoothing (seconds)\n"+
       "-smtype <string> {median}\n"+
       "                 Smooth type, median or mean\n"+
       "-style <string>  Convert track to other form.  Forms supported are:\n"+
       "                 label		pos/neg decision\n"+
       "                 pm		labels at time points\n"+
       "-t <float>       threshold for track to label conversion\n"+
       "-neg             Name of negative values in label conversion\n"+
       "-pos             Name of positive values in label conversion\n"+
       "-pc <string>     Combine given tracks in parallel.  If option\n"+
       "                 is longest, pad shorter tracks to longest, else if\n"+
       "                 first pad/cut to match first input track\n"+
       options_subtrack() + 
       options_track_filetypes_long(),
			files, al);

    override_lib_ops(settings, al);
    out_file = al.present("-o") ? al.val("-o") : (EST_String)"-";

    EST_TokenStream ts;
    
//    ts.open(files.first());
//    tr.load(ts);
//    cout << tr;

    if (read_TrackList(trlist, files, al) != read_ok)
	exit(0);

    if (files.length() == 0)
    {
	cerr << argv[0] << ": no input files specified\n";
	exit(-1);
    }

    if (al.present("-info"))
    {
	for (p = trlist.head(); p; p = next(p))
	    track_info(trlist(p));
	exit(0);
    }

    if (al.present("-pc"))       // parallelize them
	ParallelTracks(tr, trlist, al.val("-pc"));

    else if (al.val("-otype", 0) == "snns")
    {   // sometime this will generalise for multiple input files
	EST_TrackList inpat, outpat;
	inpat.append(trlist.nth(0));
	outpat.append(trlist.nth(1));
	save_snns_pat(out_file, inpat, outpat);
	exit(0);
    }
    else                         // concatenate them
    {
	tr.resize(0, tr.num_channels());
	// Reorg -- fix += to resize to largest num_channels (with warning)
	for (p = trlist.head(); p; p = next(p))
	    tr += trlist(p);
    }

    if (al.present("-c"))
    {
	cout << "al:" << al.val("-c") << endl;
	tr.set_contour_type(EST_Contour::type(al.val("-c")));
    }
    // if (al.present("-color"))
    // tr.color = al.val("-color");
    if (al.present("-S"))
	tr.sample(al.fval("-S"));
    if (al.present("-sm"))
	track_smooth(tr, al.fval("-sm"),al.val("-smtype"));
    if (al.present("-diff"))
    {
	tr.set_contour_type(ct_diff_f0);
	tr = differentiate(tr);
    }
    if (al.present("-extract"))
    {
	EST_StrList s;
	EST_Track ntr;
	EST_IList il;
	StringtoStrList(al.val("-extract"), s);
	StrListtoIList(s, il);
	extract_channel(tr, ntr, il);
	tr = ntr;
    }

    if (al.present("-start") || al.present("-end") 
	|| al.present("-to") || al.present("-from"))
	extract(tr, al);

//    tr.assign_map(&LPCTrackMap);
//    tr.set_contour_type(ct_lpc);
//    tr.set_space_type("VARI");

    // Write out file in appropriate format
    
    if (al.val("-style",0) == "label")
    {
	EST_Stream lab;
	if (al.present("-t"))
	    track_to_label(tr, lab, al.fval("-t"));
	else
	    track_to_label(tr, lab);
	if (al.present("-pos"))
	    change_label(lab, "pos", al.val("-pos"));
	if (al.present("-neg"))
	    change_label(lab, "neg", al.val("-neg"));
	if (lab.save(out_file) != write_ok)
	    exit(-1);
    }
    else if (al.val("-style",0) == "pm")
    {
	EST_Stream lab;
	
	if (!al.present("-f"))
	  {
	    cerr << "must specify sample rate (with -f) for pm style\n";
	    exit(-1);
	  }
	int sample_rate = al.ival("-f", 0);

	track_to_pm(tr, sample_rate, lab);

	if (lab.save(out_file) != write_ok)
	    exit(-1);
    }
    else
	if (tr.save(out_file, al.val("-otype")) != write_ok)
	    exit(-1);
    
    return 0;
}

void override_lib_ops(EST_Option &a_list, EST_Option &al)
{
    a_list.override_val("ishift", al.val("-s", 0));
    a_list.override_val("color", al.val("-color", 0));
    a_list.override_val("in_track_file_type", al.val("-itype", 0));
    a_list.override_val("out_track_file_type", al.val("-otype", 0));
    a_list.override_val("tr_to_label_thresh", al.val("-t", 0));
    a_list.override_fval("time_scale", DEFAULT_TIME_SCALE);
    
    if (al.val("-style", 0) == "label")
	a_list.override_val("lab_file_type", al.val("-otype", 0));
    if (al.present("-time_scale"))
	a_list.override_fval("time_scale", al.fval("-time_scale", 1));
    if (al.present("-time_channel"))
	a_list.override_val("time_channel", al.sval("-time_channel", 1));
}

