/*
 * settings.c : Set program parameters on a popup panel
 *
 * George Ferguson, ferguson@cs.rochester.edu, 22 Oct 1991.
 * Version 2.0: 23 Apr 1993.
 */

#include <stdio.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Cardinals.h>
#include "xarchie.h"
#include "types.h"
#include "appres.h"
#include "m-defs.h"
#include "rdgram.h"
#include "alert.h"
#include "status.h"
#include "popups.h"
#include "xutil.h"
#include "stringdefs.h"
#include "weight.h"
#include "tilde.h"
extern void resortBrowser();		/* browser.c */

/*
 * Functions declared in this file
 */
void initSettings(),reinitSettings();
void setSettingsShellState();
void updateSettingsHost(),updateSettingsSearchType();
void updateSettingsSortType(),updateSettingsNiceLevel();
void updateSettingsAutoScroll();
void updateSettingsFtpType(),updateSettingsFtpPrompt();
void updateSettingsFtpTrace(),updateSettingsFtpStrip();
void setSettingsChangedFlag();

static void initSettingsWidgets();
static void addTextEventHandler(),textEventProc();
static void popupSettingsAction();
static void applySettingsAction(),defaultSettingsAction(),doneSettingsAction();
static void setHostAction(),setSearchTypeAction();
static void setSortTypeAction(),setNiceLevelAction();
static void setHostNowAction(),setSearchTypeNowAction();
static void setSortTypeNowAction(),setNiceLevelNowAction();
static int settingsConfirm();
static void settingsConfirmCallback();

/*
 * Data declared in this file
 */
static Widget settingsShell;
static Widget setApplyButton;
static Widget setHostText;
static Widget setSearchLabel,setSortLabel;
static Widget setNiceText,setMaxHitsText;
static Widget setTimeoutText,setRetriesText;
static Widget setAutoScrollLabel;
static Widget ftpLocalDirText,ftpTypeLabel;
static Widget ftpPromptLabel,ftpTraceLabel,ftpStripLabel;
static Widget ftpMailAddressText;
static Widget setHostWeightsLabel,setHostWeightsText;

static char *defArchieHost;
static SearchType currentSearchType,defSearchType;
static SortType currentSortType,defSortType;
static int defMaxHits,defTimeout,defRetries,defNiceLevel;
static Boolean currentAutoScroll,defAutoScroll;
static char *defFtpLocalDir,*defFtpType;
static Boolean currentFtpPrompt,defFtpPrompt;
static Boolean currentFtpTrace,defFtpTrace;
static Boolean currentFtpStrip,defFtpStrip;
static char *defFtpMailAddress;
static char *defHostWeights;
static Boolean settingsChanged,isPoppedUp;

static XtActionsRec actionTable[] = {
    { "popup-settings",		popupSettingsAction },
    { "settings-apply",		applySettingsAction },
    { "settings-default",	defaultSettingsAction },
    { "settings-done",		doneSettingsAction },
    { "set-host",		setHostAction },
    { "set-host-now",		setHostNowAction },
    { "set-search-type",	setSearchTypeAction },
    { "set-search-type-now",	setSearchTypeNowAction },
    { "set-sort-type",		setSortTypeAction },
    { "set-sort-type-now",	setSortTypeNowAction },
    { "set-nice-level",		setNiceLevelAction },
    { "set-nice-level-now",	setNiceLevelNowAction },
};

/*	-	-	-	-	-	-	-	-	*/
/*
 * initSettings() : Stores away the values of the application resources
 *	at startup for use by the default-settings() action. Also
 *	create the settings panel and register the action procedures.
 */
void
initSettings()
{
    defArchieHost = XtNewString(appResources.archieHost);
    defSearchType = appResources.searchType;
    defSortType = appResources.sortType;
    defNiceLevel = appResources.niceLevel;
    defMaxHits = appResources.maxHits;
    defTimeout = appResources.timeout;
    defRetries = appResources.retries;
    defAutoScroll = appResources.autoScroll;
    defFtpLocalDir = XtNewString(appResources.ftpLocalDir);
    defFtpType = XtNewString(appResources.ftpType);
    defFtpPrompt = appResources.ftpPrompt;
    defFtpTrace = appResources.ftpTrace;
    defFtpStrip = appResources.ftpStrip;
    defFtpMailAddress = XtNewString(appResources.ftpMailAddress);
    defHostWeights = XtNewString(appResources.hostWeights);
    isPoppedUp = False;
    XtAppAddActions(appContext,actionTable,XtNumber(actionTable));
    initSettingsWidgets();
}

/*
 * reinitSettings() : Sets the values in the settings editor from the
 *	current state of the application resources.
 */
void
reinitSettings()
{
    char buf[16];
    static int firsttime = 1;

    setWidgetString(setHostText,appResources.archieHost);
    updateSettingsSearchType(appResources.searchType);
    updateSettingsSortType(appResources.sortType);
    sprintf(buf,"%d",appResources.niceLevel);
    setWidgetString(setNiceText,buf);
    sprintf(buf,"%d",appResources.maxHits);
    setWidgetString(setMaxHitsText,buf);
    sprintf(buf,"%d",appResources.timeout);
    setWidgetString(setTimeoutText,buf);
    sprintf(buf,"%d",appResources.retries);
    setWidgetString(setRetriesText,buf);
    updateSettingsAutoScroll(appResources.autoScroll);
    /* Kludge to prevent tilde from being expanded in the Text item */
    if (firsttime) {
	setWidgetString(ftpLocalDirText,appResources.ftpLocalDir);
	firsttime = 0;
    }
    updateSettingsFtpType(appResources.ftpType);
    updateSettingsFtpPrompt(appResources.ftpPrompt);
    updateSettingsFtpTrace(appResources.ftpTrace);
    updateSettingsFtpStrip(appResources.ftpStrip);
    setWidgetString(ftpMailAddressText,appResources.ftpMailAddress);
    setWidgetString(setHostWeightsText,appResources.hostWeights);
    setSettingsChangedFlag(False);
    updateSettingsMenuMarks();
}

/*
 * initSettingsWidgets() : Create the popup settings editor.
 */
static void
initSettingsWidgets()
{
    Widget form;

    settingsShell = XtCreatePopupShell("settingsShell",
				       topLevelShellWidgetClass,toplevel,
				       NULL,0);
    form = XtCreateManagedWidget("settingsForm",formWidgetClass,
				 settingsShell,NULL,0);
    (void) XtCreateManagedWidget("setDoneButton",commandWidgetClass,
				 form,NULL,0);
    setApplyButton = XtCreateManagedWidget("setApplyButton",commandWidgetClass,
					   form,NULL,0);
    (void)XtCreateManagedWidget("setDefaultButton",commandWidgetClass,
				form,NULL,0);
    (void)XtCreateManagedWidget("setHostButton",menuButtonWidgetClass,
				form,NULL,0);
    setHostText = XtCreateManagedWidget("setHostText",asciiTextWidgetClass,
					form,NULL,0);
    (void)XtCreateManagedWidget("setSearchButton",menuButtonWidgetClass,
				form,NULL,0);
    setSearchLabel = XtCreateManagedWidget("setSearchLabel",labelWidgetClass,
					   form,NULL,0);
    (void)XtCreateManagedWidget("setSortButton",menuButtonWidgetClass,
				form,NULL,0);
    setSortLabel = XtCreateManagedWidget("setSortLabel",labelWidgetClass,
					 form,NULL,0);
    (void)XtCreateManagedWidget("setNiceButton",menuButtonWidgetClass,
				form,NULL,0);
    setNiceText = XtCreateManagedWidget("setNiceText",asciiTextWidgetClass,
					form,NULL,0);
    (void)XtCreateManagedWidget("setMaxHitsLabel",labelWidgetClass,
				form,NULL,0);
    setMaxHitsText = XtCreateManagedWidget("setMaxHitsText",
					   asciiTextWidgetClass,
					   form,NULL,0);
    (void)XtCreateManagedWidget("setTimeoutLabel",labelWidgetClass,
				form,NULL,0);
    setTimeoutText = XtCreateManagedWidget("setTimeoutText",
					   asciiTextWidgetClass,
					   form,NULL,0);
    (void)XtCreateManagedWidget("setRetriesLabel",labelWidgetClass,
				form,NULL,0);
    setRetriesText = XtCreateManagedWidget("setRetriesText",
					   asciiTextWidgetClass,
					   form,NULL,0);
    (void)XtCreateManagedWidget("setAutoScrollButton",menuButtonWidgetClass,
				form,NULL,0);
    setAutoScrollLabel = XtCreateManagedWidget("setAutoScrollLabel",
					       labelWidgetClass,
					       form,NULL,0);
    (void)XtCreateManagedWidget("ftpMailAddressLabel",labelWidgetClass,
				form,NULL,0);
    ftpMailAddressText = XtCreateManagedWidget("ftpMailAddressText",
					       asciiTextWidgetClass,
					       form,NULL,0);
    (void)XtCreateManagedWidget("ftpLocalDirLabel",labelWidgetClass,
				form,NULL,0);
    ftpLocalDirText = XtCreateManagedWidget("ftpLocalDirText",
					    asciiTextWidgetClass,
					    form,NULL,0);
    (void)XtCreateManagedWidget("ftpTypeButton",menuButtonWidgetClass,
				form,NULL,0);
    ftpTypeLabel = XtCreateManagedWidget("ftpTypeLabel",labelWidgetClass,
					 form,NULL,0);
    (void)XtCreateManagedWidget("ftpPromptButton",menuButtonWidgetClass,
				form,NULL,0);
    ftpPromptLabel = XtCreateManagedWidget("ftpPromptLabel",labelWidgetClass,
					   form,NULL,0);
    (void)XtCreateManagedWidget("ftpTraceButton",menuButtonWidgetClass,
				form,NULL,0);
    ftpTraceLabel = XtCreateManagedWidget("ftpTraceLabel",labelWidgetClass,
					   form,NULL,0);
    (void)XtCreateManagedWidget("ftpStripButton",menuButtonWidgetClass,
				form,NULL,0);
    ftpStripLabel = XtCreateManagedWidget("ftpStripLabel",labelWidgetClass,
					   form,NULL,0);
    setHostWeightsLabel = XtCreateManagedWidget("setHostWeightsLabel",
						labelWidgetClass,
						form,NULL,0);
    setHostWeightsText = XtCreateManagedWidget("setHostWeightsText",
					       asciiTextWidgetClass,
					       form,NULL,0);
    /* Add event handler for detecting changes */
    addTextEventHandler(setHostText);
    addTextEventHandler(setMaxHitsText);
    addTextEventHandler(setTimeoutText);
    addTextEventHandler(setRetriesText);
    addTextEventHandler(setNiceText);
    addTextEventHandler(setHostWeightsText);
    addTextEventHandler(ftpMailAddressText);
    addTextEventHandler(ftpLocalDirText);
    XtRealizeWidget(settingsShell);
    /* Enable WM_DELETE_WINDOW handling */
    (void)XSetWMProtocols(XtDisplay(settingsShell),XtWindow(settingsShell),
			  &WM_DELETE_WINDOW,1);
}

static void
addTextEventHandler(w)
Widget w;
{
    if (w != NULL)
	XtAddEventHandler(w,KeyPressMask|ButtonPressMask,False,
			  textEventProc,(XtPointer)NULL);
}

void
setSettingsShellState(state)
int state;
{
    if (!isPoppedUp)
	return;
    switch (state) {
	case NormalState:
	    XtMapWidget(settingsShell);
	    break;
	case IconicState:
	    XtUnmapWidget(settingsShell);
	    break;
    }
}

/*	-	-	-	-	-	-	-	-	*/
/* Action procedures */

#define ACTION_PROC(NAME)	void NAME(w,event,params,num_params) \
					Widget w; \
					XEvent *event; \
					String *params; \
					Cardinal *num_params;

/*
 * popupSettingsAction() : Pops up the settings editor, and fills it with
 *	the information from the current values of the application settings.
 */
/*ARGSUSED*/
static
ACTION_PROC(popupSettingsAction)
{
    if (isPoppedUp) {
	XRaiseWindow(display,XtWindow(settingsShell));
    } else {
	reinitSettings();
	XtPopup(settingsShell,XtGrabNone);
	isPoppedUp = True;
    }
}

/*
 * applySettingsAction() : Callback for apply button - Set the application
 *	resources from the items on the settings editor panel. Some of these
 *	require special action when changed, and this routine does that.
 */
/*ARGSUSED*/
static
ACTION_PROC(applySettingsAction)
{
    char *s;
    int n;

    /* Host */
    s = getWidgetString(setHostText);
    if (strcmp(appResources.archieHost,s) != 0) {
	XtFree(appResources.archieHost);
	appResources.archieHost = XtNewString(s);
	setHostMenuMark(appResources.archieHost);
    }
    /* Search type */
    if (appResources.searchType != currentSearchType) {
	appResources.searchType = currentSearchType;
	setSearchMenuMark(appResources.searchType);
    }
    /* Sort type */
    if (appResources.sortType != currentSortType) {
	appResources.sortType = currentSortType;
	setSortMenuMark(appResources.searchType);
	resortBrowser();
    }
    /* Nice level */
    s = getWidgetString(setNiceText);
    n = atoi(s);
    if (n < RDGRAM_MIN_PRI)		/* leave -32766 to -32768 alone */
	n = RDGRAM_MIN_PRI;
    else if (n > RDGRAM_MAX_SPRI)
	n = RDGRAM_MAX_PRI;
    if (appResources.niceLevel != n) {
	appResources.niceLevel = n;
	setNiceMenuMark(appResources.niceLevel);
    }
    /* Max hits */
    s = getWidgetString(setMaxHitsText);
    appResources.maxHits = atoi(s);
    /* Timeout */
    s = getWidgetString(setTimeoutText);
    appResources.timeout = atoi(s);
    /* Retries */
    s = getWidgetString(setRetriesText);
    appResources.retries = atoi(s);
    /* Auto Scroll */
    appResources.autoScroll = currentAutoScroll;
    /* Ftp mail address */
    s = getWidgetString(ftpMailAddressText);
    if (strcmp(appResources.ftpMailAddress,s) != 0) {
	XtFree(appResources.ftpMailAddress);
	appResources.ftpMailAddress = XtNewString(s);
    }
    /* Ftp local dir */
    s = getWidgetString(ftpLocalDirText);
    s = tildeExpand(s);
    if (strcmp(appResources.ftpLocalDir,s) != 0) {
	XtFree(appResources.ftpLocalDir);
	appResources.ftpLocalDir = XtNewString(s);
    }
    /* Ftp type (uses Label not Text) */
    s = getWidgetLabel(ftpTypeLabel);
    if (strcmp(appResources.ftpType,s) != 0) {
	XtFree(appResources.ftpType);
	appResources.ftpType = XtNewString(s);
    }
    /* Ftp prompt */
    appResources.ftpPrompt = currentFtpPrompt;
    /* Ftp trace */
    appResources.ftpTrace = currentFtpTrace;
    /* Ftp strip */
    appResources.ftpStrip = currentFtpStrip;
    /* Host Weights */
    s = getWidgetString(setHostWeightsText);
    if (strcmp(appResources.hostWeights,s) != 0) {
	XtFree(appResources.hostWeights);
	appResources.hostWeights = XtNewString(s);
	reinitHostWeights();
	resortBrowser();
    }
    /* All done */
    setSettingsChangedFlag(False);
}

/*
 * defaultSettingsAction() : Callback for default button - Reset the items
 *      to their default values.
 */
/*ARGSUSED*/
static
ACTION_PROC(defaultSettingsAction)
{
    char buf[16];

    setWidgetString(setHostText,defArchieHost);
    updateSettingsSearchType(defSearchType);
    updateSettingsSortType(defSortType);
    sprintf(buf,"%d",defNiceLevel);
    setWidgetString(setNiceText,buf);
    sprintf(buf,"%d",defMaxHits);
    setWidgetString(setMaxHitsText,buf);
    sprintf(buf,"%d",defTimeout);
    setWidgetString(setTimeoutText,buf);
    sprintf(buf,"%d",defRetries);
    setWidgetString(setRetriesText,buf);
    updateSettingsAutoScroll(defAutoScroll);
    setWidgetString(ftpMailAddressText,defFtpMailAddress);
    setWidgetString(ftpLocalDirText,defFtpLocalDir);
    updateSettingsFtpType(defFtpType);
    updateSettingsFtpPrompt(defFtpPrompt);
    updateSettingsFtpTrace(defFtpTrace);
    updateSettingsFtpStrip(defFtpStrip);
    setWidgetString(setHostWeightsText,defHostWeights);
    setSettingsChangedFlag(True);
}

/*
 * doneSettingsAction() : Callback for done button - Pop down the editor.
 */
/*ARGSUSED*/
static
ACTION_PROC(doneSettingsAction)
{
    if (!settingsChanged || settingsConfirm()) {
	XtPopdown(settingsShell);
	isPoppedUp = False;
    }
}

/*	-	-	-	-	-	-	-	-	*/
/*
 * setHostAction() : Action procedure to set the host.
 */
/*ARGSUSED*/
static
ACTION_PROC(setHostAction)
{
    if (*num_params != 1) {
	alert0("Incorrect number of arguments to set-host()");
	return;
    }
    if (setHostText == NULL) {
	alert0("set-host() has no effect since setHostText was not created");
	return;
    }
    setWidgetString(setHostText,*params);
    setSettingsChangedFlag(True);
}

/*
 * setSearchTypeAction() : Action procedure to set the search type.
 */
/*ARGSUSED*/
static
ACTION_PROC(setSearchTypeAction)
{
    XrmValue from,to;

    if (*num_params != 1) {
	alert0("Incorrect number of arguments to set-search-type()");
	return;
    }
    from.addr = *params;
    from.size = sizeof(String);
    to.addr = NULL;
    XtConvertAndStore(w,XtRString,&from,GfRSearchType,&to);
    if (to.addr != NULL)
	updateSettingsSearchType((SearchType)(*(to.addr)));
    setSettingsChangedFlag(True);
}

/*
 * setSortTypeAction() : Action procedure to set the sort type.
 */
/*ARGSUSED*/
static
ACTION_PROC(setSortTypeAction)
{
    XrmValue from,to;

    if (*num_params != 1) {
	alert0("Incorrect number of arguments to set-sort-type()");
	return;
    }
    from.addr = *params;
    from.size = sizeof(String);
    to.addr = NULL;
    XtConvert(w,XtRString,&from,GfRSortType,&to);
    if (to.addr != NULL)
	updateSettingsSortType((SortType)(*(to.addr)));
    setSettingsChangedFlag(True);
}

/*
 * setNiceLevelAction() : Action procedure to set rdgram_priority
 */
/*ARGSUSED*/
static
ACTION_PROC(setNiceLevelAction)
{
    char buf[8];
    int n;

    if (*num_params != 1) {
	alert0("Incorrect number of arguments to set-nice-level()");
	return;
    }
    if (setNiceText == NULL) {
       alert0("set-nice-level() has no effect since niceText was not created");
	return;
    }
    n = atoi(*params);
    if (n < RDGRAM_MIN_PRI) {
	alert1("Nice level too negative: %d",(char *)n);
	sprintf(buf,"%d",RDGRAM_MIN_PRI);
	setWidgetString(setNiceText,buf);
    } else if (n > RDGRAM_MAX_PRI) {
	alert1("Nice level too positive: %d",(char *)n);
	sprintf(buf,"%d",RDGRAM_MAX_PRI);
	setWidgetString(setNiceText,buf);
    } else {
	setWidgetString(setNiceText,*params);
    }
    setSettingsChangedFlag(True);
}

/*	-	-	-	-	-	-	-	-	*/
/* These actions are like their non-Now counterparts, expect that
 * (a) they set appResources immediately rather than waiting for
 *     apply-settings() to be called, and
 * (b) they do not set the changedFlag since they have made the change
 *     globally already.
 * Still, they really aren't meant to be used when the settingsPanel is
 * being displayed.
 */

/*ARGSUSED*/
static
ACTION_PROC(setSearchTypeNowAction)
{
    XrmValue from,to;
    SearchType type;

    if (*num_params != 1) {
	alert0("Incorrect number of arguments to set-search-type-now()");
	return;
    }
    from.addr = *params;
    from.size = sizeof(String);
    to.addr = NULL;
    XtConvertAndStore(w,XtRString,&from,GfRSearchType,&to);
    if (to.addr != NULL) {
	type = (SearchType)(*(to.addr));
	appResources.searchType = type;
	status1("Set search type to %s",*params);
	updateSettingsSearchType(type);
    }
}

/*ARGSUSED*/
static
ACTION_PROC(setSortTypeNowAction)
{
    XrmValue from,to;
    SortType type;

    if (*num_params != 1) {
	alert0("Incorrect number of arguments to set-sort-type-now()");
	return;
    }
    from.addr = *params;
    from.size = sizeof(String);
    to.addr = NULL;
    XtConvertAndStore(w,XtRString,&from,GfRSortType,&to);
    if (to.addr != NULL) {
	type = (SortType)(*(to.addr));
	appResources.sortType = type;
	status1("Set sort type to %s",*params);
	updateSettingsSortType(type);
    }
}

/*ARGSUSED*/
static
ACTION_PROC(setHostNowAction)
{
    if (*num_params != 1) {
	alert0("Incorrect number of arguments to set-host-now()");
	return;
    }
    XtFree(appResources.archieHost);
    appResources.archieHost = XtNewString(*params);
    status1("Set host to %s",*params);
    setWidgetString(setHostText,*params);
}

/*ARGSUSED*/
static
ACTION_PROC(setNiceLevelNowAction)
{
    int n;

    if (*num_params != 1) {
	alert0("Incorrect number of arguments to set-nice-level-now()");
	return;
    }
    n = atoi(*params);
    if (n < RDGRAM_MIN_PRI) {
	alert1("Nice level too negative: %d -- not set",(char *)n);
    } else if (n > RDGRAM_MAX_PRI) {
	alert1("Nice level too positive: %d -- not set",(char *)n);
    } else {
	appResources.niceLevel = n;
	status1("Set niceLevel to %d",(char *)n);
	setWidgetString(setNiceText,*params);
    }
}

/*	-	-	-	-	-	-	-	-	*/
/*
 * textEventProc() : Called whenever the user types in any Text item.
 *      Note that this does NOT detect, eg., selection pastes, as
 *	documented in the BUGS section of the man page.
 */
/*ARGSUSED*/
static void
textEventProc(w,client_data,event,continue_flag)
Widget w;
XtPointer client_data;
XEvent *event;
Boolean *continue_flag;
{
    setSettingsChangedFlag(True);
}

/*	-	-	-	-	-	-	-	-	*/

void
updateSettingsHost(str)
char *str;
{
    setWidgetString(setHostText,str);
}

void
updateSettingsSearchType(type)
SearchType type;
{
    currentSearchType = type;
    setWidgetLabel(setSearchLabel,searchTypeToString(type));
}

void
updateSettingsSortType(type)
SortType type;
{
    currentSortType = type;
    setWidgetLabel(setSortLabel,sortTypeToString(type));
    if (type == GfWeight) {
	XtMapWidget(setHostWeightsLabel);
	XtMapWidget(setHostWeightsText);
    } else {
	XtUnmapWidget(setHostWeightsLabel);
	XtUnmapWidget(setHostWeightsText);
    }
}

void
updateSettingsNiceLevel(level)
int level;
{
    char buf[16];

    sprintf(buf,"%d",level);
    setWidgetString(setNiceText,buf);
}

void
updateSettingsAutoScroll(flag)
Boolean flag;
{
    currentAutoScroll = flag;
    setWidgetLabel(setAutoScrollLabel,(flag?"yes":"no"));
}

void
updateSettingsFtpType(type)
String type;
{
    setWidgetLabel(ftpTypeLabel,type);
}

void
updateSettingsFtpPrompt(flag)
Boolean flag;
{
    currentFtpPrompt = flag;
    setWidgetLabel(ftpPromptLabel,(flag?"yes":"no"));
}

void
updateSettingsFtpTrace(flag)
Boolean flag;
{
    currentFtpTrace = flag;
    setWidgetLabel(ftpTraceLabel,(flag?"yes":"no"));
}

void
updateSettingsFtpStrip(flag)
Boolean flag;
{
    currentFtpStrip = flag;
    setWidgetLabel(ftpStripLabel,(flag?"yes":"no"));
}

/*	-	-	-	-	-	-	-	-	*/

void
setSettingsChangedFlag(value)
Boolean value;
{
    if (setApplyButton != NULL)
	XtSetSensitive(setApplyButton,value);
    settingsChanged = value;
}

/*	-	-	-	-	-	-	-	-	*/
/* Routines for the popup Dismiss confirmer */

static Widget settingsConfirmShell;
static int settingsConfirmResult;

static int
settingsConfirm()
{
    if (settingsConfirmShell == NULL)
	settingsConfirmShell =
	    createPopup("settingsConfirm",3,settingsConfirmCallback);
    setPopupLabel(settingsConfirmShell,"settingsConfirm",
		  "Dismiss without applying changes?");
    popupMainLoop(settingsConfirmShell);
    return(settingsConfirmResult);
}
 
/*ARGSUSED*/
static void
settingsConfirmCallback(w,client_data,call_data)
Widget w;
XtPointer client_data;		/* button number */
XtPointer call_data;
{
    switch ((int)client_data) {
      case 0:			/* Dismiss */
	settingsConfirmResult = 1;
	break;
      case 1:			/* Apply */
	applySettingsAction(w,NULL,NULL,0);
	settingsConfirmResult = 1;
	break;
      case 2:			/* Cancel */
	settingsConfirmResult = 0;
	break;
    }
    popupDone();
}
