//
// File:        StubDoc.java
// Package:     gov.llnl.babel.backend.fortran
// Revision:    @(#) $Revision: 4434 $
// Description: Generate a pseudo-code documentation file for FORTRAN clients
//
// Copyright (c) 2000-2002, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
//
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
//
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA


package gov.llnl.babel.backend.fortran;

import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.fortran.Fortran;
import gov.llnl.babel.backend.fortran.StubSource;
import gov.llnl.babel.backend.writers.LanguageWriterForFortran;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * This class provides the ability to write a FORTRAN file with a
 * subroutine template for each method the end user has to implement to
 * implement a sidl class. The class will retain the previous user provided
 * implmentation when overwriting a implementation file.
 */
public class StubDoc {

  /**
   * This is the output device.
   */
  private LanguageWriterForFortran d_lw;

  /**
   * Indicator of version of FORTRAN being generated.
   */
  /**
   * Generate an instance to generate documentation for a FORTRAN 
   * client.
   * 
   * @param writer    the output device to which the FORTRAN documentation
   *                  should be written.
   */
  public StubDoc(LanguageWriterForFortran writer) {
    d_lw     = writer;
  }

  /**
   * Write the FORTRAN documentation for a subroutine that 
   * connects to a sidl method. This writes the method
   * signature and declares the types of the arguments.
   * 
   * @param m     the method whose implementation template is to be
   *              written.
   * @param id    the name of the class that owns this method.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception to indicate problems in the code generation
   *   phase.
   */
  private void extendAndGenerate(Method   m,
                                 SymbolID id)
    throws CodeGenerationException
  {
    List extendedArgs = StubSource.extendArgs(id, m, false);
    d_lw.writeComment(m, false);
    d_lw.print("subroutine " + Fortran.getMethodStubName(id, m) +
               "(");
    Iterator i = extendedArgs.iterator();
    while (i.hasNext()) {
      d_lw.print(((Argument)i.next()).getFormalName());
      if (i.hasNext()) {
        d_lw.print(", ");
      }
    }
    d_lw.println(")");
    d_lw.println("implicit none");
    i = Fortran.reorderArguments(extendedArgs).iterator();
    while (i.hasNext()) {
      Argument a = (Argument)i.next();
      d_lw.writeCommentLine(a.getArgumentString());
      d_lw.print(Fortran.getReturnString(a.getType()) + " " 
                 + a.getFormalName());
      if (a.getType().isRarray()) {
        d_lw.println(Fortran.arrayIndices(a.getType().getArrayIndices()));
      }
      else {
        d_lw.println();
      }
    }
    d_lw.println("end");
  }

  /**
   * Generate a create method object.
   */
  private Method create(SymbolID id)
  {
    Method m = new Method();
    m.setMethodName("_create");
    m.setDefinitionModifier(Method.STATIC);
    String[] s = new String[1];
    s[0] = "Create an instance of class " + id.getFullName();
    m.setComment(new Comment(s));
    m.setReturnType(new Type(id));
    return m;
  }

  /**
   * Add the implicit stub methods to the list of those that must
   * be documented.
   *
   * @param ext the class whose implementation is being written.
   */
  private Collection extendMethods(Extendable ext) 
    throws CodeGenerationException {
    Collection allMethods = ext.getMethods(true);
    final SymbolID id = ext.getSymbolID();
    ArrayList  extendedMethods = new ArrayList(allMethods.size()+3);
    if (!ext.isAbstract()) {
      extendedMethods.add(create(id));
    }
    extendedMethods.add(Fortran.createCast(id));
    extendedMethods.add(Fortran.createCastTwo(id));

    extendedMethods.addAll(allMethods);
    return extendedMethods;
  }

  /**
   * Generate the implementation FORTRAN file for a sidl class.  The
   * implementation file contains all the subroutines that need to be
   * implemented for the class, and when replacing an implementation file, 
   * the previous implementation is retained.
   * 
   * @param ext    the sidl class whose implementation is to be written.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception to indicate problems in the code generation
   *   phase.
   */
  public void generateCode(Extendable ext)
    throws CodeGenerationException
  {
    Collection methods = extendMethods(ext);
    final SymbolID id = ext.getSymbolID();

    d_lw.writeBanner(ext, Fortran.getImplFile(id), false,
                     CodeConstants.C_FORTRAN_DESC_STUB_PREFIX 
                     + id.getFullName());
    d_lw.println("THIS FILE IS DOCUMENTATION - DO NOT COMPILE IT!");
    d_lw.println();
    d_lw.writeComment(ext, false);
    d_lw.println();

    Iterator i = methods.iterator();
    while (i.hasNext()) {
      d_lw.println();
      d_lw.println();

      extendAndGenerate((Method)i.next(), id);
    }
  }

  /**
   * Generate the implementation FORTRAN file for a sidl class.  The
   * implementation file contains all the subroutines that need to be
   * implemented for the class, and when replacing an implementation file, 
   * the previous implementation is retained.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception to indicate problems in the code generation
   *   phase.
  */
  public static void generateCode(Symbol                   sym,
                                  LanguageWriterForFortran writer)
    throws CodeGenerationException
  {
    if (sym instanceof Extendable) {
      StubDoc doc = new StubDoc(writer);
      doc.generateCode((Extendable)sym);
    }
  }
}
