//
// File:        PythonClientCSource.java
// Package:     gov.llnl.babel.backend.python
// Revision:    @(#) $Revision: 4434 $
// Date:        $Date: 2005-03-17 09:05:29 -0800 (Thu, 17 Mar 2005) $
// Description: Generate the C source file for a Python for an extendable
// 
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.python;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.LevelComparator;
import gov.llnl.babel.backend.python.Python;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Interface;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.Type;
import java.util.LinkedList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;


/**
 * This class will write the C source file for a Python C extension
 * module to provide access from Python to a class or interface.
 */
public class PythonClientCSource {
  /**
   * The short name for my extendable.
   */
  private String             d_shortName     = null;

  private Set                d_includedFiles;

  /**
   * The class or interface for which a Python C extension module
   * is being written.
   */
  private Extendable         d_ext           = null;

  /**
   * A writer for the C source file.
   */
  private LanguageWriterForC d_lw            = null;

  /**
   * A table that knows how to convert a <code>SymbolID</code> or
   * string into a <code>Symbol</code>.
   */
  private SymbolTable        d_symtab        = null;

  /**
   * The types that this extendable references as a method argument,
   * return value, parent interface or something like that.
   */
  private Collection         d_references    = null;

  /**
   * <code>true</code> means this extenable has one or more static
   * methods; <code>false</code> means this extendable has
   * exactly zero static methods.
   */
  private boolean            d_hasStaticMethods;

  private boolean            d_isBaseInterface;

  private String []          d_externalMethods;

  /**
   * Create an object capable of generating the source C file for a
   * sidl extendable (i.e. class or interface).
   *
   * @param ext    an interface or class that needs a header
   *               file for a Python C extension class.
   */
  public PythonClientCSource(Extendable ext) {
    Type extType    = new Type(ext.getSymbolID());
    d_ext           = ext;
    d_shortName     = d_ext.getSymbolID().getShortName();
    d_symtab        = SymbolTable.getInstance();
    d_references = Utilities.convertIdsToSymbols(ext.getSymbolReferences());
    d_hasStaticMethods = d_ext.hasStaticMethod(true);
    d_includedFiles = new HashSet();
    d_isBaseInterface = BabelConfiguration.getBaseInterface().
      equals(ext.getFullName());
    if (d_isBaseInterface) {
      d_externalMethods = new String[10];
      d_externalMethods[9] = Python.getBorrowArrayFromSIDL(null);
    }
    else {
      d_externalMethods = new String[9];
    }
    d_externalMethods[0] = Python.getExtendableWrapper(d_ext);
    d_externalMethods[1] = Python.getExtendableConverter(d_ext);
    d_externalMethods[2] = Python.getBorrowArrayFromPython(extType);
    d_externalMethods[3] = Python.getBorrowArrayFromSIDL(extType);
    d_externalMethods[4] = Python.getExtendableBorrow(d_ext);
    d_externalMethods[5] = Python.getExtendableDeref(d_ext);
    d_externalMethods[6] = Python.getExtendableNewRef(d_ext);
    d_externalMethods[7] = Python.getExtendableAddRef(d_ext);
    d_externalMethods[8] = Python.getExtendableType(d_ext);
  }

  /**
   * Write a comment explaining the contents of the source file to 
   * anyone who might happen to read it.
   */
  private void explainExtensionSource() {
    d_lw.beginBlockComment(false);
    d_lw.println("THIS CODE IS AUTOMATICALLY GENERATED BY THE BABEL");
    d_lw.println("COMPILER. DO NOT EDIT THIS!");
    d_lw.println();
    d_lw.println("This file contains the implementation of a Python C");
    d_lw.println("extension type (i.e. a Python type implemented in C).");
    d_lw.println("This extension type provides Python interface to the");
    d_lw.print("sidl type ");
    d_lw.print(d_ext.getFullName());
    d_lw.println(".");
    d_lw.endBlockComment(false);
    d_lw.println();
  }

  private void addInclude(String filename, boolean useGuard) {
    if (!d_includedFiles.contains(filename)) {
      d_includedFiles.add(filename);
      d_lw.generateInclude(filename, useGuard);
    }
  }

  /**
   * Add <code>#include</code> lines for all the system headers and
   * the referenced types.
   */
  private void includeHeaderFiles() {
    d_lw.printUnformatted("#define ");
    d_lw.printUnformatted(Python.getInternalGuard(d_ext));
    d_lw.printlnUnformatted(" 1");
    addInclude(Python.getCHeaderPath(d_ext, "Module"), false);
    addInclude(IOR.getHeaderFile(d_ext.getSymbolID()), true);
    addInclude("sidlObjA.h", false);
    addInclude("sidlPyArrays.h", false);
    addInclude("Numeric/arrayobject.h", false);
    addInclude("sidl_Loader.h", true);
    addInclude("sidl_header.h", true);
    addInclude("sidl_interface_IOR.h", true);
    Iterator i = d_references.iterator();
    while (i.hasNext()) {
      Symbol sym = (Symbol)i.next();
      if (!(Symbol.ENUM == sym.getSymbolType())) {
        addInclude(Python.getCHeaderPath(sym, "Module"), false);
      }
    }
    d_lw.printlnUnformatted("#include <stdlib.h>");
    d_lw.printlnUnformatted("#include <string.h>");
    d_lw.println();
  }

  /**
   * Return <code>true</code> if and only if the extendable is
   * a class that is the base exception class, is an interface that is
   * the base exception interface, or it has the base exception class or 
   * interface in its type ancestry.
   */
  private boolean isException() {
    final Symbol exceptionInterface = d_symtab
      .lookupSymbol(BabelConfiguration.getBaseExceptionInterface());
    final Symbol exceptionClass = d_symtab
      .lookupSymbol(BabelConfiguration.getBaseExceptionClass());
    SymbolID exceptionCID = null;
    SymbolID exceptionIID = null;
    if (exceptionClass != null) {
      exceptionCID = exceptionClass.getSymbolID();
    }
    if (exceptionInterface != null) {
      exceptionIID = exceptionInterface.getSymbolID();
    }
    if ( (exceptionCID != null) && (exceptionIID != null) ) {
      if (d_ext instanceof Class) {
        return 
          d_ext.getSymbolID().equals(exceptionCID)
          || ((Class)d_ext).hasParentClass(exceptionCID, true)
          || d_ext.hasParentInterface(exceptionIID, true);
      } else if (d_ext instanceof Interface) {
        return 
          d_ext.getSymbolID().equals(exceptionIID)
          || d_ext.hasParentInterface(exceptionIID, true);
      }
    }
    return false;
  }

  /**
   * If the extendable has static methods, create a static
   * variable to hold the classes static entry point
   * vector.
   */
  private void staticMethods() {
    d_lw.println("staticforward PyTypeObject _" +
                 d_ext.getFullName().replace('.','_')
                 + "Type;");
    d_lw.println();
    if (d_hasStaticMethods){
      d_lw.print("static ");
      d_lw.print(IOR.getSEPVName(d_ext.getSymbolID()));
      d_lw.print(" *_sepv");
      d_lw.println(" = NULL;");
      d_lw.println();
    }
    if (!d_ext.isInterface()) {
      d_lw.println("static const " + 
                   IOR.getExternalName(d_ext.getSymbolID()) + 
                   " *_implEPV = NULL;");
      d_lw.println();
    }
  }

  /**
   * Write out the function signature for the function to
   * handle cast and create requests.
   */
  private void pythonCastSignature() {
    d_lw.println("static int");
    d_lw.println(d_ext.getFullName().replace('.','_') +
                 "_createCast(PyObject *self, PyObject *args, PyObject *kwds) {");
    d_lw.tab();
  }

  /**
   * Write out code to do the input argument processing for the
   * cast function. The success of input processing is stored
   * the in the variable <code>_okay</code>.
   */
  private void pythonCastInArguments() {
    final SymbolID id = d_ext.getSymbolID();
    d_lw.print(IOR.getObjectName(id));
    d_lw.println(" *optarg = NULL;");
    d_lw.println("static char *_kwlist[] = { \"sobj\", NULL };");
    d_lw.print("int _okay = PyArg_ParseTupleAndKeywords(args, kwds, \"");
    if (!d_ext.isAbstract()) {
      d_lw.print("|");
    }
    d_lw.print("O&\", _kwlist, (void *)");
    d_lw.print(Python.getExtendableConverter(d_ext));
    d_lw.println(", &optarg);");
  }

  /**
   * If this a create object request and the class is not abstract,
   * call the constructor to create a new sidl object.
   */
  private void pythonCastCreate() {
    if (!d_ext.isAbstract()) {
      d_lw.println("if (!optarg) {");
      d_lw.tab();
      d_lw.print("optarg = (*(_implEPV->createObject))");
      d_lw.println("();");
      d_lw.backTab();
      d_lw.println("}");
    }
  }

  /**
   * Wrap up the IOR in a wrapper Python class.
   */
  private void pythonCastWrapAndReturn() {
    d_lw.println("return sidl_Object_Init(");
    d_lw.tab();
    d_lw.println("(SPObject *)self,");
    if (d_ext.isInterface()) {
      d_lw.println("(struct sidl_BaseInterface__object *)optarg->d_object,");
    }
    else {
      d_lw.println("(struct sidl_BaseInterface__object *)optarg,");
    }
    d_lw.println("sidl_PyStealRef);");
    d_lw.backTab();
  }

  /**
   * Write the Python cast/create function. This method is called
   * when the Python user says X.X() or X.X(obj).  This function
   * is exposed as a static method.
   */
  private void pythonCastConstructor() {
    SymbolID id = d_ext.getSymbolID();
    pythonCastSignature();
    pythonCastInArguments();
    d_lw.println("if (_okay) {");
    d_lw.tab();
    pythonCastCreate();
    pythonCastWrapAndReturn();
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("return -1;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  private void pythonDocComment()
  {
    if ((d_ext.getComment() != null) && 
        (d_ext.getComment().getComment() != null)) {
      copyComment(d_ext.getComment());
    }
    else {
      d_lw.printlnUnformatted("SIDL wrapper for the " +
                   d_ext.getFullName() + " type.\\");
    }
  }

  private void writePythonType() {
    final SymbolID id = d_ext.getSymbolID();
    d_lw.println("static PyTypeObject _" + id.getFullName().replace('.','_')
                 + "Type = {");
    d_lw.tab();
    d_lw.println("PyObject_HEAD_INIT(NULL)");
    d_lw.println("0,      /* ob_size */");
    d_lw.println("\"" + id.getFullName() + "\", /* tp_name */");
    d_lw.println("0,      /* tp_basicsize */");
    d_lw.println("0,      /* tp_itemsize */");
    d_lw.println("0,      /* tp_dealloc */");
    d_lw.println("0,      /* tp_print */");
    d_lw.println("0,      /* tp_getattr */");
    d_lw.println("0,      /* tp_setattr */");
    d_lw.println("0,      /* tp_compare */");
    d_lw.println("0,      /* tp_repr */");
    d_lw.println("0,      /* tp_as_number */");
    d_lw.println("0,      /* tp_as_sequence */");
    d_lw.println("0,      /* tp_as_mapping */");
    d_lw.println("0,      /* tp_hash  */");
    d_lw.println("0,      /* tp_call */");
    d_lw.println("0,      /* tp_str */");
    d_lw.println("0,      /* tp_getattro */");
    d_lw.println("0,      /* tp_setattro */");
    d_lw.println("0,      /* tp_as_buffer */");
    d_lw.println("Py_TPFLAGS_DEFAULT, /* tp_flags */");
    d_lw.println("\"\\");
    pythonDocComment();
    d_lw.printlnUnformatted("\", /* tp_doc */");
    d_lw.println("0,      /* tp_traverse */");
    d_lw.println("0,       /* tp_clear */");
    d_lw.println("0,       /* tp_richcompare */");
    d_lw.println("0,       /* tp_weaklistoffset */");
    d_lw.println("0,       /* tp_iter */");
    d_lw.println("0,       /* tp_iternext */");
    d_lw.println("_" + d_shortName + 
                 "ObjectMethods, /* tp_methods */");
    d_lw.println("0,       /* tp_members */");
    d_lw.println("0,       /* tp_getset */");
    d_lw.println("0,       /* tp_base */");
    d_lw.println("0,       /* tp_dict */");
    d_lw.println("0,       /* tp_descr_get */");
    d_lw.println("0,       /* tp_descr_set */");
    d_lw.println("0,       /* tp_dictoffset */");
    d_lw.println(id.getFullName().replace('.','_') +
                 "_createCast,   /* tp_init */");
    d_lw.println("0,       /* tp_alloc */");
    d_lw.println("0,       /* tp_new */");
    d_lw.backTab();
    d_lw.println("};");
    d_lw.println();
  }

  /**
   * Write a method signature for a static class method or a normal
   * class method. The argument list is determine by the Python
   * extension standards.  For normal class methods, it includes
   * code to fetch the IOR and test if it is non-<code>NULL</code>.
   *
   * @param m   the method whose signature is to be written.
   */
  private void writeMethodSignature(Method m) {
    d_lw.println("static PyObject *");
    d_lw.print(Python.getStubMethod(d_ext.getSymbolID(),m));
    if (m.isStatic()){
      d_lw.println("(PyObject *_ignored, PyObject *_args, PyObject *_kwdict) {");
    }
    else {
      d_lw.println("(PyObject *_self, PyObject *_args, PyObject *_kwdict) {");
    }
    d_lw.tab();
    d_lw.println("PyObject *_return_value = NULL;");
    if (!m.isStatic()) {
      SymbolID id = d_ext.getSymbolID();
      d_lw.print(IOR.getObjectName(id));
      d_lw.println(" *_self_ior =");
      d_lw.tab();
      d_lw.print("((");
      d_lw.print(IOR.getObjectName(id));
      d_lw.println(" *)");
      d_lw.print(" sidl_Cast(_self, \"");
      d_lw.print(id.getFullName());
      d_lw.println("\"));");
      d_lw.backTab();
      d_lw.println("if (_self_ior) {");
      d_lw.tab();
    }
  }

  /**
   * Write a type declaration for a variable.
   * 
   * @param t        the type of the variable.
   * @param name     the formal name of the variable.
   * @param mode     the mode (in, inout or out) of the parameter.
   * @exception gov.llnl.babel.backend.CodeGeneration
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void declareVariable(Type t,
                               String name,
                               final int mode)
    throws CodeGenerationException
  {
    d_lw.print(IOR.getReturnString(t));
    d_lw.print(" ");
    d_lw.print(name);
    if (Utilities.isPointer(t)) {
      d_lw.print(" = NULL");
    } else if (t.getDetailedType() == Type.DCOMPLEX) {
      d_lw.print(" = { 0.0, 0.0 }");
    } else if (t.getDetailedType() == Type.FCOMPLEX) {
      d_lw.print(" = { 0.0, 0.0 }");
    } else {
      d_lw.print(" = (" + IOR.getReturnString(t) + ") 0");
    }
    d_lw.println(";");
  }

  /**
   * Declare variables to hold the incoming and outgoing arguments.
   *
   * @param m   the method whose arguments will be declared.
   * @exception gov.llnl.babel.backend.CodeGeneration
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void declareArguments(Method m) 
    throws CodeGenerationException 
  {
    Iterator i = m.getArgumentList().iterator();
    while (i.hasNext()) {
      Argument arg = (Argument)i.next();
      declareVariable(arg.getType(), arg.getFormalName(), arg.getMode());
    }
    if (!m.getThrows().isEmpty()) {
      d_lw.print(IOR.getExceptionFundamentalType());
      d_lw.println("_exception = NULL;");
    }
  }

  /**
   * Return <code>true</code> if and only if this method has a Python return
   * value. A method has a return value in Python if it has <code>out</code>
   * or <code>inout</code> parameters or if it has a non-<code>void</code>
   * return type.
   *
   * @param m  the method to interrogate about whether it has
   *           a return value in Python.
   * @return   <code>true</code> if and only if this method has a Python
   *           return value.
   */
  private boolean hasReturn(Method m) {
    if (m.getReturnType().getType() != Type.VOID) {
      return true;
    }
    Iterator i = m.getArgumentList().iterator();
    while (i.hasNext()) {
      if (isOutgoing((Argument)i.next())) {
        return true;
      }
    }
    return false;
  }

  /**
   * Return <code>true</code> if and only if the argument is an 
   * <code>out</code> or <code>inout</code> argument.
   *
   * @param arg  the argument whose direction is interrogated.
   * @return     <code>true</code> means the argument has an
   *             output value.
   */
  private final static boolean isOutgoing(Argument arg) {
    final int mode = arg.getMode();
    return (mode == Argument.OUT) || (mode == Argument.INOUT);
  }

  /**
   * Return <code>true</code> if and only if the argument is an 
   * <code>in</code> or <code>inout</code> argument.
   *
   * @param arg  the argument whose direction is interrogated.
   * @return     <code>true</code> means the argument has an
   *             input value.
   */
  private final static boolean isIncoming(Argument arg) {
    final int mode = arg.getMode();
    return (mode == Argument.IN) || (mode == Argument.INOUT);
  }
  
  /**
   * Write code to declare and initialize a static variable holding
   * a <code>NULL</code> terminated list of parameter names.
   * 
   * @param m    the method whose argument names are to be listed.
   */
  private void listArgumentNames(Method m) {
    Iterator i = m.getArgumentList().iterator();
    d_lw.println("static char *_kwlist[] = {");
    d_lw.tab();
    while (i.hasNext()) {
      Argument a = (Argument)i.next();
      if (isIncoming(a)) {
        d_lw.print("\"");
        d_lw.print(a.getFormalName());
        d_lw.println("\",");
      }
    }
    d_lw.println("NULL");
    d_lw.backTab();
    d_lw.println("};");
  }


  /**
   * Write code to call the sidl method using the static entry point vector
   * or the entry point vector in the IOR.
   *
   * @param m           the method to call.
   * @param isInterface <code>true</code> if the method is on an interface;
   *                    <code>false</code> if the method is on an object.
   */
  private void callMethod(Method m, boolean isInterface) {
    boolean addComma = false;
    if (m.getReturnType().getType() != Type.VOID) {
      d_lw.print("_return = ");
    }
    if (m.isStatic()) {
      d_lw.print("(*(_sepv->");
    }
    else {
      d_lw.print("(*(_self_ior->d_epv->");
    }
    d_lw.print(IOR.getVectorEntry(m.getLongMethodName()));
    d_lw.print("))(");
    if (!m.isStatic()) {
      if (isInterface) {
        d_lw.print("_self_ior->d_object");
      }
      else {
        d_lw.print("_self_ior");
      }
      addComma = true;
    }
    Iterator i = m.getArgumentList().iterator();
    while (i.hasNext()) {
      Argument arg = (Argument)i.next();
      if (addComma) {
        d_lw.print(", ");
      }
      if (isOutgoing(arg)) {
        d_lw.print("&");
      }
      d_lw.print(arg.getFormalName());
      addComma = true;
    }
    if (!m.getThrows().isEmpty()) {
      if (addComma) {
        d_lw.print(", ");
      }
      d_lw.print("&_exception");
      addComma = true;
    }
    d_lw.println(");");
  }

  /**
   * Write the code to determine which exception was thrown and to create
   * the corresponding Python exception.
   *
   * @param m   a method with throws.
   * @exception gov.llnl.babel.backend.CodeGeneration
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void processExceptions(Method m)
    throws CodeGenerationException 
  {
    Object [] exceptions = m.getThrows().toArray();
    Arrays.sort(exceptions, new LevelComparator(d_symtab));
    int i;
    /* declare exception pointers */
    for(i = 0; i < exceptions.length ; ++i) {
      d_lw.print(IOR.getObjectName((SymbolID)exceptions[i]));
      d_lw.print(" *_ex");
      d_lw.print(Integer.toString(i));
      d_lw.println(";");
    }
    /* test each exception in order */
    for(i = 0; i < exceptions.length; ++i) {
      SymbolID id = (SymbolID)exceptions[i];
      Symbol sym = d_symtab.lookupSymbol(id);
      if (i > 0) {
        d_lw.print("else ");
      }
      d_lw.print("if ((_ex");
      d_lw.print(Integer.toString(i));
      d_lw.print(" = ");
      d_lw.print("(");
      d_lw.print(IOR.getObjectName(id));
      d_lw.println(" *)");
      d_lw.tab();
      d_lw.print("sidl_PyExceptionCast(_exception, \"");
      d_lw.print(id.getFullName());
      d_lw.println("\")))");
      d_lw.backTab();
      d_lw.println("{");
      d_lw.tab();
      d_lw.print("PyObject *obj = ");
      d_lw.print(Python.getExtendableWrapper(sym));
      d_lw.print("(_ex");
      d_lw.print(Integer.toString(i));
      d_lw.println(");");
      d_lw.print("PyErr_SetObject(");
      d_lw.print(Python.getExceptionType(sym));
      d_lw.println(", obj);");
      d_lw.println("Py_XDECREF(obj);");
      d_lw.backTab();
      d_lw.println("}");
    }
  }

  /**
   * Write the code to check for exceptions, build the return value and/or
   * prepare a void return.  First check for exceptions if necessary. If
   * there is no exception, either build the return value or prepare to
   * return <code>Py_None</code> (the Python void equivalent).
   *
   * @param m   the method whose return value processing will be written.
   * @exception gov.llnl.babel.backend.CodeGeneration
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void processReturnValues(Method m,
                                   TranslateArguments translator)
    throws CodeGenerationException 
  {
    boolean hasThrows = !m.getThrows().isEmpty();
    if (hasThrows) {
      d_lw.println("if (_exception) {");
      d_lw.tab();
      processExceptions(m);
      d_lw.backTab();
      d_lw.println("}");
      d_lw.println("else {");
      d_lw.tab();
    }
    if (hasReturn(m)) {
      d_lw.print("_return_value = ");
      translator.setConvertIncoming(false);
      translator.convertSidlToPython();
    }
    else {
      d_lw.println("_return_value = Py_None;");
      d_lw.println("Py_INCREF(_return_value);");
    }
    if (hasThrows) {
      d_lw.backTab();
      d_lw.println("}");
    }
  }

  /**
   * Write the closing part of method.
   * 
   * @param m   the method whose closing will be written.
   */
  private void writeMethodClosing(Method m) {
    if (!m.isStatic()) {
      d_lw.backTab();
      d_lw.println("}");
      d_lw.println("else {");
      d_lw.tab();
      d_lw.println("PyErr_SetString(PyExc_TypeError, ");
      d_lw.tab();
      d_lw.disableLineBreak();
      d_lw.print("\"self pointer is not a ");
      d_lw.print(d_ext.getFullName());
      d_lw.println("\");");
      d_lw.enableLineBreak();
      d_lw.backTab();
      d_lw.backTab();
      d_lw.println("}");
    }
    d_lw.println("return _return_value;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  /**
   * Declare a variable to hold the method return value.
   * 
   * @exception gov.llnl.babel.backend.CodeGeneration
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void declareReturnValue(Method m,
                                  TranslateArguments translator) 
    throws CodeGenerationException 
  {
    if (m.getReturnType().getType() != Type.VOID) {
      declareVariable(m.getReturnType(), "_return", Argument.OUT);
      translator.declareProxy(m.getReturnType(),
                              "_return", Argument.OUT);
    }
  }

  /**
   * Call the array destructor for this argument.
   *
   * @param arrayType  the type of the array.
   * @param argName    the name of the argument to destroy.
   */
  private void destroyArrayArg(Type arrayType,
                               String argName)
  {
    d_lw.print(Python.getDestroyArray(arrayType));
    d_lw.print("((struct sidl__array *)");
    d_lw.print(argName);
    d_lw.println(");");
  }

  /**
   * Perform any clean up activities required.
   *
   * @param m  the method.
   * @exception gov.llnl.babel.backend.CodeGeneration
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void cleanUpArguments(Method m)
    throws CodeGenerationException
  {
    Iterator i = m.getArgumentList().iterator();
    while (i.hasNext()) {
      Argument a = (Argument)i.next();
      switch(a.getType().getDetailedType()) {
      case Type.STRING:
        if (isOutgoing(a)) {
          d_lw.print("free((void *)");
          d_lw.print(a.getFormalName());
          d_lw.println(");");
        }
        break;
      case Type.INTERFACE:
      case Type.CLASS:
        if (Argument.IN == a.getMode()) {
          Extendable ext = 
            (Extendable)Utilities.lookupSymbol(a.getType().getSymbolID());
          d_lw.println(Python.getExtendableDeref(ext) +
                       "(" + a.getFormalName() + ");");
        }
        break;
      case Type.ARRAY:
        destroyArrayArg(a.getType().getArrayType(),
                        a.getFormalName());
        break;
      }
    }
    if (Type.STRING == m.getReturnType().getType()) {
      d_lw.println("free((void *)_return);");
    }
    else if (Type.ARRAY == m.getReturnType().getType()) {
      destroyArrayArg(m.getReturnType().getArrayType(), "_return");
    }
  }

  /**
   * Write a function in C that will be exposed in Python to execute a sidl
   * method (static or normal). This is the overall manager for writing
   * the method.
   * 
   * @param m  the method.
   * @param isInterface <code>true</code> if the method is on an interface;
   *                    <code>false</code> if the method is on an object.
   * @exception gov.llnl.babel.backend.CodeGeneration
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void convertMethod(Method m, boolean isInterface) 
    throws CodeGenerationException 
  {
    TranslateArguments 
      translator  = new TranslateArguments(d_lw, m, true, true);
    writeMethodSignature(m);
    declareArguments(m);
    translator.declareProxies();
    listArgumentNames(m);
    translator.setBorrowArrays(true);
    translator.setConvertIncoming(true);
    d_lw.print("const int _okay = ");
    translator.convertPythonToSidl("_args", "_kwdict", "_kwlist");
    d_lw.println("if (_okay) {");
    d_lw.tab();
    declareReturnValue(m, translator);
    translator.convertIncomingArguments(false);
    callMethod(m, isInterface);
    translator.setConvertIncoming(false);
    translator.setBorrowArrays(false);
    translator.convertOutgoingArguments(true);
    processReturnValues(m, translator);
    cleanUpArguments(m);
    d_lw.backTab();
    d_lw.println("}");
    writeMethodClosing(m);
  }

  /**
   * Return the list of methods only defined first in this extendable. This
   * excludes any methods that were defined in a parent class or interface.
   * 
   * @param includeStatics if <code>true</code> the return value has
   *                       static and non-static methods.
   */
  private Collection pythonMethodList(boolean includeStatics)
  {
    HashSet localMethods = new HashSet
      (includeStatics ? d_ext.getMethods(false) 
       : d_ext.getNonstaticMethods(false));
    if (!localMethods.isEmpty()) {
      Iterator i = d_ext.getParentInterfaces(false).iterator();
      while (!localMethods.isEmpty() && i.hasNext()) {
        localMethods.removeAll(((Extendable)i.next()).
                               getNonstaticMethods(true));
      }
    }
    return localMethods;
  }

  /**
   * Write the functions in C that will be exposed in Python to execute the
   * sidl methods (static and/or normal). This converts each method in 
   * turn.
   * 
   * @param isInterface <code>true</code> if the method is on an interface;
   *                    <code>false</code> if the method is on an object.
   * @exception gov.llnl.babel.backend.CodeGeneration
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void convertMethods(boolean isInterface)
    throws CodeGenerationException 
  {
    Iterator i = pythonMethodList(true).iterator();
    while (i.hasNext()) {
      convertMethod((Method)i.next(), isInterface);
    }
  }

  /**
   * Convert a comment into a C string.
   *
   * @param c   the comment to convert
   */
  private void copyComment(Comment c) {
    if (c != null) {
      final String [] lines = c.getComment();
      if (lines != null) {
        int i = 0;
        d_lw.printlnUnformatted("\\");
        while (i < lines.length) {
          d_lw.printUnformatted
            (LanguageWriterForC.toSafeString(lines[i]));
          if (++i < lines.length) {
            d_lw.printlnUnformatted("\\n\\");
          }
        }
      }
    }
  }

  /**
   * Create a Python doc comment for this method.
   *
   * @parameter m the method to document
   */
  private void documentMethod(Method m)
  {
    d_lw.printlnUnformatted("\"\\");
    printMethodSignature(m);
    if (m.getComment() != null && 
        m.getComment().getComment() != null) {
      d_lw.printlnUnformatted("\\n\\");
      copyComment(m.getComment());
    }
    d_lw.printlnUnformatted("\"");
  }

  private String indentString(int width)
  {
    StringBuffer buf = new StringBuffer(width);
    while (width-- > 0){
      buf.append(' ');
    }
    return buf.toString();
  }

  private void writeArgumentList(Iterator args,
                                 final int excluding,
                                 final int indent,
                                 boolean notFirst)
  {
    final String indStr = indentString(indent);
    while (args.hasNext()) {
      Argument a = (Argument)args.next();
      if (excluding != a.getMode()) {
        if (notFirst) {
          d_lw.printlnUnformatted(",\\n\\");
          d_lw.printUnformatted(indStr);
        }
        else {
          notFirst = true;
        }
        d_lw.printUnformatted(a.getArgumentString());
      }
    }
    d_lw.printlnUnformatted(")\\n\\");
  }

  private void printMethodSignature(Method m)
  {
    final String name = m.getLongMethodName();
    d_lw.printUnformatted(name + "(");
    writeArgumentList(m.getArgumentList().iterator(), Argument.OUT, 
                      name.length() + 1, false);
    d_lw.printlnUnformatted("RETURNS\\n\\");
    if (hasReturn(m)){ 
      d_lw.printUnformatted("   (");
      if (m.getReturnType().getType() != Type.VOID) {
        d_lw.printUnformatted(m.getReturnType().getTypeString() +
                              " _return");
      }
      writeArgumentList(m.getArgumentList().iterator(), Argument.IN, 
                        4, m.getReturnType().getType() != Type.VOID);
    }
    else {
      d_lw.printlnUnformatted("    None\\n\\");
    }
    if (!m.getThrows().isEmpty()) {
      Iterator i = m.getThrows().iterator();
      d_lw.printlnUnformatted("RAISES\\n\\");
      while(i.hasNext()) {
        SymbolID id = (SymbolID)i.next();
        d_lw.printUnformatted("    ");
        d_lw.printUnformatted(id.getFullName());
        d_lw.printlnUnformatted("\\n\\");
      }
    }
  }

  /**
   * Create a Python virtual function table for a set of methods.
   *
   * @param i              the set of methods to include in the
   *                       virtual function table.
   */
  private void makeVirtualTable(Iterator i) {
    SymbolID id = d_ext.getSymbolID();
    while (i.hasNext()) {
      Method m = (Method)i.next();
      d_lw.print("{ \"");
      d_lw.print(m.getLongMethodName());
      d_lw.print("\", (PyCFunction)");
      d_lw.print(Python.getStubMethod(id, m));
      d_lw.println(",");
      d_lw.println("(METH_VARARGS | METH_KEYWORDS),");
      documentMethod(m);
      d_lw.println(" },");
    }
  }

  /**
   * Write the static method virtual function table for the Python module.
   * This table is needed to initialize the Python C extension module.
   * It includes the cast/create function and any static class methods
   * from the extendable.
   */
  private void pythonStaticVirtualTable() {
    d_lw.print("static PyMethodDef _");
    d_lw.print(d_shortName);
    d_lw.println("ModuleMethods[] = {");
    d_lw.tab();
    makeVirtualTable(Utilities.sort(d_ext.getStaticMethods(false)).iterator());
    d_lw.println("{ NULL, NULL }");
    d_lw.backTab();
    d_lw.println("};");
    d_lw.println();
  }

  /**
   * Write the virtual function table for the Python extension type.
   * This table is needed when creating an instance of the Python
   * object that wraps an IOR.
   */
  private void pythonObjectVirtualTable() {
    d_lw.print("static PyMethodDef _");
    d_lw.print(d_shortName);
    d_lw.println("ObjectMethods[] = {");
    d_lw.tab();
    makeVirtualTable(Utilities.sort(pythonMethodList(false))
                                         .iterator());
    d_lw.println("{ NULL, NULL }");
    d_lw.backTab();
    d_lw.println("};");
    d_lw.println();
  }

  /**
   * Write a function to "wrap" an IOR inside a Python C extension type.
   * This create a Python object implemented in C to expose the sidl
   * object/interface to Python clients.  This function is part of the
   * external API of the Python extension module.
   */
  private void wrapperMethod() {
    SymbolID id = d_ext.getSymbolID();
    String fullname = id.getFullName();
    String wrapper = Python.getExtendableWrapper(d_ext);
    d_lw.print(wrapper);
    d_lw.println("_RETURN");
    d_lw.print(wrapper);
    d_lw.print(" ");
    d_lw.print(wrapper);
    d_lw.println("_PROTO {");
    d_lw.tab();
    d_lw.println("PyObject *result;");
    d_lw.println("if (sidlobj) {");
    d_lw.tab();
    d_lw.println("result = _" +
                 id.getFullName().replace('.','_')
                 + "Type.tp_new(&_" +
                 id.getFullName().replace('.','_') + "Type, NULL, NULL);");
    d_lw.println("if (result) {");
    d_lw.tab();
    d_lw.println("if (sidl_Object_Init(");
    d_lw.tab();
    d_lw.println("(SPObject *)result,");
    d_lw.println("(" + IOR.getInterfaceType()+  ")(sidlobj" +
                 (d_ext.isInterface() ? "->d_object)," : "),"));
    d_lw.println("sidl_PyStealRef))");
    d_lw.backTab();
    d_lw.println("{");
    d_lw.tab();
    d_lw.println("Py_DECREF(result);");
    d_lw.println("result = NULL;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("else {");
    d_lw.tab();
    d_lw.println("result = Py_None;");
    d_lw.println("Py_INCREF(result);");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("return result;");
    
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  /**
   * Write a function to call delete reference on an extendable of this
   * type.
   */
  private void derefMethod()
  {
    final SymbolID id = d_ext.getSymbolID();
    final String fullname = id.getFullName();
    final String deref = Python.getExtendableDeref(d_ext);
    d_lw.print(deref);
    d_lw.println("_RETURN");
    d_lw.print(deref);
    d_lw.print(" ");
    d_lw.print(deref);
    d_lw.println("_PROTO {");
    d_lw.tab();
    d_lw.println("if (sidlobj) {");
    d_lw.tab();
    if (d_ext.isInterface()){
      d_lw.println("(*(sidlobj->d_epv->" + 
                   IOR.getVectorEntry("deleteRef") +
                   "))(sidlobj->d_object);");
    }
    else {
      d_lw.println("(*(sidlobj->d_epv->" + 
                   IOR.getVectorEntry("deleteRef") +
                   "))(sidlobj);");
    }
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  /**
   * Write a function to "wrap" an IOR inside a Python C extension type
   * with a weak reference. This create a Python object implemented in C
   * to expose the sidl object/interface to Python clients.  This function
   * is part of the external API of the Python extension module.
   */
  private void borrowMethod() {
    final SymbolID id = d_ext.getSymbolID();
    final String fullname = id.getFullName();
    final String borrow = Python.getExtendableBorrow(d_ext);
    d_lw.print(borrow);
    d_lw.println("_RETURN");
    d_lw.print(borrow);
    d_lw.print(" ");
    d_lw.print(borrow);
    d_lw.println("_PROTO {");
    d_lw.tab();
    d_lw.println("PyObject *result;");
    d_lw.println("if (sidlobj) {");
    d_lw.tab();
    d_lw.println("result = _" +
                 id.getFullName().replace('.','_')
                 + "Type.tp_new(&_" +
                 id.getFullName().replace('.','_') + "Type, NULL, NULL);");
    d_lw.println("if (result) {");
    d_lw.tab();
    d_lw.println("if (sidl_Object_Init(");
    d_lw.tab();
    d_lw.println("(SPObject *)result,");
    d_lw.println("(" + IOR.getInterfaceType() + ")(sidlobj" +
                 (d_ext.isInterface() ? "->d_object)," : "),"));
    d_lw.println("sidl_PyWeakRef))");
    d_lw.backTab();
    d_lw.println("{");
    d_lw.tab();
    d_lw.println("Py_DECREF(result);");
    d_lw.println("result = NULL;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("else {");
    d_lw.tab();
    d_lw.println("result = Py_None;");
    d_lw.println("Py_INCREF(result);");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("return result;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  /**
   * Write a function to "wrap" an IOR inside a Python C extension type
   * with a new reference. This create a Python object implemented in C
   * to expose the sidl object/interface to Python clients.  This function
   * is part of the external API of the Python extension module.
   */
  private void newRefMethod() {
    final SymbolID id = d_ext.getSymbolID();
    final String fullname = id.getFullName();
    final String newRef = Python.getExtendableNewRef(d_ext);
    d_lw.print(newRef);
    d_lw.println("_RETURN");
    d_lw.print(newRef);
    d_lw.print(" ");
    d_lw.print(newRef);
    d_lw.println("_PROTO {");
    d_lw.tab();
    d_lw.println("PyObject *result;");
    d_lw.println("if (sidlobj) {");
    d_lw.tab();
    d_lw.println("result = _" +
                 id.getFullName().replace('.','_')
                 + "Type.tp_new(&_" +
                 id.getFullName().replace('.','_') + "Type, NULL, NULL);");
    d_lw.println("if (result) {");
    d_lw.tab();
    d_lw.println("if (sidl_Object_Init(");
    d_lw.tab();
    d_lw.println("(SPObject *)result,");
    d_lw.println("(" + IOR.getInterfaceType() + ")(sidlobj" +
                 (d_ext.isInterface() ? "->d_object)," : "),"));
    d_lw.println("sidl_PyNewRef))");
    d_lw.backTab();
    d_lw.println("{");
    d_lw.tab();
    d_lw.println("Py_DECREF(result);");
    d_lw.println("result = NULL;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("else {");
    d_lw.tab();
    d_lw.println("result = Py_None;");
    d_lw.println("Py_INCREF(result);");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("return result;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  /**
   * Write a function to call addRef on a sidl class or interface.
   * This function is part of the external API of the Python extension module.
   */
  private void addRefMethod() {
    final SymbolID id = d_ext.getSymbolID();
    final String fullname = id.getFullName();
    final String addRef = Python.getExtendableAddRef(d_ext);
    d_lw.print(addRef);
    d_lw.println("_RETURN");
    d_lw.print(addRef);
    d_lw.print(" ");
    d_lw.print(addRef);
    d_lw.println("_PROTO {");
    d_lw.tab();
    d_lw.println("if (sidlobj) {");
    d_lw.tab();
    if (d_ext.isInterface()){
      d_lw.println("(*(sidlobj->d_epv->" +
                   IOR.getVectorEntry("addRef") + "))(sidlobj->d_object);");
    }
    else {
      d_lw.println("(*(sidlobj->d_epv->" +
                   IOR.getVectorEntry("addRef") + "))(sidlobj);");
    }
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  /**
   * Write a function to call addRef on a sidl class or interface.
   * This function is part of the external API of the Python extension module.
   */
  private void typeMethod() {
    final SymbolID id = d_ext.getSymbolID();
    final String fullname = id.getFullName();
    final String typeMethod = Python.getExtendableType(d_ext);
    d_lw.print(typeMethod);
    d_lw.println("_RETURN");
    d_lw.print(typeMethod);
    d_lw.print(" ");
    d_lw.print(typeMethod);
    d_lw.println("_PROTO {");
    d_lw.tab();
    d_lw.println("Py_INCREF(&_" + id.getFullName().replace('.','_')
                 + "Type);");
    d_lw.println("return &_" +id.getFullName().replace('.','_')
                 + "Type;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  /**
   * Write a function to try to convert a <code>PyObject*</code> into an IOR
   * pointer for this extendable.  This function is part of the external API
   * of the Python extension module.
   */
  private void converterMethod() {
    SymbolID id = d_ext.getSymbolID();
    String fullname = id.getFullName();
    String convert = Python.getExtendableConverter(d_ext);
    d_lw.print(convert);
    d_lw.println("_RETURN");
    d_lw.print(convert);
    d_lw.print(" ");
    d_lw.print(convert);
    d_lw.println("_PROTO {");
    d_lw.tab();
    d_lw.print("*sidlobj = sidl_Cast(obj, \"");
    d_lw.print(fullname);
    d_lw.println("\");");
    d_lw.println("if (*sidlobj) {");
    d_lw.tab();
    if (d_ext.isInterface()) {
      d_lw.println("(*((*sidlobj)->d_epv->" +
                   IOR.getVectorEntry("addRef") +
                   "))((*sidlobj)->d_object);");
    }
    else {
      d_lw.println("(*((*sidlobj)->d_epv->" +
                   IOR.getVectorEntry("addRef") +
                   "))(*sidlobj);");
    }
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("else if (obj != Py_None) {");
    d_lw.tab();
    d_lw.println("PyErr_SetString(PyExc_TypeError, ");
    d_lw.tab();
    d_lw.print("\"argument is not a(n) ");
    d_lw.print(fullname);
    d_lw.println("\");");
    d_lw.backTab();
    d_lw.println("return 0;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("return 1;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  private void convertAndStore() {
    final SymbolID id = d_ext.getSymbolID();
    d_lw.println("static int");
    d_lw.println
      ("_convertPython(void *sidlarray, const int *ind, PyObject *pyobj)");
    d_lw.println("{");
    d_lw.tab();
    d_lw.print(IOR.getObjectName(id));
    d_lw.println(" *sidlobj;");
    d_lw.print("if (");
    d_lw.print(Python.getExtendableConverter(d_ext));
    d_lw.println("(pyobj, &sidlobj)) {");
    d_lw.tab();
    d_lw.print("sidl_interface__array_set((struct sidl_interface__array *)");
    d_lw.println("sidlarray,");
    d_lw.println("ind, (struct sidl_BaseInterface__object *)sidlobj);");
    d_lw.println("if (sidlobj) {");
    d_lw.tab();
    d_lw.println("sidl_BaseInterface_deleteRef((struct sidl_BaseInterface__object *)sidlobj);");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("return FALSE;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("return TRUE;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  private void convertPythonArray() {
    final SymbolID id = d_ext.getSymbolID();
    final String convert = 
      Python.getBorrowArrayFromPython(new Type(id));
    d_lw.print(convert);
    d_lw.println("_RETURN");
    d_lw.print(convert);
    d_lw.print(" ");
    d_lw.print(convert);
    d_lw.println("_PROTO {");
    d_lw.tab();
    d_lw.println("int result = 0;");
    d_lw.println("*sidlarray = NULL;");
    d_lw.println("if (obj == Py_None) {");
    d_lw.tab();
    d_lw.println("result = TRUE;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("else {");
    d_lw.tab();
    d_lw.println
      ("PyObject *pya = PyArray_FromObject(obj, PyArray_OBJECT, 0, 0);");
    d_lw.println("if (pya) {");
    d_lw.tab();
    d_lw.println
      ("if (PyArray_OBJECT == ((PyArrayObject *)pya)->descr->type_num) {");
    d_lw.tab();
    d_lw.println("int dimen, lower[SIDL_MAX_ARRAY_DIMENSION],");
    d_lw.tab();
    d_lw.println("upper[SIDL_MAX_ARRAY_DIMENSION],");
    d_lw.println("stride[SIDL_MAX_ARRAY_DIMENSION];");
    d_lw.backTab();
    d_lw.println("if (sidl_array__extract_python_info");
    d_lw.tab();
    d_lw.println("(pya, &dimen, lower, upper, stride))");
    d_lw.backTab();
    d_lw.println("{");
    d_lw.tab();
    d_lw.print("*sidlarray = ("+ IOR.getArrayName(id) + "*)");
    d_lw.tab();
    d_lw.println("sidl_interface__array_createRow");
    d_lw.println("(dimen, lower, upper);");
    d_lw.backTab();
    d_lw.println("result = sidl_array__convert_python");
    d_lw.tab();
    d_lw.println("(pya, dimen, *sidlarray, _convertPython);");
    d_lw.backTab();
    d_lw.println("if (*sidlarray && !result) {");
    d_lw.tab();
    d_lw.println("sidl_interface__array_deleteRef(");
    d_lw.tab();
    d_lw.println("(struct  sidl_interface__array *)*sidlarray);");
    d_lw.backTab();
    d_lw.println("*sidlarray = NULL;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("Py_DECREF(pya);");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("return result;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  private void getAndConvert() {
    final SymbolID id = d_ext.getSymbolID();
    d_lw.println("static int");
    d_lw.println
      ("_convertSIDL(void *sidlarray, const int *ind, PyObject **dest)");
    d_lw.println("{");
    d_lw.tab();
    d_lw.print(IOR.getObjectName(id));
    d_lw.println(" *sidlobj = (" + IOR.getObjectName(id) + "*)");
    d_lw.println("sidl_interface__array_get((struct sidl_interface__array *)");
    d_lw.tab();
    d_lw.println("sidlarray, ind);");
    d_lw.backTab();
    d_lw.print("*dest = ");
    d_lw.print(Python.getExtendableWrapper(d_ext));
    d_lw.println("(sidlobj);");
    d_lw.println("return (*dest == NULL);");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  private void convertSIDLArray() {
    final SymbolID id = d_ext.getSymbolID();
    final String convert = Python.getBorrowArrayFromSIDL(new Type(id));
    d_lw.println(convert + "_RETURN");
    d_lw.println(convert + " " + convert + "_PROTO {");
    d_lw.tab();
    d_lw.println("PyObject *pya = NULL;");
    d_lw.println("if (sidlarray) {");
    d_lw.tab();
    d_lw.println("const int dimen = sidlArrayDim(sidlarray);");
    d_lw.println("int i;");
    d_lw.println("int *lower = (int *)malloc(sizeof(int) * dimen);");
    d_lw.println("int *upper = (int *)malloc(sizeof(int) * dimen);");
    d_lw.println("int *numelem = (int *)malloc(sizeof(int) * dimen);");
    d_lw.println("for(i = 0; i < dimen; ++i) {");
    d_lw.tab();
    d_lw.println("lower[i] = sidlLower(sidlarray, i);");
    d_lw.println("upper[i] = sidlUpper(sidlarray, i);");
    d_lw.println("numelem[i] = 1 + upper[i] - lower[i];");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("pya = PyArray_FromDims(dimen, numelem, PyArray_OBJECT);");
    d_lw.println("if (pya) {");
    d_lw.tab();
    d_lw.println("if (!sidl_array__convert_sidl(pya, dimen, lower, upper,");
    d_lw.tab();
    d_lw.println("numelem, sidlarray, _convertSIDL))");
    d_lw.backTab();
    d_lw.println("{");
    d_lw.tab();
    d_lw.println("Py_DECREF(pya);");
    d_lw.println("pya = NULL;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("free(numelem);");
    d_lw.println("free(upper);");
    d_lw.println("free(lower);");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("else {");
    d_lw.tab();
    d_lw.println("Py_INCREF(Py_None);");
    d_lw.println("pya = Py_None;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("return pya;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println();
  }

  private void convertGenericSIDL()
  {
    final String convert = Python.getBorrowArrayFromSIDL(null);
    d_lw.println(convert + "_RETURN");
    d_lw.println(convert + " " + convert + "_PROTO {");
    d_lw.tab();
    d_lw.println("if (sidlarray &&");
    d_lw.tab();
    d_lw.println("(sidl_interface_array == sidl__array_type(sidlarray))) {");
    d_lw.println("return " + 
                 Python.getBorrowArrayFromSIDL(new Type(d_ext.getSymbolID()))
                 + "(sidlarray);");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("else {");
    d_lw.tab();
    d_lw.println("return sidl_python_borrow_array(sidlarray);");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.backTab();
    d_lw.println("}");
  }

  /**
   * Write the methods that are part of the external C API of this
   * Python C extension module.
   */
  private void externalMethods() {
    wrapperMethod();
    borrowMethod();
    derefMethod();
    newRefMethod();
    addRefMethod();
    typeMethod();
    converterMethod();
    convertAndStore();
    convertPythonArray();
    getAndConvert();
    if (d_isBaseInterface) {
      convertGenericSIDL();
    }
    convertSIDLArray();
  }

  private void preparePythonType()
  {
    final String typeObj = "_" + d_ext.getFullName().replace('.','_') + "Type";
    final String shortName = d_ext.getSymbolID().getShortName();
    LinkedList parents = new LinkedList(d_ext.getParentInterfaces(false));
    if (d_ext instanceof Class) {
      Extendable parent = ((Class)d_ext).getParentClass();
      if (parent != null) {
        parents.addFirst(parent);
      }
    }
    if (parents.isEmpty()) {
      d_lw.println(typeObj + ".tp_base = sidl_PyType();");
    }
    else {
      Extendable base = (Extendable)parents.getFirst();
      d_lw.println(typeObj + ".tp_base = " +
                   Python.getExtendableType(base) +
                   "();");
      d_lw.println(typeObj + ".tp_bases = PyTuple_New(" + 
                   parents.size() + ");");
      Iterator i = parents.iterator();
      int count = 0;
      while (i.hasNext()) {
        d_lw.println("PyTuple_SetItem(" + typeObj + ".tp_bases," + count++ 
                     + ", (PyObject *)" + 
                     Python.getExtendableType((Symbol)i.next()) +
                     "());");
      }
    }
    d_lw.println("if (PyType_Ready(&" + typeObj + ") < 0) {");
    d_lw.tab();
    d_lw.println("PyErr_Print();");
    d_lw.println("fprintf(stderr, \"PyType_Ready on " +
                 d_ext.getFullName() + " failed.\\n\");");
    d_lw.println("return;");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("Py_INCREF(&" + typeObj +");");
    d_lw.println("PyDict_SetItemString(dict, \"" + shortName +
                 "\", (PyObject *)&" + typeObj + ");");
  }

  /**
   * Write the C extension module's initialization function. This 
   * function is called when the module is loaded into memory or
   * when Python is started in cases where static linking is used.
   */
  private void initModule() {
    final SymbolID id = d_ext.getSymbolID();
    final boolean isBaseClass = BabelConfiguration.isSIDLBaseClass(id);
    boolean docComment = (d_ext.getComment() != null) && 
      (d_ext.getComment().getComment() != null);
    d_lw.println("void");
    d_lw.print("init");
    d_lw.print(d_shortName);
    d_lw.println("(void) {");
    d_lw.tab();
    d_lw.println("PyObject *module, *dict, *c_api;");
    d_lw.print("static void *ExternalAPI[");
    d_lw.print(Python.getAPIVarName(d_ext));
    d_lw.println("_NUM];");
    if (!(isBaseClass || d_ext.isInterface())){
      d_lw.println("sidl_DLL dll = NULL;");
      d_lw.println(IOR.getExternalName(id) + "*(*_extFunc)(void) = NULL;");
    }
    if (docComment) {
      d_lw.print("module = Py_InitModule3(\"");
    }
    else {
      d_lw.print("module = Py_InitModule(\"");
    }
    d_lw.print(d_shortName);
    d_lw.print("\", _");
    d_lw.print(d_shortName);
    d_lw.print("ModuleMethods");
    if (docComment) {
      d_lw.println(", \"\\");
      copyComment(d_ext.getComment());
      d_lw.printlnUnformatted("\"");
    }
    d_lw.println(");");
    d_lw.println("dict = PyModule_GetDict(module);");
    for(int i = 0; i < d_externalMethods.length; ++i) {
      d_lw.print("ExternalAPI[");
      d_lw.print(d_externalMethods[i]);
      d_lw.print("_NUM] = (void*)");
      d_lw.print(d_externalMethods[i]);
      d_lw.println(";");
    }
    d_lw.println("import_SIDLObjA();");
    d_lw.println("if (PyErr_Occurred()) {");
    d_lw.tab();
    d_lw.println("Py_FatalError(\"Error importing sidlObjA module.\");");
    d_lw.backTab();
    d_lw.println("}");
    if (isException()) {
      final String extType = Python.getExceptionType(d_ext);
      d_lw.print(extType);
      d_lw.print(" = PyErr_NewException(\"");
      d_lw.print(d_ext.getFullName());
      d_lw.println(".Exception\", (PyObject *)sidl_PyType(), NULL);");
      d_lw.print("PyDict_SetItemString(dict, \"Exception\", ");
      d_lw.print(extType);
      d_lw.println(");");
      d_lw.print("ExternalAPI[");
      d_lw.print(extType);
      d_lw.print("_NUM] = ");
      d_lw.print(extType);
      d_lw.println(";");
    }
    d_lw.println
      ("c_api = PyCObject_FromVoidPtr((void *)ExternalAPI, NULL);");
    d_lw.println("PyDict_SetItemString(dict, \"_C_API\", c_api);");
    d_lw.println("Py_XDECREF(c_api);");
    d_lw.println("import_SIDLPyArrays();");
    d_lw.println("if (PyErr_Occurred()) {");
    d_lw.tab();
    d_lw.println("Py_FatalError(\"Error importing sidlPyArrays module.\");");
    d_lw.backTab();
    d_lw.println("}");
    d_lw.println("import_array();");
    d_lw.println("if (PyErr_Occurred()) {");
    d_lw.tab();
    d_lw.println("Py_FatalError(\"Error importing Numeric Python module.\");");
    d_lw.backTab();
    d_lw.println("}");
    Iterator i = d_references.iterator();
    while (i.hasNext()) {
      Symbol sym = (Symbol)i.next();
      if (!sym.equals((Symbol)d_ext) && (sym instanceof Extendable)) {
        d_lw.print(Python.getExtendableImport(sym));
        d_lw.println("();");
      }
    }
    preparePythonType();
    if (!d_ext.isInterface()) {
      if (isBaseClass) {
        d_lw.println("_implEPV = " + IOR.getExternalFunc(id)
                     + "();");
      }
      else {
        d_lw.writeCommentLine("Load the implementation after initializing the module.");
        d_lw.writeCommentLine("Try search global namespace first");
        d_lw.println("dll = sidl_DLL__create();");
        d_lw.println("if (dll && sidl_DLL_loadLibrary(dll, \"main:\", TRUE, FALSE)) {");
        d_lw.tab();
        d_lw.println("_extFunc = (" + IOR.getExternalName(id) + "*(*)(void))");
        d_lw.tab();
        d_lw.println("sidl_DLL_lookupSymbol(dll, \"" +
                     IOR.getExternalFunc(id) + "\");");
        d_lw.backTab();
        d_lw.println("if (_extFunc) {");
        d_lw.tab();
        d_lw.println("_implEPV = (*_extFunc)();");
        d_lw.backTab();
        d_lw.println("}");
        d_lw.backTab();
        d_lw.println("}");
        d_lw.println("if (dll) sidl_DLL_deleteRef(dll);");
        
        d_lw.println("if (!_implEPV) {");
        d_lw.tab();
        d_lw.println("dll = sidl_Loader_findLibrary(\"" + 
                     id.getFullName() + "\",");
        d_lw.tab();
        d_lw.println("\"ior/impl\", sidl_Scope_SCLSCOPE,");
        d_lw.println("sidl_Resolve_SCLRESOLVE);");
        d_lw.backTab();
        d_lw.println("if (dll) {");
        d_lw.tab();
        d_lw.println("_extFunc = (" + IOR.getExternalName(id) + "*(*)(void))");
        d_lw.tab();
        d_lw.println("sidl_DLL_lookupSymbol(dll, \"" +
                     IOR.getExternalFunc(id) + "\");");
        d_lw.backTab();
        d_lw.println("if (_extFunc) {");
        d_lw.tab();
        d_lw.println("_implEPV = (*_extFunc)();");
        d_lw.backTab();
        d_lw.println("}");
        d_lw.println("sidl_DLL_deleteRef(dll);");
        d_lw.backTab();
        d_lw.println("}");
        d_lw.backTab();
        d_lw.println("}");
      }
      if (d_hasStaticMethods){
        d_lw.println("if (_implEPV) {");
        d_lw.tab();
        d_lw.print("_sepv");
        d_lw.println(" = (*_implEPV->getStaticEPV)();");
        d_lw.println("if (PyErr_Occurred()) {");
        d_lw.tab();
        d_lw.print("Py_FatalError(\"Cannot initialize Python module ");
        d_lw.print(d_ext.getFullName());
        d_lw.println(".\");");
        d_lw.backTab();
        d_lw.println("}");
        d_lw.backTab();
        d_lw.println("}");
        d_lw.println("else {");
      }
      else {
        d_lw.println("if (!_implEPV) {");
      }
      d_lw.tab();
      d_lw.println("Py_FatalError(\"Cannot load implementation for sidl" +
                   (d_ext.isInterface() ? " interface " : " class ") +
                   id.getFullName() + "\");");
      d_lw.backTab();
      d_lw.println("}");
    }
    d_lw.backTab();
    d_lw.println("}");
  }

  /**
   * Generate the source file for the extendable with which this object was
   * created.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception for problems during the code
   *    generation phase.
   */
  public synchronized void generateCode()
    throws CodeGenerationException
  {
    try {
      d_lw = Python.createStub
        (d_ext, "implement a C extension type for a sidl extendable");
      d_lw.enableLineBreak(78, null, "\\");
      explainExtensionSource();
      d_lw.writeComment(d_ext, true);
      includeHeaderFiles();
      staticMethods();
      convertMethods(d_ext.isInterface());
      pythonCastConstructor();
      pythonStaticVirtualTable();
      pythonObjectVirtualTable();
      writePythonType();
      externalMethods();
      initModule();
    }
    finally {
      if (d_lw != null) {
        d_lw.close();
        d_lw = null;
      }
    }
  }
}
