// File:        Method.java
// Package:     gov.llnl.babel.symbols
// Revision:    @(#) $Id: Method.java 4434 2005-03-17 17:05:29Z epperly $
// Description: sidl method (modifiers, return type, name, args, and throws)
//
// Copyright (c) 2000-2003, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.symbols;

import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Assertion;
import gov.llnl.babel.symbols.AssertionException;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolUtilities;
import gov.llnl.babel.symbols.Type;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.Set;
import java.util.Collection;

/**
 * The <code>Method</code> class defines a SIDL method, including modifiers,
 * return type, name, arguments, and exceptions.  Methods may have one of
 * three definition modifiers: final, static, or abstract.  They may also
 * have one of two communication modifiers: local or oneway.  The return type
 * is either a type or null (which indicates void).  The method name is a
 * simple identifier string.  The optional arguments are an ordered collecton
 * or arguments.  The method may throw any number of exceptions.
 */
public class Method implements Cloneable {
  public static final int NORMAL   = 0; // communication/definition modifier
  public static final int ABSTRACT = 1; // definition modifier
  public static final int FINAL    = 2; // definition modifier
  public static final int STATIC   = 3; // definition modifier
  public static final int LOCAL    = 1; // communication modifier
  public static final int ONEWAY   = 2; // communication modifier

  public static final String[] s_def_mod = {
    "", "abstract", "final", "static",
  };

  public static final String[] s_comm_mod = {
    "", "local", "oneway",
  };

  private ArrayList d_arguments;
  private Comment   d_comment;
  private int       d_communication_modifier;
  private int       d_definition_modifier;
  private String    d_short_name;      // for langs that support overloading
  private String    d_extension;       // short + extension = long name
  private String    d_long_name;       // keep for convenience
  private HashSet   d_references;
  private HashSet   d_basicarrays;     // basic array types references
  private boolean   d_return_copy;
  private Type      d_return_type;
  private HashSet   d_throws;
  private ArrayList d_preconditions;	// Local preconditions only
  private ArrayList d_postconditions;	// Local postconditions only
  private HashSet   d_rarray_indices; //All the rarray indices required in the arg list	
  private HashMap   d_rarray_index_map; //A Hashmap of rarray index names to
                                        //a collection of RarrayInfo objects.  And
                                        //RarrayInfo objects represents
                                        //what rarray uses the index, and where.

  /**
   * Create an empty <code>Method</code> object that will be built by
   * calls to other member functions.
   */
  public Method() {
    d_arguments              = new ArrayList();
    d_comment                = null;
    d_communication_modifier = NORMAL;
    d_definition_modifier    = NORMAL;
    d_short_name             = null;
    d_extension              = null;
    d_long_name              = null;
    d_references             = new HashSet();
    d_basicarrays            = new HashSet();
    d_return_copy            = false;
    d_return_type            = null;
    d_throws                 = new HashSet();
    d_preconditions          = new ArrayList();
    d_postconditions         = new ArrayList();
    d_rarray_indices	     = null;	
  }

  /**
   * Return a shallow copy of this method object.  This method should
   * never fail since this method is cloneable.  Send in the clones.
   */
  public Method cloneMethod() {
    Method m = null;
    try {
      m = (Method) super.clone();
    } catch (CloneNotSupportedException ex) {
      throw new java.lang.InternalError(ex.getMessage());
    }
    return m;
  }

  /**
   * Add another argument to the end of the list of method arguments.
   *
   * @param  arg  The argument to be appended to the object's argument list
   */
  public void addArgument(Argument arg) {
    d_arguments.add(arg);
    addTypeReferences(arg.getType());
  }

  /**
   * Return the array of arguments in an <code>ArrayList</code>, where each
   * element is an <code>Argument</code>.
   */
  public ArrayList getArgumentList() {
    return getArgumentListWithOutIndices();
  }

  /**
   * Return the array of arguments in an <code>ArrayList</code> container.
   * This method returns only arguments that are not used as indices to 
   * an Rarray
   */
  public ArrayList getArgumentListWithOutIndices() {
    if(d_rarray_indices != null) {
      ArrayList ret = new ArrayList(d_arguments.size() -
                                    d_rarray_indices.size());
      for(Iterator a = d_arguments.iterator(); a.hasNext();) {
        Argument arg = (Argument) a.next();
        if (!d_rarray_indices.contains(arg.getFormalName())) {
          ret.add(arg);
        }
      }      
      return ret;
    } else {
      return d_arguments;
    }
  }

  /**
   * Return the array of arguments in an <code>ArrayList</code> container.
   * This returns all the arguments in the argument list, including those
   * that are indices to an Rarray.
   */
  public ArrayList getArgumentListWithIndices() {
    return d_arguments;
  }


  /**
   * add a required rarray index to the set of indices
   */	
  public void addRarrayIndex(String s) {
    if(d_rarray_indices == null)
      d_rarray_indices = new HashSet();
    d_rarray_indices.add(s);
  }

  /**
   * add a set of required rarray indices to the set of indices
   */	
  public void addRarrayIndex(Collection new_indices) {
    if(d_rarray_indices == null)
      d_rarray_indices = new HashSet();
    d_rarray_indices.addAll(new_indices);
  }
  
  /**
   * Return the set of rarray indices required in the arg list.  
   * (Returns null if there are no indices required)
   */
  public Set getRarrayIndices() {
    return d_rarray_indices;
  } 

  /**
   * Returns true if this method has an rarray in it's signature
   */
   public boolean hasRarray() {
      return d_rarray_indices != null;
   }

  /**
   * Return the <code>Type</code> of the argument with the specified formal 
   * name, if any; otherwise, return null.
   *
   * @param  name  The formal argument name whose type is to be returned.
   */
  public Type getArgumentType(String name) {
    Type t = null;
    for (Iterator i=d_arguments.iterator(); (i.hasNext() && (t == null)); ) {
      Argument arg = (Argument) i.next();
      if (arg.getFormalName().equals(name)) {
        t = arg.getType();
      }
    }
    return t;
  }

  /**
   * Return the mode of the argument with the specified formal name, if any; 
   * otherwise, return -1.
   *
   * @param  name  The formal argument name whose type is to be returned.
   */
  public int getArgumentMode(String name) {
    int m = -1;
    for (Iterator i=d_arguments.iterator(); (i.hasNext() && (m == -1)); ) {
      Argument arg = (Argument) i.next();
      if (arg.getFormalName().equals(name)) {
        m = arg.getMode();
      }
    }
    return m;
  }

  /**
   * Set the comment for the method.  
   *
   * @param  comment  The comment associated with the method.  May be null.
   */
  public void setComment(Comment comment) {
    d_comment = comment;
  }

  /**
   * Return the comment for the method.  This may be null if there is no
   * comment.
   */
  public Comment getComment() {
    return d_comment;
  }

  /**
   * Set the communication modifier for the method.  
   *
   * @param  modifier  The method's communication modifier.  Valid values
   *                   are NORMAL, LOCAL, and ONEWAY.
   */
  public void setCommunicationModifier(int modifier) {
    d_communication_modifier = modifier;
  }

  /**
   * Return the communication modifier for this method.  
   */
  public int getCommunicationModifier() {
    return d_communication_modifier;
  }

  /**
   * Return the communication modifier string for this method.  
   */
  public String getCommunicationModifierString() {
    return s_comm_mod[d_communication_modifier];
  }

  /**
   * Set the definition modifier for the method.
   *
   * @param  modifier  The method's definnition modifier.  Valid values are
   *                   NORMAL, ABSTRACT, FINAL, and STATIC.
   */
  public void setDefinitionModifier(int modifier) {
    d_definition_modifier = modifier;
  }

  /**
   * Return the definition modifier for the method.  
   */
  public int getDefinitionModifier() {
    return d_definition_modifier;
  }

  /**
   * Return the explicit definition modifier string for the method based
   * on the type of extendable in which it belongs.  
   *
   * @param  is_interface  If TRUE, then the extendable is an interface so
   *                       do NOT include ABSTRACT since it is implicit.
   * @return the string associated with the explicit definition modifier
   */
  public String getDefinitionModifier(boolean is_interface) {
    String mod = null;
    if (  (!is_interface)
       || (d_definition_modifier == NORMAL) ) {
      mod = s_def_mod[d_definition_modifier];
    }
    return mod;
  }

  /**
   * Return TRUE if the method is abstract; otherwise, return FALSE.
   */
  public boolean isAbstract() {
    return d_definition_modifier == ABSTRACT;
  }

  /**
   * Return TRUE if the method is final; otherwise, return FALSE.
   */
  public boolean isFinal() {
    return d_definition_modifier == FINAL;
  }

  /**
   * Return TRUE if the method is static; otherwise, return FALSE.
   */
  public boolean isStatic() {
    return d_definition_modifier == STATIC;
  }

  /**
   * Return TRUE if and only if at least one argument of this method is an 
   * array with an ordering specification.  For example, calling this on 
   * methods without array arguments will return FALSE while calling it on 
   * something like <code>void doIt(in array&lt;int, 2, row-major&gt; x);</code>
   * will return TRUE.
   */
  public boolean hasArrayOrderSpec() {
    if (d_return_type.hasArrayOrderSpec()) return true;
    for(Iterator i = d_arguments.iterator(); i.hasNext(); ){
      if (((Argument)i.next()).hasArrayOrderSpec()) return true;
    }
    return false;
  }

  /**
   * Set the names of the method (a standard SIDL identifier).
   *
   * @param  shortName  The short name of the method
   * @param  extension  The method name extension used in combination
   *                    with the short name to uniquely identify this 
   *                    method
   */
  public void setMethodName(String shortName, String extension) {
    d_short_name = shortName;
    d_extension  = extension;
    d_long_name  = shortName + extension;
  }

  /**
   * Set the names of the method (a standard SIDL identifier).  Use of
   * this method will result in the long and short name being identical.
   *
   * @param  shortName  The short name of the method
   */
  public void setMethodName(String shortName) {
    setMethodName(shortName, "");
  }

  /**
   * Return the short method name (a standard SIDL identifier).
   */
  public String getShortMethodName() {
    return d_short_name;
  }

  /**
   * Return the method name extension (a standard SIDL identifier).
   */
  public String getNameExtension() {
    return d_extension;
  }

  /**
   * Return the long method name (a standard SIDL identifier).
   */
  public String getLongMethodName() {
    return d_long_name;
  }

  /**
   * Set the copy mode for the return type.  
   *
   * @param  copy  If TRUE then the return interface/class is to be copied 
   *               to the caller; otherwise, it is not.
   */
  public void setReturnCopy(boolean copy) {
    d_return_copy = copy;
  }

  /**
   * Return the copy mode for the return type.  
   */
  public boolean isReturnCopy() {
    return d_return_copy;
  }

  /**
   * Set the return type for the method.  
   *
   * @param  type  The return type to be used.  Note that a void return
   *               type must be represented by a <code>Type.VOID</code>
   *               NOT a null Type reference.
   */
  public void setReturnType(Type type) {
    d_return_type = type;
    addTypeReferences(type);
  }

  /**
   * Return the return type for the method.  
   */
  public Type getReturnType() {
    return(d_return_type);
  }

  /**
   * Add a symbol identifier to the list of supported exceptions for this
   * method.  No error checking is performed to ensure that the specified
   * symbol is valid in the throws clause; such checking must be performed
   * by the parser.
   *
   * @param  id  The symbol identifier to be added 
   */
  public void addThrows(SymbolID id) {
    d_throws.add(id);
    d_references.add(id);
  }

  /**
   * Return the <code>Set</code> of exceptions that may be thrown by this 
   * method.  Each element of the set is a <code>SymbolID</code>.
   */
  public Set getThrows() {
    return d_throws;
  }

  /**
   * Return the <code>Set</code> of symbols referred to by this method.  
   * Each element of the set is a <code>SymbolID</code>.
   */
  public Set getSymbolReferences() {
    return d_references;
  }

  /**
   * Return the <code>Set</code> of basic array references including
   * arrays of fundamental types such as double, int, etc.  Each element
   * of the set is a <code>SymbolID</code>.
   */
  public Set getBasicArrays() {
    return d_basicarrays;
  }


  /**
   * Return the string corresponding to the return type for this method.  Note
   * the string may represent an abbreviated return type (i.e., a type
   * stripped of its package information).
   *
   * @param  parent_pkg  The string containing the parent package.  When not
   *                     null, it is used to strip the package from the return
   *                     string if it is in the specified package.
   * @return             the string containing the possibly abbreviated return
   *                     type.
   */
  public String getReturnType(String parent_pkg) {
    StringBuffer result = new StringBuffer();
    String rtype = d_return_type.getTypeString();
    if (parent_pkg != null) {
      result.append(SymbolUtilities.getSymbolName(rtype, parent_pkg));
    } else {
      result.append(rtype);
    }
    return result.toString();
  }


  /**
   * Return the concatenation of the explicit definition modifier, copy,
   * return type, name and extension.
   *
   * @param  is_interface  TRUE if an interface and want the implicit 
   *                       definition modifier excluded from the result; 
   *                       otherwise, FALSE.
   * @param  parent_pkg    The string containing the parent package.  When not
   *                       null, it is used to strip the package from the return
   *                       string if it is in the specified package.
   * @return               the string containing the preface
   */
  public String getSignaturePreface(boolean is_interface, String parent_pkg) {
    StringBuffer preface = new StringBuffer();

    String dmod = getDefinitionModifier(is_interface);
    if (dmod != null) {
      preface.append(dmod + " ");
    }
    if (d_return_copy) {
      preface.append("copy ");
    }
    preface.append(getReturnType(parent_pkg));
    preface.append(" ");
    preface.append(d_short_name);
    if ((d_extension != null) && (d_extension != "")) {
      preface.append(" [");
      preface.append(d_extension);
      preface.append("] ");
    }
    return preface.toString();
  }


  /**
   * Return the signature of the method, including the definition modifier
   * based on the extendable type.  Also, optionally abbreviate type if in 
   * specified package.
   *
   * @param  is_interface  TRUE if interface and want the implicit definition
   *                       modifier excluded from the result; otherwise, FALSE
   * @param  parent_pkg    The string containing the parent package.  When not
   *                       null, it is used to strip the package from the return
   *                       string if it is in the specified package.
   * @return               The string containing the full signature.
   */
  public String getSignature(boolean is_interface, String parent_pkg) {
    StringBuffer signature = new StringBuffer();

    signature.append(getSignaturePreface(is_interface, parent_pkg));
    signature.append("(");
    for (Iterator a = d_arguments.iterator(); a.hasNext(); ) {
      signature.append(((Argument) a.next()).getArgumentString());
      if (a.hasNext()) {
        signature.append(", ");
      }
    }
    signature.append(") " + getCommunicationModifierString());

    if (!d_throws.isEmpty()) {
      signature.append(" throws ");
      for (Iterator t = d_throws.iterator(); t.hasNext(); ) {
        SymbolID sid = (SymbolID) t.next();
        String fname = sid.getFullName();
        if (parent_pkg != null) {
          signature.append(SymbolUtilities.getSymbolName(fname, parent_pkg));
        } else {
          signature.append(fname);
        }
        if (t.hasNext()) {
          signature.append(", ");
        }
      }
    }

    return signature.toString();
  }

  /**
   * Return the signature of the method.  The signature does not include
   * the attributes abstract, final, or static.  It also does not abbreviate
   * package names.
   */
  public String getSignature() {
    return getSignature(false, null);
  }

  /**
   * Return TRUE if the signature of the specified method matches that of
   * this method; otherwise, return FALSE.  For signatures to match, the 
   * methods must have the same return types and mode, the same names, the 
   * same arguments, and the same throws clauses.  They must also have the 
   * same communication modifiers.  The signature does not include modifiers 
   * static, abstract, or final.
   *
   * @param  m  The method whose signature is being compared with.
   */
  public boolean sameSignature(Method m) {
    /*
     * Check modifiers, method names, and return types
     */
    if (d_communication_modifier != m.d_communication_modifier) {
      return false;
    }
    if (!d_short_name.equals(m.d_short_name)) {
      return false;
    }
    if (!d_extension.equals(m.d_extension)) {
      return false;
    }
    if (d_return_copy != m.d_return_copy) {
      return false;
    }
    if (d_return_type == null) {
      if (m.d_return_type != null) {
        return false;
      }
    } else if (!d_return_type.equals(m.d_return_type)) {
      return false;
    }

    /*
     * Check that the two throws lists are the same
     */
    if (d_throws.size() != m.d_throws.size()) {
      return false;
    }
    for (Iterator i = d_throws.iterator(); i.hasNext(); ) {
      if (!m.d_throws.contains(i.next())) {
        return false;
      }
    }

    return sameArguments(m);
  }

  /**
   * Return TRUE if the base signature of the specified method matches
   * that of this method; otherwise, return FALSE.  For them to match, 
   * the methods must have the same short and the same arguments.
   *
   * @param  m  The method whose base signature is being compared.
   */
  public boolean sameBaseSignature(Method m) {
    /*
     * Check the names
     */
    if (!d_short_name.equals(m.d_short_name)) {
      return false;
    }

    return sameArguments(m);
  }

  /**
   * Return TRUE if the arguments of the specified method match those of
   * this method; otherwise, return FALSE.  Note that ordering counts!  
   * That is, the arguments are the same if the same types appear in the 
   * same order.
   *
   * @param  m  The method whose argument list is being compared.
   */
  private boolean sameArguments(Method m) {
    /*
     * Check that the arguments are the same
     */
    int size = d_arguments.size();
    if (size != m.d_arguments.size()) {
      return false;
    } else {
      for (int a = 0; a < size; a++) {
        if (!d_arguments.get(a).equals(m.d_arguments.get(a))) {
          return false;
        }
      }
    }

    return true;
  }

  /**
   * Return an the argument whose formal name matches the passed in string.
   * Return null if the string does not match anything.
   */
  public Argument getArgumentByName(String s) {
    for(Iterator a = d_arguments.iterator(); a.hasNext();) {
      Argument arg = (Argument) a.next();
      if(arg.getFormalName().compareTo(s) == 0)
        return arg;
    }
      
    return null;
  }

  /**
   * Add the symbol identifier of the specified type to the symbol reference
   * set ONLY if the specified type is an enumerated type, an interface, a 
   * class, or an array of these objects.
   *
   * @param  type  The symbol/type to be added
   */
  private void addTypeReferences(Type type) {
    if (type != null) {
      if (type.getType() == Type.SYMBOL) {
        d_references.add(type.getSymbolID());
      } else if (type.getType() == Type.ARRAY) {
        type = type.getArrayType();
        if (null != type) {
          addTypeReferences(type);
          if ((type.getType() >= Type.BOOLEAN) &&
              (type.getType() <= Type.STRING)) {
            Version v = new Version();
            v.appendVersionNumber(0);
            d_basicarrays.add(new SymbolID("sidl." + type.getTypeString(), v));
          }
        }
      }
    }
  }

  /**
   * Return TRUE if PURE clause appears in the assertions associated
   * with this method; otherwise, return FALSE.
   */
  public boolean hasPureAssertion() {
    boolean   found = false;
    ArrayList list  = getPostconditions();
    for (ListIterator iter = list.listIterator();
         iter.hasNext() && !found; ) {
      Assertion item = (Assertion) iter.next();
      if (item.hasPureClause()) {
        found = true;
      }
    }
    return found;
  }

  /**
   * Add the specified assertion to the proper assertion list for this method.
   *
   * @param   assertion  The assertion to be added.
   * @throws  gov.llnl.babel.symbols.AssertionException
   *                     The exception thrown if the assertion is not valid.
   */
  public void addAssertion(Assertion assertion) throws AssertionException 
  {
    if ( assertion.isPrecondition() ) {
      d_preconditions.add(assertion);
    } else if ( assertion.isPostcondition() ) {
      d_postconditions.add(assertion);
    } else {
      throw new AssertionException("Method: " + d_long_name, "Cannot add a(n) "
                + "\"" + assertion.getTypeName() + "\" assertion to a "
                + "method\'s assertion list.");
    }
  }

  /**
   * Return the list of preconditions as an <code>ArrayList</code> of
   * <code>Assertion</code> elements.  
   */
  public ArrayList getPreconditions() {
    return d_preconditions;
  }

  /**
   * Return the list of postconditions as an <code>ArrayList</code> of
   * <code>Assertion</code> elements.
   */
  public ArrayList getPostconditions() {
    return d_postconditions;
  }

  /**
   * Return TRUE if the list of assertions has the specified reserved method 
   * assertion; FALSE otherwise.
   */
  public boolean hasReservedMethodAssertion(ArrayList list, int type) {
    boolean hasIt = false;
    for (ListIterator iter = list.listIterator(); iter.hasNext() && !hasIt; ) {
      Assertion item = (Assertion) iter.next();
      if (item.hasReservedMethod(type)) {
        hasIt = true;
      }
    }
    return hasIt;
  }

  /**
   * Return the number of array iteration macros in the assertions of the 
   * method of the specified return type.  Valid types are given in 
   * MethodCall.java as MACRO_RETURNS_* values.
   */
  public int getMaxArrayIterMacros(char type) {
    int num;
    Assertion as;
    int max = 0;

    Iterator i = d_preconditions.iterator();
    while (i.hasNext()) {
      as  = (Assertion)i.next();
      num = as.getNumArrayIterMacrosByType(type);
      if (num > max) {
        max = num;
      }
    }

    i = d_postconditions.iterator();
    while (i.hasNext()) {
      as  = (Assertion)i.next();
      num = as.getNumArrayIterMacrosByType(type);
      if (num > max) {
        max = num;
      }
    }

    return max;
  }

  /**
   * Return TRUE if the method has the specified reserved method assertion; 
   * FALSE otherwise.
   */
  public boolean hasReservedMethodAssertion(int type) {
    boolean hasIt = hasReservedMethodAssertion(d_preconditions, type);
    if (!hasIt) {
      hasIt = hasReservedMethodAssertion(d_postconditions, type);
    }
    return hasIt;
  }

  /**
   * Return TRUE if the list of assertions has the type of unreserved method 
   * assertion; FALSE otherwise.
   */
  public boolean hasUnreservedMethodAssertion(ArrayList list, boolean any) {
    boolean hasIt = false;
    for (ListIterator iter = list.listIterator(); iter.hasNext() && !hasIt; ) {
      Assertion item = (Assertion) iter.next();
      if (item.hasUnreservedMethod(any)) {
        hasIt = true;
      }
    }
    return hasIt;
  }

  /**
   * Return TRUE if the method has any unreserved method (when any is TRUE)
   * assertion or has an unreserved method assertion with a throws clause 
   * (if any is FALSE); otherwise, return FALSE.
   */
  public boolean hasUnreservedMethodAssertion(boolean any) {
    boolean hasIt = hasUnreservedMethodAssertion(d_preconditions, any);
    if (!hasIt) {
      hasIt = hasUnreservedMethodAssertion(d_postconditions, any);
    }
    return hasIt;
  }

  /**
   * This returns an array list of RarrayInfo objects that contains the info
   * for every rarray index used in this method.  
   */
  public HashMap getRarrayInfo() {
    if(d_rarray_index_map == null)
      createRarrayMap();
    return d_rarray_index_map;
  }

  private void createRarrayMap() {
    d_rarray_index_map = new HashMap();
    Iterator i = d_arguments.iterator();
    while (i.hasNext()) {
      Argument a = (Argument)i.next();
      Type argType = a.getType();
      if (argType.isRarray()) {
        Iterator j = argType.getArrayIndices().iterator();
        int position = 0;
        while (j.hasNext()) {
          String indexName = j.next().toString();
          Set indexSet = (Set) d_rarray_index_map.get(indexName);
          Argument index_arg = null;
          if (indexSet == null) {
            indexSet = new HashSet();
            d_rarray_index_map.put(indexName, indexSet);
          }
          
          for(Iterator a1 = d_arguments.iterator(); a1.hasNext();) {
            Argument arg = (Argument) a1.next();
            if(indexName.compareTo(arg.getFormalName()) == 0) {
              index_arg = arg;
              break;
            }
          }
          indexSet.add(new RarrayInfo(a, index_arg, position));
          ++position;
        }
      }
    }
  }

  public int hashCode() {
    return d_long_name.hashCode() +
      d_short_name.hashCode() +
      d_communication_modifier +
      d_throws.hashCode() +
      d_arguments.hashCode() +
      (d_return_copy ? 1 : 0);
  }

  public boolean equals(Object o) {
    try {
      Method m = (Method)o;
      return (d_communication_modifier == m.d_communication_modifier) &&
        (d_return_copy == m.d_return_copy) &&
        d_arguments.equals(m.d_arguments) &&
        d_return_type.equals(m.d_return_type) &&
        d_long_name.equals(m.d_long_name) &&
        d_short_name.equals(m.d_short_name) &&
        d_throws.equals(m.d_throws);
    }
    catch(Exception e){
    }
    return false;
  }

  
  public class RarrayInfo {
    
    public final Argument rarray;
    public final Argument index;
    public final int dim;

    public RarrayInfo(Argument r, Argument i, int d) {
      rarray = r;
      index = i;
      dim = d;
    }
    
    public int hashCode() {
      return rarray.getFormalName().hashCode() + index.getFormalName().hashCode() + dim;
    }

    public boolean equals(Object o) {
      RarrayInfo r = null;
      try {
        r = (RarrayInfo)o;
        if(r.rarray == rarray &&
           r.index == index &&
           r.dim == dim)
          return true;
      } catch(Exception ex) {
        return false;
      }
      return false;
    }

  }
}
