/*
 * $Id: gnovd.c,v 1.13 2002/01/09 23:20:42 binnema Exp $
 */

/*
** Copyright (C) 2001 Dirk-Jan C. Binnema <dirk-jan@djcbsoftware.nl>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/

#include <config.h>

#include <gnome.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#ifdef HAVE_GCONF
#include <gconf/gconf.h>
#endif /* Maybe GTKHTML needs gconf */

#include <gtkhtml/gtkhtml.h>

#ifdef HAVE_GNOME_ICON
#include <libgnome/gnome-util.h>
#include <libgnomeui/gnome-window-icon.h>
#endif /* HAVE_GNOME_ICON */

#include <libgnuvd.h>

typedef struct _app_data 	 AppData;
struct _app_data {
	GtkEntry       		*entry;
	GtkButton		*ok_button;
	GtkStatusbar   		*status_bar;
	GtkProgressBar 		*progress_bar;
	GtkHTML        		*html_widget;
	
};


static GtkWidget* gvd_app_new             (AppData   *app_data);
static void       on_ok                   (GtkWidget *w, gpointer data);
static void       on_about                (GtkWidget *w, gpointer data);
static void       on_enter                (GtkWidget *widget, GdkEventKey* event,
					   gpointer data);
static void       on_quit                  (GtkWidget* widget, gpointer data);
static void       show_results             (VDQuery *query, GtkHTML *html_widget);
static void       show_not_found           (VDQuery *query, GtkHTML *html_widget);
static void       show_error               (VDException *vd_ex, GtkHTML *html_widget);

int
main (int argc, char *argv[])
{
	GtkWidget *app;
	AppData   app_data;
#ifdef HAVE_GCONF
	GError *err;
#endif /*HAVE_GCONF*/
	
	gnome_init ("GNOVD", VERSION, argc, argv);
	
#ifdef HAVE_GCONF
	if (gtk_html_build_with_gconf())
		gconf_init (argc, argv, &err);
#endif /*HAVE_GCONF*/
	
	app = gvd_app_new (&app_data);
	gtk_widget_show (GTK_WIDGET (app));

	/* handle cmdline arg */
	if (argc == 2) {
		gtk_entry_set_text (app_data.entry, argv[1]);
		on_ok (NULL, &app_data);
	}
	
	gtk_main ();
	
	return 0;
}



GtkWidget*
gvd_app_new (AppData *app_data)
{
	GtkWidget 	*win, *vbox, *entrybox, *label;
	GtkWidget 	*ok,  *about, *quit, *statusbox, *sbar, *entry;
	GtkWidget 	*swin, *html;
	GtkHTMLStream 	*stream;
 
	win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title (GTK_WINDOW(win), PACKAGE);
 
#ifdef HAVE_GNOME_ICON
	{
	 gchar *icon_path = gnome_unconditional_datadir_file ("pixmaps/");
	 gchar *master_icon = g_strconcat (icon_path, "gnovd.png", NULL);
	 gnome_window_icon_set_from_file(GTK_WINDOW(win), master_icon);
	 g_free (icon_path);
	 g_free (master_icon);
	}
#endif /*HAVE_GNOME_ICON*/
	
	vbox      = gtk_vbox_new   (FALSE, 1);
	entrybox  = gtk_hbox_new   (FALSE, 1);
	statusbox = gtk_hbox_new   (FALSE, 2);
	label     = gtk_label_new  ("Woord: ");
	about     = gtk_button_new_with_label ("Info");
	ok        = gtk_button_new_with_label ("Ok");
	quit      = gtk_button_new_with_label ("Sluiten");
	
	html = gtk_html_new ();
	gtk_widget_set_usize (html, 350, 280);
	
	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_container_add (GTK_CONTAINER(swin), html);
	
	entry = gtk_entry_new_with_max_length (30);
	sbar  = gtk_statusbar_new (); 
	
	gtk_container_add  (GTK_CONTAINER(win), vbox);
	gtk_box_pack_start (GTK_BOX(vbox), entrybox,   FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), swin,       TRUE,  TRUE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), statusbox,  FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX(statusbox), sbar, FALSE, FALSE, 0);
	
	gtk_box_pack_start (GTK_BOX(entrybox), label, FALSE,  FALSE, 0);
	gtk_box_pack_start (GTK_BOX(entrybox), entry, TRUE , FALSE, 0);
	gtk_box_pack_start (GTK_BOX(entrybox), ok,    FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(entrybox), about, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(entrybox), quit,  FALSE, FALSE, 0);

	app_data->ok_button   = GTK_BUTTON(ok);
	app_data->status_bar  = GTK_STATUSBAR(sbar);
	app_data->entry       = GTK_ENTRY(entry);
	app_data->html_widget = GTK_HTML(html);
	
	gtk_signal_connect (GTK_OBJECT(ok),    "clicked", on_ok, app_data);
	gtk_signal_connect (GTK_OBJECT(about), "clicked", on_about, app_data);
	gtk_signal_connect (GTK_OBJECT(quit),  "clicked", on_quit, app_data);
	gtk_signal_connect (GTK_OBJECT(win),   "destroy", on_quit, app_data);
	gtk_signal_connect (GTK_OBJECT(entry), "key_press_event", on_enter, app_data);

	on_about (vbox, app_data);
	
	gtk_widget_show_all (GTK_WIDGET(vbox));
		
	return win;
}



static void
on_enter (GtkWidget *widget, GdkEventKey* event,  gpointer data)
{
	if (event->keyval == GDK_Return)
		on_ok (widget, data);
}


static void
on_ok (GtkWidget *widget, gpointer data)
{
	char 		 buf[50];
	AppData 	*app_data;
	VDQuery 	*vd_query;
	VDException 	*vd_ex;
	int 		 result;
	int 		 offset;
	static guint     context_id = 0;

	g_return_if_fail (data);

	vd_ex = vd_exception_new ();
	g_return_if_fail (vd_ex);
	
	app_data = (AppData*) data;
	snprintf (buf, 50, "aan het zoeken naar '%s'", 
		  gtk_entry_get_text (app_data->entry));
	gtk_statusbar_push (app_data->status_bar, ++context_id, buf);

	/* make sure it's shown before the query starts */
	if (widget)
		while (gtk_events_pending())
			gtk_main_iteration();
	
	vd_query = vd_query_new (gtk_entry_get_text(app_data->entry),
				 vd_ex);

	if (!vd_query) {
		
		gtk_statusbar_push (app_data->status_bar, ++context_id,
				    "er is fout opgetreden");
		show_error (vd_ex, app_data->html_widget);
		vd_exception_destroy (vd_ex);
		return;
	}

	result = vd_query_perform (vd_query, vd_ex);
	switch (result) {

	case VD_ERR_OK:
		gtk_statusbar_push (app_data->status_bar, ++context_id,
				    "gevonden");
		show_results (vd_query, app_data->html_widget);
		break;

	case VD_ERR_NOT_FOUND:
		gtk_statusbar_push (app_data->status_bar, ++context_id,
				    "niet gevonden");
		show_not_found (vd_query, app_data->html_widget);
		break;

	default:
		gtk_statusbar_push (app_data->status_bar, ++context_id,
				    "er is een fout opgetreden");
		show_error (vd_ex, app_data->html_widget);
		break;
	}

	vd_exception_destroy (vd_ex);
	vd_query_destroy (vd_query);
	
	gtk_widget_show (GTK_WIDGET(app_data->html_widget));
}




static void
on_quit (GtkWidget* widget, gpointer data)
{
	gtk_main_quit ();
}




static void
on_about (GtkWidget *widget, gpointer data)
{
	GtkHTMLStream 	*stream;
	AppData 	*app_data;
	static const gchar *html = 
		"<html>"
		"<body>"
		"<h1><font color=\"red\">" PACKAGE ", versie " VERSION "</font></h1>"
		"<p><b><i>Het online-woordenboek</i></b><br>"
		"<hr>"
		"<br><br><br><br>"	
		"<center>"
		"GNUVD-homepage: "
		"<b>http://www.djcbsoftware.nl/projecten/gnuvd</b>"
		"</center>"
		"<br><br><br><br>"	
		"<hr>"	
		"&copy; 2001 "
		"<b>Dirk-Jan C. Binnema</b><br>"
		"uitgebracht volgens de voorwaarden van de GNU GPL versie 2 of later"
		"</body></html>";
	
	g_return_if_fail (data);
	
	app_data = (AppData*) data;
	
	stream    = gtk_html_begin (GTK_HTML(app_data->html_widget));
	gtk_html_write (GTK_HTML (app_data->html_widget), stream,
			html, strlen(html));
	gtk_html_end   (GTK_HTML (app_data->html_widget), stream,
			GTK_HTML_STREAM_OK);
}




static void
show_results (VDQuery *vd_query, GtkHTML *html_widget)
{
	GtkHTMLStream *stream;

	static const gchar* start  = "<html><body>";
	static const gchar* finish = "</body></html>";
	
	stream = gtk_html_begin (html_widget);
	gtk_html_write (html_widget, stream, start, strlen(start));
	gtk_html_write (html_widget, stream, 
			buffer_data(vd_query->_result_buffer), 
			buffer_length (vd_query->_result_buffer));
	gtk_html_write (html_widget, stream, finish, strlen(finish));
	gtk_html_end   (html_widget, stream, GTK_HTML_STREAM_OK);
}


static void
show_not_found (VDQuery *vd_query, GtkHTML *html_widget)
{
	GtkHTMLStream 	*stream;
	static const gchar *html1 = 
		"<html>"
		"<body>"
		"<h1><font color=\"red\">Niets gevonden</font></h1>"
		"<hr>"
		"Niet gevonden: <i>";
	static const gchar *html2 = 
		"</i></body></html>";

	const char* search_term = vd_query_search_term (vd_query);
		
	stream    = gtk_html_begin (GTK_HTML(html_widget));
	gtk_html_write (GTK_HTML (html_widget), stream,
			html1, strlen(html1));
	gtk_html_write (GTK_HTML (html_widget), stream,
			search_term, strlen(search_term));
	gtk_html_write (GTK_HTML (html_widget), stream,
			html2, strlen(html2));
	gtk_html_end   (GTK_HTML (html_widget), stream,
			GTK_HTML_STREAM_OK);
}


static void
show_error (VDException *vd_ex, GtkHTML *html_widget)
{
	GtkHTMLStream 	*stream;

	const gint 	buf_size = 50;
	gchar 		buf[buf_size];
	
	static const gchar *html_before = 
		"<html>"
		"<body>"
		"<h1><font color=\"red\">Fout</font></h1><hr>";
	static const gchar *html_after =
		"<hr>"
		"</body>"
		"</html>";
	
	stream = gtk_html_begin (GTK_HTML(html_widget));

	gtk_html_write (GTK_HTML (html_widget), stream,
			html_before, strlen(html_before));

	/* print the err message if vd_ex contains one */
	if (vd_ex->_msg) 
		gtk_html_write (GTK_HTML (html_widget), stream,
				vd_ex->_msg, strlen (vd_ex->_msg));
			
	gtk_html_write (GTK_HTML (html_widget), stream,
			html_after, strlen(html_after));

	gtk_html_end   (GTK_HTML (html_widget), stream, GTK_HTML_STREAM_OK);
}
