/*  gtktiemu - a TI89/92/92+ emulator
 *  (c) Copyright 2000-2001, Romain Lievin and Thomas Corvazier
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
  This unit manage configurable options and config file
*/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "platform.h"
#include "paths.h"
#include "struct.h"
#include "version.h"
#include "paths.h"
#include "img_fmt.h"
#include "files.h"
#include "specific.h"
#include "const.h"

static char *rc_file; 

#define MAXCHARS 256

static char *rc_file; 

int debug_options()
{
  fprintf(stdout, "******************************************************************\n");
  /*
  fprintf(stdout, "# Calculator type\n");
  fprintf(stdout, "calctype=");
  switch((options.lp)->calc_type)
    {
    case CALC_TI92P:
      fprintf(stdout, "ti92+\n");
      break;
    case CALC_TI92:
      fprintf(stdout, "ti92\n");
      break;
    case CALC_TI89:
      fprintf(stdout, "ti89\n");
      break; 
    case CALC_TI86:
      fprintf(stdout, "ti86\n");
      break;
    case CALC_TI85:
      fprintf(stdout, "ti85\n");
      break;
    case CALC_TI83P:
      fprintf(stdout, "ti83+\n");
      break; 
    case CALC_TI83:
      fprintf(stdout, "ti83\n");
      break;
    case CALC_TI82:
      fprintf(stdout, "ti82\n");
      break;
    }
  fprintf(stdout, "\n");
  */
  fprintf(stdout, "# Link cable type\n");
  fprintf(stdout, "linktype=");
  switch((options.lp)->link_type)
    {
    case LINK_PAR:
      fprintf(stdout, "parallel\n");
      break;
    case LINK_SER:
      fprintf(stdout, "serial\n");
      break;
    case LINK_TGL:
      fprintf(stdout, "TIGraphLink\n");
      break;
    }
  fprintf(stdout, "\n");
  fprintf(stdout, "# Parallel/serial port address\n");
  fprintf(stdout, "adr_port=0x%03X\n", (options.lp)->io_addr);
  fprintf(stdout, "\n");
  fprintf(stdout, "# Serial device\n");
  fprintf(stdout, "serial_device=%s\n", (options.lp)->device);
  fprintf(stdout, "\n");
  fprintf(stdout, "# Timeout value in 0.1 seconds\n");
  fprintf(stdout, "timeout=%i\n", (options.lp)->timeout);
  fprintf(stdout, "\n");
  fprintf(stdout, "# Baud rate\n");

  fprintf(stdout, "# Delay value\n");
  fprintf(stdout, "delay=%i\n", (options.lp)->delay);
  fprintf(stdout, "\n");
  
  return 0;
}


int get_rcfile_path(char **path)
{
#if defined(__UNIX__)
  char *home_dir;
  
  get_home_path(&home_dir);
  *path = g_strconcat(home_dir, DIR_SEPARATOR, CONF_DIR, 
		      DIR_SEPARATOR, RC_FILE, NULL);
  g_free(home_dir);
#elif defined(__WIN32__)
  *path = g_strconcat(inst_paths.base_dir, DIR_SEPARATOR, CONF_DIR,
		     DIR_SEPARATOR, RC_FILE, NULL);
#endif

  return 0;
}

/* Print an error msg */
void stop (int line)
{
  fprintf(stderr, gettext("Configuration file error at line %i.\n"), line);
}

char *find_str(char *s, const char *t)
{
  char *p = strstr(s, t);

  if(p==NULL) return NULL;
  else return p+strlen(t);
}

/* Chech whether a RC file exists */
int is_rcfile_exist(void)
{
  get_rcfile_path(&rc_file);
  
  return !access(rc_file, F_OK);
}


/* Read the RC file and configure the corresponding variables */
void read_rc_file(void)
{
  FILE *txt;
  char buffer[256];
  char *p;
  int l=0;

  get_rcfile_path(&rc_file);

  txt=fopen(rc_file, "rt");
  if(txt == NULL)
    {
      DISPLAY(_("Configuration file not found, use default values. You can create one by the 'File|Save config' command menu.\n"));
      //gif->msg_box(_("Information"), _("Configuration file not found, use default values. You can create one by the 'File|Save config' command menu.\n"));
      
      return;
    }

  while(!feof(txt))
    {
      fgets(buffer, 256, txt);
      l++;
      buffer[strlen(buffer)-1]='\0';
      if(!strcmp(buffer, "RC_END")) 
	{
	  fclose(txt);
	  return;
	}
      if(buffer[0]=='#' || !strlen(buffer)) continue;
      //printf("<%s>\n", buffer);
      /* Common part with TiLP: hardware section */
      if( (p=find_str(buffer, "calctype=")) )
	{/*
	  if(!strcmp(p, "ti92+")) (options.lp)->calc_type=CALC_TI92P;
	  else if(!strcmp(p, "ti92")) (options.lp)->calc_type=CALC_TI92;
	  else if(!strcmp(p, "ti89")) (options.lp)->calc_type=CALC_TI89;
	  else if(!strcmp(p, "ti86")) (options.lp)->calc_type=CALC_TI86;
	  else if(!strcmp(p, "ti85")) (options.lp)->calc_type=CALC_TI85;
	  else if(!strcmp(p, "ti83+")) (options.lp)->calc_type=CALC_TI83P;
	  else if(!strcmp(p, "ti83")) (options.lp)->calc_type=CALC_TI83;
	  else if(!strcmp(p, "ti82")) (options.lp)->calc_type=CALC_TI82;
	  else stop(l);
	 */
	  continue;
	}
      
      if( (p=find_str(buffer, "linktype=")) )
	{
	  if(!strcmp(p, "parallel")) (options.lp)->link_type=LINK_PAR;
	  else if(!strcmp(p, "serial")) (options.lp)->link_type=LINK_SER;
	  else if(!strcmp(p, "TIGraphLink")) (options.lp)->link_type=LINK_TGL;
	  else if(!strcmp(p, "fastAVRlink")) (options.lp)->link_type=LINK_AVR;
	  else if(!strcmp(p, "VTi")) (options.lp)->link_type=LINK_VTI;
	  else if(!strcmp(p, "TiEmulator")) (options.lp)->link_type=LINK_TIE;
	  else if(!strcmp(p, "virtual")) (options.lp)->link_type=LINK_VTL;
	  else if(!strcmp(p, "TiPcUsbLink")) (options.lp)->link_type=LINK_TPU;
	  else if(!strcmp(p, "UsbGraphLink")) (options.lp)->link_type=LINK_UGL;
	  else stop(l);
	  continue;
	}
      if( (p=find_str(buffer, "adr_port=")) )
	{
	  sscanf(p, "0x%03X", &((options.lp)->io_addr));
	  //check_access();
	  continue;
	}
      if( (p=find_str(buffer, "device=")) )
	{
	  strcpy((options.lp)->device, p);
	}
      if( (p=find_str(buffer, "timeout=")) )
	{
	  sscanf(p, "%i", &((options.lp)->timeout));
	  continue;
	}
      if( (p=find_str(buffer, "baudrate=")) )
	{
	  sscanf(p, "%i", &((options.lp)->baud_rate));
	  continue;
	}
      if( (p=find_str(buffer, "delay=")) )
	{
	  sscanf(p, "%i", &((options.lp)->delay));
	  continue;
	}
      if( (p=find_str(buffer, "rts_cts=")) )
        {
	  if(!strcmp(p, "on")) 
	    (options.lp)->hfc = HFC_ON;
          else if(!strcmp(p, "off")) 
	    (options.lp)->hfc = HFC_OFF;
          else stop(l);
          continue;
        }
            if( (p=find_str(buffer, "port=")) )
	{
	  if(!strcmp(p, "parallel port #1")) (options.lp)->port=PARALLEL_PORT_1;
	  else if(!strcmp(p, "parallel port #2")) (options.lp)->port=PARALLEL_PORT_2;
	  else if(!strcmp(p, "parallel port #3")) (options.lp)->port=PARALLEL_PORT_3;
	  else if(!strcmp(p, "serial port #1")) (options.lp)->port=SERIAL_PORT_1;
	  else if(!strcmp(p, "serial port #2")) (options.lp)->port=SERIAL_PORT_2;
	  else if(!strcmp(p, "serial port #3")) (options.lp)->port=SERIAL_PORT_3;
	  else if(!strcmp(p, "serial port #4")) (options.lp)->port=SERIAL_PORT_4;
	  else if(!strcmp(p, "virtual port #1")) (options.lp)->port=VIRTUAL_PORT_1;
	  else if(!strcmp(p, "virtual port #2")) (options.lp)->port=VIRTUAL_PORT_2;
	  else if(!strcmp(p, "USB port #1")) (options.lp)->port=USB_PORT_1;
	  else if(!strcmp(p, "USB port #2")) (options.lp)->port=USB_PORT_2;
	  else if(!strcmp(p, "USB port #3")) (options.lp)->port=USB_PORT_3;
	  else if(!strcmp(p, "USB port #4")) (options.lp)->port=USB_PORT_4;
	  else stop(l);
	  continue;
	}
      if( (p=find_str(buffer, "method=")) )
	{
	  if(!strcmp(p, "automatic")) (options.lp)->method=IOM_AUTO;
	  else if(!strcmp(p, "asm"))	    
	    (options.lp)->method=IOM_ASM;
	  else if(!strcmp(p, "dcb")) 
	    (options.lp)->method=IOM_DCB;
	  else if(!strcmp(p, "kernel driver"))
	    (options.lp)->method=IOM_DRV;
	  else stop(l);
	  continue;
	}

      /* GtkTiEmu specific part: emulator section */
      if( (p=find_str(buffer, "background=")) )
	{
	  sscanf(p, "%i", options.background);
	  continue;
	}
      if( (p=find_str(buffer, "rom_file=")) )
	{
	  g_free(options.rom_file);
	  options.rom_file = g_strdup(p);
	  continue;
	}
      if( (p=find_str(buffer, "ram_file=")) )
	{
	  g_free(options.ram_file);
	  options.ram_file = g_strdup(p);
	  continue;
	}
      if( (p=find_str(buffer, "tib_file=")) )
	{
	  g_free(options.tib_file);
	  options.tib_file = g_strdup(p);
	  continue;
	}
      if( (p=find_str(buffer, "gray_planes=")) )
	{
	  sscanf(p, "%i", options.gray_planes);
	  continue;
	}
      if( (p=find_str(buffer, "tick_rate=")) )
	{
	  sscanf(p, "%lu", options.tick_rate);
	  continue;
	}
      if( (p=find_str(buffer, "cycle_rate=")) )
	{
	  sscanf(p, "%lu", options.cycle_rate);
	  continue;
	}
      if( (p=find_str(buffer, "itick=")) )
	{
	  sscanf(p, "%i", options.i_tick);
	  continue;
	}
      if( (p=find_str(buffer, "dark_color=")) )
	{
	  sscanf(p, "0x%06x", (unsigned int *)(options.dark_color));
	  continue;
	}
      if( (p=find_str(buffer, "light_color=")) )
	{
	  sscanf(p, "0x%06x", (unsigned int *)(options.light_color));
	  continue;
	}
      if( (p=find_str(buffer, "sync_one=")) )
	{
	  sscanf(p, "%i", options.sync_one);
	  continue;
	}
      if( (p=find_str(buffer, "code_lines=")) )
	{
	  sscanf(p, "%i", &(options.code_lines));
	  continue;
	}
      if( (p=find_str(buffer, "mem_lines=")) )
	{
	  sscanf(p, "%i", &(options.mem_lines));
	  continue;
	}
      if( (p=find_str(buffer, "stack_lines=")) )
	{
	  sscanf(p, "%i", &(options.stack_lines));
	  continue;
	}
      if( (p=find_str(buffer, "img_format=")) )
	{
	  if(!strcmp(p, "pcx")) options.img_format=IMG_PCX;
	  else if(!strcmp(p, "xpm")) options.img_format=IMG_XPM;
	  else if(!strcmp(p, "jpg")) options.img_format=IMG_JPG;
	  else stop(l);
	  continue;
	}
      if( (p=find_str(buffer, "img_type=")) )
	{
	  if(!strcmp(p, "bw")) options.img_type = IMG_BW;
	  else if(!strcmp(p, "color")) options.img_type = IMG_COL;
	  else stop(l);
	  continue;
	}
      if( (p=find_str(buffer, "img_size=")) )
	{
	  if(!strcmp(p, "lcd")) options.img_size = IMG_LCD;
	  else if(!strcmp(p, "skin")) options.img_size = IMG_SKIN;
	  else stop(l);
	  continue;
	}
      if( (p=find_str(buffer, "screen_file=")) )
	{
	  g_free(options.screen_file);
	  options.screen_file = g_strdup(p);
	  continue;
	}
      if( (p=find_str(buffer, "screen_counter=")) )
	{
	  sscanf(p, "%i", &(options.screen_counter));
	  continue;
	}
	if( (p=find_str(buffer, "console=")) )
	{
	  if(!strcmp(p, "no")) options.console = 0;
	  else if(!strcmp(p, "yes")) options.console = 1;
	  else if(!strcmp(p, "boot")) options.console = 2;
	  else stop(l);
	  continue;
	}
    }
  fclose(txt);

  fprintf(stderr, "rcfile, 0x%03x\n", (options.lp)->io_addr);

  return;
}

/* Write the contents of some variables to the RC file in a plain text format */
void write_rc_file(void)
{
  FILE *txt;

  get_rcfile_path(&rc_file);
  mkdir(CONF_DIR, 0666);
  txt=fopen(rc_file, "wt");
  if(txt==NULL)
    {
      //gif->msg_box(_("Error"), _("Unable to write the config file (~/.tilp or tilp.ini).\n"));
      return;
    }

  fprintf(txt, "# Config file for GtkTiEmu\n");
  fprintf(txt, "# Copyright (C) 2000-2001 Thomas Corvazier <corvazier@yahoo.com>\n");
  fprintf(txt, "# and Romain Lievin <rlievin@mail.com>\n");
  fprintf(txt, "# Warning: any comments that you add to this file WILL be overwritten\n");
  fprintf(txt, "\n");

  /* Common part with TiLP */
  fprintf(txt, "version=%s\n", GTKTIEMU_VERSION);
  fprintf(txt, "\n");
  fprintf(txt, "#\n");
  fprintf(txt, "# HARDWARE SECTION\n");
  fprintf(txt, "#\n");
  fprintf(txt, "\n");
  fprintf(txt, "# Link cable type\n");
  fprintf(txt, "linktype=");
  switch((options.lp)->link_type)
    {
    case LINK_PAR:
      fprintf(txt, "parallel\n");
      break;
    case LINK_SER:
      fprintf(txt, "serial\n");
      break;
    case LINK_TGL:
      fprintf(txt, "TIGraphLink\n");
      break; 
    case LINK_AVR:
      fprintf(txt, "fastAVRlink\n");
      break;
    case LINK_VTL:
      fprintf(txt, "virtual\n");
      break;
    case LINK_TIE:
      fprintf(txt, "TiEmulator\n");
      break;
    case LINK_VTI:
      fprintf(txt, "VTi\n");
      break;
    case LINK_UGL:
      fprintf(txt, "UsbGraphLink\n");
      break;
    case LINK_TPU:
      fprintf(txt, "TiPcUsbLink\n");
      break;
    }
  fprintf(txt, "\n");
  fprintf(txt, "# Port to use (serial, parallel, ...\n");
  fprintf(txt, "port=");
  switch((options.lp)->port)
    {
    case PARALLEL_PORT_1:
      fprintf(txt, "parallel port #1\n");
      break;
    case PARALLEL_PORT_2:
      fprintf(txt, "parallel port #2\n");
      break;
    case PARALLEL_PORT_3:
      fprintf(txt, "parallel port #3\n");
      break;
    case SERIAL_PORT_1:
      fprintf(txt, "serial port #1\n");
      break;
    case SERIAL_PORT_2:
      fprintf(txt, "serial port #2\n");
      break;
    case SERIAL_PORT_3:
      fprintf(txt, "serial port #3\n");
      break;
    case SERIAL_PORT_4:
      fprintf(txt, "serial port #4\n");
      break;
    case VIRTUAL_PORT_1:
      fprintf(txt, "virtual port #1\n");
      break;
    case VIRTUAL_PORT_2:
      fprintf(txt, "virtual port #2\n");
      break;
    case USB_PORT_1:
      fprintf(txt, "USB port #1\n");
      break;
    case USB_PORT_2:
      fprintf(txt, "USB port #2\n");
      break;
    case USB_PORT_3:
      fprintf(txt, "USB port #3\n");
      break;
    case USB_PORT_4:
      fprintf(txt, "USB port #4\n");
      break;
    }
  fprintf(txt, "\n");
  fprintf(txt, "# Method to use for I/O accesses\n");
  fprintf(txt, "method=");
  switch((options.lp)->method)
    {
    case IOM_AUTO:
      fprintf(txt, "automatic\n");
      break;
    case IOM_ASM:
      fprintf(txt, "asm\n");
      break;
    case IOM_DCB:
      fprintf(txt, "dcb\n");
      break;
    case IOM_DRV:
      fprintf(txt, "kernel driver\n");
      break;
    }
  fprintf(txt, "\n");
  fprintf(txt, "# Parallel/serial/virtual port address (0=automatic)\n");
  fprintf(txt, "adr_port=0x%03X\n", (options.lp)->io_addr);
  fprintf(txt, "\n");
  fprintf(txt, "# Serial device or character device (empty=automatic)\n");
  fprintf(txt, "serial_device=%s\n", (options.lp)->device);
  fprintf(txt, "\n");
  fprintf(txt, "# Baud rate for the fastAVRlink\n");
  fprintf(txt, "baudrate=%i\n", (options.lp)->baud_rate);
  fprintf(txt, "\n");
  fprintf(txt, "# Timeout value in 0.1 seconds\n");
  fprintf(txt, "timeout=%i\n", (options.lp)->timeout);
  fprintf(txt, "\n");
  fprintf(txt, "# Delay value\n");
  fprintf(txt, "delay=%i\n", (options.lp)->delay);
  fprintf(txt, "\n");
  fprintf(txt, "# Hardware flow control for fastAVRlink.\n");
  fprintf(txt, "rts_cts=%s\n", ((options.lp)->hfc == HFC_ON) ? "on" : "off");
  fprintf(txt, "\n");
  /*
  fprintf(txt, "# Use or not the Linux 'tidev' kernel module ?\n");
  fprintf(txt, "use_tidev=%s\n", (options.lp)->tidev ? "yes" : "no");
  fprintf(txt, "\n");
  fprintf(txt, "# Use or not the Win32 'DLPortI/O' kernel module ?\n");
  fprintf(txt, "use_dlportio=%s\n", (options.lp)->dlportio ? "yes" : "no");
  fprintf(txt, "\n");
  */

  /* Specific part to GtkTiEmu */
  fprintf(txt, "#\n");
  fprintf(txt, "# EMULATOR SECTION\n");
  fprintf(txt, "#\n");
  fprintf(txt, "\n");
  fprintf(txt, "# Background (0 for LCD only, 1 with skin)\n");
  fprintf(txt, "background=%i\n", *(options.background));
  fprintf(txt, "\n");
  fprintf(txt, "# ROM file\n");
  fprintf(txt, "rom_file=%s\n", options.rom_file);
  fprintf(txt, "\n");
  fprintf(txt, "# RAM file\n");
  fprintf(txt, "ram_file=%s\n", options.ram_file);
  fprintf(txt, "\n");
  fprintf(txt, "# TIB file\n");
  fprintf(txt, "tib_file=%s\n", options.tib_file);
  fprintf(txt, "\n");
  fprintf(txt, "# Number of gray planes (0, 2, 5, 11)\n");
  fprintf(txt, "gray_planes=%i\n", *(options.gray_planes));
  fprintf(txt, "\n");
  fprintf(txt, "# Timer Tick rate\n");
  fprintf(txt, "tick_rate=%lu\n", *(options.tick_rate));
  fprintf(txt, "\n");
  fprintf(txt, "# Cycle rate\n");
  fprintf(txt, "cycle_rate=%lu\n", *(options.cycle_rate));
  fprintf(txt, "\n");
  fprintf(txt, "# Instructions per timertick\n");
  fprintf(txt, "itick=%i\n", *(options.i_tick));
  fprintf(txt, "\n");
  fprintf(txt, "# Dark color (0xrrggbb)\n");
  fprintf(txt, "dark_color=0x%06X\n", (unsigned int) *(options.dark_color));
  fprintf(txt, "\n");
  fprintf(txt, "# Light color (0xrrggbb)\n");
  fprintf(txt, "light_color=0x%06X\n", (unsigned int) *(options.light_color));
  fprintf(txt, "\n");
  fprintf(txt, "# Synchronization of the refresh on IRQ1/timer (0 or 1)\n");
  fprintf(txt, "sync_one=%i\n", *(options.sync_one));
  fprintf(txt, "\n");
  fprintf(txt, "# Number of lines in the disassembly window\n");
  fprintf(txt, "code_lines=%i\n", options.code_lines);
  fprintf(txt, "\n");
  fprintf(txt, "# Number of lines in the stack window\n");
  fprintf(txt, "stack_lines=%i\n", options.stack_lines);
  fprintf(txt, "\n");
  fprintf(txt, "# Number of lines in the memory window\n");
  fprintf(txt, "mem_lines=%i\n", options.mem_lines);
  fprintf(txt, "\n");
  fprintf(txt, "\n");
  fprintf(txt, "#\n");
  fprintf(txt, "# SCREENSHOT SECTION\n");
  fprintf(txt, "#\n");
  fprintf(txt, "\n");
  fprintf(txt, "# Screenshot: image format (xpm, pcx, jpg, bmp)\n");
  fprintf(txt, "img_format=");
  switch(options.img_format)
    {
    case IMG_XPM: fprintf(txt, "xpm\n"); break;
    case IMG_PCX: fprintf(txt, "pcx\n"); break;
    case IMG_JPG: fprintf(txt, "jpg\n"); break;
	case IMG_BMP: fprintf(txt, "bmp\n"); break;
    }
  fprintf(txt, "\n");
  fprintf(txt, "# Screenshot: image type (bw, color)\n");
  fprintf(txt, "img_type=%s\n", (options.img_type == IMG_BW) ? "bw" : "color");
  fprintf(txt, "\n");
  fprintf(txt, "# Screenshot: image size (lcd, skin)\n");
  fprintf(txt, "img_size=%s\n", (options.img_size == IMG_LCD) ? "lcd" : "skin");
  fprintf(txt, "\n");

  fprintf(txt, "# Screenshot base file\n");
  fprintf(txt, "screen_file=%s\n", options.screen_file);
  fprintf(txt, "\n");
  fprintf(txt, "# Screenshot counter\n");
  fprintf(txt, "screen_counter=%i\n", options.screen_counter);
  fprintf(txt, "\n");
  fprintf(txt, "\n");
  fprintf(txt, "#\n");
  fprintf(txt, "# MISC SECTION\n");
  fprintf(txt, "#\n");
  fprintf(txt, "\n");
  fprintf(txt, "# Console mode (no, yes, boot)\n");
  fprintf(txt, "console=");
  switch(options.console)
  {
  case 0: fprintf(txt, "no\n"); break;
  case 1: fprintf(txt, "yes\n"); break;
  case 2: fprintf(txt, "boot\n"); break;
  default: fprintf(txt, "no\n"); break;
  }

  fprintf(txt, "\n");
  fprintf(txt, "RC_END\n");
  fflush(txt);

  //setup_link_and_calc(options.server);

  fclose(txt);
}

/*
  Share the libti68k fields of the glob_inf structure
 */
int init_options(struct GlobalInformation *gi)
{
  options.background = (int *) &(gi->background);
  options.rom_file = gi->romFile;
  options.ram_file = gi->ramFile;
  options.tib_file = gi->tibFile;
  options.gray_planes = &(gi->grayPlanes);
  options.tick_rate = (unsigned long *) &(gi->tickRate);
  options.cycle_rate = (unsigned long *) &(gi->cycleRate);
  options.i_tick = &(gi->itick);
  options.dark_color = (unsigned long *) &(gi->darkCol);
  options.light_color = (unsigned long *) &(gi->lightCol);
  options.sync_one = &(gi->syncOne);
  options.lp = &(gi->linkParam);

  return 0;
}

/*
  Initialize the options structure with some default values
*/
int load_default_config()
{
  // share structures fields
  init_options(&glob_inf);

  // Same initialization as in libti68k
  *(options.background) = 1;
  *(options.gray_planes) = 2;
  *(options.i_tick) = 640;
  *(options.tick_rate) = 40000;
  *(options.cycle_rate) = 150;
  *(options.dark_color) = 0x000034;
  *(options.light_color) = 0xA8B4A8;
  *(options.sync_one) = 0;

  // default link cable config
  //(options.lp)->calc_type = CALC_TI92;
#ifndef __WIN32__
  (options.lp)->link_type = LINK_TIE;
  (options.lp)->io_addr = VLINK0;
#else
  (options.lp)->link_type = LINK_TIE;
  (options.lp)->io_addr = VLINK0;
#endif
  strcpy((options.lp)->device, TTY1);
  (options.lp)->timeout = DFLT_TIMEOUT;
  (options.lp)->delay = DFLT_DELAY;
  (options.lp)->baud_rate = BR9600;

  // default link cable config (fill lp struct with default values)
  ticable_get_default_param(options.lp);

  // default ROM & RAM at startup
  options.rom_file = g_strconcat(inst_paths.rom_dir, "ti.rom", NULL);
  options.ram_file = g_strdup("");
  options.tib_file = g_strdup("");

  options.code_lines = DEBUG_LINES;
  options.stack_lines = STACK_LINES;
  options.mem_lines   = MEM_LINES;

  options.img_type = IMG_BW;
  options.img_format = IMG_PCX;
  options.img_size = IMG_LCD;

  options.screen_counter = 0;
  options.screen_file = g_strdup("screenshot");
  options.console=0;

  return 0;
}

