#! /usr/bin/perl -w

## ===========================================================================
##  This is the absorption portion of hephaestus

sub absorption {
  unless ($current =~ /$uses_periodic_regex/) {
    $periodic_table -> pack(-side=>'top', -padx=>4, -pady=>4, -fill=>'x');
  };
  $bottom{$current} -> packForget if $current;
  $frames{$current} -> configure(-relief=>'flat') if ($current and ($current ne 'help'));
  $current = "absorption";
  $frames{$current} -> configure(-relief=>'ridge');
  $bottom{absorption} -> pack(-side=>'top', -anchor=>'n', -fill=>'x', -expand=>1);
  $title->configure(-text=>'Periodic Table of Absorption Data');
  $data{pt_resource} -> grid(-column=>3, -columnspan=>7, -row=>0, -rowspan=>3, , -sticky=>'w');
};


sub setup_absorption {
  my $frame = $_[0] -> Frame(-borderwidth=>2, -relief=>'flat');


  ## energy and thickness entry widgets
  $data{abs_energy_label} = $frame -> Label(-text=>'Energy', @label_args)
    -> grid(-column=>0, -row=>4, -sticky=>'w');
  my $entry = $frame -> Entry(-width=>9, -textvariable=>\$data{abs_energy},
			      -validate=>'key', -validatecommand=>\&set_variable)
    -> grid(-column=>1, -row=>4, -sticky=>'ew');
  $data{abs_units_label} = $frame -> Label(-text=>"eV", @label_args)
    -> grid(-column=>2, -row=>4, -sticky=>'w');

  my $label = $frame -> Label(-text=>'Thickness', @label_args)
    -> grid(-column=>0, -row=>5, -sticky=>'w');
  $entry = $frame -> Entry(-width=>9, -textvariable=>\$data{abs_thickness},
			   -validate=>'key', -validatecommand=>\&set_variable)
    -> grid(-column=>1, -row=>5, -sticky=>'ew');
  $label = $frame -> Label(-text=>'m', @label_args)
    -> grid(-column=>2, -row=>5, -sticky=>'w');


  my $r = -1;
  foreach my $l ('Name', 'Number', 'Weight', 'Density',
		 'Absorption Length', 'Transmitted Fraction') {
    $r=5 if ($l eq 'Absorption Length');
    $label = $frame -> Label(-text=>$l, @label_args)
      -> grid(-column=>0, -row=>++$r, -sticky=>'w', -padx=>2);
    $entry = $frame -> Label(-relief=>'flat', -textvariable=>\$data{"abs_$l"},
			     -width=>12, -anchor=>'w', -font=>'Helvetica 10', @answer_args)
      -> grid(-column=>1, -row=>$r, -sticky=>'e', -padx=>2);
  };



  ## Table of Edge energies
  my $edges = $frame -> Scrolled("HList",
				 -columns    => 2,
				 -header     => 1,
				 -scrollbars => 'oe',
				 -background => $bgcolor,
				 -selectmode => 'extended',
				 #-selectbackground => $bgcolor,
				 -highlightcolor => $bgcolor,
				 -width      => 15,
				 -relief     => 'ridge',
				 )
      -> grid(-column=>4, -row=>0, -rowspan=>9, -padx=>3);
  my @header_style_params = ('text', -font=>'Helvetica 10 bold', -anchor=>'center', -foreground=>'blue4');
  my @label_style_params  = ('text', -anchor=>'center', -foreground=>'blue4');
  my $header_style = $edges -> ItemStyle(@header_style_params);
  my $label_style  = $edges -> ItemStyle(@label_style_params);
  $edges -> headerCreate(0, -text	   => "Edge",
			 -style		   => $header_style,
			 -headerbackground => $bgcolor,
			 -borderwidth	   => 1);
  $edges -> headerCreate(1, -text          => "Energy",
			 -style            => $header_style,
			 -headerbackground => $bgcolor,
			 -borderwidth	   => 1);
  $edges -> columnWidth(0, -char=>6);
  $edges -> columnWidth(1, -char=>8);
  $edges -> Subwidget("yscrollbar")
    -> configure(-background=>$bgcolor, ($is_windows) ? () : (-width=>8));
  foreach my $e (qw(K L1 L2 L3 M1 M2 M3 M4 M5 N1 N2 N3 N4 N5 N6 N7
		    O1 O2 O3 O4 O5 P1 P2 P3)) {
    $edges -> add($e);
    $edges -> itemCreate($e, 0, -text=>$e, -style=>$label_style);
    $edges -> itemCreate($e, 1);
  };
  $energies{edges} = $edges;

  ## Table of Line energies
  my $lines = $frame -> Scrolled("HList",
				 -columns    => 4,
				 -header     => 1,
				 -scrollbars => 'oe',
				 -background => $bgcolor,
				 -selectmode => 'extended',
				 #-selectbackground => $bgcolor,
				 -highlightcolor => $bgcolor,
				 -width      => 36,
				 -relief     => 'ridge',
				 )
      -> grid(-column=>5, -row=>0, -rowspan=>9, -padx=>3, -sticky=>'ew');
  $header_style = $lines -> ItemStyle(@header_style_params);
  $label_style  = $edges -> ItemStyle(@label_style_params);
  $lines -> headerCreate(0, -text	   => "Line",
			 -style		   => $header_style,
			 -headerbackground => $bgcolor,
			 -borderwidth	   => 1);
  $lines -> headerCreate(1, -text	   => "Trans.",
			 -style		   => $header_style,
			 -headerbackground => $bgcolor,
			 -borderwidth	   => 1);
  $lines -> headerCreate(2, -text	   => "Energy",
			 -style		   => $header_style,
			 -headerbackground => $bgcolor,
			 -borderwidth	   => 1);
  $lines -> headerCreate(3, -text	   => "Prob.",
			 -style		   => $header_style,
			 -headerbackground => $bgcolor,
			 -borderwidth	   => 1);
  $lines -> columnWidth(0, -char=>9);
  $lines -> columnWidth(1, -char=>10);
  $lines -> columnWidth(2, -char=>9);
  $lines -> columnWidth(3, -char=>7);
  $lines -> Subwidget("yscrollbar")
    -> configure(-background=>$bgcolor, ($is_windows) ? () : (-width=>8));
  foreach my $e (qw(Ka1 Ka2 Ka3 Kb1 Kb2 Kb3 Kb4 Kb5
		    La1 La2 Lb1 Lb2 Lb3 Lb4 Lb5 Lb6
		    Lg1 Lg2 Lg3 Lg6 Ll Ln Ma Mb Mg Mz)) {
    $lines -> add($e);
    $lines -> itemCreate($e, 0, -text=>Xray::Absorption -> get_Siegbahn_full($e), -style=>$label_style);
    $lines -> itemCreate($e, 1, -text=>Xray::Absorption -> get_IUPAC($e),         -style=>$label_style);
    $lines -> itemCreate($e, 2);
    $lines -> itemCreate($e, 3);
  };
  $energies{lines} = $lines;

  return $frame;
};





sub get_foils_data {
  my $elem = $_[0];
  my $in_resource = Xray::Absorption -> in_resource($elem);
  map {$probs{$_} = ''} keys(%probs);
  ## enable writing in the entry widgets
  #map {$_ -> configure(-state=>'normal')} @all_entries;
  $data{abs_Name}    = get_name($elem);
  $data{abs_Number}  = get_Z($elem);
  my $z              = $data{abs_Number};
  $data{abs_Weight}  = Xray::Absorption -> get_atomic_weight($elem);
  $data{abs_Weight}  = ($data{abs_Weight}) ? $data{abs_Weight} . ' amu' : '' ;
  my $density    = Xray::Absorption -> get_density($elem);
  $data{abs_Density} = ($density) ? $density . ' g/cm^3' : '' ;

  my @edges = (qw(K L1 L2 L3 M1 M2 M3 M4 M5 N1 N2 N3 N4 N5 N6 N7
		    O1 O2 O3 O4 O5 P1 P2 P3));
  my @lines = (qw(Ka1 Ka2 Ka3 Kb1 Kb2 Kb3 Kb4 Kb5
		    La1 La2 Lb1 Lb2 Lb3 Lb4 Lb5 Lb6
		    Lg1 Lg2 Lg3 Lg6 Ll Ln Ma Mb Mg Mz));

  foreach my $e (@edges, @lines) {
    $energies{$e} = Xray::Absorption -> get_energy($elem, $e);
    $energies{$e} ||= '';
    unless ($e =~ /^(K|([LMNOP][1-7]))$/) {
      next unless $energies{$e};
      if ($Xray::Absorption::resource eq 'elam') {
	$probs{$e} =
	  sprintf "%6.4f", Xray::Absorption -> get_intensity($elem, $e);
      };
    };
  };

  if (($z >= 22) and ($z <= 29)) {
    $energies{M4} = '';
    $energies{M5} = '';
  };
  if ($z <= 17) {
    $energies{M1} = '';
    $energies{M2} = '';
    $energies{M3} = '';
  };
  if ($data{units} eq "Wavelengths") {
    foreach (keys(%energies)) {
      next if ($_ eq 'lines');
      next if ($_ eq 'edges');
      $energies{$_} = &e2l($energies{$_});
    };
  };

  ## fill Edge and Line tables with these values
  my @data_style_params = ('text', -font=>'Helvetica 10', -anchor=>'e', -foreground=>'black');
  my $data_style   = $energies{edges} -> ItemStyle(@data_style_params);
  foreach my $e (@edges) {
    $energies{edges} -> itemConfigure($e, 1, -text=>$energies{$e}, -style=>$data_style);
  };
  $energies{edges} -> selectionClear;
  $energies{edges} -> anchorClear;
  $data_style   = $energies{lines} -> ItemStyle(@data_style_params);
  foreach my $l (@lines) {
    $energies{lines} -> itemConfigure($l, 2, -text=>$energies{$l}, -style=>$data_style);
    $energies{lines} -> itemConfigure($l, 3, -text=>$probs{$l}, -style=>$data_style);
  };
  $energies{lines} -> selectionClear;
  $energies{lines} -> anchorClear;


  ##My $is_gas = ($elem =~ /\b(Ar|Br|Cl|F|H|He|Kr|N|Ne|O|Rn|Xe)\b/);
  my $is_gas = ($elem =~ /\b(Ar|Cl|H|He|Kr|N|Ne|O|Rn|Xe)\b/);

  $data{'abs_Absorption Length'} = '';
  $data{'abs_Transmitted Fraction'}       = '';
  my $bail = 0;
  if ($data{abs_energy} and $in_resource) {
    if ((lc($data{resource}) eq "henke") and ($data{abs_energy} > 30000)) {
      my $dialog =
	$top -> Dialog(-bitmap         => 'info',
		       -text           => "The Henke tables only include data up to 30 keV.",
		       -title          => 'Hephaestus warning',
		       -buttons        => [qw/OK/],
		       -default_button => 'OK')
	  -> Show();
      return;
    };
    if (($data{abs_energy} < $data{abs_odd_value}) and ($data{units} eq "Energies")) {
      my $dialog = $top -> DialogBox(-title=>"Hephaestus warning!",
				     -buttons=>['OK', 'Cancel'],);
      $dialog -> add("Label", qw/-padx .25c -pady .25c -text/,
		     "You have chosen a very low energy.  Should I$/" .
		     "try to calculate the absorption length?$/" .
		     "(There might be no data at that energy!)",)
	-> pack(-side=>'left');
      my $answer = $dialog -> Show;
      ($answer eq 'Cancel') and $bail = 1;
    } elsif (($data{abs_energy} > $data{abs_odd_value}) and ($data{units} eq "Wavelengths")) {
      my $dialog = $top -> DialogBox(-title=>"Hephaestus warning!",
				     -buttons=>['OK', 'Cancel'],);
      $dialog -> add("Label", qw/-padx .25c -pady .25c -text/,
		     "You have chosen a very large wavelnegth.  Should I$/" .
		     "try to calculate the absorption length?$/" .
		     "(There might be no data at that wavelength!)",)
	-> pack(-side=>'left');
      my $answer = $dialog -> Show;
      ($answer eq 'Cancel') and $bail = 1;
    };
    unless ($bail) {
      my $conv   = Xray::Absorption -> get_conversion($elem);
      ($data{units} eq "Wavelengths") and $data{abs_energy} = &e2l($data{abs_energy});
      my $barns  = Xray::Absorption -> cross_section($elem, $data{abs_energy}, $data{xsec});
      ($data{units} eq "Wavelengths") and $data{abs_energy} = &e2l($data{abs_energy});
      my $factor = ($is_gas) ? 1 : 10000;
      my $abslen = ($conv and $barns and $density) ?
	$factor/($barns*$density/$conv) : 0;
      $data{'abs_Absorption Length'} = '';
      if ($abslen) {
	$data{'abs_Absorption Length'}  = 	sprintf "%8.2f", $abslen;
	$data{'abs_Absorption Length'} .= ($is_gas) ? ' cm' : ' m';
	$data{'abs_Absorption Length'} =~ s/^\s+//;
      };

      $data{'abs_Transmitted Fraction'} = '';
      ##print join("  ", $conv, $barns, $density, $thickness, $abslen, $is_gas, $/);
      if ($data{abs_thickness} and $abslen) {
	my $factor = $data{abs_thickness} / $abslen;
	$data{'abs_Transmitted Fraction'} = sprintf ("%6.4g", exp(-1 * $factor));
      };
    };
  };
  ## and disable writing in the entry widgets once again
  #map {$_ -> configure(-state=>'disabled')} @all_entries;
}
