/********************************************************************
 *
 *	Copyright (C) 1999-2002 Nathan Fiedler
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * PROJECT:     Abstract Data Types
 * MODULE:      Fibonacci Heap ADT
 * FILE:        TestFibonacciHeap.java
 *
 * AUTHOR:      Nathan Fiedler
 *
 * REVISION HISTORY:
 *      Name    Date            Description
 *      ----    ----            -----------
 *      nf      03/30/98        Initial version
 *      nf      01/21/01        Changed to use Node inner class
 *      nf      01/05/02        Greatly improved
 *
 * DESCRIPTION:
 *      Implements the test wrapper for the FibonacciHeap data structure.
 *
 * $Id: TestFibonacciHeap.java,v 1.2 2002/01/06 08:54:07 nfiedler Exp $
 *
 *******************************************************************/

package com.bluemarsh.adt.test;

import com.bluemarsh.adt.FibonacciHeap;

/**
 * Test wrapper for the FibonacciHeap class.
 *
 * @author  Nathan Fiedler
 */
public class TestFibonacciHeap {

    /**
     * Test wrapper for FibonacciHeap class.
     *
     * @param  args  array of string arguments
     */
    public static void main(String[] args) {
        System.out.println();
        System.out.println("  + = good");
        System.out.println("  - = bad");
        System.out.println("  ! = information");
        System.out.println("  ? = verify manually");
        System.out.println();

	FibonacciHeap heap = new FibonacciHeap();
	if (heap.isEmpty() && heap.size() == 0) {
            System.out.println("+ heap is empty");
        } else {
	    System.out.println("- heap is not empty");
            return;
	}

	System.out.println("! populating the heap...");
	heap.insert(new Double(50), 50);
	FibonacciHeap.Node min1 = heap.min();
	heap.insert(new Double(10), 10);
	heap.insert(new Double(100), 100);
	heap.insert(new Double(80), 80);
	heap.insert(new Double(30), 30);
	heap.insert(new Double(-10), -10);
	FibonacciHeap.Node min2 = heap.min();
	heap.insert(new Double(-30), -30);
	heap.insert(new Double(40), 40);
	heap.insert(new Double(0), 0);
	heap.insert(new Double(20), 20);
	heap.insert(new Double(-20), -20);
	heap.insert(new Double(-0), -0);
	heap.insert(new Double(Double.POSITIVE_INFINITY),
                    Double.POSITIVE_INFINITY);
	// don't bother trying -Infinity, delete() will fail
	heap.insert(new Double(-Double.MAX_VALUE),
                    -Double.MAX_VALUE);
	heap.insert(new Double(Double.MAX_VALUE),
                    Double.MAX_VALUE);

        if (!heap.isEmpty() && heap.size() == 15) {
            System.out.println("+ size() seems to work");
        } else {
            System.out.println("- size() failed");
            return;
        }

        FibonacciHeap.Node node = heap.min();
        Double d = (Double) node.data;
        if (d.doubleValue() == -Double.MAX_VALUE) {
            System.out.println("+ min() appears to work");
        } else {
            System.out.println("- min() failed");
            return;
        }

	heap.delete(min2);
        if (!heap.isEmpty() && heap.size() == 14) {
            System.out.println("+ delete() seems to work");
        } else {
            System.out.println("- delete() failed to change the size");
            return;
        }

        d = (Double) heap.removeMin();
        if (d.doubleValue() == -Double.MAX_VALUE) {
            System.out.println("+ removeMin() appears to work");
        } else {
            System.out.println("- removeMin() failed");
            return;
        }

	System.out.println("! decreasing key of " + min1 + " to -100");
	heap.decreaseKey(min1, -100);
        node = heap.min();
        d = (Double) node.data; // value is 50.0
        if (d.doubleValue() == 50) {
            System.out.println("+ decreaseKey() appears to work");
        } else {
            System.out.println("- decreaseKey() failed, got " + d);
            return;
        }

	System.out.println("! creating two heaps and joining them...");

	FibonacciHeap heap2 = new FibonacciHeap();
	heap2.insert(new Double(12), 12);
	heap2.insert(new Double(-12), -12);
	heap2.insert(new Double(20), 20);
	heap2.insert(new Double(-4), -4);
	heap2.insert(new Double(35), 35);

	heap = FibonacciHeap.union(heap, heap2);
        if (!heap.isEmpty() && heap.size() == 18) {
            System.out.println("+ joined heap has correct size");
        } else {
            System.out.println("- joined heap has incorrect size, " +
                               heap.size());
            return;
        }

        System.out.println("! clearing the heap");
        heap.clear();
	if (heap.isEmpty() && heap.size() == 0) {
            System.out.println("+ heap is empty");
        } else {
	    System.out.println("- heap is not empty");
            return;
	}

        // Insert a bunch of values in mixed order.
        for (double ii = 0; ii < 100; ii++) {
            heap.insert(new Double(ii), ii);
        }
        for (double ii = 200; ii < 300; ii++) {
            heap.insert(new Double(ii), ii);
        }
        for (double ii = 100; ii < 200; ii++) {
            heap.insert(new Double(ii), ii);
        }
        for (double ii = 400; ii < 500; ii++) {
            heap.insert(new Double(ii), ii);
        }
        for (double ii = 800; ii < 1000; ii++) {
            heap.insert(new Double(ii), ii);
        }
        for (double ii = 500; ii < 800; ii++) {
            heap.insert(new Double(ii), ii);
        }
        for (double ii = 300; ii < 400; ii++) {
            heap.insert(new Double(ii), ii);
        }

	if (!heap.isEmpty() && heap.size() == 1000) {
            System.out.println("+ heap contains 1000 elements");
        } else {
	    System.out.println("- heap size is incorrect, got " + heap.size());
            return;
	}

        System.out.println("! verifying those elements");
        double jj = 0;
        while (!heap.isEmpty()) {
            d = (Double) heap.removeMin();
            if (d.doubleValue() != jj) {
                System.out.println("- min element incorrect at " + jj +
                                   ", got " + d);
                return;
            }
            jj++;
        }

	System.out.println();
	System.out.println("+ Tests completed successfully.");
    } // main
} // TestFibonacciHeap
