// -*- C++ -*-
/* This file is part of
 * ====================================================== 
 * 
 *           LyX, The Document Processor
 * 	 
 *	    Copyright 1997 Asger Alstrup
 *           and the LyX Team.
 *
 * ====================================================== */

#include <config.h>
#include <cmath>	// fabs()

#ifdef __GNUG__
#pragma implementation "FontLoader.h"
#endif

#include "gettext.h"
#include "FontLoader.h"
#include "FontInfo.h"
#include "debug.h"
#include "lyxrc.h"	// lyxrc.font_*
#include "BufferView.h"
#include "LyXView.h"
#include "minibuffer.h"

using std::endl;

extern BufferView * current_view;


// The global fontloader
FontLoader fontloader;


// Initialize font loader
FontLoader::FontLoader()
{
	reset();
}

// Destroy font loader
FontLoader::~FontLoader()
{
	unload();
}

// Update fonts after zoom, dpi, font names, or norm change
// For now, we just ditch all fonts we have. Later, we should
// reuse the ones that are already loaded.
void FontLoader::update()
{
	unload();
}

// Reset font loader
void FontLoader::reset()
{
	// Clear font infos, font structs and font metrics
	for (int i1 = 0; i1 < 4; ++i1)
		for (int i2 = 0; i2 < 2; ++i2)
			for (int i3 = 0; i3 < 4; ++i3) {
#ifdef I18N
                            for (int i4=0; i4<3; ++i4)
                                fontinfo[i4][i1][i2][i3] = 0;
#else			
				fontinfo[i1][i2][i3] = 0;
#endif				
				for (int i4 = 0; i4<10; ++i4) {
#ifdef I18N
                                        fontset[i1][i2][i3][i4] = 0;
                                        math[i1][i2][i3][i4] = 0;
#else				
					fontstruct[i1][i2][i3][i4] = 0;
#endif					
				}
			}
}

// Unload all fonts
void FontLoader::unload()
{
	// Unload all fonts
	for (int i1 = 0; i1 < 4; ++i1)
		for (int i2 = 0; i2 < 2; ++i2)
			for (int i3 = 0; i3 < 4; ++i3) {
#ifdef I18N
                            for (int i4=0; i4<3; ++i4)
                                if (fontinfo[i4][i1][i2][i3]) {
                                    delete fontinfo[i4][i1][i2][i3];
                                    fontinfo[i4][i1][i2][i3] = 0;
                                };
#else
				if (fontinfo[i1][i2][i3]) {
					delete fontinfo[i1][i2][i3];
					fontinfo[i1][i2][i3] = 0;
				}
#endif
				for (int i4 = 0; i4 < 10; ++i4) {
#ifdef I18N
                                    if (fontset[i1][i2][i3][i4]){
                                        XFreeFontSet(fl_display, fontset[i1][i2][i3][i4]);
                                        fontset[i1][i2][i3][i4] = 0;
                                    }
                                    if (math[i1][i2][i3][i4]) {
                                        XFreeFont(fl_display, math[i1][i2][i3][i4]);
                                        math[i1][i2][i3][i4] = 0;
                                    }
#else
					if (fontstruct[i1][i2][i3][i4]) {
						XFreeFont(fl_get_display(), fontstruct[i1][i2][i3][i4]);
						fontstruct[i1][i2][i3][i4] = 0;
					}
#endif
				}
			}
}

// Get font info
/* Takes care of finding which font that can match the given request. Tries
different alternatives. */
#ifdef I18N
 void FontLoader::getFontinfo(int fonttype,
                              LyXFont::FONT_FAMILY family,
 			     LyXFont::FONT_SERIES series,
 			     LyXFont::FONT_SHAPE shape)
 {
      if (fontinfo[fonttype][family][series][shape] != 0)
#else
void FontLoader::getFontinfo(LyXFont::FONT_FAMILY family,
			     LyXFont::FONT_SERIES series,
			     LyXFont::FONT_SHAPE shape)
{
	// Do we have the font info already?
	if (fontinfo[family][series][shape] != 0)
#endif
		return;

	// Special code for the symbol family
	if (family == LyXFont::SYMBOL_FAMILY){
#ifdef I18N
                 fontinfo[fonttype][family][series][shape] = new FontInfo("-*-symbol-*-adobe-fontspecific");
#else
		fontinfo[family][series][shape] = new FontInfo("-*-symbol-*-adobe-fontspecific");
#endif
		return;
	}

	// Normal font. Let's search for an existing name that matches.
	string ffamily;
	string fseries;
	string fshape;
#ifdef I18N
         string norm;
         if (fonttype==1 || fonttype==2)
             norm = lyxrc.i18n_font_norm[fonttype-1];
         else
             norm = lyxrc.font_norm;
#else	
	string norm = lyxrc.font_norm;
#endif	
	string fontname;

	FontInfo * fi = new FontInfo();
#ifdef I18N
         fontinfo[fonttype][family][series][shape] = fi;
#else	
	fontinfo[family][series][shape] = fi;
#endif

#ifdef I18N
         for (int cfam = 0; cfam < 2; ++cfam) {
#else
	for (int cfam = 0; cfam < 2; ++cfam) {
#endif	
	
#ifdef I18N
                 if (fonttype==1 || fonttype==2)
                     switch(series){
                         case LyXFont::MEDIUM_SERIES:
                             switch (cfam) {
                                 case 0: ffamily = lyxrc.i18n_normal_font_name[fonttype-1]; break;
                                 case 1: ffamily = "-*-mincho";
                                 default: cfam = 100;
                             }
                             break;
                         case  LyXFont::BOLD_SERIES:
                             switch (cfam) {
                                 case 0: ffamily = lyxrc.i18n_gothic_font_name[fonttype-1]; break;
                                 case 1: ffamily = "-*-gothic";
                                 default: cfam = 100;
                             }
                             break;
                         default: ;
                     }
                 else{
#endif
		// Determine family name
		switch (family) {
		case LyXFont::ROMAN_FAMILY:
			switch (cfam) {
			case 0: ffamily = lyxrc.roman_font_name; break;
			case 1: ffamily = "-*-times";
			default: cfam = 100;
			}
			break;
		case LyXFont::SANS_FAMILY:
			switch (cfam) {
			case 0: ffamily = lyxrc.sans_font_name; break;
			case 1: ffamily = "-*-helvetica";
			default: cfam = 100;
			}
			break;
		case LyXFont::TYPEWRITER_FAMILY:
			switch (cfam) {
			case 0: ffamily = lyxrc.typewriter_font_name; break;
			case 1: ffamily = "-*-courier";
			default: cfam = 100;
			}
			break;
		default: ;
		}
#ifdef I18N
                }
#endif

#ifdef I18N
                for (int cser=0; cser < 5; ++cser) {
#else
		for (int cser = 0; cser < 4; ++cser) {
#endif		
			// Determine series name
			switch (series) {
			case LyXFont::MEDIUM_SERIES:
				switch (cser) {
				case 0: fseries = "-medium"; break;
				case 1: fseries = "-book"; break;
#ifdef I18N
                                case 2: fseries = "-light"; break;
                                case 3: fseries = "-*";
#else				
				case 2: fseries = "-light";
#endif								
				default: cser = 100;
				}
				break;
			case LyXFont::BOLD_SERIES:
				switch (cser) {
				case 0: fseries = "-bold"; break;
				case 1: fseries = "-black"; break;			
				case 2: fseries = "-demi"; break;
#ifdef I18N
                                case 3: fseries = "-demibold"; break;
                                case 4: fseries = "-*";
#else				
				case 3: fseries = "-demibold";
#endif				
				default: cser = 100;
				}
				break;
			default: ;
			}
#ifdef I18N
                        for (int csha = 0; csha < 2; ++csha) {
#else
			for (int csha = 0; csha < 2; ++csha) {
#endif			
				// Determine shape name
				switch (shape) {
				case LyXFont::UP_SHAPE:
				case LyXFont::SMALLCAPS_SHAPE:
					switch (csha) {
					case 0: fshape = "-r";
					default: csha = 100;
					}
					break;
				case LyXFont::ITALIC_SHAPE:
					switch (csha) {
					case 0: fshape = "-i"; break;
					case 1: fshape = "-o";
					default: csha = 100;
					}
					break;
				case LyXFont::SLANTED_SHAPE:
					switch (csha) {
					case 0: fshape = "-o"; break;
					case 1: fshape = "-i";
					default: csha = 100;
					}
					break;
				default: ;
				}
				//
				fontname = ffamily + fseries + fshape +
					   "-normal-*-*-*-*-*-*-*-" + norm;
				fi->setPattern(fontname);
				if (fi->exist()) {
					return;
				}
			}
		}
	}
}


// A dummy fontstruct used when there is no gui. 
static XFontStruct dummyXFontStruct;
static bool dummyXFontStructisGood = false;

/// Do load font
#ifdef I18N
XFontSet FontLoader::doLoad(LyXFont::FONT_FAMILY family,
				LyXFont::FONT_SERIES series,
				LyXFont::FONT_SHAPE shape,
				LyXFont::FONT_SIZE size)
{
        int fonttype = 0;
	getFontinfo(fonttype,family, series, shape);
#else
XFontStruct * FontLoader::doLoad(LyXFont::FONT_FAMILY family, 
				LyXFont::FONT_SERIES series, 
				LyXFont::FONT_SHAPE shape, 
				LyXFont::FONT_SIZE size)
{
	if (!lyxrc.use_gui) {
		if (!dummyXFontStructisGood) {
			// no character specific info
			dummyXFontStruct.per_char = 0; 
			// unit ascent on character displays
			dummyXFontStruct.ascent = 1; 
			// no descent on character displays
			dummyXFontStruct.descent = 0; 
			dummyXFontStructisGood = true;
		}

		return &dummyXFontStruct;
	}

	getFontinfo(family, series, shape);
#endif	
	int fsize = int( (lyxrc.font_sizes[size] * lyxrc.dpi * 
			  (lyxrc.zoom/100.0) ) / 72.27 + 0.5 );
#ifdef I18N
        string font = fontinfo[fonttype][family][series][shape]->getFontname(fsize);
#else
	string font = fontinfo[family][series][shape]->getFontname(fsize);
#endif

#ifdef I18N
        for (int i=1;i<3;i++){
            if (lyxrc.i18n_font_use[i-1]){
                getFontinfo(i,family,series,shape);
                font = font +","+ fontinfo[i][family][series][shape]->getFontname(fsize);
            }
        }
#endif
	if (font.empty()) {
		lyxerr << "No font matches request. Using 'fixed'." << endl;
		lyxerr << "Start LyX as 'lyx -dbg 515' to get more information." << endl;
		font = "fixed";
	}
#ifdef I18N
         XFontSet fs=0;
#else
	XFontStruct * fs = 0;
#endif

	current_view->owner()->getMiniBuffer()->Store();
	current_view->owner()->getMiniBuffer()->Set(_("Loading font into X-Server..."));
#ifdef I18N
        char		**missing_list, *def_string;
        int		missing_count;

 //       XFontSet fs=0;
//        lyxerr << "Load fontset. " + font << endl;
        fs=XCreateFontSet(fl_display, font.c_str(),&missing_list,&missing_count,&def_string);
        if (!fs) {
            lyxerr << "Could not get fontset. " << font << " Using 'fixed'." << endl;
            font = font +",*-fixed-*";
            fs=XCreateFontSet(fl_display,font.c_str(),&missing_list,&missing_count,&def_string);
        }

        if (missing_count > 0) {
            lyxerr << "Could not get fontset. " << font << endl;
            for (int i = 0; i < missing_count; i++) {
                lyxerr << "Cannot load fontset " << string(missing_list[i]) << endl;
            }

            XFreeStringList(missing_list);
            lyxerr << " Using 'fixed'." << endl;
            XFreeFontSet(fl_display, fs);
            font = font +",*-fixed-*";
            fs=XCreateFontSet(fl_display,font.c_str(),&missing_list,&missing_count,&def_string);
        }
#else	
	fs = XLoadQueryFont(fl_get_display(), font.c_str());
	
	if (fs == 0) {
		if (font == "fixed") {
			lyxerr << "We're doomed. Can't get 'fixed' font." << endl;
		} else {
			lyxerr << "Could not get font. Using 'fixed'." << endl;
			fs = XLoadQueryFont(fl_get_display(), "fixed");
		}
	} else if (lyxerr.debugging(Debug::FONT)) {
		// Tell user the font matching
		LyXFont f;
		f.setFamily(family);
		f.setSeries(series);
		f.setShape(shape);
		f.setSize(size);
		// The rest of the attributes are not interesting
		f.setEmph(LyXFont::INHERIT);
		f.setUnderbar(LyXFont::INHERIT);
		f.setNoun(LyXFont::INHERIT);
		f.setLatex(LyXFont::INHERIT);
		f.setColor(LColor::inherit);
		lyxerr << "Font '" << f.stateText(0) 
		       << "' matched by\n" << font << endl;
	}
#endif

	current_view->owner()->getMiniBuffer()->Reset();
#ifdef I18N
        fontset[family][series][shape][size] = fs;
#else
	fontstruct[family][series][shape][size] = fs;
#endif
	return fs;
}

#ifdef I18N
XFontStruct * FontLoader::doLoadMath(LyXFont::FONT_FAMILY family,
                                  LyXFont::FONT_SERIES series,
                                  LyXFont::FONT_SHAPE shape,
                                  LyXFont::FONT_SIZE size)
{
	getFontinfo(0,family,series,shape);
	int fsize = (int) ( (lyxrc.font_sizes[size] * lyxrc.dpi *
				(lyxrc.zoom/100.0) ) / 72.27 + 0.5 );

	string font = fontinfo[0][family][series][shape]->getFontname(fsize);

	if (font.empty()) {
		lyxerr << "No font matches request. Using 'fixed'." << endl;
		lyxerr << "Start LyX as 'lyx -dbg 515' to get more information." << endl;
		font = "fixed";
	}

	current_view->owner()->getMiniBuffer()->Store();
	current_view->owner()->getMiniBuffer()->Set(_("Loading font into X-Server..."));

	XFontStruct * fs = XLoadQueryFont(fl_display, font.c_str());

	if (fs == 0) {
		if (font == "fixed") {
			lyxerr << "We're doomed. Can't get 'fixed' font." << endl;
		} else {
			lyxerr << "Could not get font. Using 'fixed'." << endl;
			fs = XLoadQueryFont(fl_display, "fixed");
		}
	} else {
		if (lyxerr.debugging(Debug::FONT)) {
			// Tell user the font matching
			LyXFont f;
			f.setFamily(family);
			f.setSeries(series);
			f.setShape(shape);
			f.setSize(size);
			// The rest of the attributes are not interesting
			f.setEmph(LyXFont::INHERIT);
			f.setUnderbar(LyXFont::INHERIT);
			f.setNoun(LyXFont::INHERIT);
			f.setLatex(LyXFont::INHERIT);
			f.setColor(LColor::inherit);
			lyxerr << "Font '" << f.stateText(0)
			       << "' matched by\n" << font << endl;
		}
	}
	current_view->owner()->getMiniBuffer()->Reset();

	math[family][series][shape][size] = fs;
	return fs;
}
#endif
