/* ExpressionSaver.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.expr;

import org.grinvin.invariants.Invariant;
import org.grinvin.list.ExpressionList;

import org.jdom.Element;

/**
 * Helper methods for saving expressions to disk.
 */
public class ExpressionSaver implements Expression.Visitor {
    
    private ExpressionSaver() {
        // not to be instantiated
    }
    
    //
    private Element element;
    
    private void setElement(Element element) {
        this.element = element;
    }
    
    //
    @SuppressWarnings("PMD.SingularField")
    private ExpressionSaver childVisitor;
    
    /**
     * Get a visitor for visiting children of expressions.
     */
    private ExpressionSaver getChildVisitor() {
        if (childVisitor == null)
            childVisitor = new ExpressionSaver();
        return childVisitor;
    }
    
    /**
     * Save the given list of expressions into the given JDOM element.
     */
    public static void save(ExpressionList list, Element element) {
        ExpressionSaver saver = new ExpressionSaver();
        saver.setElement(element);
        for (Expression expr: list)
            expr.accept(saver);
    }
    
    /**
     * Save the given expression into the given JDOM element.
     */
    public static void save(Expression expression, Element element) {
        if(expression==null)
            return;
        ExpressionSaver saver = new ExpressionSaver();
        saver.setElement(element);
        expression.accept(saver);
    }
    
    public void visitInvariant(Invariant invariant) {
        element.addContent(
                new Element("invariant").setAttribute("id", invariant.getId())
                );
    }
    
    public void visitCompound(Compound compound) {
        Element child = new Element ("compound");
        Operator operator = compound.getOperator();
        child.setAttribute("operator", operator.toString());
        getChildVisitor().setElement(child);
        for (int i=0; i < operator.getNumberOfArguments(); i++)
            compound.getChild(i).accept(childVisitor);
        element.addContent(child);
    }
    
    public void visitIntegerLiteral(IntegerLiteral literal) {
        element.addContent(
                new Element("integer").setAttribute("value", literal.toString())
                );
    }
    
}
