/* GraphLoader_1_0.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io.compat;

import java.io.IOException;
import java.io.InputStream;

import org.grinvin.graphs.Annotation;
import org.grinvin.graphs.Edge;
import org.grinvin.graphs.Graph;
import org.grinvin.graphs.Vertex;
import org.grinvin.io.IOFormatException;
import org.grinvin.io.LoaderSaverHelper;

import org.jdom.Element;

/**
 * Loads an old (1.0) <tt>graph.xml</tt>-section from an input stream.
 */
public class GraphLoader_1_0 {
    
    //
    private GraphLoader_1_0() { }
    
    /**
     * Construct the abstract graph from the given JDOM element
     * and fill the translation table.
     */
    private static void toGraph(Graph graph, Annotation annotation, Element element) throws IOFormatException {

        int numberOfVertices = Integer.parseInt(element.getAttributeValue("nrofvertices"));
        
        // annotations
        String[] annotations = new String[numberOfVertices];
        for (Object obj: element.getChildren("vertex")) {
            Element v = (Element)obj;
            int id = Integer.parseInt(v.getAttributeValue("id"));
            annotations[id] = v.getAttributeValue("annotation");
        }
        
        // vertices
        Vertex[] vertices = new Vertex[numberOfVertices];
        for (int i=0; i < numberOfVertices; i++) {
            vertices[i] = graph.addNewVertex();
            annotation.setAnnotation(vertices[i], annotations[i]);
        }
        
        // edges
        for (Object obj: element.getChildren("edge")) {
            Element e = (Element)obj;
            int first = Integer.parseInt(e.getAttributeValue("from"));
            int second = Integer.parseInt(e.getAttributeValue("to"));
            Edge edge = graph.addNewEdge(vertices[first], vertices[second]);
            annotation.setAnnotation(edge, e.getAttributeValue("annotation"));
        }
    }
    
    /**
     * Load the graph from the input stream. After the call the input stream
     * should no longer be used.
     */
    public static void load(Graph graph, Annotation annotation, InputStream input) throws IOException {
        toGraph(graph, annotation, LoaderSaverHelper.loadElementNonValidating(input, "graph"));
    }

    public static void load(Graph graph, Annotation annotation, Element element) throws IOException {
        toGraph(graph, annotation, LoaderSaverHelper.checkElement(element, "graph"));
    }
    
}
