/**************************************************************
*   
*   Creation Date: <1998-11-21 16:07:47 samuel>
*   Time-stamp: <2001/06/27 22:00:51 samuel>
*   
*	<emu.c>
*	
*	Emulation of some assembly instructions
*
*   Copyright (C) 1998, 1999, 2000, 2001 Samuel Rydh
*  
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation;
*
**************************************************************/

#include "compat.h"
#include <linux/sys.h>
#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <asm/unistd.h>

#include "mmu.h"
#include "kernel_vars.h"
#include "emu.h"
#include "asmfuncs.h"
#include "rvec.h"

#define PERFORMANCE_INFO
#include "performance.h"

#define BAT_PERFORMANCE_HACK
// #define DEBUG

/* If BAT_PERFORMANCE_HACK is defined, PTEs corresponding to a mac bat
 * mapping will not necessary be flushed when the bat registers are 
 * touched. This gives a huge performance gain in MacOS 9.1 (which 
 * clears the bat registers in the idle loop). Of course, this break
 * compatibility (although most operating systems initializes the
 * BATs once and for all).
 */

#ifdef BAT_PERFORMANCE_HACK
 #define BAT_HACK(kv) (kv->mmu.bat_hack_count++ < 0x100)
#else
 #define BAT_HACK(kv) 1
#endif

#define MREGS	(kv->mregs)
#define MMU	(kv->mmu)


/************************************************************************/
/*	F U N C T I O N S						*/
/************************************************************************/


asmlinkage int 
do_mtsr( kernel_vars_t *kv, int sr, ulong value )
{
	ulong	user_sr,sv_sr;
	
#ifdef DEBUG
	if( sr<0 || sr>15) {
		printk("do_mtsr: sr out of range!\n");
		return RVEC_INTERNAL_ERROR;
	}
#endif
	MREGS.segr[sr]=value;

	/* vsid_lookup translates 20 bits, *NOT* the lowest four bits!
	 */
	vsid_lookup( kv, value, &user_sr, &sv_sr );

	/* Both user_sr and sv_sr should have Ku(2) set but not Ks(1)
	 * user_sr is used if key=1, otherwise sv_sr is used.
	 * Bit3 is the no-execute bit (now supported).
	 */
	user_sr = (user_sr + (value & 0xf) * MUNGE_ESID_ADD) & VSID_MASK;
	user_sr |= (value & BIT(3)) | BIT(2);
	sv_sr = (sv_sr + (value & 0xf) * MUNGE_ESID_ADD) & VSID_MASK;
	sv_sr |= (value & BIT(3)) | BIT(2);

	kv->mmu.user_sr[sr] = (value & BIT(2)) ? user_sr : sv_sr;
	kv->mmu.sv_sr[sr] = (value & BIT(1)) ? user_sr : sv_sr;

	/* Splitmode segment registers need to be updated */
	invalidate_splitmode_sr(kv);

	BUMP(do_mtsr);
	return RVEC_NOP;
}


asmlinkage int 
do_mtsdr1( kernel_vars_t *kv, ulong value )
{
	ulong	mbase, mask;
	char	*base;
	
	MREGS.spr[S_SDR1] = value;

	mask = ((value & 0x1ff)<<16) | 0xffff;
	mbase = value & ~mask;

	base = mbase - MMU.mac_ram_base + MMU.linux_ram_base;

	if( base < MMU.linux_ram_base
	    || base + mask >= MMU.linux_ram_base + MMU.ram_size ) 
	{
		/* S_SDR1 out of range */
		printk("WARNING, S_SDR1, %08lX is out of range\n", value);

		/* fallback... */
		base = MMU.linux_ram_base;
		mbase = 0;
		mask = 0xffff;
	}
	MMU.hash_base = (ulong*)base;
	MMU.hash_mbase = mbase;
	MMU.hash_mask = mask;
	/* Clear old tlbhash matching */
	MMU.pthash_sr = -1;
	
	/* Due to the new tlbie implementation, we must flush everything */
	flush_vsid_table( kv );

	BUMP(do_mtsdr1);
	return RVEC_NOP;
}

/* This function is _very_ slow, since it must destroy a lot of PTEs.
 * Fortunately, BAT-maps are normally static.
 */
asmlinkage int 
do_mtbat( kernel_vars_t *kv, int sprnum, ulong value, int force )
{
	mac_bat_t *d;
	int batnum;

	BUMP(do_mtbat);

	if( !force && MREGS.spr[sprnum] == value )
		return RVEC_NOP;

	/* printk("do_mtbat %d %08lX\n", sprnum, value); */

	MREGS.spr[sprnum] = value;

	/* upper bat register have an even number */
	batnum = (sprnum -S_IBAT0U) >>1;
	d = &MMU.bats[batnum];

	/* First we must make sure that all PTEs corresponding to 
	 * the old BAT-mapping are purged from the hash table.
	 */
	if( BAT_HACK(kv) && d->valid )
		unmap_ea_range(kv, d->base, d->size );
	d->valid = 0;

	if( MREGS.processor==1 ){
		ulong *p = &MREGS.spr[sprnum & ~1];
		/* 601 unified BATs. Note: P601_BAT is 
		 * not correctly defined in the headerfile <asm/mmu.h>  
		 */
		if(batnum>3)
			return RVEC_NOP;
		d->valid = d->vs = d->vp = p[1] & 1;
		d->wimg = (p[0] >> 3) & 0xe;
		d->ks = p[0] & BIT(28) ? 1:0;
		d->ku = p[0] & BIT(29) ? 1:0;
		d->pp = p[0] & 0x3;
		d->size = ((p[1] & 0x3f)+1) <<17;
		d->base = p[0] & (~(p[1] & 0x3f)<<17);
		d->mbase = p[1] & (~(p[1] & 0x3f)<<17);
	} else {
		/* IBAT/DBATs
		 * Currently, we implement IBAT/DBATs as unified BATs
		 */
		BAT *p = (BAT*)&MREGS.spr[sprnum & ~1];
		d->valid = p->batu.vs | p->batu.vp;
		d->vs = p->batu.vs;
		d->vp = p->batu.vp;
		d->wimg = (p->batl.w<<3) | (p->batl.i<<2) | (p->batl.m<<1) | p->batl.g;
		/* IBAT/DBATs, behaves as if key==1 */
		d->ks = d->ku = 1;
		d->pp = p->batl.pp;
		d->size = (p->batu.bl+1)<<17;
		d->base = (p->batu.bepi & ~p->batu.bl)<<17;
		d->mbase = (p->batl.brpn & ~p->batu.bl)<<17;
	}

	/* Next, we must make sure that no PTEs refer to the new
	 * BAT-mapped area.
	 */

	if( BAT_HACK(kv) && d->valid )
		unmap_ea_range( kv, d->base, d->size );

	return RVEC_NOP;
}

/*
 *  do_tlbie
 *    Translation Lookaside Buffer Invalidate Entry
 */
asmlinkage int 
do_tlbie( kernel_vars_t *kv, ulong ea ) 
{
	table_tlbie( kv, ea );

	BUMP(do_tlbie);
	return RVEC_NOP;
}

