/* NEStra - The NES translator */
/* written by Quor */
/* Version 0.66 of December 14, 1999 */

/* Public Domain */


#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <stdio.h>
#include <stdlib.h>

#include "mapper.h"
#include "io.h"
#include "globals.h"

u_char *ROM_BASE;
u_char *VROM_BASE;
unsigned int ROM_PAGES;
unsigned int VROM_PAGES;
unsigned int ROM_MASK;
unsigned int VROM_MASK;
unsigned int VROM_MASK_1k;
int MAPPERNUMBER;
extern void(*MapperInit[])();
extern void translate();
char filename[1024];

int verbose=0;
int disassemble=0;
int ignorebadinstr=0;
int staticcolors=0;
int dolink=0;
int mapmirror=0;
int oldxcode=0;
int irqflag=0;

int main(int argc,char **argv)
{
  int x;
  int r;
  int romfd;
  int size;
  int cmirror=0;

  /* Parse args */
  if (argc<2) {help();exit(1);}
  for (x=1;x<argc;x++)
  {
    if(strncmp(argv[x],"-v",3)==0) verbose=1;
    else if(strncmp(argv[x],"-d",2)==0) disassemble=1;
    else if(strncmp(argv[x],"-l",2)==0) dolink=1;
    else if(strncmp(argv[x],"-o",2)==0) oldxcode=1;
    else if(strncmp(argv[x],"-i",2)==0) ignorebadinstr=1;
    else if(strncmp(argv[x],"-s",2)==0) staticcolors=1;
    else if(strncmp(argv[x],"-hm",3)==0) cmirror=1;
    else if(strncmp(argv[x],"-vm",3)==0) cmirror=2;
    else if(strncmp(argv[x],"-sm",3)==0) cmirror=3;
    else if(strncmp(argv[x],"-nm",3)==0) cmirror=4;
    else if(strncmp(argv[x],"-mh",3)==0) cmirror=1;
    else if(strncmp(argv[x],"-mv",3)==0) cmirror=2;
    else if(strncmp(argv[x],"-ms",3)==0) cmirror=3;
    else if(strncmp(argv[x],"-mn",3)==0) cmirror=4;
    else {strncpy(filename,argv[x],1020);break;}
  }
  if(staticcolors&&oldxcode)
  {
    printf("Static colormap not supported in old X11 display code\n");
    staticcolors=0;
  }
  
  /* Open ROM file */
  romfd=open(filename,O_RDONLY);
  if(romfd<0) {fprintf(stderr,"Unable to open %s\n",filename);fflush(stderr);exit(1);}
  size=lseek(romfd,0,SEEK_END);lseek(romfd,0,SEEK_SET); /* Get file size */
  if(size<0) {fprintf(stderr,"Unable to read %s\n",filename);fflush(stderr);exit(1);}
  if(size==0) {fprintf(stderr,"Unable to read %s (empty file)\n",filename);fflush(stderr);exit(1);}
  r=(int)mmap(ROM,0x200000,PROT_READ,MAP_FIXED|MAP_PRIVATE,romfd,0);
  if(r<=0) {fprintf(stderr,"Unable to read %s\n",filename);fflush(stderr);exit(1);}

  /* Allocate memory */
  r=(int)mmap(RAM,0x8000,PROT_READ|PROT_WRITE,MAP_FIXED|MAP_PRIVATE|MAP_ANON,-1,0);
  if(r<=0) {fprintf(stderr,"Out of memory!\n");fflush(stderr);exit(1);}
  r=(int)mmap(CODE_BASE,0x800000,PROT_READ|PROT_WRITE|PROT_EXEC,MAP_FIXED|MAP_PRIVATE|MAP_ANON,-1,0);
  if(r<=0) {fprintf(stderr,"Out of memory!\n");fflush(stderr);exit(1);}
  r=(int)mmap((void *)INT_MAP,0x400000,PROT_READ|PROT_WRITE,MAP_FIXED|MAP_PRIVATE|MAP_ANON,-1,0);
  if(r<=0) {fprintf(stderr,"Out of memory!\n");fflush(stderr);exit(1);}

  /* Initialize ROM mapper */
  MAPTABLE[0]=RAM;
  MAPTABLE[1]=RAM;
  MAPTABLE[2]=RAM;
  MAPTABLE[4]=RAM;
  MAPTABLE[16]=RAM-65536; /* Wrap around accesses which index above FFFF */
  MAPTABLE[6]=MAPTABLE[7]=RAM; /* Save RAM */
  MAPTABLE[3]=RAM+0x2000;MAPTABLE[5]=RAM; /* 3xxx and 5xxx really shouldn't be mapped to anything, but they're mapped to some unused ram just to prevent pagefaults by weird code that tries to access these areas. */
  MAPPERNUMBER=-1;
  if(*((int *)ROM)==0x1a53454e) /* .NES */
  {
    ROM_BASE=ROM+16; /* 16 bytes for .nes header */
    /* check for a 512-byte trainer */
    if(ROM[6]&4) {
      if(verbose) printf("512-byte trainer present\n");
      memcpy(RAM+0x7000,ROM_BASE,512);
      ROM_BASE+=512;
    }
    VROM_BASE=ROM_BASE+(ROM[4]<<14);
    if(ROM[5]) memcpy(VRAM,VROM_BASE,8192);
    MAPPERNUMBER=ROM[6]>>4;
    ROM_PAGES=ROM[4];
    VROM_PAGES=ROM[5];
    hvmirror=(ROM[6]&1)^1;
    nomirror=ROM[6]&8;
  }
  else if(size==40960)
  {
    /* No header, assume raw image */
    ROM_BASE=ROM;
    MAPPERNUMBER=0;
    memcpy(VRAM,ROM+32768,8192);
    ROM_PAGES=2;
    VROM_PAGES=2;
    VROM_BASE=ROM+32768;
    nomirror=1;
    mapmirror=0;
  }
  else {fprintf(stderr,"Unrecognized ROM file format\n");fflush(stderr);exit(1);}
  if(MAPPERNUMBER>MAXMAPPER||((int *)MapperInit)[MAPPERNUMBER]==0)
    {fprintf(stderr,"Unknown Mapper: %2x\n",MAPPERNUMBER);fflush(stderr);exit(1);}
  if(verbose) printf("Mapper %d  Program ROM %dk  Video ROM %dk\n",MAPPERNUMBER,ROM_PAGES*16,VROM_PAGES*8);

  /* How many address bits are used */
  if(ROM_PAGES<=2) ROM_MASK=1;
  else if(ROM_PAGES<=4) ROM_MASK=3;
  else if(ROM_PAGES<=8) ROM_MASK=7;
  else if(ROM_PAGES<=16) ROM_MASK=15;
  else if(ROM_PAGES<=32) ROM_MASK=31;
  else ROM_MASK=63; /* I don't think there are any games with >1MB prg-rom */

  if(VROM_PAGES<=2) VROM_MASK=1;
  else if(VROM_PAGES<=4) VROM_MASK=3;
  else if(VROM_PAGES<=8) VROM_MASK=7;
  else if(VROM_PAGES<=16) VROM_MASK=15;
  else if(VROM_PAGES<=32) VROM_MASK=31;
  else if(VROM_PAGES<=64) VROM_MASK=63;
  else VROM_MASK=127; /* I don't think there are any games with >1MB vrom */

  VROM_MASK_1k=(VROM_MASK<<3)|7; /* VROM mask for 1k-sized pages */

  MapperInit[MAPPERNUMBER]();

  if(cmirror==1&&mapmirror==0)
  {
    if(verbose&&!hvmirror) printf("Overriding default vertical mirroring.\n");
    hvmirror=1;
  }
  if(cmirror==2&&mapmirror==0)
  {
    if(verbose&&hvmirror) printf("Overriding default horizontal mirroring.\n");
    hvmirror=0;
  }
  if(cmirror==3&&mapmirror==0)
  {
    if(verbose&&!nomirror&&!osmirror) printf("Overriding default mirroring.\n");
    osmirror=1;
  }
  if(cmirror==4&&mapmirror==0)
  {
    if(verbose&&!nomirror) printf("Overriding default mirroring.\n");
    nomirror=1;
  }
  if(verbose&&mapmirror==0)
  {
    if(nomirror) printf("Using no mirroring.\n");
    else if(osmirror) printf("Using one-screen mirroring.\n");
    else if(hvmirror) printf("Using horizontal mirroring\n");
    else printf("Using vertical mirroring\n");
  }

  restoresavedgame();

  /* Initialize graphic display */
  loadpal();
  InitDisplay();

  /* start the show */
  STACKPTR=(int)STACK+0xFF;
  translate(*((unsigned short *)(MAPTABLE[15]+0xFFFC)));
  START(); /* execute translated code */

  /* Not Reached */
  return 0;
}

char *homedir;
char savefile[1024];

restoresavedgame()
{
  int fd,result,x;
  char buffer[1024];
  struct stat statbuf;
  *savefile=0;
  if(homedir=getenv("HOME"))
  {
    /*printf("%s\n",homedir);*/
    strncpy(buffer,homedir,1010);
    if(*buffer!=0&&buffer[strlen(buffer)-1]=='/') buffer[strlen(buffer)-1]=0;
    strcat(buffer,"/.nestra");
    result=stat(buffer,&statbuf);
    /*printf("%s = %d \n",buffer,result);*/
    if(result==0)
    {
      for(x=strlen(filename)-1;x>0&&filename[x-1]!='/';x--);
      strcat(buffer,"/");
      strncat(buffer,filename+x,1019-strlen(buffer));
      strcat(buffer,".sav");
      strcpy(savefile,buffer);
    }
  }
  if(!*savefile)
  {
    strncpy(buffer,filename,1013);savefile[1013]=0;
    for(x=strlen(buffer);x>=0&&buffer[x]!='/';x--) buffer[x]=0;
    strcat(buffer,"nestra.tmp");
    /*printf("%s\n",buffer);*/
    result=open(buffer,O_CREAT|O_RDWR,0666);unlink(buffer);close(result);
    if(result<0)
    {
      buffer[strlen(buffer)-10]=0;
      printf("Warning: can not write to directory %s\n",buffer);
      printf("Create directory ~/.nestra if you want to save games there instead.\n");
    }
    strncpy(savefile,filename,1019);savefile[1019]=0;
    strcat(savefile,".sav");
  }
  /*printf("%s\n",savefile);*/
  if((fd=open(savefile,O_RDWR,0666))>=0)
  {
    read(fd,RAM+0x6000,8192);
    close(fd);
  }
  else
  {
    /* In this special case, there is a ~/.nestra directory but no save
       file in it.  If there is a save file in the romfile directory,
       use that one. */
    strncpy(buffer,filename,1019);buffer[1019]=0;
    strcat(buffer,".sav");
    if((fd=open(buffer,O_RDWR,0666))>=0)
    {
      read(fd,RAM+0x6000,8192);
      close(fd);
      strcpy(savefile,buffer);
    }
    else
    {
      /* There is a ~/.nestra directory with no save file in it; If there
         is a save file in the romfile directory, but it is not writable,
         copy it into the ~/.nestra directory. */
      if((fd=open(buffer,O_RDONLY,0666))>=0)
      {
        read(fd,RAM+0x6000,8192);
        close(fd);
      }
      else
      {
        /* Finally, if no save file is found either place, look for a
           iNES-type save file. */
        strncpy(buffer,filename,1023);buffer[1023]=0;
        buffer[strlen(buffer)-4]=0;
        strcat(buffer,".sav");
        /*printf("%s\n",buffer);*/
        if((fd=open(buffer,O_RDWR,0666))>=0)
        {
          read(fd,RAM+0x6000,8192);
          close(fd);
          strcpy(savefile,buffer);
        }
      }
    }
  }  
}  

void quit()
{
  int fd;
  int x;
  /* Save ram */
  for(x=0;((long long *)(RAM+0x6000))[x]==0;x++) 
    if(x>=1023) exit(0); /* Nothing to save */
  fd=open(savefile,O_CREAT|O_RDWR,0666);
  if(fd>0)
  {
    write(fd,RAM+0x6000,8192);
    close(fd);
    exit(0);
  }
  fprintf(stderr,"Could not write to save file %s\n",savefile);
  exit(1);
}  

/* Load a NESticle type palette */
/* Thanks to Benjamin Sittler */
loadpal()
{
  char buffer[1024]; /* dual-use region! */
  int pen;
  int len;
  int fd;
  strncpy(buffer,filename,1020);buffer[1020]=0;
  if ((len=strlen(buffer))>4)
    if(buffer[len-4]=='.') buffer[len-4]=0;
  strcat(buffer,".pal");
  if ((fd=open(buffer,O_RDONLY)) < 0)
    if ((fd=open("nestra.pal",O_RDONLY)) < 0) {
      buffer[1012]=0; /* to prevent buffer overflows */
      for (len=strlen(buffer)-1;len&&(buffer[len]!='/');len--);
      buffer[len]='/';
      buffer[len+1]='\0';
      strcat(buffer,"nestra.pal");
      if ((fd=open(buffer,O_RDONLY)) < 0) {
        if(verbose) printf("Using built-in palette\n");
        return;
      }
    } else {
      strcpy(buffer,"nestra.pal");
    }
  if (verbose) printf("Reading palette from %s\n", buffer);
  /* clobber the filename with the file contents */
  read(fd,buffer,768);
  close(fd);
  /* convert the palette */
  for(pen=0;pen<64;pen++)
    palette_24[pen]=((unsigned char)buffer[pen*3])*0x10000
      + ((unsigned char)buffer[pen*3+1])*0x100
      + ((unsigned char)buffer[pen*3+2]);
}  

help()
{
  fprintf(stderr,"Nestra v0.66\n\n");
  fprintf(stderr,"Usage: nestra [options] filename\n");
  fprintf(stderr,"  -v Verbose output\n");
  fprintf(stderr,"  -d Disassemble\n");
  fprintf(stderr,"  -l Link branches optimization (may improve speed)\n");
  fprintf(stderr,"  -i Ignore unknown instructions\n");
  fprintf(stderr,"  -s Use static colormap (required for some games on 8-bit displays)\n");
  fprintf(stderr,"  -o Use old x11 display subroutines\n");
  fprintf(stderr,"  -mh Use horizontal mirroring\n");
  fprintf(stderr,"  -mv Use vertical mirroring\n");
  fprintf(stderr,"  -ms Use single-screen mirroring\n");
  fprintf(stderr,"  -mn Use no mirroring\n");
  fflush(stderr);
}
