/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.cnd.highlight.semantic.ifdef;

import java.awt.Color;
import java.lang.ref.WeakReference;
import java.util.Collections;
import java.util.List;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.AttributeSet;
import javax.swing.text.Document;
import javax.swing.text.StyleConstants;
import org.netbeans.api.editor.mimelookup.MimeLookup;
import org.netbeans.api.editor.mimelookup.MimePath;
import org.netbeans.api.editor.settings.AttributesUtilities;
import org.netbeans.api.editor.settings.EditorStyleConstants;
import org.netbeans.api.editor.settings.FontColorSettings;
import org.netbeans.editor.BaseDocument;
import org.netbeans.modules.cnd.api.model.CsmFile;
import org.netbeans.modules.cnd.api.model.CsmOffsetable;
import org.netbeans.modules.cnd.api.model.services.CsmFileInfoQuery;
import org.netbeans.modules.cnd.modelutil.CsmUtilities;
import org.netbeans.spi.editor.highlighting.support.OffsetsBag;
import org.openide.util.Lookup;
import org.openide.util.LookupEvent;
import org.openide.util.LookupListener;
import org.openide.util.RequestProcessor;
import org.openide.util.WeakListeners;

/**
 * Semantic C/C++ code highlighter responsible for "graying out"
 * inactive code due to preprocessor definitions
 *
 * @author Sergey Grinev
 */
public class InactiveCodeHighlighter implements DocumentListener, Runnable, LookupListener {

    private final OffsetsBag bag;
    private final WeakReference<BaseDocument> weakDoc;
    private final static String TOKEN = "cc-highlighting-inactive"; // NOI18N
    private final String mime; 
    private AttributeSet colors;
    private final AttributeSet cleanUp = AttributesUtilities.createImmutable(
                                            StyleConstants.Underline, null,
                                            StyleConstants.StrikeThrough, null,
                                            StyleConstants.Background, null,
                                            EditorStyleConstants.WaveUnderlineColor, null
                                        );


    public InactiveCodeHighlighter(Document doc) {
        mime = (String)doc.getProperty("mimeType"); //NOI18N
        bag = new OffsetsBag(doc);

        if (doc instanceof BaseDocument) {
            weakDoc = new WeakReference<BaseDocument>((BaseDocument)doc);
            ((BaseDocument) doc).addDocumentListener(this);

            Lookup lookup = MimeLookup.getLookup(MimePath.get(mime));
            Lookup.Result<FontColorSettings> result = 
                    lookup.lookup(new Lookup.Template<FontColorSettings>(FontColorSettings.class));
            result.addLookupListener(WeakListeners.create(LookupListener.class, this, null));
            result.allInstances();
            resultChanged(null);
        } else {
            weakDoc = null;
        }
    }
    
    // LookupListener
    public void resultChanged(LookupEvent ev) {
        initFontColors();
        scheduleUpdate();
    }
    
    private void initFontColors() {
        Lookup lookup = MimeLookup.getLookup(MimePath.get(mime));
        FontColorSettings fcs = lookup.lookup(FontColorSettings.class);
        colors = AttributesUtilities.createComposite(fcs.getTokenFontColors(TOKEN), cleanUp);
    }
    
    public void insertUpdate(DocumentEvent e) {
        scheduleUpdate();
    }

    public void removeUpdate(DocumentEvent e) {
        scheduleUpdate();
    }

    public void changedUpdate(DocumentEvent e) {
    }
    
    private static final AttributeSet macroColors = AttributesUtilities.createImmutable(StyleConstants.Foreground, new Color(0, 105, 0));
    private static final boolean USE_MORE_SEMANTIC = Boolean.getBoolean("cnd.semantic.advanced"); // NOI18N
    
    private void update() {
        BaseDocument doc = weakDoc.get();
        if (doc != null) {
            OffsetsBag newBag = new OffsetsBag(doc);
            newBag.clear();
            for (CsmOffsetable block : getInactiveCodeBlocks(doc)) {
                newBag.addHighlight(block.getStartOffset(), block.getEndOffset(), colors);
            }
            
            if (USE_MORE_SEMANTIC) {
                for (CsmOffsetable block : getMacroBlocks(doc)) {
                    newBag.addHighlight(block.getStartOffset(), block.getEndOffset(), macroColors);
                }
            }
            
            getHighlightsBag().setHighlights(newBag);
        }
    }

    private static List<CsmOffsetable> getInactiveCodeBlocks(BaseDocument doc) {
        List<CsmOffsetable> out = Collections.<CsmOffsetable>emptyList();
        CsmFile file = CsmUtilities.getCsmFile(doc, false);
        if (file != null) {
            out = getInactiveCodeBlocks(file);
        }
        return out;
    }

    private static List<CsmOffsetable> getMacroBlocks(BaseDocument doc) {
        List<CsmOffsetable> out = Collections.<CsmOffsetable>emptyList();
        CsmFile file = CsmUtilities.getCsmFile(doc, false);
        if (file != null) {
            out = getMacroBlocks(file);
        }
        return out;
    }
 
    /*package*/ static List<CsmOffsetable> getInactiveCodeBlocks(CsmFile file) {
        return CsmFileInfoQuery.getDefault().getUnusedCodeBlocks(file);
    }
 
    /*package*/ static List<CsmOffsetable> getMacroBlocks(CsmFile file) {
        return CsmFileInfoQuery.getDefault().getMacroUsages(file);
    }
 
    public OffsetsBag getHighlightsBag() {
        return bag;
    }

    private RequestProcessor.Task task = null;
    private final static int DELAY = 1001;
    
    public void scheduleUpdate() {
        // TODO: validation: switch to PositionsBag and
        // do nothing if lines with sharps are edited (at least for insertUpdates)

        if (task==null) {
            task = RequestProcessor.getDefault().create(this, true);
            task.setPriority(Thread.MIN_PRIORITY);
        }
        task.cancel();
        task.schedule(DELAY);
    }

    // Runnable
    public void run() {
        try {
            update();
        } catch (AssertionError ex) {
            ex.printStackTrace();
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }
}