/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.sql.framework.highlighter;

import java.io.Reader;

import javax.swing.text.AbstractDocument;
import javax.swing.text.BadLocationException;

/**
 * A reader interface for an abstract sqlDocument. Since the syntax highlighting packages
 * only accept Stings and Readers, this must be used. Since the close() method does
 * nothing and a seek() method has been added, this allows us to get some performance
 * improvements through reuse. It can be used even after the lexer explicitly closes it by
 * seeking to the place that we want to read next, and reseting the lexer.
 * 
 * @author Ahimanikya Satapathy
 * @version $Revision: 1.2 $
 */

class DocumentReader extends Reader {

    /**
     * The sqlDocument that we are working with.
     */
    private AbstractDocument document;

    /**
     * Saved position used in the mark and reset methods.
     */
    private long mark = -1;

    /**
     * Current position in the sqlDocument. Incremented whenever a character is read.
     */
    private long position = 0;

    /**
     * Construct a reader on the given sqlDocument.
     * 
     * @param sqlDocument the sqlDocument to be read.
     */
    public DocumentReader(AbstractDocument document) {
        this.document = document;
    }

    /**
     * Has no effect. This reader can be used even after it has been closed.
     */
    public void close() {
    }

    /**
     * Save a position for reset.
     * 
     * @param readAheadLimit ignored.
     */
    public void mark(int readAheadLimit) {
        mark = position;
    }

    /**
     * This reader support mark and reset.
     * 
     * @return true
     */
    public boolean markSupported() {
        return true;
    }

    /**
     * Read a single character.
     * 
     * @return the character or -1 if the end of the sqlDocument has been reached.
     */
    public int read() {
        if (position < document.getLength()) {
            try {
                char c = document.getText((int) position, 1).charAt(0);
                position++;
                return c;
            } catch (BadLocationException x) {
                return -1;
            }
        }
        return -1;
    }

    /**
     * Read and fill the buffer. This method will always fill the buffer unless the end of
     * the sqlDocument is reached.
     * 
     * @param cbuf the buffer to fill.
     * @return the number of characters read or -1 if no more characters are available in
     *         the sqlDocument.
     */
    public int read(char[] cbuf) {
        return read(cbuf, 0, cbuf.length);
    }

    /**
     * Read and fill the buffer. This method will always fill the buffer unless the end of
     * the sqlDocument is reached.
     * 
     * @param cbuf the buffer to fill.
     * @param off offset into the buffer to begin the fill.
     * @param len maximum number of characters to put in the buffer.
     * @return the number of characters read or -1 if no more characters are available in
     *         the sqlDocument.
     */
    public int read(char[] cbuf, int off, int len) {
        if (position < document.getLength()) {
            int length = len;
            if (position + length >= document.getLength()) {
                length = document.getLength() - (int) position;
            }
            if (off + length >= cbuf.length) {
                length = cbuf.length - off;
            }
            try {
                String s = document.getText((int) position, length);
                position += length;
                for (int i = 0; i < length; i++) {
                    cbuf[off + i] = s.charAt(i);
                }
                return length;
            } catch (BadLocationException x) {
                return -1;
            }
        }
        return -1;
    }

    /**
     * @return true
     */
    public boolean ready() {
        return true;
    }

    /**
     * Reset this reader to the last mark, or the beginning of the sqlDocument if a mark
     * has not been set.
     */
    public void reset() {
        if (mark == -1) {
            position = 0;
        } else {
            position = mark;
        }
        mark = -1;
    }

    /**
     * Seek to the given position in the sqlDocument.
     * 
     * @param n the offset to which to seek.
     */
    public void seek(long n) {
        if (n <= document.getLength()) {
            position = n;
        } else {
            position = document.getLength();
        }
    }

    /**
     * Skip characters of input. This method will always skip the maximum number of
     * characters unless the end of the file is reached.
     * 
     * @param n number of characters to skip.
     * @return the actual number of characters skipped.
     */
    public long skip(long n) {
        if (position + n <= document.getLength()) {
            position += n;
            return n;
        }
        long oldPos = position;
        position = document.getLength();
        return (document.getLength() - oldPos);
    }

    /**
     * Modifying the sqlDocument while the reader is working is like pulling the rug out
     * from under the reader. Alerting the reader with this method (in a nice thread safe
     * way, this should not be called at the same time as a read) allows the reader to
     * compensate.
     */
    public void update(int pos, int adjustment) {
        if (pos < this.position) {
            if (this.position < pos - adjustment) {
                this.position = pos;
            } else {
                this.position += adjustment;
            }
        }
    }
}
