/*
 * The contents of this file are subject to the terms of the Common
 * Development and Distribution License (the License). You may not use this
 * file except in compliance with the License.  You can obtain a copy of the
 * License at http://www.netbeans.org/cddl.html

 * When distributing Covered Code, include this CDDL Header Notice in each
 * file and include the License. If applicable, add the following below the
 * CDDL Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"

 * Copyright 2006 Sun Microsystems, Inc. All Rights Reserved
 *
 */

package org.netbeans.modules.aspect.editor.model;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * This class represents the aspect model.
 * 
 * @author karthikeyan s
 */
public class Aspect {

	private Element root;

	private boolean isDirty = false;

	private String exchangeType;

	private String id;

	private Input input;

	private List<Output> outputs;

	private List<Advice> advices;

	private Map<Integer, Advice> adviceMap;

	/** Creates a new instance of Aspect */
	public Aspect(Element aspectElement) {
		root = aspectElement;
		outputs = new ArrayList<Output>();
		advices = new ArrayList<Advice>();
		adviceMap = new HashMap<Integer, Advice>();
		parse();
	}

	/** Creates a new instance of Aspect */
	public Aspect(String type, String aspId) {
		exchangeType = type;
		id = aspId;
		outputs = new ArrayList<Output>();
		advices = new ArrayList<Advice>();
		input = new Input("{http://com.sun.jbi/aspect/aspect}aspectPartner",
				"aspectPortTypeRole",
				"{http://com.sun.jbi/aspect/aspect}aspectPortType",
				"aspectOperation", "{http://com.sun.jbi/aspect/aspect}inputMsg");

		Output o = new Output(null, null, "partnerLink", "roleName",
				"portType", "operation", "messageType", "1");
		outputs.add(o);
	}

	public Input getInput() {
		return input;
	}

	public List<Output> getOutputs() {
		return outputs;
	}

	public List<Advice> getAdvices() {
		return advices;
	}

	public void setAdvices(List<Advice> adviceList) {
		advices = adviceList;
	}

	public void addAdvice(Advice adv) {
		advices.add(adv);
	}

	public void replace(Advice adv) {
		advices.set(advices.indexOf(adv), adv);
	}

	public void removeAdvice(Advice adv) {
		// decrement orderno for other elements.
		boolean startDecrement = false;
		Iterator it = advices.iterator();
		while (it.hasNext()) {
			Advice advObj = (Advice) it.next();
			if (adv.getOrderNo().equals(advObj.getOrderNo())) {
				startDecrement = true;
				continue;
			}
			if (startDecrement) {
				advObj.setOrderNo(String.valueOf(Integer.parseInt(advObj
						.getOrderNo()) - 1));
			}
		}
		advices.remove(adv);
	}

	public void clearAdvices() {
		advices.clear();
	}

	public void setInput(Input in) {
		input = in;
	}

	public void setOutputs(List<Output> outs) {
		outputs = outs;
	}

	public void addOutput(Output out) {
		Iterator it = outputs.iterator();
		boolean exists = false;
		int i = -1;
		while (it.hasNext()) {
			Output output = (Output) it.next();
			i++;
			if (output.getId().equals(out.getId())) {
				exists = true;
				break;
			}
		}
		if (exists) {
			outputs.set(i, out);
		} else {
			outputs.add(out);
		}
	}

	public void clearOutput() {
		outputs.clear();
	}

	public void removeOutput(Output out) {
		outputs.remove(out);
	}

	public void setExchangeType(String type) {
		exchangeType = type;
	}

	public String getExchangeType() {
		return exchangeType;
	}

	public void setID(String aspID) {
		id = aspID;
	}

	public String getID() {
		return id;
	}

	public List<Object> getChildren() {
		List<Object> list = new ArrayList<Object>();
		list.add(input);
		list.addAll(Util.getSortedAdviceList(advices));
		list.addAll(outputs);
		return list;
	}

	private void parse() {
		exchangeType = root.getAttribute(AspectMap.EXCHANGE_TYPE);
		id = root.getAttribute(AspectMap.ID_ATTR);
		NodeList inputElements = root
				.getElementsByTagName(AspectMap.INPUT_ELEMENT);
		parseInput(inputElements);
		NodeList outputElements = root
				.getElementsByTagName(AspectMap.OUTPUT_ELEMENT);
		parseOutput(outputElements);
		NodeList adviceElements = root.getElementsByTagName(AspectMap.ADVICE);
		parseAdvice(adviceElements);
	}

	private void parseInput(NodeList inputElements) {
		// expecting only one input for a aspect tag.
		Element inputElem = (Element) inputElements.item(0);
		input = new Input(inputElem.getAttribute(AspectMap.PARTNERLINK_ATTR),
				inputElem.getAttribute(AspectMap.ROLE_NAME_ATTR), inputElem
						.getAttribute(AspectMap.PORTTYPE_ATTR), inputElem
						.getAttribute(AspectMap.OPERATION_ATTR), inputElem
						.getAttribute(AspectMap.MESSAGE_TYPE_ATTR));
	}

	private void parseOutput(NodeList outputElements) {
		for (int i = 0; i < outputElements.getLength(); i++) {
			Element outputElem = (Element) outputElements.item(i);
			Output output = new Output(outputElem
					.getAttribute(AspectMap.SERVICE_NAME_ATTR), outputElem
					.getAttribute(AspectMap.PORT_NAME_ATTR), outputElem
					.getAttribute(AspectMap.PARTNERLINK_ATTR), outputElem
					.getAttribute(AspectMap.ROLE_NAME_ATTR), outputElem
					.getAttribute(AspectMap.PORTTYPE_ATTR), outputElem
					.getAttribute(AspectMap.OPERATION_ATTR), outputElem
					.getAttribute(AspectMap.MESSAGE_TYPE_ATTR), outputElem
					.getAttribute(AspectMap.ID_ATTR));
			outputs.add(output);
		}
	}

	private void parseAdvice(NodeList adviceElements) {
		for (int i = 0; i < adviceElements.getLength(); i++) {
			Element adviceElem = (Element) adviceElements.item(i);
			Advice advice = new Advice(adviceElem);
			advices.add(advice);
		}
	}

	public void setDirty(boolean dirty) {
		isDirty = dirty;
	}

	public boolean isDirty() {
		return isDirty;
	}

	public String toXMLString() {
		StringBuffer xmlString = new StringBuffer();
		xmlString.append("<" + AspectMap.ASPECT + " " + AspectMap.EXCHANGE_TYPE
				+ "=\"" + exchangeType + "\" " + AspectMap.ID_ATTR + "=\"" + id
				+ "\">\n");
		xmlString.append(input.toXmlString() + "\n");
		Iterator it = outputs.iterator();
		while (it.hasNext()) {
			Output out = (Output) it.next();
			xmlString.append(out.toXmlString() + "\n");
		}
		it = Util.getSortedAdviceList(advices).iterator();
		while (it.hasNext()) {
			Advice adv = (Advice) it.next();
			xmlString.append(adv.toXmlString() + "\n");
		}
		xmlString.append("</" + AspectMap.ASPECT + ">");
		return xmlString.toString();
	}

	public Output getCurrentOutput() {
		// TODO change the impl as required
		return outputs.get(0);
	}

	public void setCurrentOutput(Output o) {
		// TODO change the impl as required
		outputs.clear();
		outputs.add(0, o);
	}
}