// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/base/compile_IA32.cpp,v 1.13 2001/12/19 07:17:20 gwu2 Exp $
//



#include "platform.h"
#include <iostream.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

#include "orp_types.h"
#include "Class.h"
#include "environment.h"
#include "method_lookup.h"
#include "stack_manipulation.h"
#include "exceptions.h"
#include "jit_intf.h"
#include "jit_intf_cpp.h"
#include "jit_runtime_support.h"

#include "../x86/x86.h"

#include "root_set_enum.h"
#include "object_layout.h"
#include "nogc.h"

#ifndef OBJECT_LOCK_V2
#include "find_natives.h"
#else
#include "find_natives_olv2.h"
#endif
 
#include "gc_for_orp.h"
 
#include "orp_utils.h"
#include "orp_synch.h"
#include "orp_threads.h"
#include "ini.h"
#include "orp_vtune.h"
#include "orp_stats.h"

#include "compile.h"

#include "jvmdi_clean.h"
#include "debugger_jvmdi_ia32.h"

#include "level_1a_jit_intf.h"

extern JIT *o1_jit;

#ifdef DUMP_JIT
#include "dump_jit.h"
#include "disasm_intf.h"
#endif


void init_object_handles_for_args(Method          *method,
                                  J2N_Saved_State *ljf,
                                  uint32          *repushed_args
                                  );

void free_local_object_handles(Object_Handle head,
                               Object_Handle tail
                               );




#ifdef DUMP_JIT

void dump_stub(Method_Handle method,
               char *x86_code,
               unsigned x86_code_length)
{
    Class_Handle ch = method_get_class(method);
    FILE *_f = acquire_dump_jit_file(ch);
    fprintf(_f, "Native method %s.%s%s\n",
        class_get_name(ch), method_get_name(method), method_get_descriptor(method));

#ifdef SHORT_DUMP_JIT
    fprintf(_f, "Generated code: %X .. %X\n\n",
            x86_code,
            x86_code + x86_code_length);
    LeaveCriticalSection(&dump_CriticalSection);
    return;
#endif

    char *bytestream = x86_code;
    char buf[800];

    char *x86_end   = bytestream + x86_code_length;
    while(bytestream < x86_end) {
        bytestream = x86_disasm(bytestream, buf, true, true);
        fprintf(_f, "\t%s\n", buf);
    }

    fprintf(_f, "End of Method %s.%s%s\n\n",
        class_get_name(ch), method_get_name(method), method_get_descriptor(method));

    release_dump_jit_file();
} //dump_stub

#endif

void * getaddress__free_local_object_handles_naked()
{
    static void *addr = 0;
    if (addr) {
        return addr;
    }

    const int stub_size = 1024;
    char *stub = (char *)gc_malloc_fixed_code_for_class_loading(stub_size);
#ifdef _DEBUG
    memset(stub, 0xcc /*int 3*/, stub_size);
#endif
    char *ss = stub;

    ss = push(ss, &eax_opnd);
    ss = push(ss, &edx_opnd);

    ss = push(ss, &edx_opnd);
    ss = push(ss, &ecx_opnd);

    ss = call(ss, (char *)free_local_object_handles);
    ss = alu(ss, add_opc, &esp_opnd, &Imm_Opnd(8));
    
    ss = pop(ss, &edx_opnd);
    ss = pop(ss, &eax_opnd);
    ss = ret(ss);

    addr = stub;
#ifdef ORP_VTUNE_SUPPORT
    //M: 
    vtune_notify_stub_load_finished("getaddress__free_local_object_handles_naked",(Byte*) stub,ss-stub);
#endif
    return addr;
} //getaddress__free_local_object_handles_naked


JIT_Result prepare_native_method(Method &method,
                                 JIT_Flags flags
                                 )
{
#ifdef ORP_STATS
    orp_stats_total.num_native_methods++;
#endif
    assert(method.is_native() || use_native_implementation(&method) );

    void *func;
    NI_TYPE native_intf;
    find_native_method(&method, &func, &native_intf);

    if(!func) {
        return JIT_FAILURE;
    }

    bool use_jni = false;
    switch(native_intf) {
    case NI_IS_JNI:
        use_jni = true;
        break;
    case NI_IS_RNI:
        break;
    case NI_IS_DIRECT:
        method.set_code_addr(func);
        return JIT_SUCCESS;
    default:
        assert(0);
        break;
    }
    unsigned size_of_object_handles_for_args =
        method.get_num_ref_args() * sizeof(Object_Handle_Struct);
    if(method.is_static()) {
        // For the extra jclass argument.
        size_of_object_handles_for_args += sizeof(Object_Handle_Struct);
    }

    Arg_List_Iterator iter = method.get_argument_list();
    unsigned num_arg_words;
    if(method.is_static()) {
        num_arg_words = 0;
    } else {
        num_arg_words = 1;
    }
    Java_Type typ;
    while((typ = curr_arg(iter)) != JAVA_TYPE_END) {
        switch(typ) {
        case JAVA_TYPE_LONG:
        case JAVA_TYPE_DOUBLE:
            num_arg_words += 2;
            break;
        default:
            num_arg_words += 1;
            break;
        }
        iter = advance_arg_iterator(iter);
    }

    Java_Type ret_type = method.get_return_java_type();

    // Calculate the length of the stub.
    unsigned stub_length = 31 + 4 * num_arg_words;
    if(num_arg_words)
        stub_length += 2;  // sizeof(ret n) - sizeof(ret)
    if(ret_type != JAVA_TYPE_VOID) {
        // to save the return address before setting ljf
        if(ret_type == JAVA_TYPE_LONG) {
            stub_length += 6;
        } else {
            stub_length += 4;
        }
    }
    if(method.is_static()) {
        // to push the class pointer
        stub_length += 5;
    }

    // Adjust for JNI
    //  5: push 0
    //  ?: mov esi, esp
    //  ?: mov esp, esi
    stub_length += 130 + 4 * num_arg_words;

    // We allow some extra length just in case our calculation of
    // stub_length is incorrect (the debug version has an assert after
    // the code has been generated to ensure that stub_length is correct).
    unsigned stub_size = stub_length + 10;
    char *stub = (char *)method.allocate_code_block(stub_size, 0);
#ifdef ORP_VTUNE_SUPPORT
    memset(stub, 0x90, stub_size);   // nop
#endif
    char *s = stub;
    //
    // push ebp
    // push ebx
    // push esi
    // push edi
    //
    // push 0           ; Local references list for JNI
    //
    //                  ; We have to call a function to get the address of the
    //                  ; thread-local version of orp_last_java_frame.
    //
    // if fastcall and ecx used
    //    push ecx
    // if fastcall and edx used
    //    push edx
    //
    // call get_addr_of_orp_last_java_frame
    //
    // if fastcall and edx used
    //    pop edx
    // if fastcall and ecx used
    //    pop ecx
    //
    // push eax         ; Push the address of orp_last_java_frame.
    // push [eax]       ; Push orp_last_java_frame.
    // move [eax], esp  ; Current value of esp will become the new value of ljf.
    //
    // re-push arguments
    // [for static methods, push the class pointer]
    //
    // call func        ; assume stdcall
    //
    // pop ecx          ; Pop orp_last_java_frame.
    // pop ebx          ; Pop the address of orp_last_java_frame.
    // mov [ebx], ecx   ; Restore the old value of orp_last_java_frame.
    //
    // add esp, 4       ; Local references list for JNI
    //
    // pop edi          ; The native function restores those anyway, but we have
    // pop esi          ; to re-do the work in case moving GC happened while we
    // pop ebx          ; were in native code.
    // pop ebp
    //
    // ret n            ; n is computed from the descriptor

#ifdef ORP_STATS
    // Increment only four bytes even though the counter is 8-byte long
    s = inc(s, &M_Opnd((unsigned)&(method.num_accesses)));
#endif

    s = push(s, &ebp_opnd);
    s = push(s, &ebx_opnd);
    s = push(s, &esi_opnd);
    s = push(s, &edi_opnd);

    // Local references list for JNI.
    s = push(s, &Imm_Opnd(0));

    unsigned arg_offset = (sizeof(J2N_Saved_State) - 8) + 4 * num_arg_words;
    if(use_jni) {
        arg_offset += size_of_object_handles_for_args;
    }

    // There are at most 64K arguments, so uint16 almost suffices.
    // Almost, because we don't take into account callee-saved registers.
    // Push offsets are stored "compressed", i.e., without 2 least
    // significant bits
    uint16 *push_offsets = (uint16 *)(stub + stub_size);
    uint16 *curr_push_offset = push_offsets;
    iter = method.get_argument_list();
    unsigned first_arg_offset = arg_offset -4 + 4 * num_arg_words;
    unsigned curr_arg_offset = first_arg_offset;
    unsigned fastcall_extra_offset = 0;
    uint16 *fastcall_ecx_push_offset = 0;
    uint16 *fastcall_edx_push_offset = 0;

    if(!method.is_static()) {
        *--curr_push_offset = (first_arg_offset - curr_arg_offset) >> 2;
        curr_arg_offset -= 8;
        fastcall_ecx_push_offset = curr_push_offset;
    }

    while((typ = curr_arg(iter)) != JAVA_TYPE_END) {
        switch(typ) {
        case JAVA_TYPE_LONG:
        case JAVA_TYPE_DOUBLE:
            *--curr_push_offset =
                ((first_arg_offset - curr_arg_offset) >> 2) + 1;
            *--curr_push_offset =
                ((first_arg_offset - curr_arg_offset) >> 2) + 1;
            curr_arg_offset -= 16;
            break;
        default:
            *--curr_push_offset = (first_arg_offset - curr_arg_offset) >> 2;
            curr_arg_offset -= 8;
            if(typ != JAVA_TYPE_FLOAT) {
                fastcall_edx_push_offset = fastcall_ecx_push_offset;
                fastcall_ecx_push_offset = curr_push_offset;
            }
            break;
        }
        iter = advance_arg_iterator(iter);
    }

    s = call(s, (char *)get_addr_of_orp_last_java_frame);

    s = push(s, &eax_opnd);
    s = push(s, &M_Base_Opnd(eax_reg, 0));
    s = mov(s, &M_Base_Opnd(eax_reg, 0), &esp_opnd);

    if(use_jni) {
        if(size_of_object_handles_for_args) {
            s = mov(s, &esi_opnd, &esp_opnd);
            s = alu(s, sub_opc, &esp_opnd, &Imm_Opnd(size_of_object_handles_for_args));
        }
    }

    while(curr_push_offset < push_offsets) {
        s = push(s,
            &M_Base_Opnd(esp_reg,
            4 +
            first_arg_offset - (*(curr_push_offset) << 2)));
        curr_push_offset++;
    }

    if(method.is_static()) {
        s = push(s, &Imm_Opnd((uint32)method.get_class()));
    }

    if(use_jni) {
        if(method.is_synchronized()) {
            // Save the this pointer: move the address of actual Java object to ebp
            s = mov(s, &ebp_opnd, &M_Base_Opnd(esp_reg, 0));
        }

        if(size_of_object_handles_for_args) {
            s = push(s, &esp_opnd);
            s = push(s, &esi_opnd);
            s = push(s, &Imm_Opnd((uint32)&method));
            s = call(s, (char *)init_object_handles_for_args);
            s = alu(s, add_opc, &esp_opnd, &Imm_Opnd(12));
        }

        if(method.is_synchronized()) {
            if(method.is_static()) {
                s = push(s, &Imm_Opnd((uint32)method.get_class()));
                s = call(s, (char *)orp_get_rt_support_addr(ORP_RT_MONITOR_ENTER_STATIC));
            } else {
                s = push(s, &ebp_opnd);
                s = call(s, (char *)orp_get_rt_support_addr(ORP_RT_MONITOR_ENTER));
                // Save the this pointer: move its JNI handle to ebp
                s = mov(s, &ebp_opnd, &M_Base_Opnd(esp_reg, 0));
            }
        }

        s = push(s, &Imm_Opnd((int)jni_native_intf));   // JNIEnv
        s = call(s, (char *)orp_enable_gc);
    }

    s = call(s, (char *)func);


#ifdef ORP_POSIX

    if(1) {
        unsigned num_arg_bytes = method.get_num_arg_bytes();
        if (use_jni) 
            num_arg_bytes += 4;

#else
    if(!use_jni) {
        unsigned num_arg_bytes = method.get_num_arg_bytes();
#endif

        if(method.is_static()) {
          num_arg_bytes += 4;
        }

        s = alu(s, add_opc, &esp_opnd, &Imm_Opnd(num_arg_bytes));
    }

    if(use_jni) {
        // Save and widen the return value if necessary
        switch(ret_type) {
        case JAVA_TYPE_DOUBLE:
        case JAVA_TYPE_FLOAT:
            break;
        case JAVA_TYPE_BYTE:
        case JAVA_TYPE_BOOLEAN:
            s = widen(s, &ebx_opnd, &eax_opnd, 1, 0);
            break;
        case JAVA_TYPE_CHAR:
            s = widen(s, &ebx_opnd, &eax_opnd, 0, 1);
            break;
        case JAVA_TYPE_SHORT:
            s = widen(s, &ebx_opnd, &eax_opnd, 1, 1);
            break;
        case JAVA_TYPE_LONG:
            s = mov(s, &edi_opnd, &edx_opnd);
        default:
            // int and long
            s = mov(s, &ebx_opnd, &eax_opnd);
            break;
        }

        s = call(s, (char *)orp_disable_gc);

        if(method.is_synchronized()) {
            if(ret_type == JAVA_TYPE_DOUBLE || ret_type == JAVA_TYPE_FLOAT) {
                s = alu(s, sub_opc, &esp_opnd, &Imm_Opnd(8));
                s = fst(s, &M_Base_Opnd(esp_reg, 0), 1, 1);
            }
            if(method.is_static()) {
                s = push(s, &Imm_Opnd((uint32)method.get_class()));
                s = call(s, (char *)orp_get_rt_support_addr(ORP_RT_MONITOR_EXIT_STATIC));
            } else {
                // Unhandle the this pointer.
                s = push(s, &M_Base_Opnd(ebp_reg, 0));
                s = call(s, (char *)orp_get_rt_support_addr(ORP_RT_MONITOR_EXIT));
            }
            if(ret_type == JAVA_TYPE_DOUBLE || ret_type == JAVA_TYPE_FLOAT) {
                s = fld(s, &M_Base_Opnd(esp_reg, 0), 1);
                s = alu(s, add_opc, &esp_opnd, &Imm_Opnd(8));
            }
        }

        // If the return value is a reference, dereference the object handle
        if(ret_type == JAVA_TYPE_CLASS || ret_type == JAVA_TYPE_ARRAY) {
            // ...but don't do it for null handles.
            s = alu(s, or_opc, &ebx_opnd, &ebx_opnd);
            s = branch8(s, cc_eq, &Imm_Opnd(2), 1);

            s = mov(s, &ebx_opnd, &M_Base_Opnd(ebx_reg, 0));
        }

        if(size_of_object_handles_for_args) {
            s = mov(s, &edx_opnd, &esp_opnd);
            s = mov(s, &ecx_opnd, &M_Base_Opnd(esi_reg, 8));
        } else {
            s = mov(s, &edx_opnd, &esp_opnd);
            s = mov(s, &ecx_opnd, &M_Base_Opnd(esp_reg, 8));
        }
        s = call(s, (char *)getaddress__free_local_object_handles_naked() );

        /////////////////////////////////////////////////////////////
        // begin exceptions

        s = call(s, (char *)get_current_thread_exception);
        s = alu(s, or_opc, &eax_opnd, &eax_opnd);
        s = branch8(s, cc_eq, &Imm_Opnd(5), 1);
        s = call(s, (char *)rethrow_current_thread_exception);

        // end exceptions
        /////////////////////////////////////////////////////////////

        if(size_of_object_handles_for_args) {
            s = mov(s, &esp_opnd, &esi_opnd);
        }
        if(ret_type != JAVA_TYPE_DOUBLE && ret_type != JAVA_TYPE_FLOAT) {
            s = mov(s, &eax_opnd, &ebx_opnd);
        }
        if(ret_type == JAVA_TYPE_LONG) {
            s = mov(s, &edx_opnd, &edi_opnd);
        }
    } else {
        // Widen the return value if necessary
        switch(ret_type) {
        case JAVA_TYPE_BYTE:
            s = widen(s, &eax_opnd, &eax_opnd, 1, 0);
            break;
        case JAVA_TYPE_CHAR:
            s = widen(s, &eax_opnd, &eax_opnd, 0, 1);
            break;
        case JAVA_TYPE_SHORT:
            s = widen(s, &eax_opnd, &eax_opnd, 1, 1);
            break;
        default:
            break;
        }
    }

    s = pop(s, &ecx_opnd);
    s = pop(s, &ebx_opnd);
    s = mov(s, &M_Base_Opnd(ebx_reg, 0), &ecx_opnd);

    // Local references list for JNI.
    s = alu(s, add_opc, &esp_opnd, &Imm_Opnd(4));

    s = pop(s, &edi_opnd);
    s = pop(s, &esi_opnd);
    s = pop(s, &ebx_opnd);
    s = pop(s, &ebp_opnd);

    s = ret(s, &Imm_Opnd((4 * num_arg_words) - fastcall_extra_offset));

    // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
#ifdef DUMP_JIT
    dump_stub((Method_Handle)&method, stub, s - stub);
#endif

    assert((int)stub_length >= (s - stub));

    method.set_code_addr(stub);

#ifdef ORP_VTUNE_SUPPORT
    vtune_notify_native_load_finished(&method, (Byte *)stub, (s - stub));
#endif

    return JIT_SUCCESS;
} //prepare_native_method


static JIT_Result prepare_bytecoded_method(Method &method,
                                           JIT_Flags flags
                                           )
{
#ifdef ORP_STATS
    orp_stats_total.num_java_methods++;
#if 0
    printf("*** %s.%s%s\n",
           method.get_class()->name->bytes,
           method.get_name()->bytes,
           method.get_descriptor());
#endif
#endif
    Compilation_Handle comp_handle;
    comp_handle.method = &method;
    comp_handle.env    = ORP_Global_State::loader_env;
    comp_handle.ld_exc = LD_NoException;

    // Iterate through all JITs and find the first one which successfully
    // compiles the method.
    JIT **jit;
    JIT_Result res = JIT_FAILURE;

#ifdef CLI_TESTING
#if 0
    // Michal's temporary debugging code.
    const char *name = method.get_name()->bytes;
    const char *cname = method.get_class()->name->bytes;
    if(!strcmp(name, ".cctor")) {
        if(!strcmp(cname, "System/Char")) {
            printf("Compiling %s.%s%s\n",
                   method.get_class()->name->bytes,
                   method.get_name()->bytes,
                   method.get_descriptor());
        }
    }
#endif
    if(method.is_cli()) {
        for(jit = jit_compilers_cli; *jit; jit++) {
            comp_handle.jit = *jit;
            res = (*jit)->compile_method((*jit)->global_compile_handle, &method, flags);
            if(res == JIT_SUCCESS) {
                break;
            }
        }
    } else {
#endif

    for(jit = jit_compilers; *jit; jit++) {
        comp_handle.jit = *jit;
        res = (*jit)->compile_method((*jit)->global_compile_handle, &method, flags);
        if(res == JIT_SUCCESS) {
            break;
        }
    }
    if(!*jit) {
        printf("Fatal error: can't compile %s.%s%s\n",
               method.get_class()->name->bytes,
               method.get_name()->bytes,
               method.get_descriptor());
        orp_exit(12);
    }
    assert(*jit);     // at least one of the JITs must work
#ifdef CLI_TESTING
    }
#endif

    assert(comp_handle.ld_exc == LD_NoException);

    if(res == JIT_SUCCESS) {
        // Commit the code if the JIT succeeded.
        method.set_code_addr(method.get_code_block_addr());

        method.set_jit(*jit);
        methods.add(method.get_JIT_specific_info(*jit));

        //set breakpoint at source level
        int zz;
        for (zz = 0; zz < breakpoints_num; zz++) 
        {
            if ( (&method == (Method *)(breakpoints[zz].method_handle) && jvmdi_debugger ) )
            {
                int n_entry = method_get_line_number_table_size(&method);

                if (n_entry == 0) {
                    cout << "no debug line number information for " <<
                    method.get_name()->bytes << ", setting bp at first ia32 instruction" << endl;

                    breakpoints[zz].loc  =  (jlocation)method.get_code_addr();
                    //toss assert( (breakpoints[zz].loc & 0xffFFffFF00000000L) == 0);

                    assert( *(uint8 *)(breakpoints[zz].loc) != 0);
                    //;;

					breakpoints[zz].old_value = *(uint8 *)(breakpoints[zz].loc);
                    *( (uint8 *) (breakpoints[zz].loc) ) = 0x6f;
                }


                JVMDI_line_number_entry line_table[1];
				int lowest_line_num = 0;
				int lowest_line_tab_idx = 0;
				uint32 lowest_line_code_off = 0;

                for (int xx = 0; xx < n_entry; xx++) {

                    memset(line_table, 0, sizeof(JVMDI_line_number_entry));

                    method_get_line_number( &method,
                                            xx,
                                            (unsigned *)&(line_table[0].start_location),
                                            (unsigned *)&(line_table[0].line_number) );

                    
					if (breakpoints[zz].line_no == BREAKPOINT_AT_BEGIN){      // breakpoint at begin
						if (xx ==0){
							uint32 code_offset =  o1_jit->get_break_point_offset(
												   o1_jit->global_compile_handle,
												   &method,
												   flags,
												   (unsigned)line_table[0].start_location);
							lowest_line_num = line_table[0].line_number;
							lowest_line_tab_idx = xx;
							lowest_line_code_off = code_offset;
						}else 
							if (line_table[0].line_number<lowest_line_num){
								uint32 code_offset =  o1_jit->get_break_point_offset(
													   o1_jit->global_compile_handle,
													   &method,
													   flags,
													   (unsigned)line_table[0].start_location);
								lowest_line_num = line_table[0].line_number;
								lowest_line_tab_idx = xx;
								lowest_line_code_off = code_offset;
							}
					} else	{								// breakpoint is not at begin
						if (line_table[0].line_number == (jlocation)breakpoints[zz].line_no) {
							uint32 code_offset =  o1_jit->get_break_point_offset(
												   o1_jit->global_compile_handle,
												   &method,
												   flags,
												   (unsigned)line_table[0].start_location);
							breakpoints[zz].loc = (jlocation)code_offset;
							breakpoints[zz].loc  +=  (jlocation)method.get_code_addr();
							//toss assert( (breakpoints[zz].loc & 0xffFFffFF00000000L) == 0);

							assert( *(uint8 *)(breakpoints[zz].loc) != 0);
							//;;

							if (*(uint8 *)(breakpoints[zz].loc)!=0x6f) {
								breakpoints[zz].old_value = *(uint8 *)(breakpoints[zz].loc);
								*( (uint8 *) (breakpoints[zz].loc) ) = 0x6f;
							}
							break;
						} //if	current line is breakpoint
					}	// breakpoint not at begin
                }	// for
				if (breakpoints[zz].line_no == BREAKPOINT_AT_BEGIN) {	// means begin
					breakpoints[zz].loc = (jlocation)lowest_line_code_off;
                    breakpoints[zz].loc  +=  (jlocation)method.get_code_addr();
                    //toss assert( (breakpoints[zz].loc & 0xffFFffFF00000000L) == 0);

                    assert( *(uint8 *)(breakpoints[zz].loc) != 0);
					//;;

					if (*(uint8 *)(breakpoints[zz].loc)!=0x6f) {
						breakpoints[zz].old_value = *(uint8 *)(breakpoints[zz].loc);
						*( (uint8 *) (breakpoints[zz].loc) ) = 0x6f;
					}	
				}
			
			}
        }

        for (zz =0;zz<bytecode_breakpoints_num;zz++){
            if ( (&method == (Method *)(bytecode_breakpoints[zz].method_handle) && jvmdi_debugger ) ) {
                uint32 code_offset =  o1_jit->get_break_point_offset(
											   o1_jit->global_compile_handle,
											   &method,
											   flags,
											   (unsigned)bytecode_breakpoints[zz].line_no);
                                               // in bytecode_breakpoint,line_no is bytecode number.
                if (code_offset == BP_LOC_UNKNOWN)
                    break;
                bytecode_breakpoints[zz].loc = (jlocation)code_offset;
				bytecode_breakpoints[zz].loc  +=  (jlocation)method.get_code_addr();
				//toss assert( (bytecode_breakpoints[zz].loc & 0xffFFffFF00000000L) == 0);

				assert( *(uint8 *)(bytecode_breakpoints[zz].loc) != 0);
				
                if (*(uint8 *)(bytecode_breakpoints[zz].loc)!=0x6f) {
					bytecode_breakpoints[zz].old_value = *(uint8 *)(bytecode_breakpoints[zz].loc);
					*( (uint8 *) (bytecode_breakpoints[zz].loc) ) = 0x6f;
				}
			}       
        }

        if ( (&method == (Method *)(bytecode_step_into_break_point.method_handle) && jvmdi_debugger ) ) {
            uint32 code_offset =  o1_jit->get_break_point_offset(
										   o1_jit->global_compile_handle,
										   &method,
										   flags,
										   (unsigned)bytecode_step_into_break_point.line_no);
                                           // in bytecode_step_into_break_point,line_no is bytecode number.
            bytecode_step_into_break_point.loc = (jlocation)code_offset;
			bytecode_step_into_break_point.loc  +=  (jlocation)method.get_code_addr();
			//toss assert( (bytecode_step_into_break_point.loc & 0xffFFffFF00000000L) == 0);

			assert( *(uint8 *)(bytecode_step_into_break_point.loc) != 0);
			
            if (*(uint8 *)(bytecode_step_into_break_point.loc)!=0x6f) {
				bytecode_step_into_break_point.old_value = *(uint8 *)(bytecode_step_into_break_point.loc);
				*( (uint8 *) (bytecode_step_into_break_point.loc) ) = 0x6f;
			}
		}
    }

    return res;
} //prepare_bytecoded_method



JIT_Result jit_recompile_method(Method *method,
                                JIT *jit
                                )
{
    assert(method->get_class()->state == ST_Initialized);
    assert(method->get_state() == Method::ST_Compiled);
    assert(!(method->get_JIT_specific_info_no_create(jit)));

    p_jit_a_method_lock->_lock();

    Compilation_Handle comp_handle;
    comp_handle.method = method;
    comp_handle.env    = ORP_Global_State::loader_env;
    comp_handle.ld_exc = LD_NoException;
    comp_handle.jit    = jit;
    JIT_Flags flags;
 
    flags.insert_write_barriers = (gc_requires_barriers() != GC_NO_BARRIER);
 
    JIT_Result res = jit->compile_method(&comp_handle, method, flags);
    assert(comp_handle.ld_exc == LD_NoException);

    if(res == JIT_SUCCESS) {
        // Commit the code if the JIT succeeded.
        method->set_code_addr(method->get_code_block_addr());

        method->set_jit(jit);
        methods.add(method->get_JIT_specific_info(jit));
        method->apply_vtable_patches();
    }

    p_jit_a_method_lock->_unlock();

    return res;
} //jit_recompile_method



static JIT_Result jit_compile_method_gc_unsafe1(Method &method,
                                                JIT_Flags flags
                                                )
{
    method.set_state(Method::ST_BeingCompiled);

    char *old_stub = (char *)method.get_code_addr();

    JIT_Result res;
    if(method.is_native() || use_native_implementation(&method)) {

#if 0
    printf("*** %s.%s%s\n",
           method.get_class()->name->bytes,
           method.get_name()->bytes,
           method.get_descriptor());
#endif

        res = prepare_native_method(method, flags);
    } else {

#ifdef CLI_TESTING

        if(is_cli_native( (Method_Handle *)&method))
        {
            
            JIT_Result prepare_cli_native_method(Method &method,
                                             JIT_Flags flags      );

            res = prepare_cli_native_method(method, flags);
        }
        else  {
#endif
      
        res = prepare_bytecoded_method(method, flags);
        }
#ifdef CLI_TESTING
    }
#endif

    if(res == JIT_SUCCESS) {
        method.set_state(Method::ST_Compiled);

        // Native stubs are not added into the method table.
        //Meth_Addr_Table::add_method(&method);

        // Re-write the old stub.  It shouldn't be necessary, because we apply
        // vtable patches.
        //jump(old_stub, (char *)method.get_code_addr());

        method.apply_vtable_patches();
    }

    return res;
} //jit_compile_method_gc_unsafe1



static JIT_Result jit_compile_method_gc_unsafe(Method &method,
                                               JIT_Flags flags
                                               )
{
    // Class must be initialized on first active use.
    Class *clss = method.get_class();
    class_initialize(clss);

    orp_enable_gc();
    p_jit_a_method_lock->_lock();

    orp_disable_gc();

    JIT_Result res;
    switch(method.get_state()) {
    case Method::ST_NotCompiled:
        res = jit_compile_method_gc_unsafe1(method, flags);
        break;
    case Method::ST_Compiled:
        res = JIT_SUCCESS;
        break;
    default:
        res = JIT_FAILURE;

#if 1
    printf("*** %s.%s%s\n",
           method.get_class()->name->bytes,
           method.get_name()->bytes,
           method.get_descriptor());
#endif
        assert(0);
    }

    p_jit_a_method_lock->_unlock();
    return res;
} //jit_compile_method_gc_unsafe



static JIT_Result jit_compile_method_gc_safe(Method *method,
                                             JIT_Flags flags
                                             )
{
    // Use the ljf to find the location of arguments on the stack.
    // Initialize the args pointer with the address of the return eip and increment
    // it by the number of words deduced form the descriptor and the static flag.
    J2N_Saved_State *ljf = get_orp_last_java_frame();
    uint32 *args = ((uint32 *)&(ljf->eip));
    unsigned num_ref_args = 0;
    if(!method->is_static()) {
        args++;
        num_ref_args++;
    }
    Arg_List_Iterator arg_iter = method->get_argument_list();
    Java_Type typ;
    while((typ = curr_arg(arg_iter)) != JAVA_TYPE_END) {
        switch(typ) {
        case JAVA_TYPE_LONG:
        case JAVA_TYPE_DOUBLE:
            args += 2;
            break;
        case JAVA_TYPE_CLASS:
        case JAVA_TYPE_ARRAY:
            args++;
            num_ref_args++;
            break;
        default:
            args++;
            break;
        }
        arg_iter = advance_arg_iterator(arg_iter);
    }
    int num_arg_words = args - (uint32 *)&(ljf->eip);

    // Allocate memory conservatively even for arguments which are not references.
    GC_Frame *gc_frames = (GC_Frame *)gc_malloc_fixed_data_C(num_arg_words * sizeof(GC_Frame));

    uint32 *args_start = args;
    //uint32 *last_arg = ((uint32 *)&(ljf->eip)) + 1;
    GC_Frame *gc_frames_start = gc_frames;

    orp_push_gc_frames1(method, num_arg_words, args, gc_frames);

    JIT_Result res = jit_compile_method_gc_unsafe(*method, flags);

    orp_pop_gc_frames1(method, args, gc_frames + num_ref_args);

    gc_free_fixed_data_C(gc_frames);

    return res;
} //jit_compile_method_gc_safe



//
// A method is compiled just-in-time.  This means one of two things:
// 1. The method is native and we have to locate the C function which implements
//    the method and generate a stub for it.
// 2. The method is "bytecoded," i.e., encoded in the class file with bytecodes.
//    In that case we have to invoke a JIT compiler to generate native code
//.
static void *jit_a_method(Method *method)
{
    JIT_Flags flags;

#if 0
 	const char *m = method->get_name()->bytes;
	const char *c = method->get_class()->name->bytes;
	printf("********* %s %s\n", c, m);
#endif
 
    flags.insert_write_barriers = (gc_requires_barriers() != GC_NO_BARRIER);
 
    JIT_Result res = jit_compile_method_gc_safe(method, flags);

    if(res == JIT_SUCCESS) {
        void *entry_point = method->get_code_addr();
#ifdef CLI_TESTING
    if(method->get_class()->is_value_type) {
		
		// Unboxers are used only for non-constructor and non-static methods.
		if (strcmp(method->get_name()->bytes, ".cctor") && !method->is_static()) { 
			entry_point = method->get_unboxer_addr();
		}
    }
#endif
        return entry_point;
    } else {
        if(method->is_native()) {
            //throw_java_exception("java/lang/UnsatisfiedLinkError", pname);
            throw_java_exception("java/lang/UnsatisfiedLinkError");
        } else {
            throw_java_exception("java/lang/InternalError");
        }
    }

    return NULL;
} //jit_a_method




//
// On entry:
//  - the stack contains the arguments for the method
//  - eax contains the method to be compiled
//
void *getaddress__compile_method_trampoline()
{
    static void *addr = 0;
    if (addr) {
        return addr;
    }

    const int stub_size = 1024;
    char *stub = (char *)gc_malloc_fixed_code_for_class_loading(stub_size);
#ifdef _DEBUG
    memset(stub, 0xcc /*int 3*/, stub_size);
#endif
    char *ss = stub;

    // Save the state for unwinding in case GC or
    // an exception must be handled.
    // The layout corresponds to the struct J2N_Saved_State.
    // Note that the words are pushed in the reverse order
    // compared to how the fields of J2N_Saved_State are
    // defined.
    ss = push(ss, &ebp_opnd);
    ss = push(ss, &ebx_opnd);
    ss = push(ss, &esi_opnd);
    ss = push(ss, &edi_opnd);
    ss = push(ss, &Imm_Opnd(0));

    // Save the method handle in a callee-saved register
    ss = mov(ss, &esi_opnd, &eax_opnd );
    
    // Add an entry for the current frame to the the LJF list
    ss = call(ss, (char *)get_addr_of_orp_last_java_frame);
    ss = push(ss, &eax_opnd);
    ss = push(ss, &M_Base_Opnd(eax_reg, +0));
    ss = mov(ss, &M_Base_Opnd(eax_reg, +0), &esp_opnd);

    // Compile the method.
    ss = push(ss, &esi_opnd);
    ss = call(ss, (char *)jit_a_method);
    ss = alu(ss, add_opc, &esp_opnd, &Imm_Opnd(4));
    // The entry point is returned in eax.

    // Remove the entry for the current frame from the the LJF list
    ss = pop(ss, &ecx_opnd);
    ss = pop(ss, &edx_opnd);
    ss = mov(ss, &M_Base_Opnd(edx_reg, +0), &ecx_opnd);

    // Restore the rest of the J2N state.
    ss = alu(ss, add_opc, &esp_opnd, &Imm_Opnd(4));
    ss = pop(ss, &edi_opnd);
    ss = pop(ss, &esi_opnd);
    ss = pop(ss, &ebx_opnd);
    ss = pop(ss, &ebp_opnd);

    // Continue execution in the newly compiled method.
    ss = jump(ss, &eax_opnd);

    addr = stub;
#ifdef ORP_VTUNE_SUPPORT
    //M: 
    vtune_notify_stub_load_finished("getaddress__compile_method_trampoline",(Byte*) stub,ss-stub);
#endif
    return addr;
} //getaddress__compile_method_trampoline



char *create_compile_me_stub(Method *method)
{
    char *stub = (char *)gc_malloc_fixed_code_for_class_loading(10);
    char *p = stub;
    p = mov(p, &eax_opnd, &Imm_Opnd((uint32)method));
    p = jump(p, (char *)getaddress__compile_method_trampoline() );
#ifdef ORP_VTUNE_SUPPORT
    //M: 
//  vtune_notify_stub_load_finished("create_compile_me_stub",(Byte*) stub,p-stub);
#endif
    return stub;
} //create_compile_me_stub



#ifdef CLI_TESTING

char *create_unboxer(Method *method)
{
    Class *clss = method->get_class();
    assert(clss->is_value_type);
    unsigned nab = method->get_num_arg_bytes();
    unsigned unboxed_data_offset = class_get_unboxed_data_offset(clss);
    char *unboxer_addr = (char *)gc_malloc_fixed_code_for_class_loading(10);
    char *s = unboxer_addr;
    s = alu(s, add_opc, &M_Base_Opnd(esp_reg, nab), &Imm_Opnd(unboxed_data_offset));
    s = jump(s, (char *)method->get_code_addr());
    return unboxer_addr;
} //create_unboxer


#endif
